#ifndef gavl
/*
    generic homogeneous AVL tree declaration for any key type.
    These are designed to realize the following goals:
        1. The messy stuff (tree balancing) should be coded up once and 
           the code should appear ONCE in the final executable -- being
           shared by trees of all types.
        2. The final interface should be strongly type-checked.
	3. The amount of redundant code generated by the strong type
	   checking should be minimal (goal 1 is just a special and the most
	   important case of this).
	4. Virtual functions should not be used (as they add an extra
	   pointer to the class). 

    The following example declares an avl tree with int keys compared
    by subtraction, a class foo which is entered in such a tree and
    a class bar which contains a pointer to the root of the tree.

    // avlint.h -- Avl tree class with int key
    #ifndef AVLINTH
    #define AVLINTH
    #include "gavl.h"
    gavlkey_define(int);
    #endif AVLINTH
    
    // avlint.c -- define the search procedure for trees keyed by integer.
    #include "avlint.h"
    #define intcompare(a, b) (a-b)
    gavlkey_search(int, intcompare);

    // foobar.h -- define class foo which has an integer value upon which
    // foos are sorted.  Also define class bar which binds a tree of foos
    // sorted on that integer value.
    #include "avlint.h"
    class foo;
    class bar;

    // NOTE: when the language is fixed so that nested class declarations
    // are treated as described in the update manual, then the following
    // can be simplified considerably, as the material in gavl_define and
    // gavl_inline can all be merged into gavl_inst -- maybe.

    // The gavl macros take 3 parameters -- avl member name, name of class
    // containing the avl member, and the key type.

    gavl_define(int_value,foo,int); 
    class foo {
        some_class grbx;  // misc. stuff
    public:
	gavl_inst(int_value,foo,int); 
	// note params -- creates member named int_value here.

	another_class x;  // misc. junk
    };
    gavl_inline(int_value,foo,int);

    class bar {
        junk stuff;
    public:
	gavlroot(int_value,foo,int) foos; // bar heads a tree of foos keyed by int.
	stuff junk;
    }

    // misc_code.c
    // The key values can be read and set directly from the tree class member
    // itself.  The operator=() function for this will insure that the
    // class being assigned a key value is NOT currently in a tree.

    bar *bar_object = new bar;
    foo *foo_object = new foo;
    foo *found;
    // ...
    foo_object->int_value = 5;   // key has now been set been set
    found = bar->foos.insert(foo_object);
    if (found != foo_object) {
        // already there
	delete foo_object;
	foo_object = found;
    }

    foo_object->int_value = 7;  // Illegal -- already in a tree -> fault

    // The following fragments are also useful and legal and do the obvious.
    // They are possible because the macros define an implicit type 
    // conversion from the class gavlkey(keytype) to the type keytype (int).

    int number = foo_object->int_value;

    if ((int)foo_object1->int_value < (int)foo_object2->int_value) {
        // ...
    }

    // The following code finds a foo of a particular int_value value and then
    // removes (picks) it from the tree and deletes it.

    foo *found = bar->foos.find(8);
    if (foo) {
        bar->foos.pick(foo);
	delete foo;
    }

    // The following code does the same more efficiently:

    foo *found = bar->foos.findpick(8);
    if (found) delete found;
*/

#include <generic.h>
#include "avlbase.h"
#include "giterator.h"


extern char *already_in_tree; // Fault message for gavlkey(keytype)::operator=()


#define gavl(member,classtype,keytype)\
    name4(avl_,keytype,classtype,member)
#define gavlkey(keytype) name2(avl_,keytype)
#define gavlkeyroot(keytype) name2(avlkr_,keytype)
#define gavlroot(member,classtype,keytype)\
    name4(avlroot_,keytype,classtype,member)
#define gavlinorder(member,classtype,keytype)\
    name4(avlinorder_,keytype,classtype,member)
#define gavlpreorder(member,classtype,keytype)\
    name4(avlpreorder_,keytype,classtype,member)

/*
    NOTE: I would like to make the key parameters to search, find and findpick
    a constant, but this compiler considers that const char* != const charP
    (for charP "typedef"ed to be a char*).  Thus the only way to make these
    constant is to use a different set of macros for pointer key values
    that simply put the * in all of the right places.  I am unwilling to
    resort to 2 different sets of macros for the same generic structure.
*/
#define gavlkey_define(keytype) \
class gavlkey(keytype) : public avlbase {\
    friend class gavlkeyroot(keytype);\
    keytype keyval;\
    gavlkey(keytype) *left()\
        { return (gavlkey(keytype) *)avlbase::left(); }\
    gavlkey(keytype) *right()\
        { return (gavlkey(keytype) *)avlbase::right(); }\
public:\
    keytype key()\
        { return keyval; }\
    keytype set_key(keytype i)\
    {\
        if (in_tree()) fault(already_in_tree);\
	return keyval = i;\
    }\
    gavlkey(keytype)() {}\
    gavlkey(keytype)(keytype key) { set_key(key); }\
};\
class gavlkeyroot(keytype) : public avlroot_base {\
public:\
    gavlkey(keytype) *search(/*const*/ keytype key, avlpath *path);\
};


#define gavl_define_classes(member,classtype,keytype,specifier) \
class classtype;\
\
\
class gavlroot(member,classtype,keytype) : public gavlkeyroot(keytype) {\
    friend class gavlinorder(member,classtype,keytype);\
    friend class gavlpreorder(member,classtype,keytype);\
    specifier classtype *to_classtype(gavlkey(keytype) *p);\
public:\
    gavlroot(member,classtype,keytype)() {}\
    specifier classtype *insert(classtype *object);\
    specifier classtype *find(/*const*/ keytype key);\
    specifier void pick(classtype *object);\
    specifier classtype *findpick(/*const*/ keytype key);\
};\
\
\
class gavl(member,classtype,keytype) : public gavlkey(keytype) {\
public:\
    operator keytype()\
	{ return key(); }\
    keytype operator=(keytype i)\
	{ return set_key(i); }\
};\
\
\
class gavlinorder(member,classtype,keytype) : public avl_inorder {\
    classtype *to_classtype(avlbase *p);\
public:\
    /* GITERATOR */\
    specifier gavlinorder(member,classtype,keytype)(const gavlroot(member,classtype,keytype)& root);\
    specifier classtype *operator()();\
\
    /* backward compatibility */\
    specifier classtype *next();\
    specifier classtype *first();\
};\
\
\
class gavlpreorder(member,classtype,keytype) : public avl_preorder {\
    specifier classtype *to_classtype(avlbase *p);\
public:\
    specifier gavlpreorder(member,classtype,keytype)(const gavlroot(member,classtype,keytype)& root);\
    specifier classtype *operator()();\
\
    specifier classtype *next();\
    specifier classtype *first();\
};\
\
\
typedef gavlinorder(member,classtype,keytype) GITERATOR(member,classtype)

#define gavl_define(member,classtype,keytype) \
    gavl_define_classes(member,classtype,keytype,)

#define gavl_inst(member,classtype,keytype)\
friend gavlroot(member,classtype,keytype);\
gavl(member,classtype,keytype) member


#define gavl_functions(member,classtype,keytype,specifier)\
specifier classtype *\
gavlroot(member,classtype,keytype)::to_classtype(gavlkey(keytype) *p)\
{\
    return p ?\
    (classtype *)((char *)p - (int)&(((classtype *)0)->member)) : 0;\
}\
specifier classtype *\
gavlroot(member,classtype,keytype)::insert(classtype *object)\
{\
    avlpath path; gavlkey(keytype) *found;\
    if (!(found = search(object->member.key(), &path)))\
	avlroot_base::insert(&object->member, path);\
    return found ? to_classtype(found) : object;\
}\
specifier classtype *\
gavlroot(member,classtype,keytype)::find(/*const*/ keytype key)\
{\
    avlpath path;\
    gavlkey(keytype) *found = search(key, &path);\
    return to_classtype(found);\
}\
specifier void \
gavlroot(member,classtype,keytype)::pick(classtype *object)\
{\
    avlpath path;\
    if (!search(object->member.key(), &path)) fault("Not found");\
    avlroot_base::pick(&object->member, path);\
}\
specifier classtype *\
gavlroot(member,classtype,keytype)::findpick(/*const*/ keytype key)\
{\
    avlpath path;\
    gavlkey(keytype) *found = search(key, &path);\
    if (found) avlroot_base::pick(found, path);\
    return to_classtype(found);\
}\
\
specifier classtype* gavlinorder(member,classtype,keytype)::to_classtype(avlbase *p)\
    {\
        return ((gavlroot(member,classtype,keytype) *)0)->to_classtype((gavlkey(keytype) *)p);\
    }\
specifier gavlinorder(member,classtype,keytype)::gavlinorder(member,classtype,keytype)\
    (const gavlroot(member,classtype,keytype)& root) : (root) {}\
specifier classtype *gavlinorder(member,classtype,keytype)::next()\
    { return to_classtype(avl_inorder::next()); }\
specifier classtype *gavlinorder(member,classtype,keytype)::operator()()\
    { return next(); }\
specifier classtype *gavlinorder(member,classtype,keytype)::first()\
    { return to_classtype(avl_inorder::first()); }\
\
\
specifier classtype *gavlpreorder(member,classtype,keytype)::to_classtype(avlbase *p)\
    {\
        return ((gavlroot(member,classtype,keytype) *)0)->to_classtype((gavlkey(keytype) *)p);\
    }\
specifier gavlpreorder(member,classtype,keytype)::\
    gavlpreorder(member,classtype,keytype) (const gavlroot(member,classtype,keytype)& root)\
    : (root) {}\
specifier classtype *gavlpreorder(member,classtype,keytype)::next()\
    { return to_classtype(avl_preorder::next()); }\
specifier classtype *gavlpreorder(member,classtype,keytype)::operator()()\
    { return next(); }\
specifier classtype *gavlpreorder(member,classtype,keytype)::first()\
    { return to_classtype(avl_preorder::first()); }


#define gavlkey_search(keytype, compare) \
gavlkey(keytype) *\
gavlkeyroot(keytype)::search(/*const*/ keytype key, avlpath *path)\
{\
    gavlkey(keytype) *p = (gavlkey(keytype)*)tree_root();\
    while (p) {\
        int comparison;\
        path->push(comparison = compare(key, p->key()));\
	if (!comparison) break;\
	if (comparison < 0) p = p->left();\
	else p = p->right();\
    }\
    if (!p) path->push(0);\
    return p;\
}


#define gavl_inline(member,type,keytype)\
    gavl_functions(member,type,keytype,inline)
#endif
