#ifndef RELOPS_H
#define RELOPS_H 1
/*
  Macro for declaring/defining relational operators on a purely staticly 
  bound class. The class is assumed to have a member function of the form 
	int cmp(const THIS&) 
  or 
	int cmp(THIS)
  already defined -- these make inline references to it.
*/

#define RELOPS(THIS) \
    operator==(const THIS& o) { return cmp(o) == 0; }\
    operator!=(const THIS& o) { return cmp(o) != 0; }\
    operator<(const THIS& o) { return cmp(o) < 0; }\
    operator<=(const THIS& o) { return cmp(o) <= 0; }\
    operator>(const THIS& o) { return cmp(o) > 0; }\
    operator>=(const THIS& o) { return cmp(o) >= 0; }

/*
  Does the same for a class with a cmp operation of the form
  	int cmp(const OTHER&) 
  or 
	int cmp(OTHER)
  where OTHER differs from THIS (the type of the containing class).
*/

#define SYMMETRICAL_RELOPS(THIS, OTHER) \
    RELOPS(OTHER)\
    friend operator==(const OTHER& o, const THIS& t) { return t.cmp(o) == 0; }\
    friend operator!=(const OTHER& o, const THIS& t) { return t.cmp(o) != 0; }\
    friend operator<(const OTHER& o, const THIS& t) { return t.cmp(o) < 0; }\
    friend operator<=(const OTHER& o, const THIS& t) { return t.cmp(o) <= 0; }\
    friend operator>(const OTHER& o, const THIS& t) { return t.cmp(o) > 0; }\
    friend operator>=(const OTHER& o, const THIS& t) { return t.cmp(o) >= 0; }


/*
  The following forms do the same, but separate the declaration from the definition.
  This is usefull where the programming style puts inline functions separately from
  definitions -- particularly where the cmp() functions are inline and so treated.
*/

#define RELOPS_DECLARE(THIS)\
    operator==(const THIS& o);\
    operator!=(const THIS& o);\
    operator<(const THIS& o);\
    operator<=(const THIS& o);\
    operator>(const THIS& o);\
    operator>=(const THIS& o)

#define RELOPS_INLINE(THIS) \
    inline THIS::operator==(const THIS& o) { return cmp(o) == 0; }\
    inline THIS::operator!=(const THIS& o) { return cmp(o) != 0; }\
    inline THIS::operator<(const THIS& o) { return cmp(o) < 0; }\
    inline THIS::operator<=(const THIS& o) { return cmp(o) <= 0; }\
    inline THIS::operator>(const THIS& o) { return cmp(o) > 0; }\
    inline THIS::operator>=(const THIS& o) { return cmp(o) >= 0; }

#define SYMMETRICAL_RELOPS_DECLARE(THIS, OTHER) \
    RELOPS_DECLARE(OTHER)\
    friend operator==(const OTHER& o, const THIS& t);\
    friend operator!=(const OTHER& o, const THIS& t);\
    friend operator<(const OTHER& o, const THIS& t);\
    friend operator<=(const OTHER& o, const THIS& t);\
    friend operator>(const OTHER& o, const THIS& t);\
    friend operator>=(const OTHER& o, const THIS& t)

#define SYMMETRICAL_RELOPS_INLINE(THIS, OTHER) \
    inline THIS::operator==(const OTHER& o) { return cmp(o) == 0; }\
    inline THIS::operator!=(const OTHER& o) { return cmp(o) != 0; }\
    inline THIS::operator<(const OTHER& o) { return cmp(o) < 0; }\
    inline THIS::operator<=(const OTHER& o) { return cmp(o) <= 0; }\
    inline THIS::operator>(const OTHER& o) { return cmp(o) > 0; }\
    inline THIS::operator>=(const OTHER& o) { return cmp(o) >= 0; }\
    inline operator==(const OTHER& o, const THIS& t) { return t.cmp(o) == 0; }\
    inline operator!=(const OTHER& o, const THIS& t) { return t.cmp(o) != 0; }\
    inline operator<(const OTHER& o, const THIS& t) { return t.cmp(o) < 0; }\
    inline operator<=(const OTHER& o, const THIS& t) { return t.cmp(o) <= 0; }\
    inline operator>(const OTHER& o, const THIS& t) { return t.cmp(o) > 0; }\
    inline operator>=(const OTHER& o, const THIS& t) { return t.cmp(o) >= 0; }

#endif
