function avl_insert(obj: avl_object_ptr;  var root: avl_ptr;
                    kind: avl_type): avl_ptr;
  { insert obj into the tree (unless already there).  Return a pointer
    to its entry in the tree }
  label
    90; { return }
    
  { a direction table is used to keep track of which way we went (left
    or right) at each level of the tree starting either with the root
    or the last node having a non-zero balance factor.
    The table is long enough to handle a tree up to 32 levels deep, which
    would contain as many elements than there are addresses in a 32 bit
    address space.
    The reason this table exists is to avoid having to redo the sometimes
    expensive avl_compare operations when rebalancing the tree. }

  const
    LAST_DIRECTION_TABLE_INDEX = 32;
  type
    direction = (GO_LEFT, GO_RIGHT);
    direction_range = 1..LAST_DIRECTION_TABLE_INDEX;
    direction_index_range = 0..LAST_DIRECTION_TABLE_INDEX;
    direction_table = array[direction_range] of direction;
  var
    last_non_zero: avl_ptr;       { lowest node above insertion with BF <> 0 }
    parent_of_LNZ: avl_ptr;       { parent of past_non_zero }
    current: avl_ptr;             { current node }
    parent: avl_ptr;              { parent of current }
    directions: direction_table;  { road map (see above) }
    direction_index: 
      direction_index_range;      { index into directions }
    newnode: avl_ptr;             { inserted node }
    current_child: avl_ptr;       { interesting child of current }
begin
  if debug_28 then
    begin
      writeln(Outfile, '-- avl_insert (', ord(kind):1, ') --');
      write(Outfile, '  insert: ');
      dump_avl_object(Outfile, obj, kind);
    end;
      
      
  last_non_zero := NIL;  parent_of_LNZ := NIL;
  current := root;  parent := NIL;
  direction_index := 0;

  { Locate insertion point }

  while current <> NIL do
    begin
      direction_index := direction_index + 1;
      if current^.balance_factor <> EVEN_BF then
        begin
	  last_non_zero := current;  parent_of_LNZ := parent;
          direction_index := 1;
	end;
      parent := current;
      case avl_compare(obj, current^.object, kind) of
        LT: 
	  begin
	    current := current^.left_child;
	    directions[direction_index] := GO_LEFT;
	  end;
	GT: 
	  begin
	    current := current^.right_child;
	    directions[direction_index] := GO_RIGHT;
	  end;
	EQ:
	  begin
	    avl_insert := current;
	    if debug_28 then
	      writeln(Outfile, '-- exit avl_insert (found) --');
	    goto 90 { return } ;
	  end;
      end;
    end;

  { insert new element }

  new_avl(newnode, kind);  newnode^.object := obj;
  avl_insert := newnode;
  
  if root = NIL then
    begin
      root := newnode;
      if debug_28 then 
        begin
	  writeln(Outfile, 'First node');
	  dump_avl_tree(Outfile, root, kind);
	  writeln(Outfile, '-- exit avl_insert --');
	end;
      goto 90 { return } ;
    end;
  if directions[direction_index] = GO_LEFT then
    parent^.left_child := newnode
  else parent^.right_child := newnode;
  
  { adjust balance factors from last non-zero BF node - all nodes
    between this node and newnode formerlay had an even balance factor. }
  
  if last_non_zero = NIL then
      { all nodes down to newnode were evenly balanced }
    if directions[1] = GO_LEFT then
      begin
        root^.balance_factor := LEFT_HEAVY_BF;
        current := root^.left_child;
      end
    else
      begin
        root^.balance_factor := RIGHT_HEAVY_BF;
	current := root^.right_child;
      end
  else 
    if directions[1] = GO_LEFT then 
      current := last_non_zero^.left_child
    else current := last_non_zero^.right_child;

  direction_index := 1;
  while current <> newnode do
    begin
      direction_index := direction_index + 1;
      
      if directions[direction_index] = GO_LEFT then
        begin
          current^.balance_factor := LEFT_HEAVY_BF;
          current := current^.left_child;
        end
      else
        begin
          current^.balance_factor := RIGHT_HEAVY_BF;
          current := current^.right_child;
        end;
    end;
    
  if last_non_zero = NIL then 
    begin
      if debug_28 then
        begin
	  writeln(Outfile, 'All path nodes were evenly balanced');
	  dump_avl_tree(Outfile, root, kind);
	  writeln(Outfile, '-- exit avl_insert --');
	end;
      goto 90 { return } ;  { balancing done }
    end;
  
  if directions[1] = GO_LEFT then
    begin
      if last_non_zero^.balance_factor = RIGHT_HEAVY_BF then
        begin
	  last_non_zero^.balance_factor := EVEN_BF;
	  if debug_28 then
	    begin
	      writeln(Outfile, 'No rotation (Left)');
	      dump_avl_tree(Outfile, root, kind);
	      writeln(Outfile, '-- exit avl_insert --');
	    end;
	  goto 90 { return } ;
	end;
      current := last_non_zero^.left_child;
      if current^.balance_factor = LEFT_HEAVY_BF then
        begin
	  { LL rotation }

	  if debug_28 then writeln(Outfile, 'LL rotation');
          last_non_zero^.left_child := current^.right_child;
	  current^.right_child := last_non_zero;
	  last_non_zero^.balance_factor := EVEN_BF;
	  current^.balance_factor := EVEN_BF;
	end
      else
        begin
	  { LR rotation }
          
	  if debug_28 then writeln(Outfile, 'LR rotation');
          current_child := current^.right_child;
	  current^.right_child := current_child^.left_child;
	  last_non_zero^.left_child := current_child^.right_child;
	  current_child^.left_child := current;
	  current_child^.right_child := last_non_zero;
	  case current_child^.balance_factor of
	    LEFT_HEAVY_BF:
	      begin
	        last_non_zero^.balance_factor := RIGHT_HEAVY_BF;
	        current^.balance_factor := EVEN_BF;
	      end;
	    RIGHT_HEAVY_BF:
	      begin
		last_non_zero^.balance_factor := EVEN_BF;
	        current^.balance_factor := LEFT_HEAVY_BF;
	      end;
	    EVEN_BF:
	      begin
	        last_non_zero^.balance_factor := EVEN_BF;
	        current^.balance_factor := EVEN_BF;
	      end;
	  end { case } ;
	  current_child^.balance_factor := EVEN_BF;
	  current := current_child;  { current is root of balanced subtree }
	end;
    end { left side checking and rotations }
  else
    begin { directions[1] = GO_RIGHT }
      if last_non_zero^.balance_factor = LEFT_HEAVY_BF then
        begin
	  last_non_zero^.balance_factor := EVEN_BF;
	  if debug_28 then
	    begin
	      writeln(Outfile, 'No rotation (Right)');
	      dump_avl_tree(Outfile, root, kind);
	      writeln(Outfile, '-- exit avl_insert --');
	    end;
	  goto 90 { return } ;
	end;
      current := last_non_zero^.right_child;
      if current^.balance_factor = RIGHT_HEAVY_BF then
        begin
	  { RR rotation }

	  if debug_28 then writeln(Outfile, 'RR rotation');
          last_non_zero^.right_child := current^.left_child;
	  current^.left_child := last_non_zero;
	  last_non_zero^.balance_factor := EVEN_BF;
	  current^.balance_factor := EVEN_BF;
	end
      else
        begin
	  { RL rotation }
          
	  if debug_28 then writeln(Outfile, 'RL rotation');
          current_child := current^.left_child;
	  current^.left_child := current_child^.right_child;
	  last_non_zero^.right_child := current_child^.left_child;
	  current_child^.right_child := current;
	  current_child^.left_child := last_non_zero;
	  case current_child^.balance_factor of
	    RIGHT_HEAVY_BF:
	      begin
	        last_non_zero^.balance_factor := LEFT_HEAVY_BF;
	        current^.balance_factor := EVEN_BF;
	      end;
	    LEFT_HEAVY_BF:
	      begin
		last_non_zero^.balance_factor := EVEN_BF;
	        current^.balance_factor := RIGHT_HEAVY_BF;
	      end;
	    EVEN_BF:
	      begin
	        last_non_zero^.balance_factor := EVEN_BF;
	        current^.balance_factor := EVEN_BF;
	      end;
	  end { case } ;
	  current_child^.balance_factor := EVEN_BF;
	  current := current_child;  { current is root of balanced subtree }
	end;
    end { right side checking and rotations } ;

  { re-attach balanced subtree rooted by current (formerly rooted by
    last_non_zero }
    
  if parent_of_LNZ = NIL then root := current
  else if parent_of_LNZ^.left_child = last_non_zero then
    parent_of_LNZ^.left_child := current
  else
    parent_of_LNZ^.right_child := current;

  if debug_28 then
    begin
      dump_avl_tree(Outfile, root, kind);
      writeln(Outfile, '-- exit avl_insert --');
    end;
90:  
end { avl_insert } ;


function avl_find(obj: avl_object_ptr;  root: avl_ptr;
                  kind: avl_type): avl_ptr;
  { find object in the tree and return a pointer to its entry.  If
    not found, return NIL }
  var
    current: avl_ptr;             { current node }
    found: boolean;               { TRUE iff found }
begin
  current := root;  found := FALSE;

  while not found and (current <> NIL) do
    case avl_compare(obj, current^.object, kind) of
      LT: current := current^.left_child;
      GT: current := current^.right_child;
      EQ: found := TRUE;
    end;

  avl_find := current;  
end { avl_insert } ;
