
procedure report_primitive(var f: textfile; prim1, prim2: primitive_ptr);
  var
    name: xtring;
begin
  detected_differences := TRUE;
  if prim1 <> NIL then name := prim1^.name
  else if prim2 <> NIL then name := prim2^.name
  else name := NIL;

  if name <> NIL then
    begin
      write(f, 'PATH_NAME ');
      writestring(f, name);
      if prim1 = NIL then write(f, ' not in file1;')
                     else write(f, ' line ', prim1^.line_number:1, ' file1;');
      if prim2 = NIL then write(f, ' not in file2.')
                     else write(f, ' line ', prim2^.line_number:1, ' file2.');
      writeln(f);
    end;

  if (prim1 <> NIL) and (prim2 <> NIL) then
    if prim1^.kind <> prim2^.kind then
      begin
        write(f, '< PRIMITIVE ');
	writestring(f, prim1^.kind);
	writeln(f);
        write(f, '> PRIMITIVE ');
	writestring(f, prim2^.kind);
	writeln(f);
      end;
end { report_primitive } ;


procedure report_property(var f: textfile; prefix: char; prop: property_ptr);
  var
    i: natural_number;
begin
  for i := 1 to prop^.count do
    begin
      write(f, prefix:1, ' ':1);
      writealpha(f, prop^.name^.name);
      write(f, '=');
      writestring(f, prop^.text);
      writeln(f, ';');
    end;
end { report_property } ;

  
function compare_properties(root1, root2: avl_ptr): boolean;
  var
    stack1: avl_ptr;
    stack2: avl_ptr;
    avl1: avl_ptr;
    avl2: avl_ptr;
    prop1: property_ptr;
    prop2: property_ptr;
    different: boolean;
begin
  avl1 := avl_inorder_init(root1, stack1);
  avl2 := avl_inorder_init(root2, stack2);
  different := FALSE;
  while (avl1 <> NIL) and (avl2 <> NIL) and not different do
    begin
      prop1 := avl1^.object.prop;
      prop2 := avl2^.object.prop;
      if (avl1^.object.prop^.name <> prop2^.name) or
         (prop1^.text <> prop2^.text) or
	 (prop1^.count <> prop2^.count) then different := TRUE;
      avl1 := avl_inorder(stack1);
      avl2 := avl_inorder(stack2);
    end;
  if (avl1 <> NIL) or (avl2 <> NIL) then compare_properties := FALSE
                                    else compare_properties := not different;
end { compare_properties } ;

    
function compare_body_properties(var f: textfile;  prim_reported: boolean;
                                 prim1, prim2: primitive_ptr): boolean;
  var
    stack1: avl_ptr;
    stack2: avl_ptr;
    avl1: avl_ptr;
    avl2: avl_ptr;
    diff1_head: avl_object_list_ptr;
    diff2_head: avl_object_list_ptr;
    diff1: avl_object_list_ptr;
    diff2: avl_object_list_ptr;
    different: boolean;
    in_difference: boolean;


  procedure append_difference(var head,tail: avl_object_list_ptr;
                              prop: property_ptr);
  begin
    different := TRUE;
    in_difference := TRUE;
    if head = NIL then
      begin
        insert_property_in_list(head, prop);
        tail := head;
      end
    else
      begin
        insert_property_in_list(tail^.next, prop);
        tail := tail^.next;
      end;
    tail^.object.prop := prop;
  end { append_difference } ;


  procedure report_difference;
    var
      p1: avl_object_list_ptr;
      p2: avl_object_list_ptr;
  begin
    if not prim_reported then
      begin
        report_primitive(f, prim1, prim2);
        prim_reported := TRUE;
      end;

    p1 := diff1_head;
    while p1 <> NIL do
      begin
        report_property(f, '<', p1^.object.prop);
        p1 := p1^.next;
      end;
    release_entire_avl_object_list(diff1_head);
    diff1 := NIL;

    p2 := diff2_head;
    while p2 <> NIL do
      begin
        report_property(f, '>', p2^.object.prop);
        p2 := p2^.next;
      end;
    release_entire_avl_object_list(diff2_head);
    diff2 := NIL;
    in_difference := FALSE;
  end { report_difference } ;


begin { compare_body_properties }
  different := FALSE;
  in_difference := FALSE;
  diff1 := NIL;
  diff1_head := NIL;
  diff2 := NIL;
  diff2_head := NIL;

  avl1 := avl_inorder_init(prim1^.properties, stack1);
  avl2 := avl_inorder_init(prim2^.properties, stack2);

  while (avl1 <> NIL) and (avl2 <> NIL) do
    begin
      case property_order(avl1^.object.prop, avl2^.object.prop) of
        LT:
	  begin
	    append_difference(diff1_head, diff1, avl1^.object.prop);
	    avl1 := avl_inorder(stack1);
          end;
        GT:
	  begin
	    append_difference(diff2_head, diff2, avl2^.object.prop);
	    avl2 := avl_inorder(stack2);
          end;
	EQ:
	  begin
            if avl1^.object.prop^.count = avl2^.object.prop^.count then
	      begin
	        if in_difference then report_difference;
	      end
	    else
	      begin
	        append_difference(diff1_head, diff1, avl1^.object.prop);
	        append_difference(diff2_head, diff2, avl2^.object.prop);
	      end;
	    avl1 := avl_inorder(stack1);
	    avl2 := avl_inorder(stack2);
          end;
      end { case } ;
    end;

  while avl1 <> NIL do
    begin
      append_difference(diff1_head, diff1, avl1^.object.prop);
      avl1 := avl_inorder(stack1);
    end;
  while avl2 <> NIL do
    begin
      append_difference(diff2_head, diff2, avl2^.object.prop);
      avl2 := avl_inorder(stack2);
    end;
  if in_difference then report_difference;
  compare_body_properties := not different;
end { compare_body_properties } ;

    
function compare_actuals(pin1, pin2: pin_ptr): boolean;
  var
    sig1: signal_ptr;
    sig2: signal_ptr;
    different: boolean;
begin
  sig1 := pin1^.signals;
  sig2 := pin2^.signals;
  different := FALSE;

  while (sig1 <> NIL) and (sig2 <> NIL) and not different do
    begin
      if sig1^.name <> sig2^.name then different := TRUE;
      if sig1^.left <> sig2^.left then different := TRUE;
      if sig1^.right <> sig2^.right then different := TRUE;
      if not different then
        if not compare_properties(sig1^.properties, sig2^.properties) then
          different := TRUE;
      sig1 := sig1^.next;
      sig2 := sig2^.next;
    end;

  if (sig1 <> NIL) or (sig2 <> NIL) then compare_actuals := FALSE
                                    else compare_actuals := not different;
end { compare_actuals } ;


procedure compare_bindings(var f: textfile; prim_reported: boolean;
                           prim1, prim2: primitive_ptr);
  type
    which_file_type = (FILE1, FILE2);
  var
    pin1: pin_ptr;
    pin2: pin_ptr;
    pin_stack1: avl_ptr;
    pin_stack2: avl_ptr;
    avl1: avl_ptr;
    avl2: avl_ptr;
    in_difference: boolean;
    diff1: avl_object_list_ptr;
    diff1_head: avl_object_list_ptr;
    diff2: avl_object_list_ptr;
    diff2_head: avl_object_list_ptr;
    compare_val: compare_type;


  procedure append_different_pin(pin: pin_ptr; which: which_file_type);
  begin
    if which = FILE1 then
      if diff1 = NIL then
        begin
          insert_pin_in_list(diff1, pin);
          diff1_head := diff1;
        end
      else 
        begin
          insert_pin_in_list(diff1^.next, pin);
          diff1 := diff1^.next;
        end
    else
      if diff2 = NIL then
        begin
          insert_pin_in_list(diff2, pin);
          diff2_head := diff2;
        end
      else 
        begin
          insert_pin_in_list(diff2^.next, pin);
          diff2 := diff2^.next;
        end;
    in_difference := TRUE;
  end { append_different_pin } ;


  procedure report_different_bindings;
    var
      pin1: avl_object_list_ptr;
      pin2: avl_object_list_ptr;


    procedure dump_pin_properties(var f: textfile; prefix: char;
                                  props: avl_ptr);
      const
#if SVS
        BACKSLASH_CHAR = '\\';
#else
        BACKSLASH_CHAR = '\';
#endif
      var
        stack: avl_ptr;
        current: avl_ptr;
        i: integer;
    begin
      current := avl_inorder_init(props, stack);
      while current <> NIL do
        begin
          with current^.object.prop^ do 
            for i := 1 to count do
              begin
                writeln(f);
                write(f, prefix:1, ' ':7);
                write(f, BACKSLASH_CHAR);
                writealpha(f, name^.name);
                write(f, '=');
                writestring(f, text);
              end;
          current := avl_inorder(stack);
        end;
    end { dump_pin_properties } ;


    procedure dump_pin_diff(var f: textfile; prefix: char; pin: pin_ptr);
      var
        sig: signal_ptr;
    begin
      write(f, prefix:1, ' ':1);

      { pin and its properties }

      writestring(f, pin^.name);
      dump_left_and_right(f, pin^.left, pin^.right);
      dump_pin_properties(f, prefix, pin^.properties);

      write(f, '=');

      { Signals and associated pin properties }

      sig := pin^.signals;
      while (sig <> NIL) do
        begin
          writestring(f, sig^.name);
          dump_left_and_right(f, sig^.left, sig^.right);
          dump_pin_properties(f, prefix, sig^.properties);
          sig := sig^.next;
          if sig <> NIL then 
            begin
              writeln(f, ':');
              write(f, prefix:1, ' ':3);
            end
          else write(f, ';');
        end;
      writeln(f);
    end { dump_pin_diff } ;


  begin { report_different_bindings } ;
    if not prim_reported then 
      begin
        report_primitive(f, prim1, prim2);
        prim_reported := TRUE;
      end;

    pin1 := diff1_head;
    while pin1 <> NIL do
      begin
        dump_pin_diff(f, '<', pin1^.object.pin);
        pin1 := pin1^.next;
      end;
    pin2 := diff2_head;
    while pin2 <> NIL do
      begin
        dump_pin_diff(f, '>', pin2^.object.pin);
        pin2 := pin2^.next;
      end;
    release_entire_avl_object_list(diff1_head);  diff1 := NIL;
    release_entire_avl_object_list(diff2_head);  diff2 := NIL;
    in_difference := FALSE;
  end { report_different_bindings } ;


begin { compare_bindings }
  in_difference := FALSE;
  diff1 := NIL;  diff2 := NIL;
  diff1_head := NIL;  diff2_head := NIL;
  avl1 := avl_inorder_init(prim1^.pins, pin_stack1);
  avl2 := avl_inorder_init(prim2^.pins, pin_stack2);
  while (avl1 <> NIL) and (avl2 <> NIL) do
    begin
      pin1 := avl1^.object.pin;
      pin2 := avl2^.object.pin;
      compare_val := pin_order(pin1, pin2);
      if compare_val = LT then
        begin
          append_different_pin(pin1, FILE1);
          avl1 := avl_inorder(pin_stack1);
        end
      else if compare_val = GT then
        begin
          append_different_pin(pin2, FILE2);
          avl2 := avl_inorder(pin_stack2);
        end
      else 
        begin
          if compare_actuals(pin1, pin2) then
            begin
              if in_difference then report_different_bindings;
            end
          else
            begin
              append_different_pin(pin1, FILE1);
              append_different_pin(pin2, FILE2);
            end;
          avl1 := avl_inorder(pin_stack1);
          avl2 := avl_inorder(pin_stack2);
        end;
    end;

  while (avl1 <> NIL) do
    begin
      append_different_pin(avl1^.object.pin, FILE1);
      avl1 := avl_inorder(pin_stack1);
    end;

  while (avl2 <> NIL) do
    begin
      append_different_pin(avl2^.object.pin, FILE2);
      avl2 := avl_inorder(pin_stack2);
    end;

  if in_difference then report_different_bindings;
end { compare_bindings } ;


function compare_signal_or_pin_sections(var f: textfile; 
                                        is_signal_section: boolean;
                                        prim1, prim2: primitive_ptr;
                                        table1, table2: avl_ptr): boolean;
  { compare the signal tables and report diffs.   is_signal_section is
    TRUE if this is a comparison of the SIGNAL to END_SIGNAL section
    (otherwise its a PIN to END_PIN section). prim1 and prim2 are 
    defined if their headings are to be printed if anything is amiss.
    Return TRUE iff comparison turns up no differences. }
  type
    which_file_type = (FILE1, FILE2);
  var
    sig1: signal_ptr;
    sig2: signal_ptr;
    sig_stack1: avl_ptr;
    sig_stack2: avl_ptr;
    avl1: avl_ptr;
    avl2: avl_ptr;
    in_difference: boolean;
    diff1: signal_ptr;
    diff1_head: signal_ptr;
    diff2: signal_ptr;
    diff2_head: signal_ptr;
    compare_val: compare_type;
    HEADING_reported: boolean;


  procedure append_different_sig(sig: signal_ptr; which: which_file_type);
  begin
    sig^.next := NIL;
    if which = FILE1 then
      if diff1 = NIL then
        begin
          diff1 := sig;
          diff1_head := diff1;
        end
      else 
        begin
          diff1^.next := sig;
          diff1 := diff1^.next;
        end
    else
      if diff2 = NIL then
        begin
          diff2 := sig;
          diff2_head := diff2;
        end
      else 
        begin
          diff2^.next := sig;
          diff2 := diff2^.next;
        end;
    in_difference := TRUE;
  end { append_different_sig } ;


  procedure dump_sig_diff(var f: textfile; prefix: char; sig: signal_ptr);


    procedure dump_sig_properties(var f: textfile; props: avl_ptr);
      var
        stack: avl_ptr;
        current: avl_ptr;
        i: integer;
    begin
      current := avl_inorder_init(props, stack);
      while current <> NIL do
        begin
          with current^.object.prop^ do 
            for i := 1 to count do
              begin
                writealpha(f, name^.name);
                write(f, '=');
                writestring(f, text);
              end;
          current := avl_inorder(stack);
          if current <> NIL then write(f, ',');
        end;
    end { dump_sig_properties } ;


  begin { dump_sig_diff }
    write(f, prefix:1, ' ':1);

    { sig and its properties }

    writestring(f, sig^.name);
    dump_left_and_right(f, sig^.left, sig^.right);
    write(f, ':');
    dump_sig_properties(f, sig^.properties);
    writeln(f, ';');
  end { dump_sig_diff } ;


  procedure report_different_sigs;
    var
      sig1: signal_ptr;
      sig2: signal_ptr;


  begin { report_different_sigs } ;
    if not HEADING_reported then 
      begin
        if is_signal_section then writeln(f, 'SIGNAL')
        else
          begin
            if (prim1 <> NIL) then report_primitive(f, prim1, prim2);
            writeln(f, 'PIN');
          end;
        HEADING_reported := TRUE;
      end;

    sig1 := diff1_head;
    while sig1 <> NIL do
      begin
        dump_sig_diff(f, '<', sig1);
        sig1 := sig1^.next;
      end;
    sig2 := diff2_head;
    while sig2 <> NIL do
      begin
        dump_sig_diff(f, '>', sig2);
        sig2 := sig2^.next;
      end;
    diff1 := NIL;
    diff2 := NIL;
    in_difference := FALSE;
  end { report_different_sigs } ;


begin { compare_signal_or_pin_sections }
  HEADING_reported := FALSE;
  in_difference := FALSE;
  diff1 := NIL;  diff2 := NIL;
  diff1_head := NIL;  diff2_head := NIL;
  avl1 := avl_inorder_init(table1, sig_stack1);
  avl2 := avl_inorder_init(table2, sig_stack2);
{ if (avl1 = NIL) then writeln(monitor, 'avl1 is NIL'); }(*DEBUG*)
{ if (avl2 = NIL) then writeln(monitor, 'avl2 is NIL'); }(*DEBUG*)
  while (avl1 <> NIL) and (avl2 <> NIL) do
    begin
      sig1 := avl1^.object.sig;
      sig2 := avl2^.object.sig;
{     dump_sig_diff(monitor, '>', sig2); }(*DEBUG*)
{     dump_sig_diff(monitor, '<', sig1); }(*DEBUG*)
      compare_val := signal_order(sig1, sig2);
      if compare_val = LT then
        begin
          append_different_sig(sig1, FILE1);
          avl1 := avl_inorder(sig_stack1);
        end
      else if compare_val = GT then
        begin
          append_different_sig(sig2, FILE2);
          avl2 := avl_inorder(sig_stack2);
        end
      else 
        begin
          if compare_properties(sig1^.properties, sig2^.properties) then
            begin
              if in_difference then report_different_sigs;
            end
          else
            begin
              append_different_sig(sig1, FILE1);
              append_different_sig(sig2, FILE2);
            end;
          avl1 := avl_inorder(sig_stack1);
          avl2 := avl_inorder(sig_stack2);
        end;
    end;

  while (avl1 <> NIL) do
    begin
      append_different_sig(avl1^.object.sig, FILE1);
      avl1 := avl_inorder(sig_stack1);
    end;

  while (avl2 <> NIL) do
    begin
      append_different_sig(avl2^.object.sig, FILE2);
      avl2 := avl_inorder(sig_stack2);
    end;

  if in_difference then report_different_sigs;

  if HEADING_reported then
    if is_signal_section then writeln(f, 'END_SIGNAL;')
                         else writeln(f, 'END_PIN;');

  compare_signal_or_pin_sections := not HEADING_reported;
end { compare_signal_or_pin_sections } ;


procedure compare_signal_sections(var f: textfile; table1, table2: avl_ptr);
begin
  if not compare_signal_or_pin_sections(f, TRUE, NIL, NIL, 
                                        table1, table2) then
    detected_differences := TRUE;
end { compare_signal_sections } ;


function compare_pin_sections(var f: textfile; prim_reported: boolean;
                              prim1, prim2: primitive_ptr): boolean;
  { compare the pin sections, returning TRUE iff they compare ok }
begin
  if prim_reported then
    compare_pin_sections :=
      compare_signal_or_pin_sections(f, FALSE, NIL, NIL,
                                     prim1^.pinprops, prim2^.pinprops)
  else
    compare_pin_sections :=
      compare_signal_or_pin_sections(f, FALSE, prim1, prim2,
                                     prim1^.pinprops, prim2^.pinprops);
end { compare_sections } ;


procedure compare_netlists(var f: textfile; table1, table2: avl_ptr);
  var
    prim1: primitive_ptr;
    prim2: primitive_ptr;
    prim_stack1: avl_ptr;
    prim_stack2: avl_ptr;
    avl1: avl_ptr;
    avl2: avl_ptr;
    compare_val: compare_type;
    prim_reported: boolean;
begin
  detected_differences := FALSE;
  avl1 := avl_inorder_init(table1, prim_stack1);
  avl2 := avl_inorder_init(table2, prim_stack2);
  while (avl1 <> NIL) and (avl2 <> NIL) do
    begin
      prim1 := avl1^.object.prim;
      prim2 := avl2^.object.prim;
      compare_val := primitive_order(prim1, prim2);
      if compare_val = LT then
        begin
          report_primitive(f, prim1, NIL);
          avl1 := avl_inorder(prim_stack1);
        end
      else if compare_val = GT then
        begin
          report_primitive(f, NIL, prim2);
          avl2 := avl_inorder(prim_stack2);
        end
      else
        begin
          if prim1^.kind <> prim2^.kind then report_primitive(f, prim1, prim2)
          else 
            begin
              prim_reported := 
                not compare_pin_sections(f, FALSE, prim1, prim2);
              if not compare_body_properties(f, prim_reported, 
                                             prim1, prim2) then
                prim_reported := TRUE;
              compare_bindings(f, prim_reported, prim1, prim2);
            end;
          avl1 := avl_inorder(prim_stack1);
          avl2 := avl_inorder(prim_stack2);
        end;
    end;

  while (avl1 <> NIL) do
    begin
      report_primitive(f, avl1^.object.prim, NIL);
      avl1 := avl_inorder(prim_stack1);
    end;

  while (avl2 <> NIL) do
    begin
      report_primitive(f, NIL, avl2^.object.prim);
      avl2 := avl_inorder(prim_stack2);
    end;
end { compare_netlist } ;
