
(**)    { ------- initialization routines ------- }


procedure init;
  { initialize global (and some local) 'constants' and vars }
  

  procedure init_keywords;
    { init macro keywords }
    var
      id: alpha;              { key word to be added }
      sy: symbols;            { corresponding symbol for the key word }
    

    procedure add_keyword;
      { add the key word to the name table and set its value }
      var
        name: name_ptr;       { name to be added to the name table }
    begin
      name := enter_name(id);
      if KEY_WORD in name^.kind then error(187 { assertion })
      else
        begin  name^.kind := name^.kind + [KEY_WORD];  name^.sy := sy;  end;
    end { add_keyword } ;
      

  begin { init_keywords }
    id := 'OR              ';  sy := ORSY;           add_keyword;
    id := 'AND             ';  sy := ANDSY;          add_keyword;
    id := 'MOD             ';  sy := MODSY;          add_keyword;
    id := 'NOT             ';  sy := NOTSY;          add_keyword;
    id := 'ABS             ';  sy := ABSSY;          add_keyword;
    id := 'ORD             ';  sy := ORDSY;          add_keyword;
    id := 'END             ';  sy := ENDSY;          add_keyword;
    id := 'MACRO           ';  sy := MACROSY;        add_keyword;
    id := 'END_MACRO       ';  sy := ENDMACROSY;     add_keyword;
    id := 'INVOKE          ';  sy := INVOKESY;       add_keyword;
    id := 'END_INVOKE      ';  sy := ENDINVOKESY;    add_keyword;
    id := 'PROPERTY        ';  sy := PROPERTYSY;     add_keyword;
    id := 'END_PROPERTY    ';  sy := ENDPROPERTYSY;  add_keyword;
    id := 'PIN             ';  sy := PINSY;          add_keyword;
    id := 'END_PIN         ';  sy := ENDPINSY;       add_keyword;
    id := 'BODY            ';  sy := BODYSY;         add_keyword;
    id := 'END_BODY        ';  sy := ENDBODYSY;      add_keyword;
    id := 'BINDINGS        ';  sy := BINDINGSY;      add_keyword;
    id := 'END_BINDINGS    ';  sy := ENDBINDSY;      add_keyword;
    id := 'PARAMETER       ';  sy := PARAMETERSY;    add_keyword;
    id := 'END_PARAMETER   ';  sy := ENDPARAMETERSY; add_keyword;
    id := 'FILE_TYPE       ';  sy := FILETYPESY;     add_keyword;
    id := 'SEPCOMP         ';  sy := SEPCOMPSY;      add_keyword;
  end { init_keywords } ;
      


  procedure init_errors;
    { init the error messages }
  begin
    error_strings[0]   := 'THIS ERROR IS ALWAYS SUPPRESSED             ';
    error_strings[1]   := 'Expected identifier                         ';
    error_strings[2]   := 'Expected =                                  ';
    error_strings[3]   := 'Not supported by ValidPAGECOMP              ';
    error_strings[4]   := 'Expected string or identifier               ';
    error_strings[5]   := 'Expected ,                                  ';
    error_strings[6]   := 'Unrecognized symbol                         ';
    error_strings[7]   := 'Expected )                                  ';
    error_strings[8]   := 'Expected =, <, >, <>, <= or >=              ';
    error_strings[9]   := 'Wrong file type for text macros             ';
    error_strings[10]  := 'Expected <                                  ';
    error_strings[11]  := 'Expected >                                  ';
    error_strings[12]  := 'Expected ;                                  ';
    error_strings[13]  := 'Expected :                                  ';
    error_strings[14]  := 'Unexpected symbol in integer expression     ';
    error_strings[15]  := 'Expected (                                  ';
    error_strings[16]  := 'Bit value invalid                           ';
    error_strings[17]  := 'Unknown SHAREABLE specification             ';
    error_strings[18]  := '                                            ';
    error_strings[19]  := '                                            ';
    error_strings[20]  := 'Unmatched closing comment character         ';
    error_strings[21]  := '                                            ';
    error_strings[22]  := 'String length exceeded                      ';
    error_strings[23]  := 'Illegal character found                     ';
    error_strings[24]  := 'Expression value overflow                   ';
    error_strings[25]  := 'Division by zero                            ';
    error_strings[26]  := 'PRIM in rule requires SCALD directory type  ';
    error_strings[27]  := 'Expected SPECIAL or PRIMITIVE               ';
    error_strings[28]  := 'PRIM (PART) not a legal SCALD directory type';
    error_strings[29]  := 'Extraneous junk after bit list              ';
    error_strings[30]  := 'Unexpected symbol in bit subscript          ';
    error_strings[31]  := 'Unknown REPORT specification                ';
    error_strings[32]  := 'Non-printing character found                ';
    error_strings[33]  := 'Expected a string                           ';
    error_strings[34]  := 'Comment not closed before end of input      ';
    error_strings[35]  := 'Specified parameter # > allowed # params    ';
    error_strings[36]  := 'Signal MUST have high assertion char        ';
    error_strings[37]  := 'Expected .                                  ';
    error_strings[38]  := 'File name has already been specified        ';
    error_strings[39]  := 'Undefined identifier in expression          ';
    error_strings[40]  := 'Expected END                                ';
    error_strings[41]  := 'Identifier length exceeded                  ';
    error_strings[42]  := 'Unknown debug control specifier             ';
    error_strings[43]  := 'Text macro parameter exceeds max length     ';
    error_strings[44]  := 'Constant width value out of range           ';
    error_strings[45]  := 'Directive has already been specified        ';
    error_strings[46]  := 'Duplicate global text macro definition      ';
    error_strings[47]  := 'Invalid specification for inheritance       ';
    error_strings[48]  := 'Unknown property attribute                  ';
    error_strings[49]  := 'Directive unsupported by SCALD compiler     ';
    error_strings[50]  := 'Extraneous junk at end of number            ';{non-parse}
    error_strings[51]  := 'Unknown compiler directive                  ';
    error_strings[52]  := 'Invalid specification for directive         ';
    error_strings[53]  := 'Input line exceeds maximum length           ';
    error_strings[54]  := 'Can''t begin signal name with ''(''            ';
    error_strings[55]  := 'Wrong file type for property attributes     ';
    error_strings[56]  := 'Text macro parameter cannot be found        ';
    error_strings[57]  := 'End of input before end of expression       ';
    error_strings[58]  := 'Extraneous characters at end of expr        ';
    error_strings[59]  := 'Identifier has not been defined             ';
    error_strings[60]  := 'Same shadow directory use more than once    ';
    error_strings[61]  := 'Radix must be in range 2..16                ';
    error_strings[62]  := 'Extraneous junk at end of menu version      ';
    error_strings[63]  := 'Extraneous junk at end of boolean expr      ';
    error_strings[64]  := 'Max text macro nesting depth exceeded       ';
    error_strings[65]  := 'Compile extension name is too long          ';
    error_strings[66]  := 'Compilation to .PRIM files not permitted    ';
    error_strings[67]  := 'Error in signal syntax specification        ';
    error_strings[68]  := 'Library file has already been specified     ';
    error_strings[69]  := 'Ill-formed cardinal tap body                ';
    error_strings[70]  := 'Version number is outside allowed range     ';
    error_strings[71]  := 'Page number is outside allowed range        ';
    error_strings[72]  := 'Duplicate page number                       ';
    error_strings[73]  := 'Tapped bit(s) not found in connected bus    ';
    error_strings[74]  := 'Multiple busses on cardinal tap body        ';
    error_strings[75]  := 'No named bus attached to cardinal tap body  ';
    error_strings[76]  := 'Bool expr already defined for this vers     ';
    error_strings[77]  := 'Specified version is not in directory       ';
    error_strings[78]  := 'Only MENU bodies are allowed here           ';
    error_strings[79]  := 'Expected a version 1 with MENU body(s)      ';
    error_strings[80]  := 'Illegal property on MENU body               ';
    error_strings[81]  := 'Illegal MENU property number                ';
    error_strings[82]  := 'Same MENU expr property found twice         ';
    error_strings[83]  := 'Expected signal name or constant            ';
    error_strings[84]  := 'Replication factor is out of range          ';
    error_strings[85]  := 'Expected FILE_TYPE specification            ';
    error_strings[86]  := 'File is not of the correct type             ';
    error_strings[87]  := 'Directory file name previously specified    ';
    error_strings[88]  := 'Illegal file name in drawing directory      ';
    error_strings[89]  := 'String not closed before the end of line    ';
    error_strings[90]  := '                                            ';
    error_strings[91]  := 'Expected shadow directory name              ';
    error_strings[92]  := 'Invalid (warnings and oversights only)      ';
    error_strings[93]  := 'Expected directory file name                ';
    error_strings[94]  := 'Invalid value for print width               ';
    error_strings[95]  := 'Drawings have not been written with 5.5     ';
    error_strings[96]  := 'Bit subscript on constant not permitted     ';
    error_strings[97]  := 'Unknown output file name                    ';
    error_strings[98]  := 'Extraneous junk at end of file              ';
    error_strings[99]  := 'This symbol cannot be used here             ';
    error_strings[100] := 'Can''t open specified expansion rules file   ';
    error_strings[101] := 'Drawing path name is missing closing )      ';
    error_strings[102] := 'Symbol must be one character                ';
    error_strings[103] := 'Library not found in master directory       ';
    error_strings[104] := 'Wrong FILE_TYPE for master library          ';
    error_strings[105] := 'Reserved text macro name                    ';
    error_strings[106] := 'Cannot find definition for text macro       ';
    error_strings[107] := 'Duplicate library entry in master lib       ';
    error_strings[108] := 'No SIZE in context for SIZEd drawing        ';
    error_strings[109] := 'Body properties are not allowed here        ';
    error_strings[110] := 'Undefined text macro (null value)           ';
    error_strings[111] := 'No directory was specified                  ';
    error_strings[112] := 'Separate AND single drawing compilation     ';
    error_strings[113] := 'Replication is not permitted on pin name    ';
    error_strings[114] := 'Text macro has already been defined         ';
    error_strings[115] := 'Same MENU version property found twice      ';
    error_strings[116] := 'Expanded string exceeds max string len      ';
    error_strings[117] := 'Textmacro and parameters exceeds max len    ';
    error_strings[118] := 'Expression value is empty                   ';
    error_strings[119] := 'NC is not permitted as a pin name           ';
    error_strings[120] := 'Path name exceeds maximum length            ';
    error_strings[121] := 'Path element name exceeds maximum length    ';
    error_strings[122] := 'Drawing has incompatible extensions         ';
    error_strings[123] := 'Selection expr and MENU expr mismatch       ';
    error_strings[124] := 'Versioned drawings not written w 7.0 GED    ';
    error_strings[125] := 'Pin properties are not permitted here       ';
    error_strings[126] := 'Text macro is not an identifier             ';
    error_strings[127] := 'ABBREV property not found for drawing       ';
    error_strings[128] := 'ABBREV value must be letters, digits & _    ';
    error_strings[129] := 'Menu entry for version is not permitted     ';
    error_strings[130] := 'Scalar reference to vector signal           ';
    error_strings[131] := 'Vector reference to scalar signal           ';
    error_strings[132] := 'Concatenated signal as pin name             ';
    error_strings[133] := 'This property has already been defined      ';
    error_strings[134] := 'Terminal drawing is not a primitive part    ';
    error_strings[135] := 'Cannot open compiler directives file        ';
    error_strings[136] := 'Signals cannot be attached to this body     ';
    error_strings[137] := 'Text macro nesting depth exceeded           ';
    error_strings[138] := 'Cannot open error documentation file        ';
    error_strings[139] := 'More than 1 selection expression is true    ';
    error_strings[140] := 'This signal cannot be DECLAREd              ';
    error_strings[141] := 'SIZE property on non SIZE-wide body         ';
    error_strings[142] := 'MENU bodies are not supported               ';
    error_strings[143] := 'Illegal rotation on this body               ';
    error_strings[144] := 'This body should not be given SIZE prop     ';
    error_strings[145] := 'Pin name does not exist                     ';
    error_strings[146] := 'Pin name does not have this bit             ';
    error_strings[147] := 'Root drawing specified as primitive.        ';
    error_strings[148] := 'No root drawing was specified               ';
    error_strings[149] := 'Synonyms must use single assertion          ';
    error_strings[150] := 'PERMIT attribute value invalid              ';
    error_strings[151] := 'This property not permitted on a SIGNAL     ';
    error_strings[152] := 'This property not permitted on a BODY       ';
    error_strings[153] := 'This property not permitted on a PIN        ';
    error_strings[154] := 'Signal''s assertion cannot be determined     ';
    error_strings[155] := 'Attempt to synonym 0 and 1                  ';
    error_strings[156] := 'Signal''s width cannot be determined         ';
    error_strings[157] := 'Error found in configuration file: fatal    ';
    error_strings[158] := 'Signal''s scope conflicts with complement    ';
    error_strings[159] := 'Synonym of unequal width signals            ';
    error_strings[160] := 'Cannot SIZE replicate plumbing drawings     ';
    error_strings[161] := '2 signals w/ timing assertions synonymed    ';
    error_strings[162] := 'Interface and local signals conflict        ';
    error_strings[163] := 'Local and global signals conflict           ';
    error_strings[164] := 'Global and interface signals conflict       ';
    error_strings[165] := 'This signal cannot have scope property      ';
    error_strings[166] := 'Cannot access CMPDRAW file for input        ';
    error_strings[167] := 'Cannot open synonyms file for input         ';
    error_strings[168] := 'Cannot close file                           ';
    error_strings[169] := 'Cannot open file for output                 ';
    error_strings[170] := 'Cannot open master library file             ';
    error_strings[171] := 'Bit subscript increment of 0 not allowed    ';
    error_strings[172] := 'Bit subscript should be right to left       ';
    error_strings[173] := 'Bit subscript should be left to right       ';
    error_strings[174] := 'Vector/scalar conflict between complements  ';
    error_strings[175] := 'Can''t use a remote shadow directory         ';
    error_strings[176] := 'PATH name is not unique                     ';
    error_strings[177] := 'Selection expr for drawing is FALSE         ';
    error_strings[178] := 'Max error value must be >= 1                ';
    error_strings[179] := 'Extraneous junk at end of signal            ';
    error_strings[180] := 'Parameter was declared twice                ';
    error_strings[181] := 'Cannot access drawing directory             ';
    error_strings[182] := 'Drawing title does not match directory      ';
    error_strings[183] := 'X_FIRST must be >= 0 (set to 0)             ';
    error_strings[184] := 'PIN_EQUIVALENT no longer supported          ';
    error_strings[185] := 'SIZE must be >= 0 (set to 1)                ';
    error_strings[186] := 'X_STEP must be > 0 (set to 1)               ';
    error_strings[187] := 'Assertion chk failure: save CMPLOG file     ';
    error_strings[188] := 'Parameters not permitted on this body       ';
    error_strings[189] := 'Timing assertion not allowed on pin name    ';
    error_strings[190] := 'No selection expression evaluates true      ';
    error_strings[191] := 'Drawing not found in the directories        ';
    error_strings[192] := 'PATH name element is not unique             ';
    error_strings[193] := 'No usable extension found for drawing       ';
    error_strings[194] := 'Text macro refers to itself (recursive)     ';
    error_strings[195] := 'Cannot open specified attributes file       ';
    error_strings[196] := 'Default value used for SIZE (1)             ';
    error_strings[197] := 'PATH property not found for body            ';
    error_strings[198] := 'Bit subscript on undefined width pin        ';
    error_strings[199] := 'Pin name conflicts with previous pin        ';
    error_strings[200] := 'Pin name and signal widths do not match     ';
    error_strings[201] := 'Signal fails bubble check on this pin       ';
    error_strings[202] := 'Pin name with NWC cannot have subscript     ';
    error_strings[203] := 'LOGIC primitive used as terminal drawing    ';
    error_strings[204] := 'Pin name cannot use signal negation         ';
    error_strings[205] := 'Cannot open DRAWING file                    ';
    error_strings[206] := 'Cannot open specified directory file        ';
    error_strings[207] := 'Cannot open syntax configuration file       ';
    error_strings[208] := 'Too many errors in this compile!            ';
    error_strings[209] := 'Cannot open specified text macro file       ';
    error_strings[210] := 'Primitive cannot have NWC pin               ';
    error_strings[211] := 'Pin name is vector but used as a scalar     ';
    error_strings[212] := 'A pin name cannot be a constant             ';
    error_strings[213] := 'Versioned drawing must have EXPR prop       ';
    error_strings[214] := 'String not closed before end of signal      ';
    error_strings[215] := 'Pin name is scalar but used as vector       ';
    error_strings[216] := 'PART not allowed; COMPILE LOGIC assumed     ';
    error_strings[217] := 'Fatal error(s) encountered - run stopped    ';
    error_strings[218] := 'Pin subscript must be a simple subrange     ';
    error_strings[219] := 'DECLARE bodies are no longer supported      ';
    error_strings[220] := 'PART_NAME property should not be used       ';
    error_strings[221] := 'Bodies with NWC cannot expand to parts      ';
    error_strings[222] := 'DELETE_DRAWING specifies unfound drawing    ';
    error_strings[223] := 'SCHEMA file is not a SCHEMA file.           ';
    error_strings[224] := 'Extraneous junk at end of command line      ';
    error_strings[225] := 'This directive is obsolete                  ';
    error_strings[226] := 'auto-SCALD dir name not in master dir       ';
    error_strings[227] := 'Cannot open expansion file                  ';
    error_strings[228] := 'Incorrect use of seplink command            ';
    error_strings[229] := 'Illegal COMMAND value                       ';
    error_strings[230] := 'Can''t create shadow directory               ';
    error_strings[231] := 'Can''t open compiler-specified file          ';
    error_strings[232] := 'Unable to access schema file                ';
    error_strings[233] := 'Expected an expansion file -- wrong type    ';
    error_strings[234] := 'Cannot access DESIGN file for output        ';
    error_strings[235] := 'Signal synonymed to its own complement      ';
    error_strings[236] := 'Cannot make a directory                     ';
    error_strings[237] := 'Cannot get file "last modified" time        ';
    error_strings[238] := 'Illegal hierarchical drawing recursion      ';
#if SVS
    error_strings[239] := 'Pin or pin bit has no \\I signal             ';
#else
    error_strings[239] := 'Pin or pin bit has no \I signal             ';
#endif
    error_strings[240] := '.PRIM and .PART both found for drawing      ';
    error_strings[241] := 'Pin connected to its own complement         ';
    error_strings[242] := 'Different pins on bodies for same module    ';
    error_strings[243] := 'Compiler errors in a drawing                ';
    error_strings[244] := 'Needed module was not loaded                ';
    error_strings[245] := 'ABBREV property conflict in drawing         ';
    error_strings[246] := 'Unable to remove file                       ';
    error_strings[247] := 'Pages of version across SCALD dirs          ';
    error_strings[248] := 'Incorrect page expansion file syntax        ';
    error_strings[249] := '                                            ';
    error_strings[250] := 'Missing ValidPAGECOMP/LINKER results        ';
    error_strings[251] := 'Unrecognized command line argument          ';
    error_strings[252] := 'Illegal drawing directory in SCALD directory';

    error_strings[253] := 'More than 1 DEFAULT selection expression    ';
    error_strings[254] := 'Same special model in more than 1 directory ';
    error_strings[255] := 'Unable to successfully read expansion rules ';

    { initialize environment variables for error display }

    num_errors := 0;
    num_warnings := 0;
    num_oversights := 0;
    last_error := 0;

    ok_to_print_error := TRUE;
    errors_encountered := [];
    exception_code := NULL_ERROR_CODE;

    { ------- error sets ------- }

    scan_past_errors := [1,2,4..8,10..13,15,33,37,40,83,85,93,179];
    fatal_errors := [65,67,100,111,112,120,135,148,157,166,169,
                     205,206,207,228,229,234,238,252,255];
    warning_errors := [3,8,49,68,107,108,141,154,158,182,184,194,196,203,
		       239,246];
    oversight_errors := [37,40,41,80,92,95,97,123,127,134,140,143,
                         144,174,176,192,197,213,216,220,224,225,251,245];
    parse_errors := [1..48,51..53,55..59,61..68,83..87,89,90,92..94,
                     96,97,99,101..104,107,171..173,178..179,184,216,225,229,
		     248,252];
    echo_to_monitor_errors := [208,217];
  end { init_errors } ;


  procedure init_sets;
    { initialize various constant sets }
  begin
    {  mulops := [ASTERISK, SLASH, MODSY];
       addops := [PLUS, MINUS];
       relops := [EQUAL, NESY, LESSTHAN, LESY, GREATERTHAN, GESY];  }

      
    valid_chars[2]  := ['0','1'];
    valid_chars[3]  := valid_chars[2]  + ['2'];
    valid_chars[4]  := valid_chars[3]  + ['3'];
    valid_chars[5]  := valid_chars[4]  + ['4'];
    valid_chars[6]  := valid_chars[5]  + ['5'];
    valid_chars[7]  := valid_chars[6]  + ['6'];
    valid_chars[8]  := valid_chars[7]  + ['7'];
    valid_chars[9]  := valid_chars[8]  + ['8'];
    valid_chars[10] := valid_chars[9]  + ['9'];
    valid_chars[11] := valid_chars[10] + ['A'];
    valid_chars[12] := valid_chars[11] + ['B'];
    valid_chars[13] := valid_chars[12] + ['C'];
    valid_chars[14] := valid_chars[13] + ['D'];
    valid_chars[15] := valid_chars[14] + ['E'];
    valid_chars[16] := valid_chars[15] + ['F'];

    constant_chars := ['0'..'9','(',')'];
    legal_chars := ['0'..'9','A'..'I','J'..'R','S'..'Z',
                    'a'..'i','j'..'r','s'..'z',
                    ' ','!','"','#','$','%','&','''','(',')',
                    '*','+',',','-','.','/',':',';','<','>',
                    '=','?','@','[',
#if SVS
                    '\\', 
#else
                    '\',
#endif
                    ']','^','{','|','}','~','_','`'];

    forbidden_symbols := [SEMI,LESSTHAN,GREATERTHAN,CONSTANT,PERCENT,SHARP];
    expression_symbols := [PLUS, MINUS, SLASH, LPAREN, RPAREN, ASTERISK];

{    bitsubendsys := [COMMA, GREATERTHAN];         }
    propbeginsys := [PINSY, BODYSY, PARAMETERSY];

    allowed_key_words := [];
{   schema_keysys := [ENDSY, FILETYPESY];         }
    expansion_keysys := [ENDSY, FILETYPESY];
    directives_keysys := [ENDSY];
    directory_keysys := [ENDSY, FILETYPESY];
    signal_keysys := [ORSY..MAXSY,low_asserted_sy..negation_sy];
    config_keysys := [low_asserted_sy..negation_sy];

    inheritance_attributes := [INHERIT_BODY, INHERIT_PIN, INHERIT_SIGNAL];
    signal_inheritance_attributes := [INHERIT_PIN, INHERIT_SIGNAL];
    permission_attributes := [PERMIT_SIGNAL, PERMIT_PIN, PERMIT_BODY];
    default_attributes := [INHERIT_BODY, INHERIT_SIGNAL] +
                          permission_attributes;

    parameter_attributes := [IS_INT_PARAMETER, IS_PARAMETER];

    bodies_with_bindings := [USER_BODY, PIN_NAMES_BODY, DECLARE_BODY];

{   PRIM_equivalent_extensions := [PRIM_EXTENSION, PART_EXTENSION];    }
  end { init_sets } ;



  procedure init_string_structures;
    { init the table describing the discrete string lengths that can be
      created on the heap }
    var
      i: 1..33;                  { index into list of free strings }
      j: hash_string_range;      { index into the string table }
  begin
    for j := 0 to HASH_STRING_TABLE_SIZE do string_table[j] := NIL;

    free_pointers := NIL;
    for i := 1 to 33 do free_strings[i] := NIL;
    s_length[1] := 4;
    s_length[2] := 8;
    s_length[3] := 12;
    s_length[4] := 16;
    s_length[5] := 20;
    s_length[6] := 24;
    s_length[7] := 28;
    s_length[8] := 32;
    s_length[9] := 36;
    s_length[10] := 40;
    s_length[11] := 44;
    s_length[12] := 48;
    s_length[13] := 52;
    s_length[14] := 56;
    s_length[15] := 60;
    s_length[16] := 64;
    s_length[17] := 68;
    s_length[18] := 72;
    s_length[19] := 76;
    s_length[20] := 80;
    s_length[21] := 84;
    s_length[22] := 88;
    s_length[23] := 92;
    s_length[24] := 96;
    s_length[25] := 100;
    s_length[26] := 120;
    s_length[27] := 140;
    s_length[28] := 160;
    s_length[29] := 180;
    s_length[30] := 200;
    s_length[31] := 220;
    s_length[32] := 240;
    s_length[33] := 256;

    create_a_string(nullstring, 0);
  end { init_string_structures } ;




  procedure init_directives;
    { initialize the linker directive tables and flags }

    
    function make_default_file_name(file_prefix: alpha;
        			    middle_name: alpha; 
				    name: alpha): xtring;
      { Return a default name made up of the concatenation of the prefix,
        middle name, and file name.  It is assumed that the file name will
        never be null, but that the prefix and/or middle_name can be. }
      var
        file_name: xtring;             { file name to be created }
        i: string_range;               { index into the names for copy }
        prefix_length: 0..ID_LENGTH;   { length of the prefix }  
        middle_len: 0..ID_LENGTH;      { length of middle part of name }
        file_length: id_range;         { length of the file name }
    begin
      prefix_length := alpha_length(file_prefix); 
      if file_prefix = NULL_ALPHA then prefix_length := 0;
      middle_len := alpha_length(middle_name);     
      if middle_name = NULL_ALPHA then middle_len := 0;

      file_length := alpha_length(name);

      create_a_string(file_name, (prefix_length 
                                  + middle_len            
				  + file_length) );

      for i := 1 to prefix_length do file_name^[i] := file_prefix[i];
      for i := 1 to middle_len do                        
        file_name^[prefix_length+i] := middle_name[i];    
      for i := 1 to file_length do 
	file_name^[i+prefix_length+middle_len] := name[i];

      make_default_file_name := enter_and_release_string(file_name);
    end { make_default_file_name } ;


    procedure add_to_table(directive: directive_type; name: alpha);
      { add the given directive to the table }
    begin
      compiler_directive[directive] := enter_name(name);
    end { add_to_table } ;


  begin { init_directives }
    add_to_table(ALLOW_PART_NAME_DIRECTIVE,         'ALLOW_PART_NAME ');
    add_to_table(AMUSING_MESSAGES_DIRECTIVE,        'AMUSING_MESSAGES');
    add_to_table(BUBBLECHECK_DIRECTIVE,             'BUBBLE_CHECK    ');
    {    add_to_table(CLASSGEN_DIRECTIVE,                'CLASS_GEN       ');  }
    add_to_table(CARDINAL_TAP_DIRECTIVE,            'CARDINAL_TAP    ');
    add_to_table(COMMAND_DIRECTIVE,                 'COMMAND         ');
    add_to_table(COMPILE_DIRECTIVE,                 'COMPILE         ');
    add_to_table(CONFIG_FILE_DIRECTIVE,             'CONFIGURE_FILE  ');
    add_to_table(CONST_BUBBLE_CHK_DIRECTIVE,        'CONST_BUBBLE_CHK');
    add_to_table(CONTEXT_DIRECTIVE,                 'CONTEXT         ');
    add_to_table(DEBUG_DIRECTIVE,                   'DEBUG           ');
    add_to_table(DEBUG_AT_PATH_directive,           'DEBUG_AT_PATH   ');
    add_to_table(DEBUG_CONTROL_DIRECTIVE,           'DEBUG_CONTROL   ');
    add_to_table(DECLARE_BODIES_DIRECTIVE,          'DECLARE_BODIES  ');
    add_to_table(DEFAULT_L_OR_G_DIRECTIVE,          'DEFAULT_SCOPE   ');
    add_to_table(DEFAULT_FILTER_DIRECTIVE,          'DEFAULT_FILTER  ');
    add_to_table(DIRECTORY_DIRECTIVE,               'DIRECTORY       ');
    add_to_table(ERROR_HELP_DIRECTIVE,              'ERROR_HELP      ');
    add_to_table(EVACUATE_DIRECTIVE,                'EVACUATE        ');
    add_to_table(EXPANSION_RULES_DIRECTIVE,         'EXP_RULES_FILE  ');
    add_to_table(FILTER_PROPERTY_DIRECTIVE,         'FILTER_PROPERTY ');
    add_to_table(HIERARCHICAL_NWC_DIRECTIVE,        'HIERARCHICAL_NWC');
    add_to_table(LIBRARY_DIRECTIVE,                 'LIBRARY         ');
    add_to_table(LOCALLY_GLOBAL_DIRECTIVE,          'LOCALLY_GLOBAL  ');
    add_to_table(MASTER_LIBRARY_DIRECTIVE,          'MASTER_LIBRARY  ');
    add_to_table(MAX_ERROR_DIRECTIVE,               'MAX_ERRORS      ');
    add_to_table(MERGE_DRAWINGS_DIRECTIVE,          'MERGE_DRAWINGS  ');
    add_to_table(NESTED_TM_PARAMS_DIRECTIVE,        'NESTED_TM_PARAMS');
    add_to_table(NET_PROCESSING_DIRECTIVE,          'NET_PROCESSING  ');
    add_to_table(OUTPUT_DIRECTIVE,                  'OUTPUT          ');
    add_to_table(OVERSIGHT_DIRECTIVE,               'OVERSIGHTS      ');
    add_to_table(PAGE_SYNONYM_DIRECTIVE,            'PAGE_SYNONYM    ');
    add_to_table(PASS_PROPERTY_DIRECTIVE,           'PASS_PROPERTY   ');
    add_to_table(PASSWORD_DIRECTIVE,                'PASSWORD        ');
    add_to_table(PERMIT_NO_ASSERTION_DIRECTIVE,     'PERMIT_NO_ASSERT');
    add_to_table(PICK_DIRECTIVE,                    'PICK            ');
    add_to_table(PRIMITIVE_DIRECTIVE,               'PRIMITIVE       ');
    add_to_table(PROPERTY_DIRECTIVE,                'PROPERTY_FILE   ');
    add_to_table(PRINT_WIDTH_DIRECTIVE,             'PRINT_WIDTH     ');
    add_to_table(READ_ALL_UDIRS_DIRECTIVE,          'READ_ALL_UDIRS  ');
    add_to_table(REPORT_DIRECTIVE,                  'REPORT          ');
    add_to_table(REPORT_UNKASSERT_DIRECTIVE,        'REPORT_UNKASSERT');
    add_to_table(ROOT_DIRECTIVE,                    'ROOT_DRAWING    ');
    add_to_table(SEPARATE_COMPILE_DIRECTIVE,        'SEPARATE_COMPILE');
    add_to_table(SHADOW_ROOT_DIRECTIVE,             'SHADOW_ROOT     ');
    add_to_table(SHAREABLE_DIRECTIVE,               'SHAREABLE       ');
    add_to_table(SINGLE_LEVEL_DIRECTIVE,            'SINGLE_DRAWING  ');
    add_to_table(SUPPRESS_DIRECTIVE,                'SUPPRESS        ');
    add_to_table(TOKENIZE_PARAMS_DIRECTIVE,         'TOKENIZE_PARAMS ');
    add_to_table(TEXT_MACRO_DIRECTIVE,              'TEXT_MACRO_FILE ');
    add_to_table(WARN_DIRECTIVE,                    'WARNINGS        ');
       
    output_file[CMPLST_FILE    ] := enter_name('LIST            ');
    output_file[CMPEXP_FILE    ] := enter_name('EXPAND          ');
    output_file[CMPERR_FILE    ] := enter_name('ERROR           ');
    output_file[CMPSYN_FILE    ] := enter_name('SYNONYM         ');
    output_file[CHIPS_FILE     ] := enter_name('CHIPS           ');
    output_file[CMPHIER_FILE   ] := enter_name('HIERARCHICAL    ');

    found_debug_password := FALSE;
    found_root_context_password := FALSE;
    ROOT_specified_in_command_line := FALSE;
    TYPE_specified_in_command_line := FALSE;
    CONTEXT_specified_in_command_line := FALSE;



    debug_directives := [DEBUG_DIRECTIVE,
                         DEBUG_AT_PATH_DIRECTIVE,
                         DEBUG_CONTROL_DIRECTIVE];

    one_time_directives := [succ(FIRST_DIRECTIVE)..pred(LAST_DIRECTIVE)] -
                           debug_directives -
                           [DIRECTORY_DIRECTIVE,
			    EXPANSION_RULES_DIRECTIVE,
                            FILTER_PROPERTY_DIRECTIVE,
                            LIBRARY_DIRECTIVE,
                            MASTER_LIBRARY_DIRECTIVE,
                            OUTPUT_DIRECTIVE,
                            PASS_PROPERTY_DIRECTIVE,
                            PICK_DIRECTIVE,
                            PRIMITIVE_DIRECTIVE,
                            PROPERTY_DIRECTIVE,
                            REPORT_DIRECTIVE,
                            SUPPRESS_DIRECTIVE,
                            TEXT_MACRO_DIRECTIVE];

    file_name_directives := [CONFIG_FILE_DIRECTIVE,
                             DIRECTORY_DIRECTIVE,
                             MASTER_LIBRARY_DIRECTIVE,
                             PROPERTY_DIRECTIVE,
                             TEXT_MACRO_DIRECTIVE];

    root_macro_name := nullstring;
    context_name := nullstring;

    tokenize_params := TRUE;

    single_level_compile := FALSE;
    default_filter_value := FALSE;
    page_synonym := TRUE;

    suppress_errors := [];
    display_warnings := true;
    display_oversights := true;
    max_errors := 1000;
    left_to_right := false;
    display_error_doc := TRUE;

    expansion_connectivity := LOGIC_connectivity;
    specified_compile_type := enter_name(LOGIC_PREFIX);


    configuration_file := make_default_file_name(DISTRIBUTION_DIR,
                                                 DIST_LANGUAGE,
                                                 DEFAULT_CONFIG_FILE);
    standard_property_file := make_default_file_name(DISTRIBUTION_DIR,
                                                     DIST_LANGUAGE,
                                                     DEFAULT_PROP_FILE);

    property_file := NIL;

    PrintCmpExp  := FALSE;
    PrintChips   := FALSE;
    PrintCmpSyn  := FALSE;
    special_compile := FALSE;

    files_to_generate := [CMPEXP_FILE, CMPSYN_FILE];
    property_enum := INIT_PROPERTY_ENUM;

  end { init_directives } ;


  procedure init_lexical_analyzer;
    { initialize the lexical analyzer }
    var
      i: radix_range;     { index into the table of radix digit sizes (bits) }
  begin
    id.next := NIL;
    sy := nullsy;
    const_val := 0;
    const_width := 0;
    read_state := FINIT;
    stack_top := 0;  parse_stack_pointer := 1;
    line_pos := 0;
    last_sym_pos := 0;
    upper_case_strings := TRUE;
    how_to_parse := PARSE_SEPARATELY;
    copy_input := FALSE;
    current_pos := 0;
    copy_pos := 0;
    current_file := UNKNOWN_FILE;
    parse_SCALDconstants := TRUE;

    create_a_string(lex_string,   MAX_STRING_LENGTH);
    create_a_string(input_buffer, MAX_STRING_LENGTH);
    create_a_string(instring,     MAX_STRING_LENGTH);
    create_a_string(last_string,  MAX_STRING_LENGTH);
    instring^[0] := chr(0);  last_string^[0] := chr(0);


    subrangesy := DOTDOTSY;  fieldsy := COLON;

    for i := MIN_RADIX to MAX_RADIX do
      radix_width[i] := round(ln(i) / 0.6931471805 + 0.4);   { ceil log2(i) }
  end { init_lexical_analyzer } ;


  procedure init_constants;
    { init some global constants }
  begin
    digits := ['0'..'9'];
    letters := ['A'..'I','J'..'R','S'..'Z'];     { EBCDIC compatibility }
    lower_case := ['a'..'i','j'..'r','s'..'z'];  { EBCDIC compatibility }
    identifier_chars := letters + digits + ['_'];
  end { init_constants } ;


  procedure init_symbols;
    { initialize symbol constants and CONFIGURE directive }
  begin
    configure_specifiers[CONFIGURE_SUBRANGE      ] :=
                                               enter_name('SUBRANGE        ');
    configure_specifiers[CONFIGURE_BIT_ORDERING  ] :=
                                               enter_name('BIT_ORDERING    ');
    configure_specifiers[CONFIGURE_LOW_ASSERTED  ] :=
                                               enter_name('LOW_ASSERTION   ');
    configure_specifiers[CONFIGURE_HIGH_ASSERTED ] :=
                                               enter_name('HIGH_ASSERTION  ');
    configure_specifiers[CONFIGURE_NEGATION      ] :=
                                               enter_name('NEGATION        ');
    configure_specifiers[CONFIGURE_NAME_PREFIX   ] :=
                                               enter_name('NAME_PREFIX     ');
    configure_specifiers[CONFIGURE_GENERAL_PREFIX] :=
                                               enter_name('GENERAL_PREFIX  ');
    configure_specifiers[CONFIGURE_CONCATENATION ] :=
                                               enter_name('CONCATENATION   ');

    signal_negation_symbol         := MINUS;
    signal_negation_char           := '-';

    signal_is_asserted_low_symbol  := ASTERISK;
    signal_is_asserted_low_char    := '*';

    signal_is_asserted_high_symbol := NULLSY;
    signal_is_asserted_high_char   := chr(255);

    name_property_prefix_symbol    := EXCLAMATION;
    name_property_prefix_char      := '!';

    general_property_prefix_symbol := BACKSLASH;
#if SVS
    general_property_prefix_char   := '\\';
#else
    general_property_prefix_char   := '\';
#endif

    concatenation_symbol           := COLON;
    concatenation_char             := ':';

    class_name_separator_symbol    := DOLLAR;

    signal_name_terminators := [general_property_prefix_char,
                                signal_is_asserted_low_char,
                                concatenation_char,
                                '<', chr(EOL)];

  end { init_symbols } ;


  procedure init_file_types;
    { initialize table of file types }
  begin
    { The ALLOWED_DIRECTORY element is the file type for the directory
      that contains the primitives that are to be output.  For example,
      when COMPILE SIM is specified, the ALLOWED_DIRECTORY type is set
      to SIM_DIR.  }

    file_type_list[LOGIC_DIRECTORY    ] := enter_name('LOGIC_DIR       ');
    file_type_list[ALLOWED_DIRECTORY  ] := enter_name('LOGIC_DIR       ');
    file_type_list[PROPERTY_ATTRIBUTES] := enter_name('ATTRIBUTES      ');
    file_type_list[CONFIGURATION_SPEC ] := enter_name('CONFIGURATION   ');

    directory_file_types := [LOGIC_DIRECTORY, ALLOWED_DIRECTORY];
  end { init_file_types } ;


  procedure init_scalars;
    { initialize standard scalars }
  begin
    column := 0;     { current output column }
  end { init_scalars } ;


  procedure init_free_lists;
    { initialize lists of "free" structures }
  begin
    free_properties := NIL;
    free_pin := NIL;
    free_signal := NIL;
    free_temp_prop := NIL;
  end { init_free_list } ;


  procedure init_special_names;
    { initialize some special names }
  begin
    PASS_string := make_and_enter_string('PASS            ');
    FILTER_string := make_and_enter_string('FILTER          ');
  end { init_special_names } ;


  procedure init_built_in_identifier_names;
    { initialize predefined identifiers }
    var
      i: name_table_range;       { index into the name table }
      j: hash_string_range;      { index into the string table }
  begin
    for i := 0 to NAME_TABLE_SIZE do name_table[i] := NIL;
    for j := 0 to HASH_STRING_TABLE_SIZE do string_table[j] := NIL;

    UNDEFINED_identifier      := enter_name('UNDEFINED       ');
    ALL_identifier            := enter_name('ALL             ');

    ABBREV_prop_name          := enter_name('ABBREV          ');
    PATH_prop_name            := enter_name('PATH            ');
    PATH_prop_name^.kind := [RESERVED] + default_attributes;
    SIZE_prop_name            := enter_name('SIZE            ');
    SIZE_prop_name^.kind := [RESERVED] + default_attributes;
    TIMES_prop_name           := enter_name('TIMES           ');
    TIMES_prop_name^.kind := [RESERVED] + default_attributes;
    EXPR_prop_name            := enter_name('EXPR            ');
    TITLE_prop_name           := enter_name('TITLE           ');
    PART_NAME_prop_name       := enter_name('PART_NAME       ');
    VERSION_prop_name         := enter_name('VERSION         ');
    SIG_NAME_prop_name        := enter_name('SIG_NAME        ');
    AUTO_GEN_prop_name        := enter_name('AUTO_GEN        ');
    SCOPE_prop_name           := enter_name('SCOPE           ');
    NO_WIDTH_prop_name        := enter_name('NOWIDTH         ');
    NO_BUBBLE_prop_name       := enter_name('NOBUBBLE        ');
    BUBBLED_prop_name         := enter_name('BUBBLED         ');
    OUTPUT_TYPE_prop_name     := enter_name('OUTPUT_TYPE     ');
    NO_ASSERT_prop_name       := enter_name('NOASSERT        ');
    REPLICATION_prop_name     := enter_name('REP             ');
    TERMINAL_prop_name        := enter_name('TERMINAL        ');
    NEEDS_NO_SIZE_prop_name   := enter_name('NEEDS_NO_SIZE   ');
    COMMENT_BODY_prop_name    := enter_name('COMMENT_BODY    ');
    HAS_FIXED_SIZE_prop_name  := enter_name('HAS_FIXED_SIZE  ');
    BODY_TYPE_prop_name       := enter_name('BODY_TYPE       ');
    NET_ID_prop_name          := enter_name('NN              ');
    XY_prop_name              := enter_name('XY              ');
    DIRECTORY_prop_name       := enter_name('DIR             ');
    ROTATION_prop_name        := enter_name('ROT             ');
    VER_prop_name             := enter_name('VER             ');
    CONTEXT_prop_name         := enter_name('CONTEXT         ');
    WARNING_prop_name         := enter_name('WARNING         ');
    OVERSIGHT_prop_name       := enter_name('OVERSIGHT       ');
    ERROR_prop_name           := enter_name('ERROR           ');
    DRAWING_prop_name         := enter_name('DRAWING         ');
    PAGE_prop_name            := enter_name('PAGE            ');
    TYPE_prop_name            := enter_name('TYPE            ');
    PATH_NAME_prop_name       := enter_name('PATH_NAME       ');
    TIME_prop_name            := enter_name('TIME            ');

    SIGNAL_prop_name          := enter_name('SIGNAL          ');
    PIN_prop_name             := enter_name('PIN             ');
    BODY_prop_name            := enter_name('BODY            ');

    PRIM_connectivity         := enter_name('PRIM_CONN       ');
    PART_connectivity         := enter_name('PART_CONN       ');
    LOGIC_connectivity        := enter_name('LOGIC_CONN      ');
    TIME_connectivity         := enter_name('TIME_CONN       ');
    SIM_connectivity          := enter_name('SIM_CONN        ');
    LOGIC_compile_type        := enter_name('LOGIC           ');
 
    null_name                 := enter_name('                ');
    LOCAL_specifier           := enter_name('LOCAL           ');
    GLOBAL_specifier          := enter_name('GLOBAL          ');
    ON_specifier              := enter_name('ON              ');
    OFF_specifier             := enter_name('OFF             ');
    L_to_R_specifier          := enter_name('LEFT_TO_RIGHT   ');
    R_to_L_specifier          := enter_name('RIGHT_TO_LEFT   ');
    X_FIRST_identifier        := enter_name('X_FIRST         ');
    X_STEP_identifier         := enter_name('X_STEP          ');
    X_identifier              := enter_name('X               ');
    X_identifier^.kind := [RESERVED, IS_PARAMETER];

    S_identifier              := enter_name('S               ');
    D_identifier              := enter_name('D               ');
    I_identifier              := enter_name('I               ');
    R_identifier              := enter_name('R               ');

    { RESERVED and UNRESERVED are non-reserved keywords, and are
      therefore not given the KEY_WORD name_type.  They are parsed as
      identifiers, and the id.name is checked against the following values. }

    RESERVED_key_name         := enter_name('RESERVED        ');
    UNRESERVED_key_name       := enter_name('UNRESERVED      ');
  end { init_built_in_identifier_names } ;


  procedure init_signal_syntax;
    { initialize the signal syntax descriptor }
  begin
    syntax_specifier_names[negation_specifier ]:=enter_name('NEGATION        ');
    syntax_specifier_names[name_specifier     ]:=enter_name('NAME            ');
    syntax_specifier_names[assertion_specifier]:=enter_name('ASSERTION       ');
    syntax_specifier_names[subscript_specifier]:=enter_name('SUBSCRIPT       ');
    syntax_specifier_names[null_specifier     ]:=enter_name('                ');
    syntax_specifier_names[property_specifier ]:=enter_name('GENERAL_PROPS   ');

    signal_syntax_table[1] := NEGATION_SPECIFIER;
    signal_syntax_table[2] := NAME_SPECIFIER;
    signal_syntax_table[3] := SUBSCRIPT_SPECIFIER;
    signal_syntax_table[4] := ASSERTION_SPECIFIER;
    signal_syntax_table[5] := PROPERTY_SPECIFIER;
  end { init_signal_syntax } ;
  
  

begin { init }


  { NOTE: these initialization routines are order sensitive.  The routines
          called first are those that use no other "constant" variables
          for initializations.  The other routines do, and some of them
          must be called in this specific order.  Check very carefully
          before changing any of this. }

  { these initializations do NOT rely on other initializations }

  init_free_lists;    { this must be first because of NEW recording }

  init_sets;
  init_string_structures;
  init_constants;
  init_scalars;

  { these initializations rely on other initializations }

  init_built_in_identifier_names;
  init_keywords;
  init_errors;
  init_signal_syntax;
  init_directives;
  init_lexical_analyzer;
  init_symbols;
  init_file_types;
  init_special_names;
  


  if not rewrite_file(monitor, MONITOR_FILE_NAME) then
    halt_with_status(FATAL_COMPLETION);

  write(monitor, WELCOME_MESSAGE);
  dump_string(monitor, vversion);
  writeln(monitor);
  writeln(monitor, COPYRIT_MESSAGE);
end { init } ;

