
(**)     { ------- lexical analyzer ------- }


procedure insymbol;
  { parse the next token from the input string 
    }
  type
    line_read_type = (NORMAL, CONTINUATION);

  var
    ch,                             { last character read in }
    next: char;                     { the next character in the input stream }
    looking_for_string,             { TRUE if reading in a string }
    done: boolean;                  { TRUE when symbol has been parsed }
    upper_case_on_input: boolean;   { TRUE if upper casing of all characters }



  function get_line(which: line_read_type): boolean;
    { read the next line from the input file.  Return TRUE if a string
      was popped from the stack instead of reading in another line.
      If a string was popped, the state of the lexical analyzer was
      restored from stack.  Otherwise, the character returned is space.
      Don't read in a new line if currently reading a string. }
   

    procedure read_a_line(var f: textfile);
      { read a line from the specified input file }
      var
        i: string_range;      { index into the parse string }
    begin
      while not eof(f) and eoln(f) do readln(f);

      i := 0;
      while not eof(f) and not eoln(f) do
        begin
          read(f, ch);  
          if i < MAX_STRING_LENGTH then
            begin  i := i + 1;  instring^[i] := ch;  end
          else
            begin
              instring^[0] := chr(MAX_STRING_LENGTH);
              error(53 { input line length exceeded });
              while not eof(f) and not eoln(f) do get(f);
            end;
        end;
      if eof(f) then ch := chr(EOL) else ch := ' ';
      instring^[0] := chr(i);  
    end { read_a_line } ;


  begin { get_line }
    get_line := FALSE;

    if parse_stack_pointer > 1 then       { pop the stack }
        ch := chr(EOL)    { end of string }

    { don't allow strings to cross lines }

    else if looking_for_string and (which = NORMAL) then
      ch := chr(EOL)

    else
      begin
        case current_file of
          DIRECTIVES_FILE:  read_a_line(infile);
          STANDARD_FILE:    read_a_line(CmpStan);
          UNKNOWN_FILE:     error(187 {assertion });
        end;
        line_pos := 0;  read_state := FINPUT;
      end;

  end { get_line } ;


  procedure getchar(var ch: char);
    { read the next char from the input buffer }
    var
      need_a_char: boolean;      { TRUE if we need to read another char }
  begin

    repeat
      if read_state = FGOT_CHAR then
        begin  ch := last_char;  need_a_char := FALSE;  end
      else
        if (read_state = FINIT) or (line_pos >= ord(instring^[0])) then 
          need_a_char := get_line(NORMAL)     { TRUE if we popped a string }
        else
          begin
            line_pos := line_pos + 1;
            ch := instring^[line_pos];
            if (ch = CONTINUATION_CHAR) and (line_pos = ord(instring^[0])) then
              begin
                need_a_char := get_line(CONTINUATION);
                need_a_char := TRUE;    { read past the ' ' at EOLN }
              end
            else
              begin
                { eat all tabs }

                if ch = chr(TAB_char) then ch := ' '

                { check for legal characters }

                else if not (ch IN legal_chars) then
                  begin
                    error(32 { illegal character in input });  ch := ' ';
                  end

                { upper case input if OK }

                else if (ch IN lower_case) and upper_case_on_input then
                  ch := chr(ord(ch) - ord('a') + ord('A'));

                need_a_char := FALSE;
              end;
          end;
    until not need_a_char;

    read_state := FINPUT;
    last_char := ch;

  end { getchar } ;
        

  function nextchar: char;
    { get the next char. Set a flag indicating that the next char was read }
  begin
    getchar(ch);  nextchar := ch;

    { set look ahead if this isn't the end of the line }

    if ch <> chr(EOL) then read_state := FGOT_CHAR;
  end { nextchar } ;



(**)     { ------- scan for an identifier ------- }


  procedure get_identifier;
    { read in an identifier }
    var
      i: 0..ID_LENGTH;            { index into the identifier }
      temp: alpha;                { identifier being parsed }
  begin
    temp := NULL_ALPHA;  id.name := NIL;

    if parse_stack_pointer < stack_top then fix_parse_stack;

    i := 0;  sy := IDENT;
    repeat
      if i >= ID_LENGTH then
        begin
          error(41 { identifier length exceeded });
          while (ch IN identifier_chars) do getchar(ch);
        end
      else
        begin  
          i := i + 1;  temp[i] := ch;  getchar(ch);
        end;
    until not (ch IN identifier_chars);

    if ch <> chr(EOL) then read_state := FGOT_CHAR;
    
        id.name := enter_name(temp);

        { check for a key word if not just copying input }

        if (KEY_WORD IN id.name^.kind) then
          if id.name^.sy IN allowed_key_words then sy := id.name^.sy;



  end { get_identifier } ;

    
(**)     { ------- scan for constant ------- }


  procedure get_constant;
    { read in one of three different constant types }
    var
      new_radix: natural_number;     { radix specified in constant }


    procedure skip_to_end_of_constant(number_radix: radix_range);
      { skip to the end of the constant;  error recovery }
    begin
      while ch in valid_chars[number_radix] do getchar(ch);
    end { skip_to_end_of_constant } ;


    function build_number(radix: radix_range): natural_number;
      { build a number with the specified radix }
      var
        temp: natural_number;      { value of the function to be returned }
        next_digit: 0..MAX_RADIX;  { numeric value of current digit }
    begin
      temp := 0;  const_width := 0;
      repeat
        const_width := const_width + 1;
        if ch <= '9' then  next_digit := ord(ch) - ord('0')
                     else  next_digit := ord(ch) - ord('A') + 10;

        if (temp > MAXINT DIV radix) or 
           ((temp = MAXINT DIV radix) and
            (next_digit > MAXINT MOD radix)) then
          begin  
            error(24 { ovf });
            skip_to_end_of_constant(radix);
          end
        else
          begin  temp := radix * temp + next_digit;  getchar(ch);  end;
      until not (ch IN valid_chars[radix]);

      const_width := const_width * radix_width[radix];

      build_number := temp;
    end { build_number } ;


  begin { get_constant }
    sy := CONSTANT;
    const_val := build_number(10);

    if parse_SCALDconstants then
      if ch in letters then
        begin
          repeat

            getchar(ch);
          until not (ch in letters);
        end

      else
        begin
          if ch = '#' then
            begin
              new_radix := const_val;
              if (new_radix < min_radix) or (new_radix > max_radix) then
                begin  error(61 { out of range });  new_radix := 10;  end;

              getchar(ch);
              const_val := build_number(new_radix);

              sy := SIGNALCONST;
            end;

          if ch = '(' then    { width specification }
            begin
              getchar(ch);
              const_width := build_number(10);

              if (const_width <= 0) or (const_width > max_bit_value) then
                begin  error(44 { invalid width });  const_width := 1;  end;

              if ch = ')' then getchar(ch) else error(7 { expected ) });

              sy := SIGNALCONST;
            end;
        end;

    read_state := FGOT_CHAR;

  end { get_constant } ;


(**)     { ------- scan for string ------- }


  procedure get_string(stopper: char);
    { read a string }
    var
      len: string_range;   { length of the string read in }
      done: boolean;       { TRUE when end of the string has been found }
  begin
    len := 0;  done := FALSE;  looking_for_string := TRUE;
#if UNIX
    if not upper_case_strings then upper_case_on_input := FALSE;
#endif
    repeat
      getchar(ch);
      if ch = stopper then
        if nextchar = stopper then getchar(ch) else done := TRUE;

      if (ch = chr(EOL)) and not done then
        begin  error(89 { string not closed });  done := TRUE;  end;

      if not done then
        if len >= MAX_STRING_LENGTH then
          begin error(22 { string length exceeded });
            while (ch <> stopper) and (ch <> chr(EOL)) do getchar(ch);
          end
        else
          begin len := len + 1;  input_buffer^[len] := ch;  end;
    until done;

    input_buffer^[0] := chr(len);
    sy := STRINGS;
    looking_for_string := FALSE;
    upper_case_on_input := TRUE;

    lex_string := enter_string(input_buffer);

  end { get_string } ;


(**)     { ------- main lexical analyzer ------- }


begin { insymbol }
  looking_for_string := FALSE;
  upper_case_on_input := TRUE;
  copy_pos := current_pos;

  if read_state = finput then last_sym_pos := line_pos
  else
    if line_pos >= 1 then last_sym_pos := line_pos-1
                     else last_sym_pos := 1;

  repeat
    done := TRUE;
    getchar(ch);
    while ch = ' ' do getchar(ch);  current_pos := copy_pos;

    if ch = chr(EOL) then sy := ENDOFDATASY
    else 
      case ch of
        '!':  sy := EXCLAMATION;
        '"':  get_string(ch);
        '#':  sy := SHARP;
        '$':  sy := DOLLAR;
        '%':  sy := PERCENT;
        '&':  sy := AMPERSAND;
       '''':  get_string(ch);
        '(':  sy := LPAREN;
        ')':  sy := RPAREN;
        '*':  sy := ASTERISK;
        '+':  sy := PLUS;
        ',':  sy := COMMA;
        '-':  sy := MINUS;
        '.':  if nextchar = '.' then sy := DOTDOTSY else sy := PERIOD;
        '/':  sy := SLASH;
        '0','1','2','3','4','5','6','7','8','9':  get_constant;
        ':':  if nextchar = ':' then sy := COLONCOLONSY else sy := COLON;
        ';':  sy := SEMI;
        '<':  begin
                next := nextchar;
                if next = '=' then sy := LESY
                else if next = '>' then sy := NESY
                else sy := LESSTHAN;
              end;
        '=':  sy := EQUAL;
        '>':  begin
                next := nextchar;
                if nextchar = '=' then sy := GESY
                else sy := GREATERTHAN;
              end;
        '?':  sy := QUESTION;
{       '@':  sy := ATSY;      this symbol is not used: @=^ in EBCDIC! }
        'A','B','C','D','E','F','G','H','I','J','K','L','M','N',
        'O','P','Q','R','S','T','U','V','W','X','Y','Z':  get_identifier;
        '[':  sy := LBRACKET;
#if SVS
        '\\': sy := BACKSLASH; 
#else
        '\': sy := BACKSLASH;
#endif
        ']':  sy := RBRACKET;
        '^':  sy := CIRCUMFLEX;
        '_':  sy := UNDERBAR;
        '`':  sy := ACCENTGRAVE;
        '{':  begin
                repeat
                  getchar(ch)
                until (ch='}')  or (ch=chr(EOL));

                if ch = chr(EOL) then error(34 { comment not closed });
                done := FALSE;
              end;
        '|':  sy := VERTICALBAR;
        '}':  begin  error(20 { unmatched symbol });  done := FALSE;  end;
        '~':  sy := TILDA;
        OTHERWISE
              error(23 { illegal character in input });
      end;
  until done;

  if parse_stack_pointer < stack_top then fix_parse_stack;

  if sy IN [COLONCOLONSY,DOTDOTSY,LESY,GESY,NESY] then read_state := FINPUT;

end { insymbol } ;

