 procedure add_to_bit_union(source: subscript_ptr; var dest: subscript_ptr);
  { Enters the bits mentioned in the source list into the dest list.
    Produces a subscript list that is in canonical order (ascending
    if left_to_right, descending if not) and which is as merged
    as possible (each representing a disjoint non-contiguous subrange). 
    Dest should be in this condition to start with.  (That assumption
    is checked.)  This is useful for keeping track of a set of
    bits. }
  var
    current_source: subscript_ptr;  { current element of source list }


  function test_canonical_order(subs: subscript_ptr): boolean;
    { Tests subscript list for disjoint elements in
      canonical order.  Returns FALSE if wrong. }
    var
      current: subscript_ptr;   { current element in list }
      first: boolean;           { TRUE iff first element }
      last_right: bit_range;    { previous right_index }
      ok: boolean;              { answer to be returned } 
  begin
    ok := TRUE;  first := TRUE;    current := subs;
    if left_to_right then
      while (current <> NIL) and ok do with current^ do
	begin
	  ok := (left_index <= right_index);
  
	  if first then first := FALSE
	  else if ok then
	    if last_right = MAX_BIT_VALUE then ok := false
	    else ok := (left_index > (last_right + 1));
  
	  last_right := right_index;
	  current := next;
	end
    else
      while (current <> NIL) and ok do with current^ do
	begin
	  ok := (left_index >= right_index);
  
	  if first then first := FALSE
	  else if ok then
	    if last_right = 0 then ok := false
	    else ok := (left_index < (last_right - 1));
  
	  last_right := right_index;
	  current := next;
	end;
    test_canonical_order := ok;
  end { test_canonical_order } ;


  procedure insert_subscript_element(left, right: bit_range;
				     var s: subscript_ptr);

    { inserts a subscript element with indicated indices into
      a list immediately ahead of s^.   S is returned pointing
      to the new element.

      Ex:
	insert_subscript_element(head_of_list);
	  or
	insert_subscript_element(parent_of_insertee^.next); }
  begin
    new_subscript(s);
    s^.left_index := left;
    s^.right_index := right;
  end { insert_subscript_element } ;


  procedure delete_subscript_element(var s: subscript_ptr);

    { deletes s^ from the list, setting s to the former s^.next. 

      Ex:
	delete_subscript_element(head_of_list);
	  or
	delete_subscript_element(parent_of_deletee^.next); }
    var
      old_s: subscript_ptr; { element to be disposed }
  begin
    if s = NIL then assert(112 { a no-no })
    else
      begin
	old_s := s;
	s := s^.next;
	old_s^.next := NIL;
	release_subscript(old_s);
      end;
  end { delete_subscript_element } ;


  procedure merge_into_subscript(left, right: bit_range;
				 var dest: subscript_ptr);
    { Merges a subrange into a subscript which is in canonical 
      order.  Merges elements whenever possible. }
    var
      current: subscript_ptr;   { current element of dest }
      parent: subscript_ptr;    { parent of current }
      found: boolean;           { TRUE when we find an element with
				  left_index to the right of left }
      done: boolean;            { TRUE when we are done merging }
      new_one: subscript_ptr;   { element containing left, right 
				  (may or may not be new) }
  begin
    { find first element with left_index to the right of left }

    current := dest;  parent := NIL;  found := FALSE;
    while (current <> NIL) and not found do 
      if left_to_right then
	if current^.left_index > left then found := TRUE
	else 
	  begin
	    parent := current;
	    current := current^.next;
	  end
      else { not left_to_right }
	if current^.left_index < left then found := TRUE
	else 
	  begin
	    parent := current;
	    current := current^.next;
	  end;

    { insert new element or merge range with parent of 
      found element }

    new_one := NIL;
    if parent = NIL then 
      begin
	insert_subscript_element(left, right, dest);
	new_one := dest;
      end
    else
      begin
	if left_to_right then
	  if parent^.right_index >= (left - 1) then
	    begin
	      if parent^.right_index < right then
		parent^.right_index := right;  
	      new_one := parent;
	    end
	  else { new range is disjoint with its parent }
	    begin
	      insert_subscript_element(left, right, parent^.next);
	      new_one := parent^.next;
	    end
	else { not left_to_right }
	  if parent^.right_index <= (left + 1) then
	    begin
	      if parent^.right_index > right then
		parent^.right_index := right;  
	      new_one := parent;
	    end
	  else { new range is disjoint with its parent }
	    begin
	      insert_subscript_element(left, right, parent^.next);
	      new_one := parent^.next;
	    end
      end;

    { merge to the right as far as possible }

    done := FALSE;
    if new_one <> NIL then
      while (new_one^.next <> NIL) and not done do 
	with new_one^ do
	  if left_to_right then
	    if (next^.left_index - 1) <= right_index then
	      begin
		if next^.right_index > right_index then
		  right_index := next^.right_index;
		delete_subscript_element(next);
	      end
	    else done := TRUE
	  else { not left_to_right }
	    if (next^.left_index + 1) >= right_index then
	      begin
		if next^.right_index < right_index then
		  right_index := next^.right_index;
		delete_subscript_element(next);
	      end
	    else done := TRUE;
  end { merge_into_subscript } ;


begin { add_to_bit_union }
  if debug_36 then
    begin
      write(OutFile,'add_to_bit_union(source=');
      dump_bit_subscript(outfile, source, VECTOR);
      write(OutFile,', dest=');
      dump_bit_subscript(outfile, dest, VECTOR);
      writeln(OutFile, ')');
    end;

  if not test_canonical_order(dest) then
    begin { fix it }
      assert(136 { we don't like it, but we can fix it});
      current_source := NIL;
      add_to_bit_union(dest, current_source);
      dest := current_source;
    end;
  current_source := source;
  while current_source <> NIL do with current_source^ do
    begin
      if left_to_right then 
	if left_index <= right_index then
	    merge_into_subscript(left_index, right_index, dest)
	else
	    merge_into_subscript(right_index, left_index, dest)
      else { right_to_left }
	if left_index >= right_index then
	    merge_into_subscript(left_index, right_index, dest)
	else
	    merge_into_subscript(right_index, left_index, dest);
      current_source := next;
    end;
  if debug_36 then
    begin
      write(outfile,'exit add_to_bit_union dest=');
      dump_bit_subscript(outfile, dest, VECTOR);
      writeln(outfile);
    end;
end { add_to_bit_union } ;


function bit_set_from_descriptor(s: signal_descriptor_ptr): subscript_ptr;
  { return a canonicalized (in order and merged) version of the 
    subscript }
  var
    dest: subscript_ptr;
begin
  dest := NIL;
  while s <> NIL do 
    begin
      add_to_bit_union(s^.bit_subscript, dest);
      s := s^.next;
    end;
  bit_set_from_descriptor := dest;
end { bit_set_from_descriptor } ;


function bit_set_from_subscript(s: subscript_ptr): subscript_ptr;
  { return a canonicalized (in order and merged) version of the 
    subscript }
  var
    dest: subscript_ptr;
begin
  dest := NIL;
  add_to_bit_union(s, dest);
  bit_set_from_subscript := dest;
end { bit_set_from_descriptor } ;


function bit_difference(subtrahend, subtractor: subscript_ptr): subscript_ptr;
  { return the set difference subtrahend - subtractor, given 2 canonical
    bit sets }
  var
    temp,result: subscript_ptr;
    left_bit: bit_range;


  procedure add_subrange(low,high: bit_range);
    { merge the subrange into the result }
  begin
    temp^.left_index := low;   { arbitrarily }
    temp^.right_index := high;
    add_to_bit_union(temp, result);
  end { add_subrange } ;


begin { bit_difference }
  if debug_36 then
    begin
      write(OutFile,'bit_difference(subtrahend=');
      dump_bit_subscript(outfile, subtrahend, VECTOR);
      write(OutFile,', subtractor=');
      dump_bit_subscript(outfile, subtractor, VECTOR);
      writeln(OutFile, ')');
    end;

  result := NIL;  temp := NIL;  new_subscript(temp);
  if left_to_right then
    begin
      if subtrahend <> NIL then left_bit := subtrahend^.left_index;
      while (subtrahend <> NIL) and (subtractor <> NIL) do with subtrahend^ do
	begin
	  if (left_bit < subtractor^.left_index) then
	    add_subrange(left_bit, 
	                 min(subtractor^.left_index - 1, right_index));

          if (right_index < subtractor^.right_index) then
	    begin
	      subtrahend := next;  left_bit := left_index;
	    end
	  else if (subtractor^.right_index < right_index) then
	    begin
	      left_bit := subtractor^.right_index + 1;
	      subtractor := subtractor^.next;
	    end
	  else
	    begin
	      subtrahend := next;  left_bit := left_index;
	      subtractor := subtractor^.next;
	    end;
	end;
    end
  else
    begin
      if subtrahend <> NIL then left_bit := subtrahend^.left_index;
      while (subtrahend <> NIL) and (subtractor <> NIL) do with subtrahend^ do
	begin
	  if (left_bit > subtractor^.left_index) then
	    add_subrange(left_bit, 
	                 max(subtractor^.left_index + 1, right_index));

          if (right_index > subtractor^.right_index) then
	    begin
	      subtrahend := next;  left_bit := left_index;
	    end
	  else if (subtractor^.right_index > right_index) then
	    begin
	      left_bit := subtractor^.right_index - 1;
	      subtractor := subtractor^.next;
	    end
	  else
	    begin
	      subtrahend := next;  left_bit := left_index;
	      subtractor := subtractor^.next;
	    end;
	end;
    end;

  if subtrahend <> NIL then 
    begin
      add_subrange(left_bit, subtrahend^.right_index);
      subtrahend := subtrahend^.next;
      add_to_bit_union(subtrahend, result);
    end;

  release_subscript(temp);

  if debug_36 then
    begin
      write(outfile,'bit_difference returns ');
      dump_bit_subscript(outfile, result, VECTOR);
      writeln(outfile);
    end;
  bit_difference := result;
end { bit_difference } ;


function bit_intersection(set1, set2: subscript_ptr): subscript_ptr;
  var
    diff, result: subscript_ptr;
begin
  if debug_36 then
    begin
      write(OutFile,'bit_intersection(');
      dump_bit_subscript(outfile, set1, VECTOR);
      write(Outfile, ', ');
      dump_bit_subscript(outfile, set2, VECTOR);
      writeln(OutFile, ')');
    end;

  diff := bit_difference(set1, set2);
  result := bit_difference(set1, diff);
  release_entire_subscript(diff);

  if debug_36 then
    begin
      write(outfile,'bit_intersection returns ');
      dump_bit_subscript(outfile, result, VECTOR);
      writeln(outfile);
    end;
  bit_intersection := result;
end { bit_intersection } ;
