function X_replicated_name(name: xtring; node: mtree_node_ptr; 
                           scope: scope_type): xtring;
  { return the name with prepended (<x_value>) in a string as needed.
    Rules --
      1. If scope <> LOCAL, then x_value = 0
      2. Otherwise, x_value := node^.x_value.
      3. If x_value = 0 then return the name unchanged }
  var
    temp: xtring;     { for concocting new name }
    x_value: longint; { x value to be used to concoct new name }
begin
  if (scope <> LOCAL) then x_value := 0
                      else x_value := node^.x_value;

  if x_value = 0 then X_replicated_name := name
  else
    begin
      create_a_string(temp, 
                      min(MAX_STRING_LENGTH,
                          2 + width_of_integer(x_value) + ord(name^[0])));
      temp^[0] := chr(0);
      if add_char_to_string(temp, '(') then ;
      if add_number_to_string(temp, x_value) then ;
      if add_char_to_string(temp, ')') then ;
      if not add_string_to_string(temp, name) then
        assert(235 { str overflow });
      X_replicated_name := enter_and_release_string(temp);
    end;
end { X_replicated_name } ;


procedure output_signal_scope(var f: pipe; scope: scope_type);
  { print the signal scope to the given file.  If the current
    line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a continuation
    character and continue on the next line. }
begin
  case scope of
    XINTERFACE:     pipe_dump_char(f, 'I');
    GLOBAL:        pipe_dump_char(f, 'G');
    DECLARED:      pipe_dump_char(f, 'D');
    OTHERWISE begin  { print nothing }  end;
  end;
end { output_signal_scope } ;


procedure output_bit_subscript(var f: pipe; bits: subscript_ptr;
                                      kind: signal_kind);
  { print the bit subscript (BITS) of the given kind (KIND).  If the current
    line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a continuation
    character and continue on the next line. }
  var
    SP: subscript_ptr;       { current subscript element }
    first: boolean;          { TRUE if no preceding comma to be printed }
begin
  if bits = NIL then
    if kind = UNDEFINED then
      begin
        pipe_dump_char(f, '<');
        pipe_dump_alpha(f, UNDEFINED_identifier^.name);
        pipe_dump_char(f, '>');
      end
    else if kind = VECTOR then
      pipe_dump_alpha(f, '<none>          ')
    else { it's a scalar, don't print anything }
  else
    if (kind = UNDEFINED) or (kind = SINGLE) then
      pipe_dump_alpha(f, '<BAD>           ')
    else
      begin
        SP := bits;  first := TRUE;
        pipe_dump_char(f, '<');
        while SP <> NIL do
          begin
            if first then first := FALSE
            else pipe_dump_char(f, ',');  

            pipe_dump_integer(f, SP^.left_index);

            if SP^.left_index <> SP^.right_index then
              begin
                pipe_dump_char(f, ':');
                pipe_dump_integer(f, SP^.right_index);
              end;

            SP := SP^.next;
          end;

        pipe_dump_char(f, '>');
      end;
end { output_bit_subscript } ;


procedure output_left_and_right(var f: pipe;
                                       left, right: bit_range);
  { output a left and right subscript with the specified bits.  If the current
    line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a continuation
    character and continue on the next line. }
begin
  pipe_dump_char(f, '<');
  pipe_dump_integer(f, left);

  if left <> right then
    begin
      pipe_dump_char(f, ':');
      pipe_dump_integer(f, right);
    end;

  pipe_dump_char(f, '>');
end { output_left_and_right } ;


procedure output_signal_name(var f: pipe;
                             polarity: signal_polarity;
                             name: xtring; 
			     scope: scope_type;
                             node: mtree_node_ptr);
  { write the signal name to f with closing quote.
    If the current line is overflowed (MAX_OUTPUT_FILE_LENGTH), print a
    continuation character and continue on the next line. }
begin
  name := X_replicated_name(name, node, scope);
  if polarity = COMPLEMENTED then pipe_dump_char(f, '-');
  lookup_and_print_string_continue(f, name, expansion_string_dictionary);
end { output_signal_name } ;


procedure output_numbered_property(var f: pipe; 
				   name: name_ptr; text: xtring);
  { writes the property to f using expansion dictionaries }
begin
  if (name <> NIL) and (text <> NIL) then
    begin
      lookup_and_print_id_continue(f, name, expansion_id_dictionary);
      lookup_and_print_string_continue(f, text, expansion_string_dictionary);
    end;
end { output_numbered_property } ;


procedure output_expanded_property(var f: pipe; node: mtree_node_ptr;
                                   property_name: name_ptr;
                                   property_val: xtring);
  { output the property to f with its value expanded. Node defines the
    scope within which the value is to be expanded. }
  var
    exp_value: xtring;   { expanded value }
begin
  exp_value := expand_property_value_TMs(node, property_name, property_val);
  output_numbered_property(f, property_name, exp_value);
end { output_expanded_property } ;


procedure output_numbered_param_properties(var f: pipe;
                                           parms: property_ptr);
  { outputs the parameter body properties using the expansion dictionaries }
  var
    current: property_ptr;      { current element }
begin
  current := parms;
  if current <> NIL then
    if (current^.name = SIZE_prop_name) then
      begin
        pipe_dump_char(f, ' ');
        pipe_dump_integer(f, string_to_natural_number(current^.text));
        current := current^.next;
      end;
  while current <> NIL do with current^ do
    begin
      lookup_and_print_id_continue(f, name, expansion_id_dictionary);
      lookup_and_print_string_continue(f, text, expansion_string_dictionary);
      current := next;
    end;
end { output_numbered_param_properties } ;


procedure enter_numbered_property(name: name_ptr; text: xtring);
  { enter the name and non-expandable text into expansion
    dictionaries }
  var
    dummy: numbered_token_ptr;  { don't care what gets returned }
begin
  dummy := enter_numbered_id(name, expansion_id_dictionary);
  if not TM_prefix_in_string(text) then
    dummy := enter_numbered_string(text, expansion_string_dictionary);
end { enter_numbered_property } ;


procedure enter_numbered_property_list(prop: property_ptr);
  { enter the name and non-expandable text into expansion
    dictionaries for each element of the list }
  var
    p: property_ptr;            { current element of prop list }
    dummy: numbered_token_ptr;  { don't care what gets returned }
begin
  p := prop;
  while (p <> NIL) do
    begin
      dummy := enter_numbered_id(p^.name, expansion_id_dictionary);
      if not TM_prefix_in_string(p^.text) then
        dummy := enter_numbered_string(p^.text, expansion_string_dictionary);
      p := p^.next;
    end;
end { enter_numbered_property_list } ;


procedure output_heading(var f: pipe; pg: natural_number);
  { write the heading section of the expansion file.  pg gives the page
    number to use if undefined (is a natural_number as 0 is used to
    send a page containing error information regarding the drawing itself). }
  var
    tempval: xtring;        { for string representations of non-string values }
    tempprop: property_ptr; { found property }


  procedure output_fake_abbrev(var f: pipe);
    { concoct an ABBREV to prevent linker from emitting missing ABBREV
      oversight on an unfound drawing or non-graphical model }
  begin
    { Checking for page_being_compiled = 0 is a hack to insure that we
      don't emit an abbrev for the case where a non-graphical model is
      being emitted as an additional page to a primitive.  We only
      want to emit it for unfound drawings and purely non-graphical models,
      both of which get to this procedure with page_being_compiled = 0. }
      
    if page_being_compiled = 0 then
      begin
	output_numbered_property(f, ABBREV_prop_name, 
				 concoct_abbrev(root_macro_name));
	pipe_dump_CRLF(f);
      end;
  end { output_fake_abbrev } ;


begin { output_heading }
  output_numbered_property(f, TIME_prop_name, expansion_compile_time);
  pipe_dump_CRLF(f);
  output_numbered_property(f, DRAWING_prop_name, root_macro_name);
  pipe_dump_CRLF(f);
  if page_being_compiled = 0 then tempval := number_to_string(pg)
  else tempval := number_to_string(page_being_compiled);
  output_numbered_property(f, PAGE_prop_name,tempval);
  pipe_dump_CRLF(f);
  tempval := nullstring;
  copy_to_string(extension_being_compiled^.name, tempval);
  output_numbered_property(f, TYPE_prop_name, tempval);
  pipe_dump_CRLF(f);
  tempval := number_to_string(version_being_compiled);
  output_numbered_property(f, VER_prop_name, tempval);
  pipe_dump_CRLF(f);
  tempval := nullstring;
  if mtree_root <> NIL then
    if mtree_root^.macro <> NIL then
      if find_property(mtree_root^.macro^.properties, TITLE_prop_name, 
			 tempprop) then
	tempval := tempprop^.text;
  if tempval = nullstring then tempval := root_macro_name;
  output_numbered_property(f, TITLE_prop_name, tempval);
  pipe_dump_CRLF(f);
  tempval := nullstring;
  if mtree_root <> NIL then
    if mtree_root^.macro <> NIL then
      if find_property(mtree_root^.macro^.properties, ABBREV_prop_name, 
			 tempprop) then
        begin
          output_numbered_property(f, ABBREV_prop_name, tempprop^.text);
          pipe_dump_CRLF(f);
        end
      else
        begin 
	  { nothing -- it's missing and linker will concoct one and 
	    issue an oversight message }
        end
    else output_fake_abbrev(f)
  else output_fake_abbrev(f);
	
  tempval := number_to_string(num_warnings);
  output_numbered_property(f, WARNING_prop_name, tempval);
  pipe_dump_CRLF(f);
  tempval := number_to_string(num_oversights);
  output_numbered_property(f, OVERSIGHT_prop_name, tempval);
  pipe_dump_CRLF(f);
  tempval := number_to_string(num_errors);
  output_numbered_property(f, ERROR_prop_name, tempval);

  if mtree_root <> NIL then
    begin
      if mtree_root^.x_value <> 0 then
        begin
         tempval := number_to_string(mtree_root^.x_value);
          output_numbered_property(f, X_FIRST_identifier, tempval);
        end;
      if (mtree_root^.next = NIL) then
        if mtree_root^.uses_SIZE_property then
          tempval := search_id(SIZE_prop_name)  { X_STEP = SIZE }
        else tempval := nullstring { X_STEP defaults to 1 in the linker }
      else tempval := 
        number_to_string(mtree_root^.next^.x_value - mtree_root^.x_value);
      if tempval <> nullstring then
        output_numbered_property(f, X_STEP_identifier, tempval);
    end;

  pipe_dump_char(f, ';');
end { output_heading } ;


procedure enter_heading;
  { enter ids and strings destined for heading section of expansion
    file into the expansion file dictionaries.  When parsing
    will allow literals, this proc can be omitted. }
  var
    tempval: xtring;        { for string representations of non-string values }
    tempprop: property_ptr; { found property }
begin
  enter_numbered_property(TIME_prop_name, expansion_compile_time);
  enter_numbered_property(DRAWING_prop_name, root_macro_name);
  tempval := nullstring;
  copy_to_string(extension_being_compiled^.name, tempval);
  enter_numbered_property(TYPE_prop_name, tempval);
  tempval := number_to_string(version_being_compiled);
  enter_numbered_property(VER_prop_name, tempval);
  tempval := number_to_string(page_being_compiled);
  enter_numbered_property(PAGE_prop_name,tempval);
  tempval := nullstring;
  if mtree_root <> NIL then
    if mtree_root^.macro <> NIL then
      if find_property(mtree_root^.macro^.properties, TITLE_prop_name, 
			 tempprop) then
	tempval := tempprop^.text;
  if tempval = nullstring then tempval := root_macro_name;
  enter_numbered_property(TITLE_prop_name, tempval);
  tempval := nullstring;
  if mtree_root <> NIL then
    if mtree_root^.macro <> NIL then
      if find_property(mtree_root^.macro^.properties, ABBREV_prop_name, 
			 tempprop) then
	enter_numbered_property(ABBREV_prop_name, tempprop^.text);
  tempval := number_to_string(num_warnings);
  enter_numbered_property(WARNING_prop_name, tempval);
  tempval := number_to_string(num_oversights);
  enter_numbered_property(OVERSIGHT_prop_name, tempval);
  tempval := number_to_string(num_errors);
  enter_numbered_property(ERROR_prop_name, tempval);
end { enter_heading } ;


procedure dump_body_property_list(var f: pipe; list: property_ptr;
                                  context_node: mtree_node_ptr);
  { Dump all properties in the list as body properties. Context_node
    identifies the node within which the property was defined (and so
    defines the context within which its value will be expanded). }
var
    property: property_ptr;     { current property being output }
begin
  property := list;
  while property <> NIL do
    begin
      if not (DONT_OUTPUT in property^.name^.kind) or
         (IS_ET_CONTROL in property^.name^.kind) then
        output_expanded_property(f, context_node,
                                 property^.name, property^.text);
      property := property^.next;
    end;
end { dump_body_property_list } ;


procedure output_drawing_properties(var f: pipe);
  { output drawing body properties section of expand file }
  var
    property: property_ptr;            { current property }
    found_PART_NAME_property: boolean; { TRUE if PART_NAME property exists
                                         and its expanded value differs from
					 the body name }
    partname: xtring;                  { expanded PART_NAME property value }
begin
  { NOTE that if the drawing is being X replicated, that this does
    not attempt to do the replication for each value of X }

  found_PART_NAME_property := FALSE;

  if mtree_root = NIL then property := NIL
  else if mtree_root^.macro = NIL then property := NIL
  else property := mtree_root^.macro^.properties;

  if property <> NIL then pipe_dump_CRLF(f);


  while property <> NIL do
    begin
      { Note that all properties are output (even those with "filter" 
        attribute.  The final expansion file (or equivalent representation)
	must list all non-inherit(body) properties on the root drawing body, 
	so the linker must have them in order to do this. }

      if property^.name = PART_NAME_prop_name then
        begin
	  partname := expand_property_value_TMs(mtree_root, property^.name,
	                                        property^.text);
          if partname <> mtree_root^.macro_name then
	    found_PART_NAME_property := TRUE;
          output_numbered_property(f, property^.name, partname);
	end
      else
        output_expanded_property(f, mtree_root, 
	                            property^.name, property^.text);

      property := property^.next;
    end;

  if found_PART_NAME_property then
    output_numbered_property(f, BODY_NAME_prop_name, mtree_root^.macro_name);
  pipe_dump_char(f, ';');
end { output_drawing_properties } ;


procedure output_context(var f: pipe);
  { outputs the context being compiled to a pipe }
begin
  if specified_context <> NIL then pipe_dump_CRLF(f);
  output_numbered_param_properties(f, specified_context);
  pipe_dump_char(f, ';');
end { output_sepcomp_context } ;


procedure output_modules(var f: pipe);
  { output the list of modules.  p and f point to the same file -- either
    can be used. }
  var
    modl: module_list_ptr;          { current element }
    i: module_table_range;          { current bucket }
    max_count: natural_number;      { max number between CRs }
    count: natural_number;          { number printed since last CR }
begin
  max_count := min(8, max(1, MAX_OUTPUT_FILE_LENGTH div 16)); { heuristic }
  count := max_count; { force immediate CRLF on first module }
  with modules_in_page do for i := 0 to LAST_MODULE_TABLE_ENTRY do
    begin
      modl := table[i];
      while modl <> NIL do with modl^ do
        begin
          if count = max_count then
	    begin
	      pipe_dump_CRLF(f);  count := 1;
	    end
	  else count := count + 1;
	  pipe_dump_char(f, '&');
          pipe_dump_integer(f, number);          
        { output_numbered_token(f, auto_dir^, STRING_NUMBER); }
          output_numbered_token(f, drawing^, STRING_NUMBER);
          output_parameters(f, context);
          modl := next;
        end;
    end;
  pipe_dump_char(f, ';');
end { output_sepcomp_modules } ;


(**)     { ------- output leaf macros to expansion file ------- }


(******************************************************************)
(*  OUTPUT of EXPANSION FILE (page-at-a-time style)               *)
(*                                                                *)
(*  This routine outputs the design to the expansion file.  Each  *)
(*  primitive is output along with properties attached to it or   *)
(*  its signals.  This routine generates only an expansion file.  *)
(******************************************************************)


procedure output_expansion(var p: pipe; node: mtree_node_ptr);
  { output the primitive macro to the expansion file }
var
    formal_actual_pair: formal_actual_ptr;   { list of formals of macro }
    actual: propertied_CS_ptr;               { the actual signal }


  procedure print_body_properties(var f: pipe; node: mtree_node_ptr);
    { print the list of properties to the given file.  Check for 
      X-replication and modify the PATH property accordingly (prepend
      (<x_value>) if x_value <> 0). }
    var
      property: property_ptr;     { current property being output }
  begin
    { output the properties of the body. Leave inheritance for the linker }

    { It is assumed that node^.called_by and father^.macro are non-NIL, as
      this routine is only called on sons of mtree_root or sons of siblings
      of mtree_root. }


    output_numbered_property(f, PATH_prop_name,
      X_replicated_name(node^.called_by^.path, node^.father_node, LOCAL));

    property := node^.called_by^.properties;
    while property <> NIL do
      begin
        if not (DONT_OUTPUT in property^.name^.kind) or
           (IS_ET_CONTROL in property^.name^.kind) then
          output_expanded_property(f, node^.father_node,
                                      property^.name, property^.text);
        property := property^.next;
      end;
  end { print_body_properties } ;


  procedure output_formal(var f: pipe;
                          formal_actual_pair: formal_actual_ptr);
    { write the formal parameter to the output file along with its NWC
      spec and followed by the BUBBLED property (if bubbled).  In the
      future, this can be followed by all pin properties that apply to
      all bits (which can then be omitted from the properties put on
      the pieces of the actual).  This change would be strictly local
      to the compiler, as the linker is set up to do the right thing
      regardless. }


    procedure output_formal_subscript(var f: pipe; subscript: subscript_ptr;
                                      kind: signal_kind);
      { output the subscript to the given file.  If the KIND is UNDEFINED,
        output a "generic" subscript width.  This is a KLUDGE for the
        MEMORY primitive of the SIMULATOR!!! }
      var
        width: bit_range;        { width of the concatenated signal }
    begin
      if kind <> UNDEFINED then output_bit_subscript(f, subscript, kind)
      else
        begin
          width := width_of_propertied_CS(formal_actual_pair^.
                                                    actual_parameter^.signal);

          if width <= 1 then
            pipe_dump_alpha(f, '<0>             ')
          else
            if left_to_right then
              begin
                pipe_dump_alpha(f, '<0:             ');
                pipe_dump_integer(f, width-1);
                pipe_dump_char(f, '>');
              end
            else
              begin
                pipe_dump_char(f, '<');
                pipe_dump_integer(f, width-1);
                pipe_dump_alpha(f, ':0>             ');
              end;
        end;
    end { output_formal_subscript } ;


  begin { output_formal }
    if formal_actual_pair^.is_NWC_pin 
      then pipe_dump_char(f, OUTPUT_NWC_CHAR);
    if formal_actual_pair^.formal_parameter <> NIL then
      with formal_actual_pair^.formal_parameter^ do
        begin
          output_signal_name(f, defined_by^.polarity,
                                     defined_by^.signal^.name,
                                     XINTERFACE, node);
          output_formal_subscript(f, bit_subscript,
                                  defined_by^.kind);
        end
    else if formal_actual_pair^.pin_name <> NIL then
      with formal_actual_pair^.pin_name^ do
        begin
          output_signal_name(f, polarity, signal_name, XINTERFACE, node);
          output_formal_subscript(f, bit_subscript, kind);
        end
    else assert(175 { NIL formal param and pin name });

    { output BUBBLED property if pin is bubbled. NOTE: this algorithm
      is brain-damaged, as it puts out a bubble on NAC pins that are
      connected to low-asserted signals.  It should be fixed up to treat
      the bubble as just another pin property that ends up getting output
      here with the others. }

    if formal_actual_pair^.polarity = COMPLEMENTED then
      begin
        lookup_and_print_id_continue(f, BUBBLED_prop_name,
                                     expansion_id_dictionary);
        lookup_and_print_string_continue(f, nullstring,
                                         expansion_string_dictionary);
      end;

    { output pin properties affecting all bits of the pin }
    { (not het implemented) }

  end { output_formal } ;


  procedure output_actual(var f: pipe;
                          formal_actual_pair: formal_actual_ptr;
                          actual_list: propertied_CS_ptr);
    { write the actual parameter(s) to the output file }
    var
      current_prop,                       { current property of actual }
      prop_list: subscript_property_ptr;  { properties of the actual }
      actual: propertied_CS_ptr;          { current signal being output }
      pins_direction: -1..1;              { index direction for pin }
      pins_left_index,                    { current left index of pin }
      pins_right_index,                   { current right index of pin }
      left, right: bit_range;             { current bits of actual }
      width: bit_range;                   { width of current actual }
      delta_width: bit_range;             { width of a delta change in width }
      first: boolean;                     { TRUE if first to be printed }
      direction: -1..1;                   { increment direction }
      one_was_output: boolean;            { TRUE if an actual was output }
      rep_value: replication_range;       { replication value }


    procedure output_property_list(var f: pipe; prop_list: property_ptr);
      { output the given property list to the given file }
      var
        prop: property_ptr;      { current property in the list }
    begin
      prop := prop_list;
      while prop <> NIL do
        begin
          if not(DONT_OUTPUT in prop^.name^.kind) then
            output_numbered_property(f, prop^.name, prop^.text);
          prop := prop^.next;
        end;
    end { output_property_list } ;


    procedure print_signal_instance_and_props(instance: signal_instance_ptr;
                                              left, right: bit_range;
                                              var first: boolean;
                                              props: subscript_property_ptr);
      { print the signal instance with the given bits. }


      procedure print_constant(var f: pipe; instance: signal_instance_ptr);
        { print the constant represented by the given instance to the file.
          It is known that constants have subscripts always numbered from 0.
          It is further known that the instance's subscript nust be a simple
          subrange (it comes from find_base... }
        var
          signal_name: xtring;       { name of the signal }
          bit: bit_range;            { current bit of the instance }
          direction: -1..1;          { delta direction }
          is_complemented: boolean;  { TRUE if signal is complemented }
          done: boolean;             { TRUE when all bits of const output }


        procedure write_bit(bit: bit_range);
          { write the given bit to the output file. }
          var
            ch: char;       { constant character to be output }
        begin
          if left_to_right then ch := signal_name^[bit+1]
                           else ch := signal_name^[ord(signal_name^[0])-bit];

          if ch = '0' then
            output_numbered_token(f, expansion_0_string^, STRING_NUMBER)
          else
            output_numbered_token(f, expansion_1_string^, STRING_NUMBER);

          if first then first := FALSE;

          if (props <> NIL) then
            output_property_list(f, props^.properties);

          pins_left_index := pins_left_index + pins_direction;
        end { write_bit } ;


      begin { print_constant }
        signal_name := instance^.defined_by^.signal^.name;
        is_complemented := (instance^.defined_by^.polarity = COMPLEMENTED);

        if left = -1 then write_bit(0)  { scalar }
        else
          begin
            if left> right then direction := -1 else direction := 1;
           
            bit := left;

            repeat
              write_bit(bit);

              done := (bit = right);
              if not done then bit := bit + direction;
            until done;
          end;
      end { print_constant } ;


    begin { print_signal_instance_and_props }
      if instance^.defined_by^.is_const then
        print_constant(f, instance)
      else
        begin
          if is_NC_signal(instance^.defined_by^.signal^.name) then
	    begin
              output_numbered_token(f, expansion_NC_string^, STRING_NUMBER);
              if instance^.defined_by^.kind = VECTOR then
	        if left <> right { no subscript needed on 1 bit NCs } then
                  output_left_and_right(f, left, right);
	    end
          else 
	    begin
	      output_signal_name(
                f, instance^.defined_by^.polarity, 
                instance^.defined_by^.signal^.name,
                instance^.defined_by^.scope,
		instance^.defined_by^.node);

              if instance^.defined_by^.kind = VECTOR then
                output_left_and_right(f, left, right);
	    end;

          if props <> NIL then
            output_property_list(f, props^.properties);

          pins_left_index := pins_left_index +
                                        pins_direction*(abs(left-right) + 1);
        end;
    end { print_signal_instance_and_props } ;


  begin { output_actual }
    if debug_14 or debug_17 then
      begin
        writeln(outfile, 'Entering output_actual: actual=');
        dump_propertied_CS(outfile, actual_list);
        writeln(outfile, '  formal/actual=');
        dump_formal_actual(outfile, formal_actual_pair);
      end;

    (************************************************************************)
    (*  Theory of operation:                                                *)
    (*                                                                      *)
    (*  Properties of the pin are output as properties of the actual        *)
    (*  connected with the pin.  If the actual does not have breaks         *)
    (*  corresponding to the sections of the pin with properties, the       *)
    (*  actual must be broken into concatenated signals.  The actual        *)
    (*  is output piece by concatenated piece.  The corresponding bits      *)
    (*  of the pin are represented by PINS_LEFT_INDEX and PINS_RIGHT_INDEX. *)
    (*  For each actual, the properties are gathered for the corresponding  *)
    (*  bits of the pin.  The property list is processed element by         *)
    (*  element and an actual is output for each one.                       *)
    (************************************************************************)

    { set up the subscript descriptors for the pin: PINS_LEFT_INDEX and
      PINS_RIGHT_INDEX. }

    with formal_actual_pair^.pin_name^ do
      if kind = VECTOR then
        if bit_subscript = NIL then
          begin
            assert(232 { NIL subs on VECTOR pin_name });
	    dump_mtree_node(cmplog, node);
	    dump_formal_actual(cmplog, formal_actual_pair);
            pins_left_index := -1;  pins_right_index := -1;
          end
        else
          begin
            pins_left_index := bit_subscript^.left_index;
            pins_right_index := bit_subscript^.right_index;
          end
      else
        begin  pins_left_index := -1;  pins_right_index := -1;  end;

    if pins_left_index > pins_right_index then pins_direction := -1
    else if pins_left_index < pins_right_index then pins_direction := 1
    else pins_direction := 0;

    { process each of the actuals in the concatenated signal }

    actual := actual_list;  first := TRUE;
    while actual <> NIL do
      begin
        if debug_14 or debug_17 then
          begin
            writeln(outfile, 'Processing actual:');
            dump_signal_instance(outfile, actual^.instance);
          end;

        if not actual^.instance^.defined_by^.is_virtual_base then
	  (* NOTE: we are ignoring a symptom here -- this assertion check
	     occurs when scope conflict errors occur.  Since this code is
	     soon to be obsolete, I am not going to bother actually fixing
	     the problem but instead am squelching the assertion failure
	     when these errors have occured. *)
	  if (scope_conflict_errors * errors_encountered = []) then
            begin
              assert(197 { MUST not be a virtual signal! });
              dump_signal_definition(CmpLog, actual^.instance^.defined_by);
              dump_virtual_defs(CmpLog, actual^.instance^.defined_by);
            end;

        { output the actual signal taking into account any replication }

        rep_value := actual^.instance^.replication_factor;
        repeat
          if (actual^.instance^.defined_by^.kind <> VECTOR) then
            begin
              if debug_14 or debug_17 then
                writeln(outfile, '-actual is non-VECTOR');

              width := 1;

              pins_right_index := pins_left_index;
              prop_list := gather_properties_from_subscript
                                              (formal_actual_pair^.properties,
                                               pins_left_index,
                                               pins_right_index);

              print_signal_instance_and_props(actual^.instance,
                                              -1, -1, first, prop_list);
            end
          else
            begin
              if debug_14 or debug_17 then
                writeln(outfile, '-actual is a VECTOR');

              { we know that the actual instance cannot be a bit list }

              width := width_of_subscript(actual^.instance^.bit_subscript,
                                          VECTOR);

              pins_right_index := pins_left_index + pins_direction*(width-1);

              { gather the properties from the corresponding bits of the
                pin to which the actual signal is connected. }

              prop_list := gather_properties_from_subscript
                                              (formal_actual_pair^.properties,
                                               pins_left_index,
                                               pins_right_index);

              left := actual^.instance^.bit_subscript^.left_index;
              right := actual^.instance^.bit_subscript^.right_index;

              if left > right then direction := -1
              else if left < right then direction := 1
              else direction := 0;

              right := left;

              current_prop := prop_list;  one_was_output := FALSE;
              while current_prop <> NIL do
                begin
                  { if there are no properties for this section of the actual,
                    output a portion of the actual including no propertied. }

                  if pins_left_index <> current_prop^.left_index then
                    begin
                      delta_width := abs(pins_left_index -
                                         current_prop^.left_index) + 1;

                      right := left + direction*(delta_width-2);

                      print_signal_instance_and_props(actual^.instance,
                                                      left, right, first,
                                                      NIL);
                      left := right + direction;
                    end;

                  { output portion of the actual including properties }

                  delta_width := abs(current_prop^.left_index -
                                     current_prop^.right_index) + 1;
                  right := left + direction*(delta_width-1);

                  print_signal_instance_and_props(actual^.instance,
                                                  left, right, first,
                                                  current_prop);

                  left := right + direction;

                  one_was_output := TRUE;

                  current_prop := current_prop^.next;
                end;

              { output any remaining portion of the actual }

              if (right <> actual^.instance^.bit_subscript^.right_index) or
                 not one_was_output { scalar with no properties } then
                print_signal_instance_and_props(actual^.instance,
                           left, actual^.instance^.bit_subscript^.right_index,
                           first, NIL);

            end { signal is vector } ;

          pins_left_index := pins_right_index + pins_direction;

          if (rep_value > 0) then rep_value := rep_value - 1;
        until (rep_value = 0);

        actual := actual^.next;
      end;
  end { output_actual } ;


begin { output_expansion }
  init_output_continue;  pipe_dump_CRLF(p);
  
  pipe_dump_char(p, '%');

  pipe_dump_integer(p, node^.module_number);
      
  print_body_properties(p, node);
      
  formal_actual_pair := node^.params;
  while formal_actual_pair <> NIL do
    begin
      pipe_dump_char(p, ':');  { bindings prefix }

      output_formal(p, formal_actual_pair);

      actual :=
           find_base_of_PCS(formal_actual_pair^.actual_parameter^.signal);
      output_actual(p, formal_actual_pair, actual);

      release_entire_propertied_CS(actual);

      formal_actual_pair := formal_actual_pair^.next;
    end;
end { output_expansion } ;


(**)
           {      +------------------------------------+      }
           {      |                                    |      }
           {      |      Generate Expansion File       |      }
           {      |                                    |      }
           {      | in which all the efforts of PASS 1 |      }
           {      |   are put to use as the mtree in   |      }
           {      |   all its glory is output to the   |      }
           {      |   (page-at-a-time) expansion file. |      }
           {      |                                    |      }
           {      +------------------------------------+      }


procedure generate_expansion_file(var p: pipe);
  { output the expansion file for the page.  We know that we only
    need to look at the sons of the root node (and its siblings).  
    The root node may have siblings if it has been size replicated. }
  var
    node: mtree_node_ptr;       { current root node }
    son: mtree_node_ptr;        { current son of root node }
    
begin
  if debug_17 then
    writeln(outfile, '--- Entering generate_sepcomp_expansion_file ---');

  node := mtree_root;
  while node <> NIL do
    begin
      son := node^.son;
      while son <> NIL do
        begin
          if son^.is_leaf_node then output_expansion(p, son);
          son := son^.next;
        end;
      node := node ^.next;
    end;

  pipe_dump_char(p, ';');

  if debug_17 then
    begin
      writeln(outfile, '--- Exiting generate_sepcomp_expansion_file ---');
      writeln(outfile);
    end;
end { generate_expansion_file } ;


(**)  { ------- output signal properties to page expansion file ------- }


procedure output_signal_properties(var f: pipe; 
                                           node: mtree_node_ptr);
  { output the properties of the signals defined within all of the nodes
    starting with the given node. }
  var
    temp_subscript: subscript_ptr;      { temporary subscript }


  procedure dump_properties_of_signals(var f: pipe; 
                                               node: mtree_node_ptr);
    { dump the properties of the signals within the current node and all of
      the sons of the current node except for leaf nodes. }
    var
      son: mtree_node_ptr;                { current son of this node }
      signal: signal_definition_ptr;      { current signal on this node }
      next_prop_list,                     { next in bit property list }
      prop_list: subscript_property_ptr;  { base bit properties }


    procedure output_signal(var f: pipe; def: signal_definition_ptr;
                                    prop_list: subscript_property_ptr);
      { output the given signal (DEF) with the given subscript (PROP_LIST)
        to the output file.  Output the properties (PROP_LIST). }
      const
        MAX_PROPERTIES_PER_LINE = 5;
      var
        first: boolean;        { TRUE if first property to be output }
        prop: property_ptr;    { current property in output lkist }
	prop_count: 0..MAX_PROPERTIES_PER_LINE; { count of properties output
	                                          to this line so far }
    begin
      prop := prop_list^.properties;  first := TRUE;  prop_count := 0;
      while prop <> NIL do
        begin
          if ([DONT_OUTPUT,INHERIT_PIN] * prop^.name^.kind) = [] then
            begin
              if first then
                begin
                  { print the signal name and its bits }

                  pipe_dump_CRLF(f);

                  pipe_dump_char(f, '(');

                  output_signal_name(f, def^.polarity,
                                             def^.signal^.name,
				             def^.scope,
                                             def^.node);

                  if prop_list^.left_index <> -1 then
                    begin
                      temp_subscript^.left_index := prop_list^.left_index;
                      temp_subscript^.right_index := prop_list^.right_index;

                      output_bit_subscript(f, temp_subscript, VECTOR);
                    end;

                  first := FALSE;
                end;

              { print the property name and value }

	      if prop_count < MAX_PROPERTIES_PER_LINE then
	        prop_count := prop_count + 1
	      else
		begin  pipe_dump_CRLF(f);  prop_count := 0;  end;

              output_numbered_property(f, prop^.name, prop^.text);

            end;

          prop := prop^.next;
        end { while } ;
    end { output_signal } ;


  begin { dump_properties_of_signals }
    { dump the properties of this node's signals }

    signal := node^.signals;
    while signal <> NIL do
      begin
        if debug_17 then
          begin
            write(outfile, 'Signal prop dump: ');
            dump_signal_definition(outfile, signal);
          end;

        { do not output constants }

        if not signal^.is_const and 
	   not is_NC_signal(signal^.signal^.name) then
          if (signal^.scope = XINTERFACE) and
	     (signal^.node^.father_node <> NIL) then
            begin
              assert(191 { Interface signal should have been released });
              writeln(CmpLog, 'Signal found: ');
              dump_signal_definition(CmpLog, signal);
            end
          else
            begin
              prop_list := base_bit_properties(signal);

              while prop_list <> NIL do
                begin
                  if prop_list^.properties <> NIL then
                    output_signal(f, signal, prop_list);

                  next_prop_list := prop_list^.next;

                  release_entire_property_list(prop_list^.properties);
                  release_subscript_property(prop_list);

                  prop_list := next_prop_list;
               end;
            end;

        signal := signal^.next;
      end;

    { dump properties of signals for the sons of the current node }

    son := node^.son;
    while son <> NIL do
      begin
        if not son^.is_leaf_node then
          dump_properties_of_signals(f, son);

        son := son^.next;
      end;
  end { dump_properties_of_signals } ;


begin { output_signal_properties }
  temp_subscript := NIL;
  new_subscript(temp_subscript);

  while node <> NIL do
    begin
      dump_properties_of_signals(f, node);
      node := node^.next;
    end;

  pipe_dump_char(f, ';');

  release_subscript(temp_subscript);
end { output_signal_properties } ;


procedure expand_non_graphical_model(var f: pipe; special_model: xtring);
  { writes expansion file that communicates existence of a non-graphical
    model and/or extension which has been forced to primitive. }
  var
    pg: page_range;
    save_ext: name_ptr;
    save_vers: version_range;


  procedure dump_description_props(var f: pipe; prim_type: xtring);
  begin
    output_numbered_property(f, PRIM_TYPE_prop_name, prim_type);
    output_numbered_property(f, PRIM_FILE_prop_name, 
			     enter_string(er_filename(module_being_compiled,
			                              ord(SPECIAL_MODEL_FILE), 
						      0, NIL)));
  end { dump_description_props } ;


  procedure backward_compatible_analog_hack(var f: pipe; prim_type: xtring);
    var
      spec: name_ptr;
  begin
    spec := name_from_string(prim_type);
    if spec = SUBCKT_prop_name then
      output_numbered_property(f, SUBCKT_prop_name, 
			       enter_string(er_filename(module_being_compiled,
			                                ord(SPECIAL_MODEL_FILE),
							0, NIL)))

    else dump_description_props(f, prim_type);
  end;


begin { write_special_expansion_file }
  init_output_continue;
  output_numbered_dictionary(f, expansion_string_dictionary, 
                             FINISHED_DICTIONARY);
  output_numbered_dictionary(f, expansion_id_dictionary, 
                             FINISHED_DICTIONARY);

  if (extension_being_compiled = null_name) or 
     er_force_to_prim(module_being_compiled) then
    begin
      save_ext := extension_being_compiled;
      save_vers := version_being_compiled;
      extension_being_compiled := PRIM_extension_name;
      version_being_compiled := 1;

      output_heading(f, 1);

      extension_being_compiled := save_ext;
      version_being_compiled := save_vers;
    end
  else
    begin
      { a non-graphical model along with a primitive. Output_heading 
	expects page_being_compiled to be non-zero for this case.
	Use the page following the one describing the primitive.
	(We ignore the boundary condition of someone defining a primitive
	with MAX_PAGE_NUMBER; do that and get a range-check error.) }

      page_being_compiled := 0;
      pg := er_page(module_being_compiled);
      while (pg <> 0) do
        begin
	  page_being_compiled := pg;
	  pg := er_page(module_being_compiled);
	end;
      page_being_compiled := page_being_compiled + 1;
      output_heading(f, page_being_compiled);
      page_being_compiled := 0;
    end;

  output_context(f);

  { Body properties }

  if special_model <> nullstring then
    if analog_designer_compile then 
      backward_compatible_analog_hack(f, special_model)
    else dump_description_props(f, special_model);

  pipe_dump_char(f, ';');  { end body properties section }
  pipe_dump_CRLF(f);

  { Rest is empty }

  pipe_dump_char(f, ';');  { dependencies section }
  pipe_dump_char(f, ';');  { modules list section }
  pipe_dump_char(f, ';');  { syonyms section }
  pipe_dump_char(f, ';');  { invokes section }
  pipe_dump_char(f, ';');  { signal properties section }
  pipe_dump_CRLF(f);
  pipe_dump_alpha(f, 'END.            ');
  pipe_dump_CRLF(f);
end { write_bogus_expansion_file } ;


