function fix_signal_name(signal: xtring): xtring;
  { scan the signal name and insert quotes where neccessary to make it
    LR(1), LL(1) parseable. }
  type
    string_range_set = set of string_range;
  var
    stopper,                            { char needed to close string }
    ch: char;                           { current char from SIGNAL }
    const_pos,                          { position of start of constant }
    pos,                                { current position in SIGNAL }
    start,                              { start of the signal name in SIGNAL }
    last,                               { end of the signal name in SIGNAL }
    num_constants,                      { number of constants in signal }
    num_inserts,                        { number of quotes to insert }
    num_double,                         { number of double quotes in signal }
    num_single: string_range;           { number of single quotes in signal }
    insert_positions,                   { places to insert quotes }
    insert_radix: string_range_set;     { places to insert binary radix }
    temp: xtring;                       { string to be returned }
    is_const,                           { TRUE if signal name is a constant }
    finished,                           { TRUE when a signal name found }
    done: boolean;                      { TRUE when entire signal scanned }
    syntax_index: signal_syntax_range;  { index into signal syntax table }
    found_non_digit: boolean;           { TRUE if non digit encountered 
                                          within potential constant }
    found_non_constant_char: boolean;   { TRUE if char not in digits + 
                                          ['(',')',' '] is found in a
					  potential constant }


  procedure read_char;
    { get the next character from SIGNAL.  Return 0 if end of string reached }
  begin
    if pos >= ord(signal^[0]) then ch := chr(EOL)
    else
      begin  pos := pos + 1;  ch := signal^[pos];  end;
  end { read_char } ;


  function previous_char: char;
    { return the previous character in the string }
  begin
    if pos > 1 then previous_char := signal^[pos-1]
               else previous_char := chr(EOL);
  end { previous_char } ;


  function next_char: char;
    { return the next character in the string }
  begin
    if pos < ord(signal^[0]) then next_char := signal^[pos+1]
                             else next_char := chr(EOL);
  end { next_char } ;


  function insert_quotes(signal: xtring): xtring;
    { create a new string.  Copy SIGNAL to it inserting quotes }
    var
     num_quotes,                        { number of quotes in signal }
     source_pos,                        { current position within SIGNAL }
     dest_pos: string_range;            { current position within TEMP }
     in_string: boolean;                { TRUE if within signal name string }
     temp: xtring;                      { string to be returned }
     separator: char;                   { string delimiter char }


    procedure add_char(ch: char);
      { add the character to the output string (TEMP) }
    begin
      dest_pos := dest_pos + 1;  temp^[dest_pos] := ch;
    end { add_char } ;


  begin { insert_quotes }
    if (num_double = 0) and (num_single <> 0) then
      begin  separator := '"';  num_quotes := num_double;  end
    else
      begin  separator := '''';  num_quotes := num_single;  end;
    create_a_string(temp,
                    (ord(signal^[0])+num_inserts+num_quotes+2*num_constants));

    source_pos := 0;  dest_pos := 0;  in_string := FALSE;
    while (source_pos < ord(signal^[0])) and (dest_pos < ord(temp^[0])) do
      begin
        source_pos := source_pos + 1;
        if source_pos IN insert_radix then
          begin  add_char(chr(default_radix + ord('0')));  add_char('#');  end
        else if source_pos IN insert_positions then
          begin  add_char(separator);  in_string := not in_string;  end;
        add_char(signal^[source_pos]);
        if in_string and (signal^[source_pos] = separator) then
          add_char(separator);
      end;
    { take care of special case for end of signal name = end of string }
    if (source_pos+1) IN insert_positions then add_char(separator);

    if (source_pos <> ord(signal^[0])) or (dest_pos <> ord(temp^[0])) then
      assert(7 { some kind of problem here! });
    insert_quotes := temp;
  end { insert_quotes } ;


  procedure skip_past_subscript;
    { skip the subscript }
    var
      num_subscripts: natural_number;      { number of subscripts found }
  begin
    read_char;  num_subscripts := 1;
    repeat
      if ch = '<' then num_subscripts := num_subscripts + 1
      else if ch = '>' then num_subscripts := num_subscripts - 1;
      read_char;
    until (num_subscripts = 0) or (ch = chr(EOL)) or
          (ch = general_property_prefix_char);
    if ch = '>' then read_char;    { eat the '>' }

    while ch = ' ' do read_char;
  end { skip_past_subscript } ;


  procedure scan_until(stopper: char);
    { scan the input until the given character is found.  Then read in the
      next character. }
  begin
    read_char;    { eat the current character (assumed to be = stopper ) }
    while (ch <> stopper) and (ch <> chr(EOL)) do read_char;
    if ch = stopper then read_char
    else
      begin
	error(214 { string not closed });
	error_dump_current_parse_environment;
	error_dump_indent(indent);
	error_dump_alpha('Signal="        ');
	error_dump_string(signal);
	error_dump_char('"');
	error_dump_CRLF;
      end;
  end { scan_until } ;

    
  procedure scan_until_end_of_signal;
    { scan until the end of this signal: another signal, end of string.
      NOTE: Concatenation character is NOT configurable }
    var
      done: boolean;                   { TRUE when end of signal is found }
      stopper: char;                   { stopper character for search }
  begin
    done := (ch = ':') or (ch = chr(EOL));  stopper := ':';
    while not done do
      begin
        if ch = '<' then skip_past_subscript
        else if ch = '''' then scan_until('''')
        else if ch = '"' then scan_until('"')
        else if (ch = chr(EOL)) or (ch = ':') then done := TRUE
        else read_char;
      end;
  end { scan_until_end_of_signal } ;


begin { fix_signal_name }
  if debug then disp_line('enter fix_signal_');

  pos := 0;
  insert_positions := [];
  insert_radix := [];
  num_constants := 0;
  num_single := 0;
  num_double := 0;
  num_inserts := 0;

  read_char;  done := FALSE;
  repeat
    for syntax_index := 1 to SYNTAX_TABLE_SIZE do
      case signal_syntax_table[syntax_index] of

        NEGATION_SPECIFIER:
            begin
              if ch = signal_negation_char then read_char;
              while ch = ' ' do read_char;
            end;

        NAME_SPECIFIER:
            begin
              while ch = ' ' do read_char;
              start := pos;  const_pos := 0;
              stopper := chr(EOL);  is_const := FALSE;
	      found_non_digit := FALSE;  found_non_constant_char := FALSE;
              repeat
                if ch <> ' ' then last := pos;
                if ch = '''' then num_single := num_single + 1
                else if ch = '"' then num_double := num_double + 1;
                if stopper = chr(EOL) then
                  begin
                    if isdigit[ch] and (const_pos = 0) then
                      const_pos := pos
                    else if ch = '#' then
                      if not found_non_digit then is_const := TRUE;
		    if not isdigit[ch] then
		      begin
		        found_non_digit := TRUE;
			if (ch <> '(') and (ch <> ')') and (ch <> ' ') then
			  found_non_constant_char := TRUE;
		      end;
                  end;
                if ch = stopper then stopper := chr(EOL)
                else
                  if stopper = chr(EOL) then
                    if (ch = '''') or (ch = '"') then stopper := ch;
                read_char;
                finished := (ch = chr(EOL));
                if stopper = chr(EOL) then
                  if is_signal_name_terminator[ch] then
                    if not isupper[ch] then finished := TRUE
                    else
                       if not (isupper[previous_char] or 
		               isdigit[previous_char] or
			       (previous_char = '$')) and
                          (  (next_char = ' ') or 
			     (is_signal_name_terminator[next_char] and
			      not isupper[next_char])  ) then
                         finished := TRUE
              until finished;
              if stopper <> chr(EOL) then
                begin
                  error(214 { string not closed });
                  error_dump_current_parse_environment;
                  error_dump_indent(indent);
                  error_dump_alpha('Signal="        ');
                  error_dump_string(signal);
                  error_dump_char('"');
                  error_dump_CRLF;
                end;
  
              if not found_non_constant_char or is_const then
                begin
                  if not is_const then   { does not already have radix spec }
                    begin
                      insert_radix := insert_radix + [const_pos];
                      num_constants := num_constants + 1;
                    end;
                end
              else
                begin
                  { this algorythm blows up when last = MAX_STRING_LENGTH }
                  insert_positions := insert_positions + [start, last+1];
                  num_inserts := num_inserts + 2;
                end;
            end;

        ASSERTION_SPECIFIER:
            if (ch = signal_is_asserted_low_char) or
               (ch = signal_is_asserted_high_char) then
              repeat
                read_char
              until ch <> ' ';

        SUBSCRIPT_SPECIFIER:
            if ch = '<' then skip_past_subscript;

        PROPERTY_SPECIFIER:
            scan_until_end_of_signal;

        NULL_SPECIFIER: ;
      end { case } ;

    if ch = concatenation_char then read_char else done := TRUE;
  until done;

  temp := insert_quotes(signal);
  fix_signal_name := enter_and_release_string(temp);

  if debug then
    begin  writestring(outfile, temp);  writeln(outfile);  end;

  if debug then disp_line('fix_signal_name  ');
end { fix_signal_name } ;
