(**)     { ------- current parse string output (error) ------- }


procedure print_input_line(var f: textfile; error_num: error_range;
                           indent: boolean);
  { print the input parse line to the given file. If indent, then
    indent 2 extra spaces. }
  var
    base_of_stack: stack_index_range;  { position in stack of base string }
    done: boolean;                     { TRUE when stack dump complete }
    position: string_range;            { position to print pointer (^) }
    base_pos,                          { starting position of current string }
    curr_pos: string_range;            { current output string length }


  procedure output_parse_string(stack_pos: natural_number);
    { outdut the parse string, expanding current text macro }
    var
      start: string_range;        { start of last half of parse string }
      i: string_range;            { index into the string }
  begin
    if (stack_pos > stack_top) or (stack_pos = 0) then
      begin
        dump_string(f, instring);
        curr_pos := curr_pos + ord(instring^[0]);
      end
    else
      with stack[stack_pos] do
        begin
          if stack_pos = base_of_stack then
            begin
              dump_string(f, str);
              writeln(f);  write(f, ' ');
            end;

          for i := 1 to last_pos do write(f, str^[i]);

          curr_pos := curr_pos + last_pos;
          i := last_pos+1;
          while (i < line_pos) and (str^[i] = ' ') do
            begin  write(f, ' ');  curr_pos := curr_pos + 1;  i := i+1;  end;

          base_pos := curr_pos;
          if (stack_pos < max(stack_top, parse_stack_pointer)) then
	    output_parse_string(stack_pos+1);

          if (state = FGOT_CHAR) and (pos < ord(str^[0])) then start := pos
                                                          else start := pos+1;
          for i := start to ord(str^[0]) do
            begin  write(f, str^[i]);  curr_pos := curr_pos + 1;  end;
        end;
  end { output_parse_string } ;


begin { print_input_line }
  curr_pos := 0;  base_pos := 0;  done := FALSE;
  base_of_stack := stack_top;
  if parse_stack_pointer > stack_top then
    begin
      base_of_stack := parse_stack_pointer;
      if how_to_parse <> PARSE_TRANSPARENTLY then done := TRUE
      else base_of_stack := base_of_stack - 1
    end;
  while (base_of_stack > 0) and not done do
    if stack[base_of_stack].how <> PARSE_TRANSPARENTLY then done := TRUE
    else base_of_stack := base_of_stack - 1;

  if indent then write(f, '   ')
            else write(f, ' ');

  output_parse_string(base_of_stack);
  writeln(f);

  { LAST_SYM_POS points to last character preceding current symbol.  If the
    the compiler does not understand the current symbol, the pointer should
    point to the 1st place in the symbol (e.g., expected > ), hence, need to
    use LAST_SYM_POS+1.  If READ_STATE = FINPUT, then LINE_POS points to the
    last character read in and should be used as is.  If READ_STATE = 
    FGOT_CHAR, then LINE_POS points to the character following the last char
    read in.  If, however, LINE_POS points to the last position in the string,
    use it as is. }

  if parse_stack_pointer <= stack_top then position := 0
  else
    if error_num IN scan_past_errors then position := last_sym_pos+1
    else
      if read_state = FINPUT then position := line_pos
      else
        if line_pos = ord(instring^[0]) then position := line_pos
        else if line_pos > 1 then position := line_pos-1
                             else position := line_pos;

  position := position {in line} + base_pos {text macro} + 1 {leading space};
  if indent then position := position + 2 { indentation };
  if sy = ENDOFDATASY then position := position + 1;
  if position > 1 then writeln(f, error_position_char:position)
                  else writeln(f, ' ', error_position_char);
end { print_input_line } ;


(**)     { ------- parse stack utilities ------- }


procedure dump_parse_state(var f: textfile; heading: boolean);
  { dump the current state of the parse stack for debugging purposes.
    If heading, then preceed the dump with a heading. }
  const
    INDENT1 = 10;  { Number of chars written before calling dump...element }
    INDENT2 = 11;  { Additional chars written before writing quoted string }
  var
    i: stack_index_range;


  procedure dump_current_string;
    { dump info for current parse string in this format }
  begin
    { Write misc data }

    write(f, ord(read_state):1); write(f, ' ');
    if islegal[last_char] then write(f, last_char)
                          else write(f, ' ');
    write(f, ' ');
    write(f, ord(sy):2);  write(f, ' ');
    case how_to_parse of
      PARSE_TRANSPARENTLY: write(outfile, 'T');
      PARSE_SEPARATELY: write(outfile, 'S');
    end { case } ;
    write(f, ' ');
    if allow_TM_expansion then write(f, 'T ')
                          else write(f, 'F ');

    { Write string }

    writestring(f, instring);  writeln(f, ' (instring)');

    { Indicate position of linepos and last_sym_pos }

    write(f, ' ':(INDENT1 + INDENT2 + min(last_sym_pos, line_pos)));
    if last_sym_pos = line_pos then write(f, '^')
    else if last_sym_pos < line_pos then
      begin
        write(f, 'S');
        if line_pos > (last_sym_pos + 1) then
          write(f, ' ':(line_pos - last_sym_pos - 1));
        write(f, '^');
      end
    else
      begin
        write(f, '^');
        if last_sym_pos > (line_pos + 1) then
          write(f, ' ':(last_sym_pos - line_pos - 1));
        write(f, 'S');
      end;
   { write(f, '(pos=', line_pos:1, 'sym=', last_sym_pos:1, ')'); }
    writeln(f);
  end { dump_current_string } ;


  procedure dump_parse_stack_element(i: stack_index_range);
    { dump the ith element of the stack }
  begin { dump_parse_stack_element }
    if i <= 0 then
      begin
        writeln(f, ' ':INDENT2, '<', i:1, '>');
      end
    else if i > stack_top then
      if i = parse_stack_pointer then dump_current_string
      else
        begin
          writeln(f, ' ': INDENT2, '<UNDEFINED>');
        end
    else with stack[i] do
      begin
        { Write misc data }

        write(f, ord(state):1); write(f, ' ');
        if islegal[last] then write(f, last)
                         else write(f, ' ');
        write(f, ' ');
        write(f, ord(symbol):2);  write(f, ' ');
        case how of
          PARSE_TRANSPARENTLY: write(outfile, 'T');
          PARSE_SEPARATELY: write(outfile, 'S');
        end { case } ;
        write(f, ' ');
        if allow_TM then write(f, 'T ')
                    else write(f, 'F ');

        { Write string }

        writestring(f, str);  writeln(f);

        { Indicate position of linepos and last_sym_pos }

        write(f, ' ':(INDENT1 + INDENT2 + min(last_pos, pos)));
        if last_pos = pos then write(f, '^')
        else if last_pos < pos then
          begin
            write(f, 'S');
            if pos > (last_pos + 1) then write(f, ' ':(pos - last_pos - 1));
            write(f, '^');
          end
        else
          begin
            write(f, '^');
            if last_pos > (pos + 1) then write(f, ' ':(last_pos - pos - 1));
            write(f, 'S');
          end;
      { write(f, '(pos=', pos:1, 'sym=', last_pos:1, ')'); }
        writeln(f);
      end;
  end { dump_parse_stack_element } ;


  procedure dump_heading;
    { Dump a heading to describe what happens in dump...element }
  begin
    write(f, ' ':INDENT1);
    writeln(f, 'S L      E');
    write(f, ' ':INDENT1);
    writeln(f, 'T A      X');
    write(f, ' ':INDENT1);
    writeln(f, 'A S  S H P');
    write(f, ' ':INDENT1);
    writeln(f, 'T T  Y O T');
    write(f, ' ':INDENT1);
    writeln(f, 'E C  M W M');
  end;
    

begin
  if heading then dump_heading;
  for i := 0 to max(stack_top, parse_stack_pointer) do
    begin
      if i = stack_top then write(f, ' TOS')
                       else write(f, '    ');
      if i = parse_stack_pointer then write(f, ' SP')
                                 else write(f, '   ');
      if (i = stack_top) or (i = parse_stack_pointer) then write(f, ' ->')
                                                      else write(f, '   ');
      dump_parse_stack_element(i);
    end;
  if stack_top >= parse_stack_pointer then
    begin
      write(f, ' ':INDENT1);
      dump_current_string;
    end;
end { dump_parse_state } ;


procedure parse_string(string_to_parse: xtring; way_to_parse: parse_type);
  { push the given string onto the parse stack }
begin
  if debug_25 then
    begin
      write(outfile, 'enter parse_string(');
      writestring(outfile, string_to_parse);
      case way_to_parse of
        PARSE_TRANSPARENTLY: write(outfile, ', T');
        PARSE_SEPARATELY: write(outfile, ', S');
      end { case } ;
      writeln(outfile, ')');
      dump_parse_state(outfile, TRUE);
    end;

  if (stack_top >= MAX_STACK) or (parse_stack_pointer >= MAX_STACK) then
    error(137 { text macro depth exceeded })
  else
    begin
      if parse_stack_pointer > stack_top then
        stack_top := parse_stack_pointer;

      { save state of current environment }
            
      with stack[parse_stack_pointer] do
        begin
          str      := instring;
          last_pos := last_sym_pos;
          pos      := line_pos;
          state    := read_state;
          last     := last_char;
          symbol   := sy;
          constant := const_val;
          how      := how_to_parse;
          allow_TM := allow_TM_expansion;
          keys     := allowed_key_words;
        end;

      parse_stack_pointer := stack_top + 1;

      instring     := string_to_parse;
      line_pos     := 0;
      last_sym_pos := 0;
      how_to_parse := way_to_parse;
      read_state   := finput;

      if way_to_parse <> PARSE_TRANSPARENTLY then
        begin
          if copy_input then allowed_key_words := []
	                     else allowed_key_words := signal_keysys;
          allow_TM_expansion := TRUE;
        end;

      if debug_25 then
        begin
          writeln(outfile, 'stack after push: ');
          dump_parse_state(outfile, FALSE);
        end;

      insymbol;
    end;
end { parse_string } ;


procedure pop_parsed_string(string_to_pop: xtring);
  { pop the top of the parse string stack until a non-transparent string
    has been popped. }
  var
    done: boolean;       { TRUE when proper string popped }
begin
  if debug_25 then
    begin
      write(outfile, 'enter pop_parsed_string(');
      writestring(outfile, string_to_pop);
      writeln(outfile, ')');
      dump_parse_state(outfile, TRUE);
    end;

  { Get rid of any "virtually" popped signals and get stack into its
    usual parse_stack_pointer = stack_top + 1 configuration. (HACK) }

  if parse_stack_pointer > 0 then stack_top := parse_stack_pointer - 1
  else stack_top := 0;

  if stack_top = 0 then assert(8 { stack underflow })
  else
    begin
      repeat
        done := (instring = string_to_pop) and 
                (how_to_parse <> PARSE_TRANSPARENTLY);
        with stack[stack_top] do
          begin
            if how_to_parse <> PARSE_TRANSPARENTLY then
              begin  sy := symbol;  const_val := constant;  end;
            instring           := str;
            last_sym_pos       := last_pos;
            line_pos           := pos;
            read_state         := state;
            last_char          := last;
            allow_TM_expansion := allow_TM;
            allowed_key_words  := keys;

            how_to_parse := how;
          end;

        stack_top := stack_top - 1;
      until done or (stack_top <= 0);
      
      if not done then assert(157 { oops! });
    end;

  parse_stack_pointer := stack_top + 1;

  if debug_25 then
    begin
      writeln(outfile, 'exit pop_parsed_string with');
      dump_parse_state(outfile, FALSE);
    end;
end { pop_parsed_string } ;


procedure virtual_pop_string;
  { move stack pointer down to next string without actually popping the
    top string.  This allows insymbol to get the next char following the
    last char of its current text macro definition without popping the
    string.  This is done so that if text macro recursion occurs at the
    end of a TM def, the stack will still grow so that infinite recursion
    will be caught.  insymbol fixes the results of this routine before
    when it proceeds on to the next character of the appropriate
    underlying string. }
begin
  if debug_25 then
    begin
      writeln(outfile, 'enter virtual_pop_string with');
      dump_parse_state(outfile, TRUE);
    end;
  if (parse_stack_pointer <= 0) then assert(8 { stack underflow })
  else if how_to_parse <> PARSE_TRANSPARENTLY then 
    begin
      assert(241 { not allowed });
      writeln(cmplog, 'Not parsing transparently');
      if debugging then writeln(outfile, 'Not parsing transparently');
    end
  else
    begin
      if (parse_stack_pointer > (stack_top + 1)) then 
        begin
          assert(241 { illegal });
          writeln(CmpLog, 'stack_top = ', stack_top:1, 
                          ' parse_stack_pointer = ', parse_stack_pointer:1);
          if debugging then
            writeln(Outfile, 'stack_top = ', stack_top:1, 
                    ' parse_stack_pointer = ', parse_stack_pointer:1);
        end;
      if parse_stack_pointer > stack_top then
        begin

          { current instring is not yet in the stack }

          if stack_top >= MAX_STACK then
            begin 
              assert(241 { This should be impossible });
              writeln(CmpLog, 'Uncaught overflow');
              if debugging then writeln(Outfile, 'Uncaught overflow');
              stack_top := MAX_STACK - 1;
            end;

          stack_top := stack_top + 1;
          
          with stack[stack_top] do
            begin
              str      := instring;
              last_pos := last_sym_pos;
              pos      := line_pos;
              state    := read_state;
              last     := last_char;
              symbol   := ENDOFDATASY; { we fell off the end of this string }
              constant := const_val;
              how      := how_to_parse;
              allow_TM := allow_TM_expansion;
              keys     := allowed_key_words;
            end;
        end;

      parse_stack_pointer := parse_stack_pointer - 1;

      with stack[parse_stack_pointer] do
        begin
          instring           := str;
          last_sym_pos       := last_pos;
          line_pos           := pos;
          read_state         := state;
          last_char          := last;
          allow_TM_expansion := allow_TM;
          allowed_key_words  := keys;
          how_to_parse       := how;
        end;
    end;

  if debug_25 then
    begin
      writeln(outfile, 'exit virtual_pop_string with');
      dump_parse_state(outfile, FALSE);
    end;
end { virtual_pop_string } ;


procedure fix_parse_stack;
  { remove wasted "virtually" popped signals from the top of the stack. 
    This is used by insymbol to clean up before returning. When done,
    parse_stack_pointer = stack_top + 1. }
begin
  if debug_25 then
    begin
      writeln(outfile, 'enter fix_parse_stack with');
      dump_parse_state(outfile, TRUE);
    end;
  if parse_stack_pointer <= 0 then 
    begin
      assert(8 { stack underflow !});  parse_stack_pointer := 1;
    end;
  while (stack_top > parse_stack_pointer) do
    begin
      if not ((stack[stack_top].how = PARSE_TRANSPARENTLY) and
             (stack[stack_top].pos >= ord(stack[stack_top].str^[0]))) then
        begin
          assert(241 { these should be spent });
          writeln(CmpLog, 'fix_parse_stack');
          if debugging then writeln(Outfile, 'fix_parse_stack');
          if stack[stack_top].how <> PARSE_TRANSPARENTLY then
            begin
              writeln(CmpLog, 'popping a non transparent string');
              if debugging then 
                writeln(Outfile, 'popping a non transparent string');
            end;
          if stack[stack_top].pos < ord(stack[stack_top].str^[0]) then
            begin
              writeln(CmpLog, 'popping an unspent string');
              writestring(CmpLog, stack[stack_top].str);  writeln(CmpLog);
              write(CmpLog, ' ':(stack[stack_top].pos + 1));
              writeln(Cmplog, '^');
              if debugging then
                begin
                  writeln(Outfile, 'popping an unspent string');
                  writestring(Outfile, stack[stack_top].str);
                  writeln(Outfile);
                  write(Outfile, ' ':(stack[stack_top].pos + 1));
                  writeln(Outfile, '^');
                end;
            end;
        end;
      stack_top := stack_top - 1;
    end;
  if stack_top = parse_stack_pointer then stack_top := stack_top - 1;

  if debug_25 then
    begin
      writeln(outfile, 'exit fix_parse_stack with');
      dump_parse_state(outfile, FALSE);
    end;
end { fix_parse_stack } ;


#include "fixsignal.p"
#include "insymbol.p"


(**)     { ------- parsing utilities ------- }


procedure skip(syms: setofsymbols);
  { used to try to gracefully recover from errors }
begin
  while not (sy IN syms+[ENDOFDATASY]) do insymbol;
end { skip } ;


function check_bit (bit: longint): bit_range;
  { check the value of a bit specifier and return }
begin
  if (bit < 0) or (bit > MAX_BIT_VALUE) then 
    begin
      error(16 { incorrect bit value });
      error_dump_indent(indent);
      error_dump_alpha('Bit value=      ');
      error_dump_char(' ');
      error_dump_integer(bit);
      error_dump_CRLF;

      bit := 0;
    end;

  check_bit := bit;
end { check_bit } ;

      
function check_addition(val1, val2: longint): longint;
  { check to see if the result of an addition of the two numbers is in range.
    If not, generate an error message and return the first argument. }
  var
    ok: boolean;      { TRUE if addition is in bounds }
begin
  ok := TRUE;
  if val1 >= 0 then
    begin
      if val2 > 0 then
        if val1 > MAXINT - val2 then ok := FALSE;
    end
  else
    if val2 < 0 then
      if val1 < -MAXINT-1 - val2 then ok := FALSE;

  if ok then check_addition := val1 + val2
  else
    begin  error(24 { overflow });  check_addition := val1;  end;

  if debug then
    begin
      disp_line('check_addition   ');
      writeln(outfile, 'CHECK: ', val1:1, '+', val2:1, '; OK=', ord(ok):1);
    end;
end { check_addition } ;


function check_subtraction(val1, val2: longint): longint;
  { check the subtraction of the two operands.  If the result is in bounds,
    return the difference otherwise generate an error and return the minuend }
  var
    ok: boolean;           { TRUE if subtraction is in bounds }
begin
  ok := TRUE;
  if val1 >= 0 then
    begin
      if val2 < 0 then
        if val1 > MAXINT + val2 then ok := FALSE;
    end
  else
    if val2 > 0 then
      if val1 < -MAXINT-1 + val2 then ok := FALSE;

  if ok then check_subtraction := val1 - val2
  else
    begin  error(24 { overflow });  check_subtraction := val1;  end;

  if debug then
    begin
      disp_line('check_subtraction');
      writeln(outfile, 'CHECK: ', val1:1, '-', val2:1, '; OK=', ord(ok):1);
    end;
end { check_subtraction } ;


(**)     { ------- expression parsing routines ------- }


#include "expression.p"


(**)     { ------- bit selection parsing routines ------- }


function check_bit_range(bit1, bit2: bit_range): bit_range;
  { check the sum of the two bits to make sure it resides in the proper
    range for a bit subscript.  It is assumed that the first bit
    is a valid bit range already. }
begin
  if bit2 > MAX_BIT_VALUE - bit1 then
    begin  error(24 { overflow });  check_bit_range := bit1;  end
  else
    check_bit_range := bit1 + bit2;
end { check_bit_range } ;


procedure reverse_bit_subscript(var sub: subscript_ptr);
  { reverse a bit subscript }
  var
    last,                 { last subscript element encountered }
    next: subscript_ptr;  { next subscript element }
begin
  last := NIL;
  while sub <> NIL do
    begin
      next := sub^.next;  sub^.next := last;  last := sub;  sub := next;
    end;
  sub := last;
end { reverse_bit_subscript } ;


function parse_bit_subscript: subscript_ptr;
  { parse a bit subscript (with associated structures) }
  var
    bit_val: bit_range;              { current bit parsed }
    done: boolean;                   { TRUE when all of subscript read }
    new_sub,                         { start of new subscript in list }
    last,                            { last subscript parsed (current) }
    sub: subscript_ptr;              { subscript representation to return }


  function end_of_list(sub: subscript_ptr): subscript_ptr;
    { find the end of the given subscript list }
    var
      next,                       { next element in the list }
      last: subscript_ptr;        { last element of the list }
  begin
    next := sub;  last := NIL;
    while next <> NIL do 
      begin  last := next;  next := next^.next;  end;
    end_of_list := last;
  end { end_of_list } ;


  procedure add_subscript(var list, last: subscript_ptr);
    { create a new subscript and add it to the end of the list }
    var
      sub: subscript_ptr;     { subscript being added }
  begin
    sub := NIL;  new_subscript(sub);
    if last <> NIL then last^.next := sub;  last := sub;
    if list = NIL then list := last;
  end { add_subscript } ;


  function process_increment(var sub: subscript_ptr): subscript_ptr;
    { parse the increment.  Check to see that the subscript is properly
      formed.  If the increment <> -1,0,1, then expand the subscript
      into a bit list and return as SUB.  Return a pointer to the last
      element in the subscript list. }
    var
      increment: longint;            { specified increment }
      new_list: subscript_ptr;       { expansion of the subscript }
      done: boolean;                 { TRUE when subscript expansion done }
      bit: longint;                  { current bit of expanded subscript }

    
    procedure swap(SP: subscript_ptr);
      { swap the subscript indices }
      var
	temp: bit_range;      { temp storage during swap }
    begin
      temp := SP^.left_index;
      SP^.left_index := SP^.right_index;  SP^.right_index := temp;
    end { swap } ;


  begin { process_increment }
    if debug then disp_line('enter process_inc');

    if sy <> fieldsy then increment := 1
    else
      begin
	insymbol;     { eat the fieldsy }
	increment := expression(NO_RELOPS);
	if increment = 0 then
	  begin  error(171 { not permitted });  increment := 1;  end;
      end;

    { legal combinations of left, right, and increment are:

	   OK            not OK
	a. 7..0:1     b. 7..0:-1        right to left
	c. 0..7:-1    d. 0..7:1         right to left

	e. 0..7:1     f. 0..7:-1        left to right
	g. 7..0:-1    g. 7..0:1         left to right

      Check to make sure this is a legal combination of above. }

    if sub^.left_index < sub^.right_index then
      begin
	if (increment < 0) and left_to_right then                 { case f }
	  begin  error(173 { should be left to right });  swap(sub);  end
	else if (increment > 0) and not left_to_right then        { case d }
	  begin  error(172 { should be right to left });  swap(sub);  end
      end
    else if sub^.left_index > sub^.right_index then
      if (increment > 0) and left_to_right then                   { case g }
	begin  error(173 { should be left to right });  swap(sub);  end
      else if (increment < 0) and not left_to_right then          { case b }
	begin  error(172 { should be right to left });  swap(sub);  end;

    { given the increment, generate a subscript }

    if abs(increment) = 1 then
      process_increment := sub        { done.  Leave as is }
    else
      begin
	{ generate a list of bits }
	{ NOTE: sub is at the end of the bit subscript list }

	if left_to_right then increment := -increment;
      
	new_list := NIL;  bit := sub^.right_index;
	repeat
	  if new_list <> NIL then new_subscript(new_list)
	  else
	    begin
	      new_subscript(new_list);
	      process_increment := new_list;
	    end;

	  new_list^.left_index := bit;
	  new_list^.right_index := bit;

	  bit := bit + increment;
	  if increment > 0 then done := (bit > sub^.left_index)
			   else done := (bit < sub^.left_index);
	until done;

	{ it is assumed that the subscript passed is the last element in
	  a subscript.  It has been initialized with the left and right
	  fields.  It should be replaced by an increment-expanded subscript.
	  Append the list and get rid of the first element since it is
	  redundant. }

	sub^ := new_list^;
	release_subscript(new_list);
      end;

    if debug then
      begin
	write(outfile, 'Subscript=');
	dump_bit_subscript(outfile, sub, VECTOR);  writeln(outfile);
	disp_line('process_increment');
      end;
  end { process_increment } ;


begin { parse_bit_subscript }
  if debug then disp_line('enter bit_subscri');

  sub := NIL;  last := NIL;  done := FALSE;
  repeat
    new_sub := last;

    bit_val := check_bit(expression(no_relops));

    if sy = subrangesy then
      begin
	insymbol;
	add_subscript(sub, last);
	last^.left_index := bit_val;
	last^.right_index := check_bit(expression(no_relops));
	last := process_increment(last);
      end

    else if sy = fieldsy then
      begin
	insymbol;
	add_subscript(sub, last);
	last^.left_index := bit_val;
	if left_to_right then
	  last^.right_index := check_bit(last^.left_index +
						  expression(no_relops) - 1)
	else
	  last^.right_index := check_bit(last^.left_index -
						 expression(no_relops) + 1);
	last := process_increment(last);
      end

    else if sy IN bitsubendsys then
      begin
	add_subscript(sub, last);
	last^.left_index := bit_val;
	last^.right_index := bit_val;
      end

    else
      begin
	if sy = IDENT then
	  error(59 { undefined identifier })
	else
	  error(30 { unexpected sy });
	skip(bitsubendsys);
	add_subscript(sub, last);
	last^.left_index := bit_val;
	last^.right_index := bit_val;
      end;

    last := end_of_list(sub);

    if sy = COMMA then insymbol else done := TRUE;
  until done;

  parse_bit_subscript := sub;

  if debug then disp_line('bit_subscript    ');
end { parse_bit_subscript } ;


function bit_selector: subscript_ptr;
  { parse the subscript and return a description of it }
begin
  if debug then disp_line('enter bit_selecto');

  if sy = LESSTHAN then 
    begin
      insymbol;
      bit_selector := parse_bit_subscript;
      if sy = GREATERTHAN then insymbol
      else
	begin  error(11 { expected > });  skip(signal_name_end_sys);  end;
    end
  else bit_selector := NIL;

  if debug then disp_line('bit selector     ');
end { bit_selector } ;


(**)     { ------- expression evaluation routines ------- }


function evaluate_string(str: xtring): longint;
  { evaluate numeric value -- no relational operators and no null strings }
  var
    t: longint;   { the expression value calculated }
begin
  if str = nullstring then
    begin  assert(4 { invalid string } );  t := 0;  end
  else
    begin
      parse_string(str, PARSE_SEPARATELY);
      t := expression(NO_RELOPS);
      if sy <> ENDOFDATASY then error(58 { extraneous junk });
      pop_parsed_string(str);
    end;
  evaluate_string := t;
end { evaluate_string } ;
  

function evaluate_boolean_expr(*str: xtring): longint*);
  { allow nullstring (return TRUE) and relational operators. Intended for
    use by C++ expansion control module -- thus the longint return rather
    than boolean.  "DEFAULT" will never appear here. }
  var
    t: longint;   { the expression value calculated }
begin
  if ord(str^[0]) = 0 then t := 1
      { string may come from C++ -- can't check str = nullstring }
  else
    begin
      parse_string(str, PARSE_SEPARATELY);
      t := expression(ALLOW_RELOPS);
      if sy <> ENDOFDATASY then error(63 { extraneous junk });
      pop_parsed_string(str);
    end;
  evaluate_boolean_expr := t;
end { evaluate_boolean_expr } ;


function evaluate_selection_expression(str: xtring): boolean;
  { str is a true string -- in compiler (pascal-side) string table. 
    "DEFAULT" always returns TRUE here -- we assume we have already
    chosen the proper version and are just checking the individual pages. }
begin
    if str = DEFAULT_string then evaluate_selection_expression := TRUE
    else evaluate_selection_expression := evaluate_boolean_expr(str) <> 0;
end {  evaluate_selection_expression } ;


