 (**)     { ------- output a string with specified format ------- }


function print_string_formatted(var f: textfile;
                                str: xtring;
                                left_margin, right_margin: print_width_range;
                                pad_the_line: boolean;
                                print_CRLF: boolean): print_width_range;
  { Print the string to the given file and break it up if it does not fit.
    The break up algorithm works as follows:
          1).  If the string length <= space available, print the string
               and pad with blanks if padding is specified (PAD_THE_LINE).
          2).  else break the string into N pieces such that each of the N
               pieces is < the space available and as large as possible.
               Print each of the pieces as follows:
                  a).  First piece is left justified at left margin.
                  b).  Last piece is right justified at right margin.
                  c).  Other pieces are left justified at
		       DEFAULT_INDENT+left margin.
          3).  Print a CRLF if so indicated.
    Strings are broken at a space.  If no space is found such that the
    resulting string is greater than half the space available, then break
    the string DEFAULT_INDENT characters in from the right margin.

    Return the character position of the last character in the last line.
    This is always = RIGHT_MARGIN unless the string is shorter than the
    space available and no padding is performed (PAD_THE_LINE = FALSE).

    It is assumed that, on entry, the current character position in the
    output file is LEFT_MARGIN. }
  var
    space_available: longint;     { space available for the string }
    i,                            { index into the string for output }
    start,                        { starting position of current substring }
    break_pos,                    { best guess break position }
    length: string_range;         { current length of the string }
    blank: longint;               { width of space to be printed }


  function find_nearest_break(last: string_range): string_range;
    { find the nearest place to break the string to the left of LAST.  If
      a position is not found before half the space available is traversed
      then return LAST. }
    var
      pos: string_range;      { index into the string }
      found: boolean;         { TRUE when a break position has been found }
  begin
    pos := last;  found := FALSE;
    while (pos > (last - (space_available DIV 2))) and not found do
      if str^[pos] = ' ' then  found := TRUE  else  pos := pos - 1;
    if found then find_nearest_break := pos
             else find_nearest_break := last;
  end { find_nearest_break } ;


  procedure post_string(start, finish: string_range);
    { print the sub string on the output file }
    var
      i: string_range;           { index into the string }
  begin
    for i := start to finish do write(f, str^[i]);  writeln(f);
    write(f, ' ':left_margin+DEFAULT_INDENT-1);
  end { post_string } ;


begin { print_string_formatted }
  space_available := right_margin - left_margin + 1;
  print_string_formatted := right_margin;
  if (space_available <= 0) then
    begin  assert(45 { invalid });
      left_margin := 1;  space_available := right_margin;
    end;
  length := ord(str^[0]);
  if length <= space_available then
    begin
      for i := 1 to length do write(f, str^[i]);
      if pad_the_line then 
        begin
	  if length < space_available then
	    write(f, ' ':space_available - length);
	end
      else print_string_formatted := left_margin + length;
    end
  else
    begin
      break_pos := find_nearest_break( (length DIV 2) );
      blank := space_available - (length - (break_pos+1)+1) - DEFAULT_INDENT;
      if (break_pos < space_available) and (blank >= 0) then
        begin  post_string(1, break_pos);
          if blank > 0 then write(f, ' ':blank);
          for i := break_pos+1 to length do write(f, str^[i]);
        end
      else
        begin
          break_pos := find_nearest_break(space_available - DEFAULT_INDENT);
          post_string(1, break_pos);
          while (length - (break_pos+1) + 1) > 
	        (space_available - DEFAULT_INDENT) do
            begin
              start := break_pos+1;
              if (start + space_available - 2*DEFAULT_INDENT) >= length then
                break_pos := length - DEFAULT_INDENT
              else
                break_pos := start + space_available - 2*DEFAULT_INDENT;
              break_pos := find_nearest_break(break_pos);
              post_string(start, break_pos);
            end;
          blank := space_available - (length - (break_pos+1)+1) - DEFAULT_INDENT;
          if blank > 0 then write(f, ' ':blank);
          for i := break_pos+1 to length do write(f, str^[i]);
        end;
    end;
  if print_CRLF then writeln(f);
end { print_string_formatted } ;
