
(**)

{ In comments the entity described by a base_descriptor, namely a piece
  of a signal instance, will be called a 'virtual signal instance'. The
  name derives from the fact that a base_descriptor together with its
  signal instance, can be take to define a collection of information
  identical to a signal instance that has a subset of the BD's signal 
  instance's bits. }

procedure synonym_1st_BD_to_2nd_BD(BD, base_BD: base_descriptor_ptr);

{ Routine to synonym BD to base_BD i.e. base_BD is baser. Part of the
  synonyming process consists of copying the properties from the appropriate
  bits of the non-base signal definition to the base signal definition.

  NOTE: This routine assumes that base_BD describes a virtual signal instance
        that is a simple (and canonical) subrange. 
	synonym_base_descriptors ensures that this is the case.              }

var
  found,                           { sig def has basescript that overlaps BD }
  done,                            { all bits of BD added to basescript list }
  too_far:           boolean;      { will not find more bits of BD on - " -- }
  direction:  -1..1;               { -1 right_to_left bit subscripts else 1  }
  head_basescript,                 { head of BD's sig def's basescript list  }
  last_basescript:   basescript_ptr; { predecessor of head_basescript        }
  basescript_to_add: basescript;     { just what is says }
  base_sig_def,
  non_base_sig_def: signal_definition_ptr;
  base_bits: subscript_ptr;
  base_BD_width,
  i,
  non_base_bit: bit_range;
  non_base_instance_direction,
  base_instance_direction: -1..1;
  props: property_ptr;


  function basescripts_overlap: boolean;
    { return TRUE if left index of basescript_to_add occurs within
      the range of head_bascript's indices. }
    var
      ans: boolean;                { => basescripts have bits in common }
  begin
    if debug_16 then writeln(outfile, ' Entered basescripts_overlap ');
        
    if left_to_right then
      ans := (basescript_to_add.left_index >= head_basescript^.left_index) AND
             (basescript_to_add.left_index <= head_basescript^.right_index)
    else
      ans := (basescript_to_add.left_index <= head_basescript^.left_index) AND
             (basescript_to_add.left_index >= head_basescript^.right_index);

    basescripts_overlap := ans;

    if debug_16 then writeln(outfile, ' Exited basescripts_overlap ', ans:5);
  end { of basescripts_overlap };


  function basescript_to_add_occurs_earlier: boolean;
    { return TRUE if left index of basescript_to_add occurs before
      the left index of head_basescript  in the list }
    var
      ans: boolean;
  begin
    if left_to_right then
      ans := basescript_to_add.left_index < head_basescript^.left_index
    else
      ans := basescript_to_add.left_index > head_basescript^.left_index;
     basescript_to_add_occurs_earlier := ans;
  end { basescript_to_add_occurs_earlier };


(**)


  function add_basescript_at_left: boolean;
  { the basescript to be added has a bit range which starts to the left of
    head_basecript and to the right of last_basescript.  It may overlap
    head_basescript on the right. }
   var
     can_expand: boolean;           { can "insert" new basescript by expanding
                                      last_basescript                        }
     temp_basescript: basescript_ptr;
     done: boolean;                 { TRUE if we didn't have to split }
  begin 
    if debug_16 then
      begin
        writeln(outfile,' Entered add_basescript_at_left, defs basescripts:');
        dump_basescript_list(outfile,BD^.instance^.defined_by^.synonym_bits);
        writeln(outfile,' basescripts from last_basescript');
        dump_basescript_list(outfile, last_basescript);
      end;

    if last_basescript = NIL then { basescript list is empty }
      begin
        new_basescript(temp_basescript);
        temp_basescript^ := basescript_to_add;
        temp_basescript^.next := BD^.instance^.defined_by^.synonym_bits;
        BD^.instance^.defined_by^.synonym_bits := temp_basescript;
	last_basescript := temp_basescript;
      end
    else
      begin
        can_expand := FALSE;  { => can insert new BS by exp last_basescript }
        with last_basescript^ do
          if (instance = basescript_to_add.instance) then
            if basescript_to_add.offset = 
	      (offset + 1 + ABS (left_index - right_index)) then
              if basescript_to_add.left_index =
                 (right_index + direction) then
                can_expand := TRUE;
        if can_expand then
          begin
            { could generalize to see if can expand head_basescript to eat
              new BS, however, first case arises from bit wise lookups so
              it is more important }
            last_basescript^.right_index := basescript_to_add.right_index;
          end
        else
          begin
            new_basescript(temp_basescript);
            temp_basescript^ := basescript_to_add;
            temp_basescript^.next := last_basescript^.next;
            last_basescript^.next := temp_basescript;
	    last_basescript := temp_basescript;
          end;
      end;

	{ handle overlap by deferring to next pass }

      if head_basescript <> NIL then
	if left_to_right then 
	  if last_basescript^.right_index >= head_basescript^.left_index then
	    begin
	      basescript_to_add.left_index := head_basescript^.left_index;
	      last_basescript^.right_index := head_basescript^.left_index - 1;
	      done := FALSE;
	    end
	  else done := TRUE
	else
	  if last_basescript^.right_index <= head_basescript^.left_index then
	    begin
	      basescript_to_add.left_index := head_basescript^.left_index;
	      last_basescript^.right_index := head_basescript^.left_index + 1;
	      done := FALSE;
	    end
	  else done := TRUE;

    add_basescript_at_left := done;

    if debug_16 then
      begin
        writeln(outfile,' Exited add_basescript_at_left(', done, 
		'), defs basescripts:');
        dump_basescript_list(outfile,BD^.instance^.defined_by^.synonym_bits);
      end;
  end { of add_basescript_at_left };

(**)

  function add_basescript_at_right: boolean;
  { add the basescript to the right }
    var
      can_expand: boolean;
  begin  { try to expand last basescript element b4 adding new one }
    can_expand := FALSE;
    with last_basescript^ do
      if (instance = basescript_to_add.instance) then
        if (basescript_to_add.offset =
           (offset + 1 +  ABS(left_index - right_index))) then
          if (basescript_to_add.left_index =
             (right_index + direction)) then
            can_expand := TRUE;
    if can_expand then
      last_basescript^.right_index := basescript_to_add.right_index
    else
      begin
        new_basescript(last_basescript^.next);
        last_basescript^.next^ := basescript_to_add;
      end;
    add_basescript_at_right := TRUE;
  end { add_basescript_at_right } ;


  function splice_in_basescript: boolean;
    { splice the basescript into the current basescript }
    var
      width_of_added_piece: bit_range;


    procedure split_basescript(B: basescript_ptr; split: bit_range);
      { split the given basescript }
      var
        new_B: basescript_ptr;
    begin
      if debug_16 then
        begin
          write(outfile, ' Entered split_basescript: ');
          writeln(outfile, ' split = ', split:1, ' basescript = ');
          dump_basescript(outfile, B);
        end;

      if (left_to_right) AND 
         ((split < B^.left_index) or (split > B^.right_index)) OR
         (NOT left_to_right) AND 
         ((split > B^.left_index) or (split < B^.right_index)) then
        assert(70 { out of range split point } )
      else
        begin
          new_basescript(new_B);   new_B^.next := B^.next;   B^.next := new_B;
          new_B^.instance := B^.instance;
          new_B^.left_index := split + direction;
          new_B^.right_index := B^.right_index;  B^.right_index := split;
          new_B^.offset := B^.offset + 1 + ABS(B^.left_index-B^.right_index);
        end;

      if debug_16 then
        begin
          write(outfile, ' Exited split_basescript');
          writeln(outfile, ' basescript = ');
          dump_basescript(outfile, B);  dump_basescript(outfile, B^.next);
        end;
    end { split_basescript } ;

(**)
  begin { splice_in_basescript }
    if debug_16 then
      begin
        writeln(outfile, ' Entered splice_in_basescript -- basescript = ');
        dump_basescript_list(outfile, head_basescript);
      end;

    with head_basescript^ do
      if left_to_right then
        { process subscripts for LEFT to RIGHT bit ordering }

        if basescript_to_add.left_index = left_index then
          begin
            if basescript_to_add.right_index = right_index then
              begin { EQ }
                instance := basescript_to_add.instance;
                offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index < right_index then
              begin { LT }
                split_basescript(head_basescript,
                                 basescript_to_add.right_index);
                instance := basescript_to_add.instance;
                offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index > right_index then
              begin { GT }
                instance := basescript_to_add.instance;
                offset   := basescript_to_add.offset;
                width_of_added_piece := 1 + (right_index - left_index);
                basescript_to_add.offset     := basescript_to_add.offset
                                                + width_of_added_piece;
                basescript_to_add.left_index := basescript_to_add.left_index
                                                + width_of_added_piece;
                splice_in_basescript := FALSE;
              end;
          end { basescript_to_add.left_index = head_basescript^.left_index }
        else
          begin
            if basescript_to_add.right_index = right_index then
              begin { EQ }
                split_basescript(head_basescript,
                                 basescript_to_add.left_index - 1);
                next^.instance := basescript_to_add.instance;
                next^.offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index < right_index then
              begin { LT }
                split_basescript(head_basescript,
                                 basescript_to_add.left_index - 1);
                split_basescript(head_basescript^.next,
                                 basescript_to_add.right_index);
                next^.instance := basescript_to_add.instance;
                next^.offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index > right_index then
              begin { GT }
                split_basescript(head_basescript,
                                 basescript_to_add.left_index - 1);
                next^.instance := basescript_to_add.instance;
                next^.offset   := basescript_to_add.offset;
                width_of_added_piece := 1 + (right_index
                                              - basescript_to_add.left_index);
                basescript_to_add.offset     := basescript_to_add.offset
                                                + width_of_added_piece;
                basescript_to_add.left_index := basescript_to_add.left_index
                                                + width_of_added_piece;
                splice_in_basescript := FALSE;
              end
          end { basescript_to_add.left_index <> head_basescript^.left_index }
      else

        { process bit subscripts for RIGHT to LEFT bit ordering }

        if basescript_to_add.left_index = left_index then
          begin
            if basescript_to_add.right_index = right_index then
              begin { EQ }
                instance := basescript_to_add.instance;
                offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index > right_index then
              begin { GT }
                split_basescript(head_basescript,
                                 basescript_to_add.right_index);
                instance := basescript_to_add.instance;
                offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index < right_index then
              begin { LT }
                instance := basescript_to_add.instance;
                offset   := basescript_to_add.offset;
                width_of_added_piece := 1 + (left_index - right_index);
                basescript_to_add.offset     := basescript_to_add.offset
                                                + width_of_added_piece;
                basescript_to_add.left_index := basescript_to_add.left_index
                                                - width_of_added_piece;
                splice_in_basescript := FALSE;
              end;
          end { basescript_to_add.left_index = head_basescript^.left_index }
        else
          begin
            if basescript_to_add.right_index = right_index then
              begin { EQ }
                split_basescript(head_basescript,
                                 basescript_to_add.left_index + 1);
                next^.instance := basescript_to_add.instance;
                next^.offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index > right_index then
              begin { GT }
                split_basescript(head_basescript,
                                 basescript_to_add.left_index + 1);
                split_basescript(head_basescript^.next,
                                 basescript_to_add.right_index);
                next^.instance := basescript_to_add.instance;
                next^.offset   := basescript_to_add.offset;
                splice_in_basescript := TRUE;
              end
            else if basescript_to_add.right_index < right_index then
              begin { LT }
                split_basescript(head_basescript,
                                 basescript_to_add.left_index + 1);
                next^.instance := basescript_to_add.instance;
                next^.offset   := basescript_to_add.offset;
                width_of_added_piece := 1 + (basescript_to_add.left_index
                                             - right_index);
                basescript_to_add.offset     := basescript_to_add.offset
                                                + width_of_added_piece;
                basescript_to_add.left_index := basescript_to_add.left_index
                                                - width_of_added_piece;
                splice_in_basescript := FALSE;
              end;
          end { basescript_to_add.left_index <> head_basescript^.left_index };

    if debug_16 then
      begin
        writeln(outfile, ' Exited splice_in_basescript -- basescript = ');
        dump_basescript_list(outfile, head_basescript);
      end;

  end { splice_in_basescript };

(**)

begin { synonym_1st_BD_to_2nd_BD }
  if debug_6 then
    begin
      writeln(outfile, 'Entered synonym_1st_BD_to_2nd_BD');

      write(outfile, ' BD: ');
      dump_base_descriptor(outfile, BD);
      write(outfile, '  BD non base bits: ');
      dump_basescript_list(outfile, BD^.instance^.defined_by^.synonym_bits);
      writeln(outfile, '  BD properties: ');
      dump_bit_properties(outfile, BD^.instance^.defined_by^.properties);

      write(outfile, ' base_BD: ');
      dump_base_descriptor(outfile, base_BD);
      writeln(outfile, '  base_BD properties: ');
      dump_bit_properties(outfile, base_BD^.instance^.defined_by^.properties);
    end;

  if left_to_right then direction := 1
                   else direction := -1;

  { initialize basescript_to_add -- note that this initialization
    guarantees a canonical simple subrange. }
  with basescript_to_add do
    begin
      next := NIL;
      instance := base_BD^.instance;
      offset := base_BD^.offset;
      left_index := BD^.offset + 1;
      if NOT nth_bit_of_signal_instance(left_index, BD^.instance) then
        assert(92 { oops, dat bit aint dere });
      right_index := left_index + direction * (BD^.width - 1);
    end;

  if debug_6 or debug_16 then
    begin
      writeln(outfile, ' Basescript to add: ');
      write(outfile , '  Instance: ');
      dump_signal_instance(outfile, basescript_to_add.instance);
      writeln(outfile, '  Offset: ', basescript_to_add.offset:1);
      write(outfile, '  left_index: ', basescript_to_add.left_index:1);
      writeln(outfile, '  right_index: ', basescript_to_add.right_index:1);
    end;
 
{ -- Synonym the first base descriptor to the second base descriptor -- }

  head_basescript := BD^.instance^.defined_by^.synonym_bits;

  if head_basescript = NIL then 
    begin
      new_basescript(head_basescript);
      head_basescript^ := basescript_to_add;
      BD^.instance^.defined_by^.synonym_bits := head_basescript;
    end
  else
    begin
      done := FALSE;  last_basescript := NIL;
      while NOT done do
        begin
          found := FALSE; too_far := FALSE;
          while (head_basescript <> NIL) AND (NOT found) AND (NOT too_far) do
            if basescript_to_add_occurs_earlier then too_far := TRUE
            else if basescripts_overlap then found := TRUE
            else
              begin
                last_basescript := head_basescript;
                head_basescript := head_basescript^.next;
              end;
          if too_far then  { add new basescript just left of head_basescript }
            done := add_basescript_at_left
          else if NOT found then { add new basescrpt at end of basescrpt lst }
            done := add_basescript_at_right
          else { found subrange that overlaps us - time to do some splitting }
            done := splice_in_basescript;
        end;
    end;

  { Do the property munging -- copy properties from non-base guy to base guy.
    This implementation can be improved in two ways:

      1) Do not do it bit-wise.
      2) MOVE the properties instead of copying them }

  base_sig_def := base_BD^.instance^.defined_by;
  non_base_sig_def := BD^.instance^.defined_by;
  if non_base_sig_def^.properties <> NIL then
    begin
      if base_sig_def^.kind = VECTOR then
        begin
          if base_BD^.instance^.bit_subscript^.left_index >
             base_BD^.instance^.bit_subscript^.right_index then
             base_instance_direction := -1
           else
             base_instance_direction := +1;

          base_bits := NIL;  new_subscript(base_bits);
          base_bits^.left_index := base_BD^.offset + 1; { ordinal of index }
	  if not nth_bit_of_signal_instance(base_bits^.left_index,
	                                    base_BD^.instance) then ;
          { base_bits^.left_index is now cardinal }
          base_bits^.right_index := base_bits^.left_index;
        end
      else
        begin  base_bits := NIL;  base_instance_direction := +1; end;

      if non_base_sig_def^.kind = VECTOR then
        begin
          if BD^.instance^.bit_subscript^.left_index >
             BD^.instance^.bit_subscript^.right_index then
            non_base_instance_direction := -1
          else
            non_base_instance_direction := +1;
          non_base_bit := BD^.offset + 1;
	  if not nth_bit_of_signal_instance(non_base_bit, BD^.instance) then
	    assert(92 { bit wasn't there });
	  { non_base_bit is now cardinal }
        end
       else
         begin  non_base_bit := 0; non_base_instance_direction := +1; end;
       
      base_BD_width := base_BD^.width;
      for i:= 1 to base_BD_width do
        begin
          props := bit_properties_of_this_bit(non_base_sig_def^.properties,
                                              non_base_bit,
                                              INHERIT_PROPERTIES);
          copy_props_to_bits_of_def(props, base_bits, base_sig_def);
          if i < base_BD_width then
            begin
              with base_bits^ do
                begin
                  left_index  := left_index + base_instance_direction;
                  right_index := left_index;
                end;
    
              non_base_bit := non_base_bit + non_base_instance_direction;
            end { if ... };
        end { for each bit of the VSI do ... };
    end { if there are properties to copies };

  if debug_6 then
    begin
      writeln(outfile, 'Exited synonym_1st_BD_to_2nd_BD: ');
      dump_signal_definition(outfile, BD^.instance^.defined_by);
      write(outfile, ' non-base bits: ');
      dump_basescript_list(outfile, BD^.instance^.defined_by^.synonym_bits);
      writeln(outfile, ' base_BD properties: ');
      dump_bit_properties(outfile, base_BD^.instance^.defined_by^.properties);
    end;
end { synonym_1st_BD_to_2nd_BD } ;


#include "bsofbs.p"
(**)


procedure synonym_base_descriptors(BD1, BD2: base_descriptor_ptr);
  { synonym the two base descriptors }
type
  base_choice = (FIRST_BD, SECOND_BD, NEITHER_BD, SAME_BDS);
var
  i: bit_range;
  base_of_BD1,
  base_of_BD2,
  next_bit_of_BD1,
  next_bit_of_BD2: base_descriptor_ptr;


  procedure init;
    { initialize the basescripts }
  begin
    base_of_BD1 := NIL;   new_base_descriptor(base_of_BD1);
    base_of_BD2 := NIL;   new_base_descriptor(base_of_BD2);
  end { init } ;


  procedure un_init;
    { release temporary basescripts }
  begin
    release_base_descriptor(base_of_BD1);release_base_descriptor(base_of_BD2);
  end { un_nit } ;


  function choose_better_base(BD1, BD2: base_descriptor_ptr): base_choice;
    { This takes two base descriptors and decides which ones virtual signal
      instance is a better base signal. A heuristic is used whose selection
      rules are (in order):

      1) Choose lower bit number if SIs are different bits of same signal
      2) Choose a constant signal.
      3) Choose the signal with name properties.
      4) Choose root level interface signal
      5) Choose the signal with most global scope.
      6) Choose the non-NC signal.
      7) Choose the signal with a nice name (no unnamed e.g.).
      8) Choose a scalar over a vector
      9) Choose the signal which is lexicographically smallest.
     10) Choose signal whose NET_ID is lexicographically smallest.

      The routine returns a "code" with the following meaning:

            FIRST_BD:     BD1 wins.
            SECOND_BD:    BD2 wins.
            NEITHER_BD:   neither wins - perform bit-by-bit synonym.
            SAME_BDS:     if single bit of same instances synonymed. }
    var
      winner: base_choice;             { first, second or no BD is best base }
      BD1_has_name_prop,               { what it says }
      BD2_has_name_prop: boolean;      { what it says }

      def1_is_interface,               { TRUE if def1 is root interface }
      def2_is_interface: boolean;      { TRUE if def2 is root interface }
      sig_def1,                        { BD1's signal definition }
      sig_def2: signal_definition_ptr; { BD2's signal definition }


    function least_funny_name(sig_def1, sig_def2: signal_definition_ptr): 
      base_choice;
      { return which base (if either) has the least funny name (NCs are 
        funnier than unnamed signals are funnier than named signals). }
    begin
      if is_NC_signal(sig_def1^.signal^.name) then
	if is_NC_signal(sig_def2^.signal^.name) then 
	  least_funny_name := NEITHER_BD
	else least_funny_name := SECOND_BD
      else if is_NC_signal(sig_def2^.signal^.name) then
	least_funny_name := FIRST_BD
      else if is_UNNAMED_signal(sig_def1^.signal^.name) then
	if is_UNNAMED_signal(sig_def2^.signal^.name) then 
	  least_funny_name := NEITHER_BD
	else least_funny_name := SECOND_BD
      else if is_UNNAMED_signal(sig_def2^.signal^.name) then
	least_funny_name := FIRST_BD
      else least_funny_name := NEITHER_BD;
    end { least_funny_name } ;


    function compare_virtuals(def1, def2: signal_definition_ptr): base_choice;
      { compare the two given signals.  It is assumed that they have the same
        signal names and signal kind.  Choose between them as follows:
            1.  Choose non-virtual over virtual.
            2.  Choose smaller NET_ID.
        It is an error to find that both are non-virtual. }
    begin
      if debug_2 then writeln(outfile, 'Entering compare_virtuals');

      compare_virtuals := FIRST_BD;

      if def1^.is_virtual_base then
        { first signal is non-virtual }

        if def2^.is_virtual_base then    { second signal is also non-virtual }
	  begin
	    if (def1^.polarity = NORMAL) and
	       (def2^.polarity = COMPLEMENTED) or
	       (def1^.polarity = COMPLEMENTED) and
	       (def2^.polarity = NORMAL) then
	      begin
	        { This message really belongs somewhere else -- won't catch
		  all possible cases here }
	        error(235 { signal synonymed to its complement });
	        error_dump_signal_def(sig_def1);
	        error_dump_signal_def(sig_def2);
	      end
	    else
	      begin
                assert(196 { 2 non-virtual signals ! });
                dump_signal_definition(CmpLog, sig_def1);
                dump_signal_definition(CmpLog, sig_def2);
	      end;

            case compare_strings(sig_def1^.net_id, sig_def2^.net_id) of
              LT: compare_virtuals := FIRST_BD;
              EQ: compare_virtuals := FIRST_BD;   {******** NOT GOOD! *******}
              GT: compare_virtuals := SECOND_BD;
            end
          end
        else
          compare_virtuals := FIRST_BD
      else
        if def2^.is_virtual_base then      { second signal is non-virtual }
          compare_virtuals := SECOND_BD

        else
          { neither signal is non-virtual: choose based on NET_ID }

          case compare_strings(sig_def1^.net_id, sig_def2^.net_id) of
            LT: compare_virtuals := FIRST_BD;
            EQ: begin
                  assert(193 { this better not happen });
                  dump_signal_definition(CmpLog, sig_def1);
                  dump_signal_definition(CmpLog, sig_def2);

                  compare_virtuals := FIRST_BD;
                end;
            GT: compare_virtuals := SECOND_BD;
          end;         
    end { compare_virtuals } ;


    procedure synonym_same_signals;
      { synonym the two given signals.  It is known that they are the
        same signal definition or share the same virtual base signal. }
      var
        bit1,                          { if BD1 is 1 bit this is bit it refs }
        bit2: bit_range;               { if BD2 is 1 bit this is bit it refs }
    begin
      if debug_2 then writeln(outfile, '-same signal');

      { we have synonymed different bits of the same signal }

      bit1 := BD1^.offset + 1;
      bit2 := BD2^.offset + 1;

      if BD1^.width = 1 then
        begin
          if nth_bit_of_signal_instance(bit1, BD1^.instance) then {nuttin} ;
          if nth_bit_of_signal_instance(bit2, BD2^.instance) then {nuttin} ;

          if bit1 < bit2 then      winner := FIRST_BD
          else if bit1 > bit2 then winner := SECOND_BD
          else                     winner := SAME_BDS;
            { quick_synonym relies on the fact that SAME_BDS is returned
              only for single bit BDs to do error detection/recovery }
        end
      else
        begin 
          { !!!! add code to handle non-overlapping subranges:
                       S<0..3> <-> S<4..7>
            for now, choose nobody forcing bitwise lookup }

          winner := NEITHER_BD;
        end;
    end { synonym_same_signals } ;


  begin { choose_better_base }
    if debug_2 then
      begin
        writeln(outfile, 'Entered choose_better_base with:');
        write(outfile, '  BD1: ');
        dump_base_descriptor(outfile, BD1);
        write(outfile, '  BD2: ');
        dump_base_descriptor(outfile, BD2);
      end;
      
    sig_def1 := BD1^.instance^.defined_by;
    sig_def2 := BD2^.instance^.defined_by;

    {-----------------------------------------------}
    { check for synonym of instances of same signal }
    {-----------------------------------------------}

    if (sig_def1 = sig_def2) then  synonym_same_signals

    { comparing 2 different signals }

    else
      begin
        winner := NEITHER_BD;

        {---------------------}
        { check for constants }
        {---------------------}

        if sig_def1^.is_const then
          begin
            if debug_2 then writeln(outfile, '-BD1 is constant');

            if sig_def2^.is_const then
              if NOT is_same_constant(BD1, BD2) then
                begin
                  error(155 { can't synonym different constants });
                  error_dump_signal_def(sig_def1);
                  error_dump_signal_def(sig_def2);

                  winner := FIRST_BD; { randomly choose the first base desc. }
                end
              else
                { same constants: choose among virtual signals }

                winner := compare_virtuals(sig_def1, sig_def2)
            else
              winner := FIRST_BD;
          end

        else
          { BD1 is not a constant }

          if sig_def2^.is_const then
            winner := SECOND_BD
          else
            begin
              if debug_2 then writeln(outfile, '-neither is constant');

              {--------------------------------------------}
              { check for the existence of name properties }
              {--------------------------------------------}

              BD1_has_name_prop := 
                     signal_name_has_timing_assertion(sig_def1^.signal^.name);
              BD2_has_name_prop := 
                     signal_name_has_timing_assertion(sig_def2^.signal^.name);

              if BD1_has_name_prop AND BD2_has_name_prop then
                if virtual_base(sig_def1) <> virtual_base(sig_def2) then
                  begin
                    error(161 { synonym of 2 sigs with name props });
                    error_dump_mtree_node(current_mtree_node);
                    error_dump_signal_instance(BD1^.instance);
                    error_dump_signal_instance(BD2^.instance);
                  end;

              if debug_2 and (BD1_has_name_prop or BD2_has_name_prop) then
                writeln(outfile, '-one has name prop');

              if BD1_has_name_prop AND NOT BD2_has_name_prop then
                winner := FIRST_BD
              else if BD2_has_name_prop AND NOT BD1_has_name_prop then
                winner := SECOND_BD

              {-------------------------------------------}
              { no name properties: check for more global }
              {-------------------------------------------}

              else if sig_def1^.node^.level <> sig_def2^.node^.level then
                begin
                  if debug_2 then writeln(outfile, '-Different levels');

                  if sig_def1^.node^.level < sig_def2^.node^.level then
                    winner := FIRST_BD
                  else if sig_def1^.node^.level > sig_def2^.node^.level then
                    winner := SECOND_BD
                end
              else if (sig_def1^.scope = GLOBAL) and
                      (sig_def2^.scope = LOCAL) then
                winner := FIRST_BD
              else if (sig_def2^.scope = GLOBAL) and
                      (sig_def1^.scope = LOCAL) then
                winner := SECOND_BD

              {-----------------------------------------------------}
              { same scope: check for a root level interface signal }
              {-----------------------------------------------------}

              else
                begin
                  def1_is_interface := (sig_def1^.node^.father_node = NIL) and
                                       (sig_def1^.scope = XINTERFACE);
                  def2_is_interface := (sig_def2^.node^.father_node = NIL) and
                                       (sig_def2^.scope = XINTERFACE);

                  if debug_2 and (def1_is_interface or def2_is_interface) then
                    writeln(outfile, '-one is interface');

                  if def1_is_interface and not def2_is_interface then
                    winner := FIRST_BD
                  else if def2_is_interface and not def1_is_interface then
                    winner := SECOND_BD
                   
                  {-------------------------------------}
                  { same scope: check for strange names }
                  {-------------------------------------}

                  else
                    begin
		      winner := least_funny_name(sig_def1, sig_def2);

                      if winner = NEITHER_BD then
                        begin
                          if debug_2 then
                            writeln(outfile, '-equally funny or non-funny');

                          {-------------------------}
                          { check VECTOR vs. SCALAR }
                          {-------------------------}

                          if (sig_def1^.kind = SINGLE) AND
                             (sig_def2^.kind = VECTOR) then
                            winner := FIRST_BD
                          else if (sig_def2^.kind = SINGLE) AND
                                  (sig_def1^.kind = VECTOR) then
                                 winner := SECOND_BD
                          else
                            begin
                              if debug_2 then
                                writeln(outfile, '-neither wins as scalar');

                              { both SCALAR or VECTOR: compare the names.
                                If the names are the same, choose among
                                virtual signals. }

                              case compare_strings(sig_def1^.signal^.name,
                                                   sig_def2^.signal^.name) of
                                LT: winner := FIRST_BD;
                                EQ: winner := compare_virtuals(sig_def1,
                                                               sig_def2);
                                GT: winner := SECOND_BD;
                              end;
                            end;
                        end;
                    end { same scope, not interface } ;
                end { signals at same scope } ;
            end { neither sig_def1 or sig_def2 are constants };
      end { BD1 and BD2 describe bits of different signal definitions } ;

    choose_better_base := winner;
      
    if debug_2 then
      begin
        write(outfile,  'Exited choose_better_base');
	writeln(outfile,' (', ord(winner):1,')');
      end;
  end { choose_better_base } ;


  function quick_synonym(BD1, BD2: base_descriptor_ptr): boolean;
    { create synonym }
    const
      shorten_chains = FALSE;
    var
      which_BD: base_choice;
  begin

  if debug_6 then
    begin
      writeln(outfile, 'Entered quick_synonym with: ');

      write(outfile, ' BD1: ');
      dump_base_descriptor(outfile, BD1);
      write(outfile, '  Signal def''s basescript list: ');
      dump_basescript_list(outfile, BD1^.instance^.defined_by^.synonym_bits);
      writeln(outfile);

      write(outfile, ' BD2: ');
      dump_base_descriptor(outfile, BD2);
      write(outfile, '  Signal def''s basescript list: ');
      dump_basescript_list(outfile, BD2^.instance^.defined_by^.synonym_bits);
      writeln(outfile);
    end;

    which_BD := NEITHER_BD;
{ !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  To ensure that we have synonym chains of length one we need to do the
  following in quick synonym (discussion below assumes that better base is 1):

         synonym base_of_BD2 to base_of_BD1
         if BD2 <> base_of_BD2 then
           remove BD2 <--> base_of_BD2 synonym
           synonym BD2 <--> base_of_BD1
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! }

    if find_base_of_base_descriptor(BD1,base_of_BD1) then
      if find_base_of_base_descriptor(BD2,base_of_BD2) then
        begin
          which_BD := choose_better_base(base_of_BD1, base_of_BD2);
          if which_BD = FIRST_BD then
            begin
              synonym_1st_BD_to_2nd_BD(base_of_BD2, base_of_BD1);
              if shorten_chains then
                if (BD2^.instance <> base_of_BD2^.instance) OR
                   (BD2^.offset   <> base_of_BD2^.offset) then
                  synonym_1st_BD_to_2nd_BD(BD2, base_of_BD1)
            end
          else if which_BD = SECOND_BD then
            begin
              synonym_1st_BD_to_2nd_BD(base_of_BD1, base_of_BD2);
              if shorten_chains then
                if (BD1^.instance <> base_of_BD1^.instance) OR
                   (BD1^.offset   <> base_of_BD1^.offset) then
                  synonym_1st_BD_to_2nd_BD(BD1, base_of_BD2)
            end
          else if which_BD = NEITHER_BD then
            { nuttin }
          else if which_BD = SAME_BDs then
            { nuttin }
        end;

  quick_synonym := (which_BD <> NEITHER_BD);

  if debug_6 AND (which_BD = FIRST_BD) then
    begin
      writeln(outfile, 'Exited quick_synonym with (FIRST_BD): ');
      write(outfile, ' sig_def1: ');
      dump_signal_definition(outfile, base_of_BD1^.instance^.defined_by);
      write(outfile, '  non-base bits: ');
      dump_basescript_list(outfile,
                           base_of_BD1^.instance^.defined_by^.synonym_bits);

      write(outfile, ' sig_def2: ');
      dump_signal_definition(outfile, BD2^.instance^.defined_by);
      writeln(outfile, '  non-base bits: ');
      dump_basescript_list(outfile, BD2^.instance^.defined_by^.synonym_bits);
    end;

  if debug_6 AND (which_BD = SECOND_BD) then
    begin
      writeln(outfile, 'Exited quick_synonym with (SECOND_BD): ');
      write(outfile, ' sig_def1: ');
      dump_signal_definition(outfile, BD1^.instance^.defined_by);
      write(outfile, '  non-base bits: ');
      dump_basescript_list(outfile, BD1^.instance^.defined_by^.synonym_bits);

      write(outfile, ' sig_def2: ');
      dump_signal_definition(outfile, base_of_BD2^.instance^.defined_by);
      writeln(outfile, '  non-base bits: ');
      dump_basescript_list(outfile,
                           base_of_BD2^.instance^.defined_by^.synonym_bits);
    end;

  if debug_6 AND (which_BD = NEITHER_BD) then
    begin
      writeln(outfile, 'Exited quick_synonym with (NEITHER_BD): ');
      write(outfile, ' sig_def1: ');
      dump_signal_definition(outfile, BD1^.instance^.defined_by);
      write(outfile, '  non-base bits: ');
      dump_basescript_list(outfile, BD1^.instance^.defined_by^.synonym_bits);

      write(outfile, ' sig_def2: ');
      dump_signal_definition(outfile, BD2^.instance^.defined_by);
      writeln(outfile, '  non-base bits: ');
      dump_basescript_list(outfile, BD2^.instance^.defined_by^.synonym_bits);
    end;

  if debug_6 AND (which_BD = SAME_BDs) then
    begin
      writeln(outfile, 'Exited quick_synonym with (SAME_BDS): ');
      write(outfile, ' sig_def1: ');
      dump_signal_definition(outfile, BD1^.instance^.defined_by);
      write(outfile, '  non-base bits: ');
      dump_basescript_list(outfile, BD1^.instance^.defined_by^.synonym_bits);

      write(outfile, ' sig_def2: ');
      dump_signal_definition(outfile, BD2^.instance^.defined_by);
      write(outfile, '  non-base bits: ');
      dump_basescript_list(outfile, BD2^.instance^.defined_by^.synonym_bits);

      write(outfile, ' base: ');
      dump_signal_definition(outfile, base_of_BD1^.instance^.defined_by);
      writeln(outfile, '  non-base bits: ');
      dump_basescript_list(outfile,
                           base_of_BD1^.instance^.defined_by^.synonym_bits);
    end;
  end { quick_synonym } ;


(**)


begin { synonym_base_descriptors }
  if debug_6 then
    begin
      writeln(outfile, 'Entered synonym_base_descriptors with: ');
      write(outfile, ' BD1: ');
      dump_base_descriptor(outfile, BD1);
      write(outfile,' sig_def1''s non_base bits: ');
      dump_basescript_list(outfile, BD1^.instance^.defined_by^.synonym_bits);

      write(outfile, ' BD2: ');
      dump_base_descriptor(outfile, BD2);
      write(outfile, ' sig_def2''s non_base bits: ');
      dump_basescript_list(outfile, BD2^.instance^.defined_by^.synonym_bits);
    end;

  init;

  if quick_synonym(BD1,BD2) then { we are all done }
  else
    begin
      next_bit_of_BD1 := NIL;    new_base_descriptor(next_bit_of_BD1);
      next_bit_of_BD1^ := BD1^;  next_bit_of_BD1^.width := 1;
      next_bit_of_BD2 := NIL;    new_base_descriptor(next_bit_of_BD2);
      next_bit_of_BD2^ := BD2^;  next_bit_of_BD2^.width := 1;
      for i := 1  to BD1^.width do
        begin
          if NOT quick_synonym(next_bit_of_BD1, next_bit_of_BD2) then
            assert(106 { quick synonym better always win on 1 bit BDs});

          with next_bit_of_BD1^ do offset := offset + 1;
          with next_bit_of_BD2^ do offset := offset + 1;
        end;

      release_base_descriptor(next_bit_of_BD1);
      release_base_descriptor(next_bit_of_BD2);
    end;

  un_init;

  if debug_6 then
    begin
      writeln(outfile, 'Exited synonym_base_descriptors with: ');
      write(outfile, ' BD1: ');
      dump_base_descriptor(outfile, BD1);
      write(outfile, ' sig_def1''s non_base bits: ');
      dump_basescript_list(outfile, BD1^.instance^.defined_by^.synonym_bits);

      write(outfile, ' BD2: ');
      dump_base_descriptor(outfile, BD2);
      write(outfile, ' sig_def2''s non_base bits: ');
      dump_basescript_list(outfile, BD2^.instance^.defined_by^.synonym_bits);
    end;
end { synonym_base_descriptors } ;

