(**)     { ------- fix width of a signal instance ------- }


procedure fix_instance_width(instance: signal_instance_ptr;
                             width: bit_range);
  { fix the given instance (INSTANCE) to the given width (WIDTH).  It
    should be noted that ALL of the instances of a given signal
    definition will be fixed to the same width when any one of them
    is fixed.  This also applies to its associated virtual defs.  }
  var
    current_def: signal_definition_ptr;     { current def or virtual def }
    current_instance: signal_instance_ptr;  { current instance to fix }
    first_instance: boolean;                { TRUE if this is the first inst }
begin
  if debug_1 then
    begin
      writeln(outfile, 'Enter fix_instance_width: width=', width:1);
      dump_signal_instance(outfile, instance);
    end;

  first_instance := TRUE;

  if instance^.defined_by^.kind <> UNDEFINED then
    begin
      assert(5 { oops! });
      dump_signal_definition(CmpLog, instance^.defined_by);
    end;

  current_def := virtual_base(instance^.defined_by);

  while current_def <> NIL do with current_def^ do
    begin
      kind := VECTOR;
      if left_to_right then
        begin  left_index := 0;  right_index := width-1;  end
      else
        begin  right_index := 0;  left_index := width-1;  end;

      { fix the widths of all instances of the signal definition }

      if instances = NIL then assert(184 { NIL instances });
      current_instance := instances;

      while current_instance <> NIL do
        begin
          if current_instance^.bit_subscript <> NIL then
            assert(167 { cannot be });
          current_instance^.bit_subscript := create_a_subscript(width);

          if first_instance then
	    begin
	      { record one instance in the synonyms table to record the
	        bits as "used". }
(* currently redundant, with the bogus_cmptmp_list hack
	      if current_def^.node^.father_node = NIL then
	        record_instance_for_synonyms(current_instance);
*)
              first_instance := FALSE;
	    end;
          current_instance := current_instance^.next;
        end;
      current_def := next_virtual_def;
    end;

  if debug_1 then
    begin
      write(outfile, 'Exit fix_instance_width: ');
      dump_signal_definition(outfile, instance^.defined_by);
    end;
end { fix_instance_width } ;


(**)     { ------- check widths of actuals with the formal ------- }


function check_and_fix_formal_actual(formal_actual_pair: formal_actual_ptr;
                                     test_all_actuals: boolean;
                                     var width_was_fixed: boolean): boolean;
  { check the actuals connected to the formal given in the formal/actual
    list (FORMAL_ACTUAL_PAIR).  If the TEST_ALL_ACTUALS flag is TRUE,
    test all of the actuals connected to the pin, otherwise check only
    the first.  If the width of any signal is fixed, return WIDTH_WAS_FIXED
    TRUE.  If there were no errors encountered, return TRUE. }
  var
    formal_instance: signal_instance_ptr;  { pin name }
    actual: actual_list_ptr;          { current actual connected to pin }
    num_unresolved: bit_range;        { number actuals of unknown width }
    total_num_signals: bit_range;     { total # signals in concat signal }
    unresolved: signal_instance_ptr;  { actual param with unknown width }
    
    width_of_formal,                  { width (bits) of formal parameter }
    width_of_actual: bit_range;       { width of actual parameter }


  function width_of_propertied_CS(signal: propertied_CS_ptr;
                                  var num_unresolved: bit_range;
                                  var unresolved: signal_instance_ptr;
                                  var number_of_signals: bit_range):
                                                                    bit_range;
    { Calculate the width of a propertied concatenated signal.  If a signal
      of unknown width is found, then return a pointer to it (UNRESOLVED).
      Return the number of unknown width signals found (NUM_UNRESOLVED).
      Return the total number of signals in NUMBER_OF_SIGNALS.
      The width returned is always the combined width of those signals whose
      width is known. }
    var
      sig: propertied_CS_ptr;   { current piece of the signal }
      total_width,              { total width of the signal }
      width: bit_range;         { width of current signal }
  begin
    if debug_1 then
      begin
        writeln(outfile, 'Entered width_of_propertied_CS: signal=');
        dump_propertied_CS(outfile, signal);
      end;

    num_unresolved := 0;  unresolved := NIL;  total_width := 0;
    number_of_signals := 0;  sig := signal;

    while sig <> NIL do
      begin
        width := width_of_subscript(sig^.instance^.bit_subscript,
                                    sig^.instance^.defined_by^.kind);
        width := width * sig^.instance^.replication_factor;
        if width = 0 then   { has unknown width }
          begin
            num_unresolved := num_unresolved + 1;
            unresolved := sig^.instance;
          end
        else
          total_width := total_width + width;

        number_of_signals := number_of_signals + 1;

        sig := sig^.next;
      end;

    width_of_propertied_CS := total_width;

    if debug_1 then
      begin
        writeln(outfile, 'Exiting width_of_propertied_CS: width=',
                         total_width:1);
        writeln(outfile, '     num_unresolved: ', num_unresolved:1);
        write(outfile,   '         unresolved: ');
        dump_signal_instance(outfile, unresolved);
        writeln(outfile, '  number_of_signals: ', number_of_signals:1);
      end;
  end { width_of_propertied_CS } ;
  

  function fixed_as_special_case: boolean;
    { check for the special case that there is an N bit signal synonymed
      to N signals all of which are either 1 bit or UNDEFINED.  This is the
      case of an NMERGE feeding N unnamed signals. }
    var
      fixed: boolean;               { TRUE if signal was fixed }
      signal: propertied_CS_ptr;    { current signal in CS }
  begin
    fixed := FALSE;

    if total_num_signals = width_of_formal then
      if num_unresolved = (width_of_formal - width_of_actual) then
        begin
          { set each of the UNDEFINED signals to 1 bit }

          signal := actual^.signal;
          while (signal <> NIL) do
            begin
              if signal^.instance^.defined_by^.kind = UNDEFINED then
                begin
                  fix_instance_width(signal^.instance, 1);
                  width_was_fixed := TRUE;
                end;

              signal := signal^.next;
            end;

          actual^.width_is_unknown := FALSE;

          fixed := TRUE;
        end;

    fixed_as_special_case := fixed;

    if debug_1 then
      writeln(outfile, 'Exiting fixed_as_special_case: ', ord(fixed):1);
  end { fixed_as_special_case } ;


  procedure display_error_and_fix;
    { display a formal/actual width mismatch error and fix actual }
  begin
    error(200 { formal/actual parameter width mismatch });
    error_dump_current_parse_environment;
    error_dump_formal_instance(formal_actual_pair^.formal_parameter);
    error_dump_propertied_CS(actual^.signal);

    actual^.signal^.next := NIL;
    actual^.signal^.instance := build_NC_signal(width_of_formal);
    actual^.signal^.control := IGNORE_ALL;

    check_and_fix_formal_actual := FALSE;
  end { display_error_and_fix } ;


begin { check_and_fix_formal_actual }
  if debug_1 then
    begin
      writeln(outfile, 'Entering check_and_fix_formal_actual:  F/A=');
      dump_formal_actual(outfile, formal_actual_pair);
    end;

  check_and_fix_formal_actual := TRUE;
  width_was_fixed := FALSE;

  if formal_actual_pair <> NIL then
    begin
      formal_instance := formal_actual_pair^.formal_parameter;
      actual := formal_actual_pair^.actual_parameter;

      { NOTE:  formal parameters have no bit lists and no replication.  The
               width calculation can be made more efficient therefore. }

      if formal_actual_pair^.width <> 0 then
        width_of_formal := formal_actual_pair^.width
      else
        width_of_formal := width_of_subscript(formal_instance^.bit_subscript,
                                          formal_instance^.defined_by^.kind);

      if debug_1 then
        writeln(outfile, 'width of formal=', width_of_formal:1);

      repeat
        width_of_actual := width_of_propertied_CS(actual^.signal,
                                                  num_unresolved,
                                                  unresolved,
                                                  total_num_signals);

        { check to see if formal's width is known }

        if width_of_formal > 0 then
          begin
            if (width_of_actual > 0) and (num_unresolved = 0) then
              if width_of_formal = width_of_actual then    { all is ok }
              else
                display_error_and_fix
            else if num_unresolved = 1 then   { we know how to fix it up }
              if width_of_formal - width_of_actual <= 0 then
                display_error_and_fix
              else
                begin
                  fix_instance_width(unresolved,
                                     (width_of_formal-width_of_actual));
                  width_was_fixed := TRUE;
                  actual^.width_is_unknown := FALSE;
                end
            else if not fixed_as_special_case then
              actual^.width_is_unknown := TRUE;
          end
    
        { formal width is unknown;  check for actual width known }

        else if (width_of_actual > 0) and (num_unresolved = 0) then
          begin
            fix_instance_width(formal_instance, width_of_actual);
            width_was_fixed := TRUE;
            formal_actual_pair^.width := width_of_actual;
            width_of_formal := width_of_actual;
          end

        { neither the formal or actual width is known }

        else
          actual^.width_is_unknown := TRUE;

        actual := actual^.next;
      until (actual = NIL) or not test_all_actuals;
    end;

  if debug_1 then
    begin
      writeln(outfile, 'Exiting check_and_fix_formal_actual: F/A=');
      dump_formal_actual(outfile, formal_actual_pair);
    end;
end { check_and_fix_formal_actual } ;


(**)     { ------- resolve the widths of all signals ------- }


function resolve_all_signal_widths(node: mtree_node_ptr): boolean;
  { resolve the widths of all of the signals in the current drawing and
    within any plumbing drawings called by this drawing.
    Return TRUE if any signal was fixed. }

  const
    MAX_PASSES = 500;                 { maximum # passes through signals }

  var
    number_of_passes: 0..MAX_PASSES;  { number of passes through signals }
    son: mtree_node_ptr;              { current son of the given node }
    some_width_was_fixed,             { TRUE if any signal was fixed }
    width_was_fixed: boolean;         { TRUE if signal on body was fixed }
    one_was_fixed: boolean;           { TRUE if any signal's width fixed }
    formal_actual_pair: formal_actual_ptr;   { current formal/actual pair }

begin
  if debug_1 then
    begin
      write(outfile, 'Entering resolve_all_signal_widths: node=');
      print_string(outfile, node^.macro_name);
      writeln(outfile);
    end;

  number_of_passes := 0;

  one_was_fixed := FALSE;
  repeat
    some_width_was_fixed := FALSE;

    son := node^.son;
    while son <> NIL do
      begin
        if son^.is_plumbing_node and not son^.is_cardinal_tap then
          if resolve_all_signal_widths(son) then
            begin
              one_was_fixed := TRUE;
              some_width_was_fixed := TRUE;
            end;

        formal_actual_pair := son^.params;
        while formal_actual_pair <> NIL do
          begin
            if check_and_fix_formal_actual(formal_actual_pair, TRUE,
                                           width_was_fixed) then ;
            if width_was_fixed then
              begin
                one_was_fixed := TRUE;
                some_width_was_fixed := TRUE;
              end;

            formal_actual_pair := formal_actual_pair^.next;
          end;

        son := son^.next;
      end;

    number_of_passes := number_of_passes + 1;
  until (number_of_passes >= MAX_PASSES) or not some_width_was_fixed;

  resolve_all_signal_widths := one_was_fixed;

  if debug_1 then
    writeln(outfile, 'Exiting resolve_all_signal_widths:', ord(one_was_fixed):1);
end { resolve_all_signal_widths } ;


