program bigfile(input, output, stderr);
  { Copies input to ouptut, merging in the SVS includes, and uncomments
    or deletes lines of code according to specified command line flags.
    Assumes 1 include per line and nothing else on that line of
    any importance (other than a system flag comment).  Tabs are expanded.

    Takes 1 or more flag arguments such as

      +370 -UNIX -BSD42
       or
      +VAX -UNIX -BSD42
       or
      +OPUS -BSD42

    
    The + arguments denote lines of code to be uncommented, the - arguments
    denote lines of code to delete.  The code itself is marked as system
    specific line-by-line by having a comment containing a flag (such as
    VAX) at the end of the line.  The flag comment should contain nothing
    else (including blanks).  Either kind of comment delimiter may be used
    for the flag comment.

    When uncommenting a line of code, braces are converted to spaces.
    Comments delmited by paren-star are left as comments.
    modified.   

    NOTE:  this applies to include lines as well -- if 
    an "ignore me" flag terminates an include line, then that include is
    ignored. 

    In addition to SVS include directives of the form
      ($i filename) or ($I filename) 
    (with the () really being comment delimiters) this program recognizes
      (%i filename) and (%I filename) 
    (with the () really being comment delimiters) as include directives.
    This allows includes to be specified as going to VAX or 370 and
    having the SVS compiler bypass them. 
    
    NOTE: both types of comment delimiters are recognized for both
    host flags and include directives.
    
    Comments may be stripped out by specifying -c in the command line 
    THE FOLLOWING ASSUMPTIONS AFFECT COMMENT HANDLING:

      1. flag comments are never nested inside of other comments.
      2. flag comments appear at the ends of lines with nothing
	 (including other comments) following them. 
      3. include directives are never nested inside of (other) comments.

    In other words, when an ignore-me flag is encountered, the line
    will be deleted and the next line will be read with the 
    assumption that we are NOT inside of a comment, also, every file
    is begun with the assumption that we are NOT inside of a comment. 
    
    Specifying the flag "-^" causes carets (^) to be replaced by at signs
    (@) (which should be done to go to the 370).

    One other thing: this program can perform s/otherwise *:/otherwise/g
    substituion.  This is done by specifying the -: flag,
    BUG: this works correctly if comments are being stripped, but if they
    are not, then
      1. otherwise: within a comment gets changed as well as
         otherwise: in code.
      2. if a comment separates the "otherwise" from the : then
         the : is not stripped.      }


procedure mainproc;
  { puts all other procedures down to level 2 so they don't
    need to have names that are unique within 8 chars. }

  const
    MAX_STRING_LENGTH = 255;
    ID_LENGTH = 16;

    BLANK = ' ';
    NULL_ALPHA = '                ';

    INPUT_FLAG     = 'i';
    OUTPUT_FLAG    = 'o';
    DECOMMENT_FLAG = 'c';
    DEBUG_FLAG     = 'd';
    DECOLON_FLAG   = ':';
    DECARET_FLAG   = '^';

    DEBUG_FILE_C    = 'debug.out       ';
    STANDARD_INPUT  = 'STANDARD_INPUT  ';
    STANDARD_OUTPUT = 'STANDARD_OUTPUT ';

    MAX_CLI_ARG_NUMBER = 100; 
    MAX_FLAG_ARRAY_INDEX = 9; 
    
    OTHERWISE_CHAR_C       = 'otherwise';
    CAPS_OTHERWISE_CHAR_C  = 'OTHERWISE';
    OTHERWISE_LENGTH       = 9;

    BUFSIZ = 1024;

  type


{-----------------------------  TYPES  --------------------------------}


  textfile = text;

  Cint = longint;
  Cfile_ptr = Cint;
  inputfile = record
    f: Cfile_ptr;
    buffer: packed array[1..BUFSIZ] of char;
  end;



{ open_mode = (READ_MODE, WRITE_MODE, UNBUF_READ, UNBUF_WRITE);  }     (*VAX*)
{ open_mode = (READ_MODE, WRITE_MODE, UNBUF_READ, UNBUF_WRITE);  }     (*370*)
{ open_mode = (READ_MODE, WRITE_MODE, UNBUF_READ, UNBUF_WRITE);  }   (*ELXSI*)


  {-----------------------------------------------------------------}
  {  An exception handler is used to intercept Pascal run-time and  }
  {  operating system errors so that the Compiler can handle them.  }
  {  This is especially true for file opening and I/O.  This type   }
  {  is used to characterize the error that occurred.               }
  {-----------------------------------------------------------------}


  { EXCEPTION_ERROR_TYPE is used to specify the exception encountered }

  exception_error_type = (NULL_ERROR_CODE,       { no exception encountered }
                          OPEN_CREATE_FAIL,      { file open/create failure }
                          OPEN_ACCESS_FAIL,      { file open/access failure }
                          CLOSE_FAIL,            { file close failure }
                          RESET_FAIL,            { file RESET failure }
                          REWRITE_FAIL,          { file REWRITE failure }
                          BUFFER_NOT_ALLOCATED,  { file buffer not allocated }
                          INAPPROPRIATE_FILE_TYPE,  { just that }
                          LINE_LENGTH_EXCEEDED,  { output line len exceeded }
                          HEAP_OVERFLOW,         { HEAP overflow }
                          STACK_OVERFLOW);       { STACK overflow }


  { special types needed by VAX/VMS exception processor }

  signal_parameter_type = array [0..9] of integer;
  mechanism_parameter_type = array [0..4] of integer;


  { special types needed by the 370/CMS exception processor }

  errortype = 1..90;               { number of exception errors }
  erroractions = (XHALT,           { terminate program }
                  XPMSG,           { print pascal diagnostic }
                  XUMSG,           { print user's message }
                  XTRACE,          { print a trace back }
                  XDEBUG,          { invoke the debugger }
                  XDECERR,         { decrement error counter }
                  XRESERVED6, XRESERVED7, XRESERVED8, XRESERVED9,
                  XRESERVEDA, XRESERVEDB, XRESERVEDC, XRESERVEDD,
                  XRESERVEDE, XRESERVEDF);

  errorset = set of erroractions;



  {---------------------------------------------------------------------}
  { A STRING is represented as a pointer to a packed array of char:     }
  {                                                                     }
  {    xtring = ^packed array [0..255] of char;                         }
  {                                                                     }
  { Each xtring, however, is usually less than 255 characters.  The     }
  { actual length of the xtring is found in the first byte:  xtring^[0].}
  { The length of the xtring is static;  it should not be changed once  }
  { the xtring has been created.                                        }
  {                                                                     }
  { Strings can be up to 255 characters long.  The programmer must make }
  { sure that characters are not written beyond the end of the xtring.  }
  {                                                                     }
  { Strings are created on the heap in quantized lengths.  There are 33 }
  { different length arrays created.  The CREATE_A_STRING routine       }
  { creates an array on the heap big enough to support the given xtring.}
  { Strings may be released for use at a later time (RELEASE_STRING).   }
  {                                                                     }
  { THIS PROGRAM DOES NOT HASH STRINGS.  String compare must be done    }
  { with the compare xtring routines.                                   }
  {---------------------------------------------------------------------}


  string_range = 0..MAX_STRING_LENGTH;            { range for xtring }

  xtring = ^char_array;
  char_array = packed array [string_range] of char;

  freeptr = ^free_element;
  free_element = record
                   next: freeptr;     { next in linked list of free xtrings }
                   str: xtring;       { ^ free xtring element }
                 end;            

{ ------- other types ------ }
  
  cli_arg_range = 0..MAX_CLI_ARG_NUMBER;
  cli_arg_index = -1..MAX_CLI_ARG_NUMBER;
  cli_arg_array_type =  array[cli_arg_range] of xtring;

  id_range = 1..ID_LENGTH;                    { range of an identifier }
  alpha = packed array [id_range] of char;    { identifier type }
  compare_type = (LT, EQ, GT);                { result of compare functions }


  flag_ptr = freeptr;      { ^ to an element of a list of strings specifying
                             a flag for deleting or uncommenting lines }

  var
    debug: boolean;        { indicates debug output to be produced }
    debug_decomment: boolean; { debug the decommenting }

    free_strings: array [1..33] of freeptr;  { free xtrings (by size) }
    s_length: array [1..33] of 0..256;       { possible xtring lengths }
    free_pointers: freeptr;                  { free "freeptr"s }
    nullstring: xtring;   { null xtring }

    cli_arg_last : cli_arg_index;
    cli_arg_array: cli_arg_array_type;

    line: xtring;                   { Read buffer }
    line_len: string_range;         { length of line }

    delete_lines: flag_ptr;         { flags specifying line deletion }
    uncomment_lines: flag_ptr;      { flags specifying line uncommenting }
    max_flag_length: string_range;  { length of longest delete/uncomment flag }

    input_file: xtring;             { input file name }
    output_file: xtring;            { output file name }
    infile: inputfile;              { intput file }
    outfile: textfile;              { output file }

    debug_file: xtring;             { name of file for debug output }
    debugf: textfile;               { file for debug output }
    end_of_file: boolean;           { set by each call to
				      read_a_line }
    decomment: boolean;             { TRUE if comments to be removed }
    in_comment: boolean;            { TRUE if decomment and in a comment }
    commented_line: boolean;        { TRUE if decomment and this line had
                                      comments }
    brace_comment: boolean;         { TRUE if comment delimiters are braces -
				      (otherwise are paren/asterisk) }
    replace_carets: boolean;        { TRUE if carets to be replaced (for IBM) }

    otherwise_char: 
      packed array[1..OTHERWISE_LENGTH] of char;  { contains "otherwise" }
    CAPS_otherwise_char: 
      packed array[1..OTHERWISE_LENGTH] of char;  { contains "OTHERWISE" }
    decolon_otherwise: boolean;     { removes colon from otherwise }
    last_tok_otherwise: boolean;    { TRUE if last non-white, non-comment 
                                      string was "otherwise" }
    
    last_line_blank: boolean;       { TRUE if decommenting and the last line
                                      printed was a blank line }

(**) {---------------- fio.c routines and system calls ------------------}


   function creset(var f: inputfile; var logical: alpha; 
                   name: xtring; buffer_size: Cint): boolean;  CEXTERNAL;
     { If name is not nullstring (has > 0 length) then reset file
       of that name.  Else reset logical file specified by "logical"
       (meaning get the value of the specified environment variable 
       and open that file.  Use specified buffer size. (NOTE: until
       the unix C library routine setbuffer is fixed, this MUST be
       the value of BUFSIZE as defined in <sdtio.h> which is currently
       512.  No other value will work until setbuffer is fixed.)
       Return TRUE iff successful }


   function crdetab(var f: inputfile;  line: xtring): Cint;  CEXTERNAL;
     { read a line from the file into the xtring, expanding tabs.  Return
         0 if completely successful
	 1 if overflow (chars lost at end of line)
	 2 if end of file (no line read)                    }


   procedure cfclose(var f: inputfile);  CEXTERNAL;
     { close the file f }


   function cfdsc(var f: inputfile): Cint;  CEXTERNAL;
     { return the file descriptor for the open file f }


   function dup2(newfd, oldfd: Cint): Cint; CEXTERNAL;


   function textfdsc(var f: textfile): Cint; CEXTERNAL;


(**)     { ------- xtring package routines ------- }


{
               ***********************************
               *                                 *
               *       String description        *
               *                                 *
               ***********************************


   A xtring is represented as a pointer to a packed array of char:

       xtring = ^packed array [0..255] of char;

   Each xtring, however, is usually less than 256 characters.  The
   actual length of the xtring is found in the first byte:  xtring^[0].
   The length of the xtring is static;  it should not be changed once
   the xtring has been created.

   Strings can be up to 255 characters long.  The programmer must make
   sure that characters are not written beyond the end of the xtring.

   Strings are created on the heap in quantized lengths.  There are 33
   different length arrays created.  The create_a_string routine
   creates an array on the heap big enough to support the given xtring. }



procedure new_free_element(var f: freeptr);
  { create a new free element for released xtrings }
begin
  new(f);
  {increment_heap_count(HEAP_FREEELEMENT, 2*POINTER_SIZE);}
  f^.next := NIL;
  f^.str := NIL;
end { new_free_element } ;


procedure create_a_string(var str: xtring; length: string_range);
  { Create a xtring on the heap of the given length.  This routine uses a
    variant record to represent xtrings of various lengths with one
    pointer.  First, the free lists are checked for a xtring of the
    appropriate length.  If none are available, a xtring is newed from
    the heap.  This scheme works only if the Pascal compiler creates only
    as much space as needed for a variant when the tag field is specified
    in the new.  }
  type
    size_type = (s4,s8,s12,s16,s20,s24,s28,s32,s36,s40,s44,
                 s48,s52,s56,s60,s64,s68,s72,s76,s80,s84,s88,
                 s92,s96,s100,s120,s140,s160,s180,s200,s220,s240,s256);

    trick_ptr = ^trick_record;
    trick_record = record case size_type of
                     s4: (f4: packed array [0..3] of char);
                     s8: (f8: packed array [0..7] of char);
                     s12: (f12: packed array [0..11] of char);
                     s16: (f16: packed array [0..15] of char);
                     s20: (f20: packed array [0..19] of char);
                     s24: (f24: packed array [0..23] of char);
                     s28: (f28: packed array [0..27] of char);
                     s32: (f32: packed array [0..31] of char);
                     s36: (f36: packed array [0..35] of char);
                     s40: (f40: packed array [0..39] of char);
                     s44: (f44: packed array [0..43] of char);
                     s48: (f48: packed array [0..47] of char);
                     s52: (f52: packed array [0..51] of char);
                     s56: (f56: packed array [0..55] of char);
                     s60: (f60: packed array [0..59] of char);
                     s64: (f64: packed array [0..63] of char);
                     s68: (f68: packed array [0..67] of char);
                     s72: (f72: packed array [0..71] of char);
                     s76: (f76: packed array [0..75] of char);
                     s80: (f80: packed array [0..79] of char);
                     s84: (f84: packed array [0..83] of char);
                     s88: (f88: packed array [0..87] of char);
                     s92: (f92: packed array [0..91] of char);
                     s96: (f96: packed array [0..95] of char);
                     s100: (f100: packed array [0..99] of char);
                     s120: (f120: packed array [0..119] of char);
                     s140: (f140: packed array [0..139] of char);
                     s160: (f160: packed array [0..159] of char);
                     s180: (f180: packed array [0..179] of char);
                     s200: (f200: packed array [0..199] of char);
                     s220: (f220: packed array [0..219] of char);
                     s240: (f240: packed array [0..239] of char);
                     s256: (f256: packed array [0..255] of char);
                    end;
var
  k: record case boolean of      { "trick" record to fiddle with pointers }
       TRUE:  (tp: trick_ptr);
       FALSE: (ap: xtring);
     end;
  p: trick_ptr;                  { pointer to the created xtring }
  fp: freeptr;                   { pointer to head of free xtrings }
  size: 1..33;                   { the size (index into table) of xtring }

begin
  if length > 100 then size := ((length+1)+420) DIV 20
                  else size := ((length+1) DIV 4) + 1;
  if free_strings[size] <> NIL then
    begin
      str := free_strings[size]^.str;
      fp := free_strings[size]^.next;
      free_strings[size]^.next := free_pointers;
      free_pointers := free_strings[size];
      free_strings[size] := fp;
    end
  else
    begin
      case s_length[size] of
          4: new(p,s4);
          8: new(p,s8);
         12: new(p,s12);
         16: new(p,s16);
         20: new(p,s20);
         24: new(p,s24);
         28: new(p,s28);
         32: new(p,s32);
         36: new(p,s36);
         40: new(p,s40);
         44: new(p,s44);
         48: new(p,s48);
         52: new(p,s52);
         56: new(p,s56);
         60: new(p,s60);
         64: new(p,s64);
         68: new(p,s68);
         72: new(p,s72);
         76: new(p,s76);
         80: new(p,s80);
         84: new(p,s84);
         88: new(p,s88);
         92: new(p,s92);
         96: new(p,s96);
        100: new(p,s100);
        120: new(p,s120);
        140: new(p,s140);
        160: new(p,s160);
        180: new(p,s180);
        200: new(p,s200);
        220: new(p,s220);
        240: new(p,s240);
        256: new(p,s256);
      end;
      k.tp := p;  str := k.ap;
      {increment_heap_count(HEAP_STRING, s_length[size]);}
    end;
  str^[0] := chr(length);
end { create_a_string } ;


procedure release_string(var str: xtring);
  { free the storage used by the given xtring and place on free list }
  var
    size: string_range;     { size (index into table) of the xtring }
    f: freeptr;             { head of list of free xtrings }
begin
  if str <> nullstring then
    begin
      if ord(str^[0]) > 100 then size := ((ord(str^[0])+1)+420) DIV 20
		      else size := ((ord(str^[0])+1) DIV 4) + 1;
      if free_pointers = NIL then new_free_element(f)
      else
        begin f := free_pointers; free_pointers := free_pointers^.next; end;
      f^.next := free_strings[size];
      free_strings[size] := f;  f^.str := str;
      str := nullstring;
    end;
end { release_string } ;


procedure copy_string(source: xtring;  var dest: xtring);
  { copy from the source to the destination.  The destination xtring must 
    exist (= nullstring or some other xtring).  If the source length is not
    equal to the destination length the destination xtring is "free"d and a
    new xtring of the proper size is created. }
  var
    pos: string_range;        { index into xtring for copy }
begin
  if source^[0] <> dest^[0] then
    begin
      release_string(dest);  create_a_string(dest, ord(source^[0]));
    end;

  for pos := 1 to ord(source^[0]) do  dest^[pos] := source^[pos];
end { copy_string } ;

    
procedure copy_to_string(name: alpha;  var str: xtring);
  { copy from an alpha to a xtring.  Trailing blanks are deleted. }
  var
    len: id_range;    { length of the identifer }
    i: string_range;  { index into the xtring }
    done: boolean;    { TRUE if end of identifier found }
begin
  len := ID_LENGTH;  done := FALSE;
  repeat
    if name[len] <> ' ' then done := TRUE else len := len - 1;
  until (len = 1) or done;

  if ord(str^[0]) <> len then 
    begin  release_string(str);  create_a_string(str, len);  end;
  for i := 1 to len do  str^[i] := name[i];
end { copy_to_string } ;


function CmpStrLEQ(s1, s2: xtring): boolean;
  { returns TRUE if s1 <= s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two xtrings }
    i: string_range;        { index into the xtrings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] > s2^[0] then 
    begin  min_length := ord(s2^[0]);  CmpStrLEQ := FALSE;  end
  else
    begin  min_length := ord(s1^[0]);  CmpStrLEQ := TRUE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] > s2^[i] then
        begin  CmpStrLEQ := FALSE;  done := TRUE;  end
      else
        if s1^[i] < s2^[i] then
          begin  CmpStrLEQ := TRUE;  done := TRUE;  end;
    end;
end { CmpStrLEQ } ;


function CmpStrLT(s1, s2: xtring): boolean;
  { returns TRUE if s1 < s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two xtrings }
    i: string_range;        { index into the xtrings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] <= s2^[0] then 
    begin  min_length := ord(s1^[0]);  CmpStrLT := TRUE;  end
  else 
    begin  min_length := ord(s2^[0]);  CmpStrLT := FALSE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] > s2^[i] then 
        begin  CmpStrLT := FALSE;  done := TRUE;  end
      else if s1^[i] < s2^[i] then
        begin  CmpStrLT := TRUE;  done := TRUE;  end;
    end;
end { CmpStrLT } ;


function CmpStrGT(s1, s2: xtring): boolean;
  { returns TRUE if s1 > s2, FALSE otherwise. }
  var
    min_length,             { minimum length of the two xtrings }
    i: string_range;        { index into the xtrings }
    done: boolean;          { TRUE if comparison complete }
begin
  if s1^[0] <= s2^[0] then 
    begin  min_length := ord(s1^[0]);  CmpStrGT := FALSE;  end
  else 
    begin  min_length := ord(s2^[0]);  CmpStrGT := TRUE;  end;

  i := 0;  done := FALSE;
  while (i < min_length) and not done do
    begin
      i := i + 1;
      if s1^[i] < s2^[i] then 
        begin  CmpStrGT := FALSE;  done := TRUE;  end
      else if s1^[i] > s2^[i] then
        begin  CmpStrGT := TRUE;  done := TRUE;  end;
    end;
end { CmpStrGT } ;


function CmpStrEQ(s1, s2: xtring): boolean;
  { returns TRUE if s1 = s2, FALSE otherwise. }
  var
    i: string_range;        { index into the xtrings }
    done: boolean;          { TRUE if comparison complete }
begin
  CmpStrEQ := FALSE;

  if s2^[0] = s1^[0] then
    begin
      i := 0;  done := FALSE;
      while (i < ord(s1^[0])) and not done do
        begin
          i := i + 1;
          if s1^[i] <> s2^[i] then done := TRUE;
        end;
      if not done then CmpStrEQ := TRUE;
    end;
end { CmpStrEQ } ;


function compare_strings(s1, s2: xtring): compare_type;
  { compare the xtrings and return the result }
  var
    min_length,             { minimum length of the two xtrings }
    i: string_range;        { index into the xtrings }
    result: compare_type;   { result of the comparison }
    still_equal: boolean;   { TRUE if xtrings are equal to current position }
begin
  if s1^[0] = s2^[0] then
    begin  min_length := ord(s1^[0]);  result := EQ;  end
  else if s1^[0] < s2^[0] then
    begin  min_length := ord(s1^[0]);  result := LT;  end
  else 
    begin  min_length := ord(s2^[0]);  result := GT;  end;

  i := 0;  still_equal := TRUE;
  while (i < min_length) and still_equal do
    begin
      i := i + 1;
      if s1^[i] < s2^[i] then
        begin  result := LT;  still_equal := FALSE;  end
      else if s1^[i] > s2^[i] then
        begin  result := GT;  still_equal := FALSE;  end;
    end;

  compare_strings := result;
end { compare_strings } ;


procedure init_string_structures;
  { init the table describing the discrete xtring lengths that can be
    created on the heap }
  var
    i: 1..33;                  { index into list of free xtrings }
    {j: hash_string_range;}    { index into the xtring table }
begin
  {for j := 0 to HASH_STRING_TABLE_SIZE do xtring_table[j] := NIL;}

  free_pointers := NIL;
  for i := 1 to 33 do free_strings[i] := NIL;
  s_length[1] := 4;
  s_length[2] := 8;
  s_length[3] := 12;
  s_length[4] := 16;
  s_length[5] := 20;
  s_length[6] := 24;
  s_length[7] := 28;
  s_length[8] := 32;
  s_length[9] := 36;
  s_length[10] := 40;
  s_length[11] := 44;
  s_length[12] := 48;
  s_length[13] := 52;
  s_length[14] := 56;
  s_length[15] := 60;
  s_length[16] := 64;
  s_length[17] := 68;
  s_length[18] := 72;
  s_length[19] := 76;
  s_length[20] := 80;
  s_length[21] := 84;
  s_length[22] := 88;
  s_length[23] := 92;
  s_length[24] := 96;
  s_length[25] := 100;
  s_length[26] := 120;
  s_length[27] := 140;
  s_length[28] := 160;
  s_length[29] := 180;
  s_length[30] := 200;
  s_length[31] := 220;
  s_length[32] := 240;
  s_length[33] := 256;

  create_a_string(nullstring, 0);
end { init_string_structures } ;


(**)     { ------- I/O utilities ------- }


function width_of_integer(i: integer): integer;
  { Returns the minimum number of places PASCAL uses to print i }
  var
    width: integer;     { width of the integer i in print positions }
begin
  width := 1;
  if i < 0 then
    begin  width := 2;  i := -1;  end;

  if i < 10 then  { ok as is }
  else if i < 100 then width := width + 1
  else if i < 1000 then width := width + 2
  else if i < 10000 then width := width + 3
  else if i < 100000 then width := width + 4
  else if i < 1000000 then width := width + 5
  else if i < 10000000 then width := width + 6
  else if i < 100000000 then width := width + 7
  else if i < 1000000000 then width := width + 8
  else width := width + 9;

  width_of_integer := width;
end { width_of_integer } ;


procedure writestring(var f: textfile; str: xtring);
  { dump the given xtring (STR) to the given file (F) as is }
  var
    i: string_range;            { index into the xtring }
begin
  for i := 1 to ord(str^[0]) do write(f, str^[i]);
end { writestring } ;


procedure writestring_quote1(var f: textfile; str: xtring);
  { dump the xtring quoted with ' }
begin
  write(f, '''');
  writestring(f, str);
  write(f, '''');
end { writestring_quote1 };
    

procedure writestring_quote2(var f: textfile; str: xtring);
  { dump the xtring quoted with " }
begin
  write(f, '"');
  writestring(f, str);
  write(f, '"');
end { writestring_quote2 };
    
  
function is_blank_string(str: xtring): boolean;
  { Tests for string which is of 0 length or all white space
    (tabs and blanks). }
  var
    i: string_range;    { index into str }
    result: boolean;    { result to be returned }
begin
  i := 0;  result := TRUE;
  while result and (i < ord(str^[0])) do
    if (str^[i + 1] = BLANK) then i := i + 1
    else result := FALSE;
  is_blank_string := result;
end { is_blank_string };


procedure substring(str : xtring; start,len : string_range;
  var subs : xtring);
  { returns the specified sub-xtring of str.  sub-xtring starts
    at start and is len chars long.  It will be truncated if 
    this start+len-1 > length of str. As usual, subs must already
    exist.  Subs may be equal to str.}
  var 
    i: string_range;           { index into str and subs }
    offset: string_range;      { index offset in str }
    actual_len : string_range; { final length of subs }
    temp: xtring;              { temp for building subs so that
				 we don't release it until the
				 last minute (in case it is
				 equal to str }
begin
  if start <= ord(str^[0]) then 
    begin
      if start-1+len > ord(str^[0]) then 
	actual_len := ord(str^[0]) - start + 1
      else actual_len := len;
      if actual_len > 0 then 
	begin
	  create_a_string(temp, actual_len);
	  offset := start-1;
	  for i := 1 to actual_len do temp^[i] := str^[i + offset];
	end;
    end;
  release_string(subs);
  subs := temp;
end; {substring}


procedure downstring(str: xtring);
  { converts all capitals in str to lower case }
  var j: string_range;    { index into str }
begin
  for j := 1 to ord(str^[0]) do case str^[j] of
    'Q': str^[j] := 'q';  'W': str^[j] := 'w';  'E': str^[j] := 'e';
    'R': str^[j] := 'r';  'T': str^[j] := 't';  'Y': str^[j] := 'y';
    'U': str^[j] := 'u';  'I': str^[j] := 'i';  'O': str^[j] := 'o';
    'P': str^[j] := 'p';  'A': str^[j] := 'a';  'S': str^[j] := 's';
    'D': str^[j] := 'd';  'F': str^[j] := 'f';  'G': str^[j] := 'g';
    'H': str^[j] := 'h';  'J': str^[j] := 'j';  'K': str^[j] := 'k';
    'L': str^[j] := 'l';  'Z': str^[j] := 'z';  'X': str^[j] := 'x';
    'C': str^[j] := 'c';  'V': str^[j] := 'v';  'B': str^[j] := 'b';
    'N': str^[j] := 'n';  'M': str^[j] := 'm';  otherwise: begin end;
  end { case };
end { downstring };


procedure upstring(str: xtring);
  { converts all lower case in str to capitals }
  var j: string_range;    { index into str }
begin
  for j := 1 to ord(str^[0]) do case str^[j] of
    'q': str^[j] := 'Q';  'w': str^[j] := 'W';  'e': str^[j] := 'E';
    'r': str^[j] := 'R';  't': str^[j] := 'T';  'y': str^[j] := 'Y';
    'u': str^[j] := 'U';  'i': str^[j] := 'I';  'o': str^[j] := 'O';
    'p': str^[j] := 'P';  'a': str^[j] := 'A';  's': str^[j] := 'S';
    'd': str^[j] := 'D';  'f': str^[j] := 'F';  'g': str^[j] := 'G';
    'h': str^[j] := 'H';  'j': str^[j] := 'J';  'k': str^[j] := 'K';
    'l': str^[j] := 'L';  'z': str^[j] := 'Z';  'x': str^[j] := 'X';
    'c': str^[j] := 'C';  'v': str^[j] := 'V';  'b': str^[j] := 'B';
    'n': str^[j] := 'N';  'm': str^[j] := 'M';  otherwise: begin end;
  end { case };
end { upstring };



function reset_file(var f: inputfile; filename: xtring): boolean;
  { reset a file of the given name.  Return true if successful }
  const
    DEVICE_NULL     = '/dev/null       ';  { must be an alpha }
  var
    logical: alpha;
begin
  if filename = nullstring then copy_to_string(DEVICE_NULL, filename);
  logical := NULL_ALPHA;
  reset_file := creset(f, logical, filename, BUFSIZ);   {UNIX}
end { reset_file } ;


function rewrite_file(var f: textfile; filename: xtring): boolean;
  { rewrite a file of the given name (if nullstring use '/dev/null').
    Return true if successful }
  const
    DEVICE_NULL = '/dev/null';  { going into an SVS string }
  var
    name: String[80];                                                   {UNIX}
    min: 1..80;                         { minimum length of name & xtring }
    index: 1..81;                       { index into the name }
begin
  {$I-}                                                             {UNIX}
  if filename = nullstring then name := '/dev/null'                 {UNIX}
  else                                                              {UNIX}
    begin                                                           {UNIX}
      if ord(filename^[0]) < 80 then min := ord(filename^[0])
				else min := 80;


      name := '';                                                   {UNIX}
      for index := 1 to min do name := concat(name, ' ');           {UNIX}
      for index := 1 to min do                                      {UNIX}
        name[index] := filename^[index];                            {UNIX}
    end;                                                            {UNIX}
  if debug then rewrite(f, name, UNBUFFERED)                        {UNIX}
  else rewrite(f, name, BUFFERED);                                  {UNIX}

  rewrite_file := (ioresult = 0);                                   {UNIX}
  {$I+}                                                             {UNIX}
end { rewrite_file } ;


procedure close_file(var f: textfile);
  { close the given file }
begin
  {$I-}                                                                {UNIX}
  close(f);
  {$I+}                                                                {UNIX}
end { close_file } ;


procedure unused_read_a_line(var f: textfile; line: xtring);
  { reads a line from the file.  Sets end_of_file if that
    condition exists and so no line has been read.  Line
    must be a xtring which has been declared to be 
    MAX_STRING_LENGTH long. }
  var
    i: string_range;   { index into line }
    done: boolean;     { loop control flag }
    ch: char;          { last char }
begin
  line^[0] := chr(0);
  if eof(f) then 
    begin
      end_of_file := TRUE;
    end
  else
    begin
      end_of_file := FALSE;
      i := 0;  done := eoln(f);
      while (i < MAX_STRING_LENGTH) and not done do
	begin
	  read(f, ch);
	  i := i + 1;
	  line^[i] := ch;
	  done := eoln(f) or eof(f);
	end;
      line^[0] := chr(i);

      if not eof(f) then 
	begin
	{ if not eoln(f) then                   }
	{   begin                               }
	{     writestring(monitor, program_p);  }
	{     writeln(monitor, OVF_LINE_S);     }
	{   end;                                }
	  if not eof(f) then readln(f);
	end;
    end;
end { unused_read_a_line };

{ These interfaces provide  UNIX like access to command line arguments.
  The idea is to map whatever command line argument access exists on 
  the host to this format.

  sargc returns the number of command line arguments (including 1 for
  the program name).

  sargv(n,str) returns the nth argument as a xtring (where xtring
  is one of the strings defined by Mike for the SCALD compiler.)

  sargv(0,str) returns the program name. (but see NOTE)
  sargv(1,str) returns the first argument following the program name,
  etc.  
  (The above is the UNIX convention - not the SVS convention)
  sargv returns nullstring if n >= sargc.  

  NOTE: The program name return is actually only implemented for
  the SVS PASCAL (UNIX) version.  On the 370 and VAX versions
  sargv(0,str) returns nullstring (although it is still counted
  in arriving at the value of sargc.)

  Null (length = 0) arguments return nullstring, so check for
  this can be done by direct comparison to nullstring.

  The following global declarations are necessary to use the 
  sargv, and sargc routines.  They ideally would be local and
  static to this package, but that is not always possible in
  PASCAL.  They must be initialized by calling 
  init_cli_arg_structures.

  const
    MAX_CLI_ARG_NUMBER = 10; (- or whatever is desired - this is the
			    maximum number of arguments other than
			    the program name -)
  type
    cli_arg_range = 0..MAX_CLI_ARG_NUMBER;
    cli_arg_index = -1..MAX_CLI_ARG_NUMBER;
    cli_arg_array_type =  array[cli_arg_range] of xtring;
  var
    cli_arg_last : cli_arg_index;
    cli_arg_array: cli_arg_array_type;

  Bill Hunsicker  28 Mar 84 }


procedure init_cli_arg_structures;
  { initializes the global variables used by sargv and sargc.
    On 370 and VAX, they are initialized to hold the command line
    arguments.  On UNIX and unimplemented machines they are 
    nulled.                                                        }
{ type  } (*VAX*)
{   VAXstring = packed array[1..MAX_STRING_LENGTH] of char; }      (*VAX*)
    
  var
  { parmstring: String; }(* command line (minus command name) *)(*370*)
  { parmstring: VAXstring; }(* command line (minus command name) *)(*VAX*)
    commline: xtring; { command line (minus command name) }
    len: string_range; { length of commline }
    i: string_range; { index into commline }
    retcode: integer;  { return code }
    debug: boolean;

{ function LIB$GET_FOREIGN(%STDESCR commline: VAXstring): } (*VAX*)
{   integer; EXTERN;                                      } (*VAX*)


  procedure cli_arg_parse(args: xtring);
    { gets the run string and parses it into arguments.  Spaces
      are delimiters.  Quoted strings are taken as 1 argument
      (without the quotes).  If a quote is not matched then
      the rest of the line is taken as that argument ( with
      trailing blanks removed.)  A quoted string may contain any
      number of the other kind of quote.  ( ' and " are recognized
      as quotes.  If one kind of quote starts an argument then
      the same kind of quote (or end of the xtring) finishes it
      and anything else (including the other kind of quote) is
      included in the argument (with the exception of trailing
      blanks occuring when a quote is not matched).) }

    { NOTE: the handling of quotes is disabled in the 370 version
      because of the way that IBM passes the args }
    label 90;
    const
      QUOTE1 = '''';
    { ORD_TAB_CHAR = 5; } (*370*)
      ORD_TAB_CHAR = 9;   (*VAXUNIX*)
      QUOTE2 = '"';
      BLANK = ' ';
    var
      TAB_CHAR: char;               { tab character constant }
      args_length: string_range;    { length of args (to be parsed )}
      i: cli_arg_index;                 (* index into cli_arg_array *)
      in_space: boolean;            (* indicates in white space *)
      in_arg: boolean;              (* indicates in argument *)
      start: string_range;          (* start of current arg *)
      finish: string_range;         (* end of current arg *)
      j: string_range;              (* index into an xtring *)
      len: string_range;            (* length of current arg *)
      temp: xtring;                 (* an intermediate result *)
      quoteset: set of char;        (* set of chars that can *)
  begin
    TAB_CHAR := chr(ORD_TAB_CHAR);
    for i := 0 to MAX_CLI_ARG_NUMBER do cli_arg_array[i] := nullstring;
    args_length := ord(args^[0]);
    start := 0;  i := 0;
    while (start < args_length) and (i < MAX_CLI_ARG_NUMBER) do
      begin
	i := i + 1;
	in_space := TRUE;
	while (start < args_length) and in_space do
	  begin
	    start := start + 1;
	    if not (args^[start] in [BLANK,TAB_CHAR]) then
	      in_space := FALSE;
	  end;
	if in_space then
	  begin
	    { done }
	    cli_arg_last := i - 1;
	    goto 90 { return };
	  end
      { else if args^[start] in [QUOTE1,QUOTE2] then        }(*VAX*)
      {   begin                                             }(*VAX*)
      {     quoteset := [args^[start]];                     }(*VAX*)
      {     finish := start;                                }(*VAX*)
      {     if start < args_length then start := start + 1; }(*VAX*)
      {   end                                               }(*VAX*)
	else
	  begin
	    quoteset := [BLANK,TAB_CHAR];
	    finish := start;
	  end;
	in_arg := TRUE;
	while (finish < args_length) and in_arg do
	  begin
	    finish := finish + 1;
	    if args^[finish] in quoteset then
	      in_arg := FALSE;
	  end;
	len := finish - start;
	if in_arg then len := len + 1;
	if len > 0 then  create_a_string(cli_arg_array[i], len);
        for j := 1 to len do 
	  cli_arg_array[i]^[j] := args^[start - 1 + j];
	if in_arg then
	  { get rid of trailing blanks (in case we were looking 
	    for an unmatched quote) }
	  begin
	    j := len;
	    while (j > 0) and (cli_arg_array[i]^[j] in
	      [BLANK,TAB_CHAR]) do j := j - 1;
	    if j = 0 then release_string(cli_arg_array[i])
	    else if j < len then
	      begin
	      create_a_string(temp, j);
	      for j := j downto 1 do temp^[j] := cli_arg_array[i]^[j];
	      release_string(cli_arg_array[i]);
	      cli_arg_array[i] := temp;
	    end;
	  end;
	start := finish;
      end;
    cli_arg_last := i;
  90:
  end { cli_arg_parse };


begin { init_cli_arg_structures }
  debug := FALSE;
  if debug then writeln(debugf, '---- init_cli_arg... ----');
  commline := nullstring;
  len := 0;                             
{ parmstring := parms;                           } (*370*)
{ if length(parmstring) > MAX_STRING_LENGTH then } (*370*)
{   len := MAX_STRING_LENGTH                     } (*370*)
{ else len := length(parmstring);                } (*370*)
{ retcode := LIB$GET_FOREIGN(parmstring);        } (*VAX*)
{ if debug then writeln(debugf, 'retcode = ',    } (*VAX*)
{   HEX(retcode));                               } (*VAX*)
{ if odd(retcode) then len := MAX_STRING_LENGTH  } (*VAX*)
{   else len := 0; (* error occurred *)          } (*VAX*)
{ create_a_string(commline, len);                } (*VAX370*)
{ for i := 1 to len do                           } (*VAX370*)
{   commline^[i] := parmstring[i];               } (*VAX370*)
  if debug then
    begin
      write(debugf, 'commline = ');
      writestring_quote2(debugf, commline);
      writeln(debugf);
    end;
  cli_arg_parse(commline);
  release_string(commline);
  if debug then writeln(debugf, '---- end init_cli_arg... ----');
end { init_cli_arg_structures };

  
  

function sargc : integer;
begin
  sargc := 0 ; { default for unimplemented hosts }
  sargc := argc;                                      {UNIX}
{ sargc := cli_arg_last + 1;                       }  (*VAX370*)
end; {sargc}


procedure sargv(which_arg : integer; var arg : xtring);
  { gets the indicated argument.  arg is 'release_string'ed
    before being set to the new arg. }
var 
  i,j,len: integer;
  done: boolean;
begin
  release_string(arg);
  if which_arg < sargc then begin
    i := which_arg + 1;                                      {UNIX}
    len := length(argv[i]^);                                 {UNIX}
    create_a_string(arg,len);                                {UNIX}
    for j := 1 to len do arg^[j] := argv[i]^[j];             {UNIX}
    if is_blank_string(arg) then release_string(arg);        {UNIX}
{   copy_string(cli_arg_array[which_arg], arg);           }  (*VAX370*)
  end;
end; {sargv}
    

  procedure command_args;
    { Gets the command line argments, initializing global flags and
      flag lists. }
    type
      file_expected_type = (INPUT_NAME, OUTPUT_NAME);
    var 
      i: integer;
      arg: xtring;
      expecting_flag: boolean; 
      file_expected: file_expected_type;
    

    procedure insert_flag(var flags: flag_ptr; str: xtring);
      { Insert the flag into the flags list (directly -- don't copy the 
        string). Update global max_flag_length. }
      var
        flag: flag_ptr;        { new flag }


      procedure ngp_secrecy_hack;
        { if str is NGP, change it to SUN -- this allows specification
	  of NGP in command line so that SUN does not show when someon
	  runs a ps }
      begin
        if (ord(str^[0])) = 3 then
	  if (str^[1] = 'N') and (str^[2] = 'G') and (str^[3] = 'P') then
	    begin
	      str^[1] := 'S';
	      str^[2] := 'U';
	      str^[3] := 'N';
	    end;
      end { ngp_secrecy_hack } ;


    begin { insert_flag }
      ngp_secrecy_hack;
      new(flag);
      flag^.str := str;
      flag^.next := flags;
      flags := flag;
      if ord(str^[0]) > max_flag_length then
        max_flag_length := ord(str^[0]);
    end { insert_flag } ;


    procedure print_usage;
      { Does just that and exits with fatal status }
    begin
      writeln(stderr, '               [+][CMS  ]');
      writeln(stderr, 'Usage: bigfile [-][VAX  ] [-c] [-:] [-^]');
      writeln(stderr, '                  [UNIX ]');
      writeln(stderr, '                  [etc.]');
      halt(1);
    end;


    function open_debug_file: boolean;
      { Open the debug file (if nullstring, open to standard error).
        Return TRUE iff successufl. }
    begin
      open_debug_file := TRUE;
      if not rewrite_file(debugf, debug_file) then
        begin
	  debug := FALSE;  debug_decomment := FALSE;
	  open_debug_file := FALSE;
	end
      else if debug_file = nullstring then
        begin
          if dup2(textfdsc(debugf), textfdsc(stderr)) < 0 then    {UNIX}
	    begin                                                 {UNIX}
	      writeln(stderr, ' Can''t map debug file to stderr');
	      debug := FALSE;  debug_decomment := FALSE;
	      open_debug_file := FALSE;
	    end;                                                  {UNIX}
	end;
    end { open_debug_file } ;


  begin
    expecting_flag := TRUE;
    arg := nullstring;
    input_file := nullstring;
    output_file := nullstring;
    for i := 1 to (sargc - 1) do
      begin
	sargv(i, arg);
	if debug then 
	  begin
	    write(debugf, 'sargv(', i:1, ') = ');
	    writestring_quote2(debugf, arg);
	    writeln(debugf);
	  end;
	if arg = nullstring then print_usage;
	if expecting_flag then
	  begin
	    if (arg^[1] = '-') and (ord(arg^[0]) = 2) then
	      begin
	        if arg^[2] = INPUT_FLAG then
	          begin
		    if input_file <> nullstring then print_usage;
		    file_expected := INPUT_NAME;
		    expecting_flag := FALSE;
	          end
	        else if arg^[2] = OUTPUT_FLAG then
	          begin
		    if output_file <> nullstring then print_usage;
		    file_expected := OUTPUT_NAME;
		    expecting_flag := FALSE;
	          end
	        else if arg^[2] = DECOMMENT_FLAG then decomment := TRUE
	        else if arg^[2] = DECOLON_FLAG then decolon_otherwise := TRUE
	        else if arg^[2] = DECARET_FLAG then replace_carets := TRUE
	        else if arg^[2] = DEBUG_FLAG then
	          begin
		    if debug or debug_decomment then debug := TRUE
		    else
		      begin
		        debug := TRUE;  { so file open will be unbuffered }
			debug := open_debug_file; { success? }
		      end;
		    debug_decomment := debug;
	          end
	        else 
	          begin
		    substring(arg, 2, ord(arg^[0]) - 1, arg);
		    insert_flag(delete_lines, arg);
	          end
	      end
	    else if arg^[1] = '-' then
	      begin
		substring(arg, 2, ord(arg^[0]) - 1, arg);
		insert_flag(delete_lines, arg);
	      end
	    else if arg^[1] = '+' then
	      begin
		substring(arg, 2, ord(arg^[0]) - 1, arg);
		insert_flag(uncomment_lines, arg);
	      end
	    else print_usage;
	  end
	else
	  begin
	    case file_expected of
	      INPUT_NAME: input_file := arg;
	      OUTPUT_NAME: output_file := arg;
	    end;
	    expecting_flag := TRUE;
	  end;
	arg := nullstring;
      end;
    release_string(arg);
  end { command_args };


  function match_flag(flag: xtring; flags: flag_ptr): boolean;
    { Checks for match of flag to any of the flags in flags.
      If a match is found, returns TRUE. Assume no nested comments. }
    label
      90; { return }
    var
      result: boolean;  { returned value }
      i: string_range; { current index into flag }
      len: string_range; { length of flag comment }
      current: flag_ptr; { current uncomment flag }
  begin
    if debug then writeln(debugf, '---- match_flag ----');
    match_flag := FALSE;  result := FALSE;
    current := flags;
    len := ord(flag^[0]);
    while (current <> NIL) and not result do
      begin
        if len = ord(current^.str^[0]) then
          begin
            result := TRUE;  i := 0;
            while (i < len) and result do
	      begin
	        i := i + 1;
		if flag^[i] <> current^.str^[i] then result := FALSE;
              end;
          end;
        current := current^.next;
      end;

    match_flag := result;
  90:
  if debug then 
    begin
      write(debugf, '---- end match_flag ');
      if result then writeln(debugf, 'TRUE ----')
                else writeln(debugf, 'FALSE ----');
    end;
  end { match_flag };
    

  function flag_comment(line: xtring): xtring;
    { Checks for trailing comment of length <= max_flag_length.
      If a match is found, return a copy of the comment, else
      return nullstring. }
    label
      90; { return }
    var
      result: xtring;  { returned value }
      i: string_range; { current index into line }
      found: boolean;  { indicates the thing looked for has been found }
      max_len: string_range;  { Max interesting comment length - is
				equal to the lesser of 
				  1. The number of chars left in line
				  2. max_flag_length + 1 (or 2)
				  (the + 1 (or 2)  accomodates the char(s)
				   which indicates start of comment) }
      len: string_range; { length of comment }
      flag: flag_ptr;    { current uncomment flag }
      k: string_range;   { index for compares }
      start_char: char;  { char which begins the comment }

  begin
    if debug then writeln(debugf, '---- flag_comment ----');
    flag_comment := nullstring;  result := nullstring;
    i := ord(line^[0]);

    { find last non-blank }

    found := FALSE;
    while (i > 0) and not found do
      if line^[i] = ' ' then i := i - 1
                        else found := TRUE;
    if not found then goto 90;
        
    { is it the end of a comment ? }

    if line^[i] = '}' then start_char := '{'
    else if line^[i] = ')' then
      if i > 1 then
        begin
          i := i - 1;
          if line^[i] = '*' then start_char := '*'
          else goto 90 { return } ; { no trailing comment }
        end
      else goto 90 { return } { no trailing comment }
    else goto 90 { return } ; { no trailing comment }
            
    if debug then writeln(debugf, 'found a comment');

    { where does it start? (how long is it?) }

    max_len := max_flag_length + 1;
    if start_char = '*' then max_len := max_len + 1;
    if (i - 1) < max_len then max_len := i - 1;
    if debug then writeln(debugf, 'max_len = ', max_len:1);
    found := FALSE;  len := 0;
    while (len < max_len) and not found do
      begin
        len := len + 1;
        if line^[i - len] = start_char then
          if start_char = '{' then found := TRUE
          else
            if len < max_len then
              begin
                len := len + 1;
                if line^[i - len] = '(' then found := TRUE;
              end;
      end;

    if not found then goto 90 { return } ; { won't match anything }

    len := len - 1;
    if start_char = '*' then len := len - 1;
        
    create_a_string(result, len);
    substring(line, i - len, len, result);

    if debug then 
      begin
        write(debugf, 'interesting comment: ');
        writestring_quote2(debugf, result);
        writeln(debugf);
      end;

    flag_comment := result;
  90:
  if debug then writeln(debugf, '---- end flag_comment ----');
  end { flag_comment };
    

  function include_file(line: xtring): xtring;
    { Checks for include directive. If there, returns the file name,
      else returns nullstring. }
    label
      90; { return }
    var
      result: xtring;  { returned string }
      i: string_range; { index into line }
      found: boolean;  { loop control flag }
      len: string_range; { length of include file name }
      term_char: char;   { char (or first char) of comment terminator }

  begin
    if debug then writeln(debugf, '---- include_file ----');
    include_file := nullstring;
    result := nullstring;
    i := 0;  found := FALSE;  term_char := '}';
    while not found do
      begin
        while (i < line_len) and not found do
          begin
            i := i + 1;
            if line^[i] = '{' then found := TRUE
	    else if line^[i] = '(' then
	      if i < line_len then
	        begin
		  i := i + 1;
	          if line^[i] = '*' then 
		    begin
		      found := TRUE;
		      term_char := '*';
		    end;
		end;
          end;
        if not found or (i > line_len - 3)  then goto 90;

        if not ( ((line^[i + 1] = '$') or (line^[i+1] = '%')) and
          ((line^[i + 2] = 'i') or (line^[i+2] = 'I')) and
          (line^[i + 3] = BLANK) ) then found := FALSE
        else 
          begin
            i := i + 3;  found := FALSE;
            while (i < line_len) and not found do
              begin
                i := i + 1;
                if (line^[i] <> BLANK) then found := TRUE;
              end;
            if found then
              begin
                { If the comment ends, or we find we cross EOLN, or if it
		  is an I- or I+ directive, then we don't have an include
		  directive }
                if line^[i] = '-' then found := FALSE;
                if line^[i] = '+' then found := FALSE;
                if line^[i] = term_char then
		  if term_char = '}' then found := FALSE
		  else if i >= line_len then found := FALSE 
		  else if line^[i+1] = ')' then found := FALSE;
              end;
          end;
      end;

    if not found then goto 90;

    { At this point, i points to the first letter of an include file name }

    found := FALSE;  len := 0;
    while ((i + len) < line_len) and not found do
      begin
        len := len + 1;
	if line^[i + len] = term_char then
	  if term_char = '}' then found := TRUE
	  else 
	    if (i + len) < line_len then
              if line^[i + len + 1] = ')' then found := TRUE;
      end;
    if not found then goto 90;

    if debug then writeln(debugf, 'found include file ');

    substring(line, i, len, result);
    include_file := result;
    if debug then
      begin
        write(debugf, 'result: ');
        writestring(debugf, result);
        writeln(debugf);
    end;
  90:
  if debug then writeln(debugf, '---- end include_file ----');
  end { include_file };


  procedure write_line(var f: textfile; line: xtring);
    { writes line to output. If decommenting is being done, then
      multiple blank lines are collapsed to 1 blank line and lines 
      that contained comments but otherwise white space are deleted entirely }
    type
      svs_string = string[MAX_STRING_LENGTH];
      svs_string_ptr = ^svs_string;
      svs_string_converter = record
        case boolean of
	  TRUE: (x: xtring);
	  FALSE: (s: svs_string_ptr);   
      end;
    var
      i,len: string_range;             { index into and length of line }
      all_white: boolean;              { TRUE if decommenting and line is 
                                         blank }
      print_it: boolean;               { TRUE if line to be printed }
      svs_hack: svs_string_converter;  { convert xtring to SVS string for
                                         fast write }
  begin
    if decomment then
      begin
        len := ord(line^[0]);  
        i := 0;
        all_white := TRUE;
        while (i < len) and all_white do
          begin
            i := i + 1;
	    if (line^[i] <> BLANK) then all_white := FALSE;
          end;
        print_it := not ((last_line_blank or commented_line) and all_white);
        if print_it then last_line_blank := all_white;
      end
    else print_it := TRUE;
    
    if print_it then
      begin  
        svs_hack.x := line;
	writeln(f, svs_hack.s^);
      end;
  end { write_line };


  procedure strip_comments;
    { strip comments and trailing blanks from line }
    var
      i: string_range;    { index of char being examined }
      j: string_range;    { index of char last (or about to be) written }
      had_star: boolean;  { TRUE if last char was a '*' }
      in_quotes: boolean; { TRUE if in a string }
  begin
    if debug_decomment then
      begin
        writeln(debugf, '**** in_comment = ', in_comment, ' original line --');
        write_line(debugf, line);
      end;
    i := 0;  j := 0; had_star := FALSE;  in_quotes := FALSE;
    commented_line := in_comment;
    while (i < line_len) do
      begin
        i := i + 1;
        if in_comment then
          begin { skip chars while looking for end of comment }
            if brace_comment then in_comment := not (line^[i] = '}')
            else
              begin
                in_comment := not (had_star and (line^[i] = ')'));
                had_star := (line^[i] = '*');
              end;
          end
        else if in_quotes then
          begin { look for end quote and copy chars }
	    j := j + 1;
	    if i <> j then line^[j] := line^[i];
            if (line^[i] = '''') then
              if (i < line_len) then
                if (line^[i+1] = '''') then 
		  begin
                    i := i + 1; { doubled quote means quote inside of string }
		    j := j + 1;
	            if i <> j then line^[j] := line^[i];
		  end
                else in_quotes := FALSE
              else in_quotes := FALSE;
          end
        else
          begin { look for start of comments or quotes and copy chars }
            if (line^[i] = '''') then in_quotes := TRUE
            else if line^[i] = '{' then
              begin
                in_comment := TRUE;  brace_comment := TRUE;
		commented_line := TRUE;
              end
            else if line^[i] = '(' then
              if i < line_len then
                if line^[i+1] = '*' then
                  begin
                    in_comment := TRUE;  brace_comment := FALSE;
		    commented_line := TRUE;
                    i := i + 1;
                  end;
            if not in_comment then
	      begin
	        j := j + 1;
		if j <> i then line^[j] := line^[i];
	      end;
          end;
      end;
    while (j > 0) and (line^[j] = BLANK) do j := j - 1;
    line_len := j;
    line^[0] := chr(j);
    if debug_decomment then
      begin
        writeln(debugf, '**** decommented line --');
        write_line(debugf, line);
        writeln(debugf, 'in_comment = ', in_comment, ' ****');
      end;
  end { strip_comments } ;
  
  
  procedure uncomment_line;
    { Change curly braces to blanks and changes paren-star combinations to
      curly braces (with * becoming blank). }
    var
      had_left_paren: boolean;
      had_star: boolean;
      i: string_range;
  begin
    had_left_paren := FALSE;  had_star := FALSE;
    if line_len > 1 then
      if line^[1] = '{' then 
        if line^[2] = '#' then
          begin
	    { line begins with a "commented out" preprocessor directive --
	      the # MUST end up in line 1 for it to work, so shift left }
            for i := 2 to line_len do line^[i-1] := line^[i];
	    line_len := line_len - 1;
	    line^[0] := chr(line_len);
	  end;
    for i := 1 to line_len do
      begin
        if line^[i] = '{' then line^[i] := ' '
        else if line^[i] = '}' then line^[i] := ' '
        else if line^[i] = ')' then
          begin
            if had_star then
              begin
                line^[i-1] := ' ';  line^[i] := '}'
              end;
          end
        else if line^[i] = '*' then
          if had_left_paren then
            begin
              line^[i] := ' ';  line^[i-1] := '{'
            end;
        had_star := (line^[i]  = '*');
        had_left_paren := (line^[i] = '(');
      end;
  end { uncomment_line } ;
  

  procedure handle_otherwise;
    { detect otherwise keyword and remove trailing colon }
    var
      i: string_range;         { line index }
      j: 1..OTHERWISE_LENGTH;  { index into const arrays }
      ch: char;                { current char }
      in_non_otherwise_id: boolean; { TRUE if we have other id string }
  begin
    j := 1;  in_non_otherwise_id := FALSE;
    for i := 1 to line_len do
      begin
        ch := line^[i];
        if not in_non_otherwise_id and 
          ((ch = otherwise_char[j]) or (ch = CAPS_otherwise_char[j])) then
          if j = OTHERWISE_LENGTH then 
            begin
              last_tok_otherwise := TRUE;  j := 1;
            end
          else j := j + 1
        else
          begin
            j := 1;
            if last_tok_otherwise and (ch = ':') then line^[i] := BLANK;
            if (ch <> BLANK) then 
              last_tok_otherwise := FALSE;
            in_non_otherwise_id := (ch in ['a'..'z','A'..'Z','_','$']);
          end;
      end;
  end { handle_otherwise } ;


  procedure handle_file(var f: inputfile; var outf: textfile);
    const
      C_END_OF_FILE = 2;
    var
      fname: xtring;     { name of include file found }
      nextf: inputfile;  { include file opened }
      i: string_range;   { index for replacing ^s }
      code: Cint;        { return from crdetab }
      flag: xtring;      { a detected trailing comment of interesting length }
  begin
    if debug or debug_decomment then writeln(debugf, '---- handle_file ----');
    in_comment := FALSE;
    fname := nullstring;
    code := crdetab(f, line);  line_len := ord(line^[0]);
    while not (code = C_END_OF_FILE) do
      begin
	flag := flag_comment(line);
        if match_flag(flag, delete_lines) then in_comment := FALSE
        else
          begin
            fname := include_file(line);
            if fname = nullstring then
              begin
                if match_flag(flag, uncomment_lines) then uncomment_line;
                if decomment then strip_comments;
                if decolon_otherwise then handle_otherwise;
                if replace_carets then
                  for i := 1 to line_len do
                    if line^[i] = '^' then line^[i] := '@';
                write_line(outf, line);
              end
            else
              begin
                if reset_file(nextf, fname) then
                  begin
                    release_string(fname);
                    handle_file(nextf, outf);
                    cfclose(nextf);
                  end
                else
                  begin
                    release_string(fname);
                    writestring(outf, line);
                    writeln(outf);
                  end;
              end;
          end;
        release_string(flag);
        code := crdetab(f, line);  line_len := ord(line^[0]);
      end;
    if debug or debug_decomment then 
      writeln(debugf, '---- end handle_file ----');
  end { handle_file };


  procedure report_flags(var f: textfile);
    { report the flag lists to f for debugging purposes }
    var
      flag: flag_ptr;       { current flag }
  begin
    flag := delete_lines;
    if flag = NIL then write(f, 'delete no lines')
    else write(f, 'delete lines flagged ');
    while flag <> NIL do
      begin
        writestring(f, flag^.str);
        flag := flag^.next;
        if flag <> NIL then write(f, ' or ');
      end;
    writeln(f);

    flag := uncomment_lines;
    if flag = NIL then write(f, 'Uncomment no lines')
    else write(f, 'Uncomment lines flagged ');
    while flag <> NIL do
      begin
        writestring(f, flag^.str);
        flag := flag^.next;
        if flag <> NIL then write(f, ' or ');
      end;
    writeln(f);
  end { report_flags } ;



begin { mainproc }
  debug := FALSE;  debug_decomment := FALSE;  last_line_blank := FALSE;

  decomment := FALSE; in_comment := FALSE;  commented_line := FALSE;
  replace_carets := FALSE;
  decolon_otherwise := FALSE;  last_tok_otherwise := FALSE;
  delete_lines := NIL;  uncomment_lines := NIL;
  max_flag_length := 0;
  
  otherwise_char := OTHERWISE_CHAR_C;
  CAPS_otherwise_char := CAPS_OTHERWISE_CHAR_C;

  init_string_structures;

  create_a_string(line, MAX_STRING_LENGTH);

  input_file := nullstring;
  output_File := nullstring;
  debug_file := nullstring;
  {copy_to_string(DEBUG_FILE_C, debug_file);}(* open as stderr for UNIX *)

  max_flag_length := 0;

  command_args;

  if debug then report_flags(debugf);

  {if input_file <> nullstring then} (* use it to buffer input *)
    if not reset_file(infile, input_file) then
      begin
        write(stderr, 'bigfile: unable to open input file ');
        writestring(stderr, input_file);
        writeln(stderr);
        halt(1);
      end;
  if input_file = nullstring then                                {UNIX}
    if dup2(cfdsc(infile), textfdsc(input)) < 0 then             {UNIX}
      begin                                                      {UNIX}
        writeln(stderr,'Can''t open buffered standard input');   {UNIX}
        halt(1);                                                 {UNIX}
      end                                                        {UNIX}
    else copy_to_string(STANDARD_INPUT, input_file);             {UNIX}


  {if output_file <> nullstring then} (*use it to buffer output *)
    if not rewrite_file(outfile, output_file) then
      begin
        if input_file <> nullstring then cfclose(infile);
        writeln(stderr, 'bigfile: unable to open output file');
        writestring(stderr, output_file);
        writeln(stderr);
        halt(1);
      end;
  if output_file = nullstring then                               {UNIX}
    if dup2(textfdsc(outfile), textfdsc(output)) < 0 then        {UNIX}
      begin                                                      {UNIX}
        writeln(output,'Can''t open buffered standard output');  {UNIX}
        halt(1);                                                 {UNIX}
      end                                                        {UNIX}
    else copy_to_string(STANDARD_OUTPUT, output_file);           {UNIX}

  if input_file = nullstring then 
    begin
      writeln(output, 'Can''t use standard input for now');
      halt(1);
      if output_file = nullstring then
        begin
          {handle_file(input, output);}
        end
      else
        begin
          {handle_file(input, outfile);}
          close_file(outfile);
        end;
    end
  else { input_file <> nullstring}
    begin
      if output_file = nullstring then
        begin
          handle_file(infile, output);
          cfclose(infile);
        end
      else
        begin
          handle_file(infile, outfile);
          cfclose(infile);
          close_file(outfile);
        end;
    end;
end { mainproc };
begin { bigfile }
  mainproc;
end { bigfile }.
