/* Insert/delete balanced binary tree: otherwise known as avl trees.
		      Algorithm from Knuth, Searching & Sorting, pp 455-468 
   Key is integer or a typedef'd integer.
									    */

#ifndef IAVL_M_INCLUDED
#define IAVL_M_INCLUDED

#include "avl.m.h"

#define Xicompare(a,b,rslt,brec,type,dirty,vr,rv,unlock)rslt=a-b

/*                M A N I P U L A T I N G      M A C R O S                    */

/*----------------------------------------------------------------------------*/

/* Access the key in an IAVL or IAVLH structure
   Access the rsp in a IAVL or IAVLH structure
   Access the lsp in a IAVL or IAVLH structure
   Access the root in a IAVLH structure

   rec_avl ==> name within record of AVL or AVLH member.
   rec     ==> -> to record containing the rec_avl structure.

***Not assertion proofed.
									      */
#define IAVLRO_NULL AVLR_NULL

#define IAVL_RSP AVL_RSP
#define IAVL_LSP AVL_LSP

#define IAVLH_RSP IAVL_RSP
#define IAVLH_LSP IAVL_LSP

#define IAVLR_TOP AVLR_TOP

#define IAVL_NULL AVL_NULL
#define IAVLO_NULL IAVL_NULL

#define IAVLH_NULL AVLH_NULL

#define IAVL_CHK(rec_avl,rec)\
\
( (IAVL_KEY(rec_avl,rec) DNE) &&\
  (AVL_CK(rec_avl,rec))\
)
#define IAVLH_CHK(rec_avl,rec)\
\
( (IAVL_CHK(rec_avl,rec) DNE) &&\
  (B(rec_avl,rec) DNE)\
)

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from any member in the tree.

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  Initially set by you to the record from which
		  you wish the search to proceed from.
		  On completion, SET to record with matching key or SET to 
		  NULL if not found. 
									    */

#define IAVL_LOCATE(key_sought,rec_avl,found,type,dirty)\
\
AVL_LOCATE(key_sought,rec_avl,found,type,Xicompare,NOP)

#define IAVLO_LOCATE(key_sought,rec_avl,found,type,dirty)\
\
AVLO_LOCATE(key_sought,rec_avl,found,type,Xicompare,dirty)

#define IAVLH_LOCATE IAVL_LOCATE
#define IAVLHO_LOCATE IAVLO_LOCATE

/*----------------------------------------------------------------------------*/

/* Locate an AVL or AVLH binary-tree member (from the root of the tree)
   whose key is lowest in a range of integers.

/* 
   lowinrange ==> lowest integer in range
   hiinrange  ==> highest integer in range
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree from which key_sought is to be sought.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  SET to record with matching key;
		  SET to NULL if not found.
									     */

#define Xrngcompare(a,b,rslt,brec,type,dirty,vr,rv,unlock)\
\
if (b<a->low) {\
     if (Xfnd) {\
          rslt=0;\
	  unlock(brec);\
	  vr(Xfnd,brec,type,dirty);\
	  }\
     else rslt= 1;\
     }\
else\
if (b>a->hi) {\
     if (Xfnd) {\
          rslt=0;\
	  unlock(brec);\
	  vr(Xfnd,brec,type,dirty);\
	  }\
     else rslt= -1;\
     }\
else {\
     rv(brec,Xfnd,type);\
     rslt = -1;\
     }
          
     
#define IAVL_RNG_FIND(lowinrange,hiinrange,hd_avlr,hd,rec_avl,found,type,dirty)\
{\
struct type *Xfnd={NULL};\
struct { int low={lowinrange}, hi={hiinrange}; } Xtmp, *key_sought={&Xtmp};\
\
AVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xrngcompare,NOP);\
}

#define IAVLO_RNG_FIND(lowinrange,hiinrange,hd_avlr,hd,rec_avl,found,type,dirty)\
{\
struct type *Xfnd={NULL};\
struct { int low={lowinrange}, hi={hiinrange}; } Xtmp, *key_sought={&Xtmp};\
\
AVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xrngcompare,dirty);\
}

#define IAVLH_RNG_FIND IAVL_RNG_FIND
#define IAVLHO_RNG_FIND IAVLO_RNG_FIND

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from the root of the tree

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree from which key_sought is to be sought.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  SET to record with matching key;
		  SET to NULL if not found.
									     */

#define IAVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,dirty)\
\
AVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xicompare,NOP)

#define IAVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,dirty)\
\
AVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xicompare,dirty)

#define IAVLH_FIND IAVL_FIND
#define IAVLHO_FIND IAVLO_FIND

/*----------------------------------------------------------------------------*/

/* Insert a key into a AVL binary-tree.

   rec_avl    ==> name within rec of AVL member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVL member to be inserted
		  into the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			 */

#define IAVL_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there)\
\
AVL_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there,Xicompare)

#define IAVLO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there)\
\
AVLO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there,Xicompare)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVL binary-tree

   rec_avl    ==> name within rec of AVL member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVL member to be deleted
		  from the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_not_there ==> operations to perform if the key is not present
		     in the tree. NOTE: it must start with a character!
			                                                   */
#define IAVL_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
AVL_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xicompare)

#define IAVLO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
AVLO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xicompare)

/*----------------------------------------------------------------------------*/

/* Insert a key into a AVLH binary-tree.

   rec_avl    ==> name within rec of AVLH member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVLH member to be inserted
		  into the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			 */
#define IAVLH_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there)\
\
AVLH_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there,Xicompare)

#define IAVLHO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there)\
\
AVLHO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there,Xicompare)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVLH binary-tree
   
   rec_avl    ==> name within rec of AVLH member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVLH member to be deleted
		  from the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_not_there ==> operations to perform if the key is not present
		     in the tree. NOTE: it must start with a character!
			                                                   */
#define IAVLH_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
AVLH_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xicompare)

#define IAVLHO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
AVLHO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xicompare)

/*----------------------------------------------------------------------------*/
#endif
