/* Insert/delete balanced binary tree: otherwise known as avl trees.
		      Algorithm from Knuth, Searching & Sorting, pp 455-468 
   Key is integer or a typedef'd integer.
									    */

#ifndef PAVL_M_INCLUDED
#define PAVL_M_INCLUDED

#include "avl.m.h"

#define Xpcompare(a,b,rslt,brec,type,dirty,vr,rv,unlock)rslt=((int)a)-((int)b)

/*                M A N I P U L A T I N G      M A C R O S                    */

/*----------------------------------------------------------------------------*/

/* Access the key in an PAVL or PAVLH structure
   Access the rsp in a PAVL or PAVLH structure
   Access the lsp in a PAVL or PAVLH structure
   Access the root in a PAVLH structure

   rec_avl ==> name within record of AVL or AVLH member.
   rec     ==> -> to record containing the rec_avl structure.

***Not asion proofed.
									      */
#define PAVLR  AVLR

#define PAVLO_KEYD(val,rec_avl,rec,type)\
\
RO_ADRCK(rec);\
ADRCK(val);\
RO_VO(val,AVL_KEY(rec_avl,rec),type)

#define PAVL_RSP AVL_RSP
#define PAVL_LSP AVL_LSP

#define PAVLH_RSP PAVL_RSP
#define PAVLH_LSP PAVL_LSP

#define PAVLR_TOP AVLR_TOP
#define PAVLRO_TOP AVLRO_TOP

#define PAVLR_NULL AVLR_NULL
#define PAVL_NULL AVL_NULL
#define PAVLH_NULL AVLH_NULL

#define PAVL_CHK(rec_avl,rec)\
\
( (PAVL_KEY(rec_avl,rec) DNE) &&\
  (AVL_CK(rec_avl,rec))\
)
#define PAVLH_CHK(rec_avl,rec)\
\
( (PAVL_CHK(rec_avl,rec) DNE) &&\
  (B(rec_avl,rec)   DNE)\
)

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from any member in the tree.

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  Initially set by you to the record from which
		  you wish the search to proceed from.
		  On completion, SET to record with matching key or SET to 
		  NULL if not found. 
									    */

#define PAVL_LOCATE(key_sought,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key could point anywhere*/\
AVL_LOCATE(key_sought,rec_avl,found,type,Xpcompare,NOP)

#define PAVLO_LOCATE(key_sought,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key could point anywhere*/\
AVLO_LOCATE(key_sought,rec_avl,found,type,Xpcompare,dirty)

#define PAVLH_LOCATE PAVL_LOCATE
#define PAVLHO_LOCATE PAVLO_LOCATE

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from the root of the tree

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree from which key_sought is to be sought.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  SET to record with matching key;
		  SET to NULL if not found.
									     */

#define PAVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key could point anywhere*/\
AVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xpcompare,NOP)

#define PAVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key could point anywhere*/\
AVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xpcompare,dirty)

#define PAVLH_FIND PAVL_FIND
#define PAVLHO_FIND PAVLO_FIND

/*----------------------------------------------------------------------------*/

/* Insert a key into a AVL binary-tree.

   rec_avl    ==> name within rec of AVL member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVL member to be inserted
		  into the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			 */

#define PAVL_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there)\
\
ADRCK(insert);\
ADRCK(PAVL_KEY(rec_avl,insert));/*could point anywhere*/\
AVL_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there,Xpcompare)

#define PAVLO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there)\
\
RO_ADRCK(insert);\
ADRCK(PAVL_KEY(rec_avl,insert));/*could point anywhere*/\
AVLO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there,Xpcompare)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVL binary-tree

   rec_avl    ==> name within rec of AVL member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVL member to be deleted
		  from the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_not_there ==> operations to perform if the key is not present
		     in the tree. NOTE: it must start with a character!
			                                                   */
#define PAVL_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
ADRCK(delete);\
ADRCK(PAVL_KEY(rec_avl,delete));/*could point anywhere*/\
AVL_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xpcompare)

#define PAVLO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
RO_ADRCK(delete);\
ADRCK(PAVL_KEY(rec_avl,delete));/*could point anywhere*/\
AVLO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xpcompare)

/*----------------------------------------------------------------------------*/

/* Insert a key into a AVLH binary-tree.

   rec_avl    ==> name within rec of AVLH member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVLH member to be inserted
		  into the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			 */
#define PAVLH_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there)\
\
ADRCK(insert);\
ADRCK(PAVL_KEY(rec_avl,insert));/*could point anywhere*/\
AVLH_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there,Xpcompare)

#define PAVLHO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there)\
\
RO_ADRCK(insert);\
ADRCK(PAVL_KEY(rec_avl,insert));/*could point anywhere*/\
AVLHO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there,Xpcompare)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVLH binary-tree
   
   rec_avl    ==> name within rec of AVLH member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVLH member to be deleted
		  from the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_not_there ==> operations to perform if the key is not present
		     in the tree. NOTE: it must start with a character!
			                                                   */
#define PAVLH_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
ADRCK(delete);\
ADRCK(PAVL_KEY(rec_avl,delete));/*could point anywhere*/\
AVLH_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xpcompare)

#define PAVLHO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
RO_ADRCK(delete);\
ADRCK(PAVL_KEY(rec_avl,delete));/*could point anywhere*/\
AVLHO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xpcompare)

/*----------------------------------------------------------------------------*/
#endif
