/*LINTLIBRARY*/
#include <stdio.h>
#include "mem.h"
#include "internals.h"
#include "../misc/assert.h"
#include "../bds/bds.m.h"
#include "../bds/stack.m.h"


#define to_power_of_2_boundary(n, exponent)\
    (((n - 1 + (1 << exponent)) >> exponent) << exponent)
#define to_word_boundary(n) to_power_of_2_boundary(n, LOG2_WORDSIZE)
#define to_block_boundary(n) to_power_of_2_boundary(n, LOG2_BLOCKSIZE)
#define BLOCKSIZE (1<<LOG2_BLOCKSIZE)
#define offset(p, n) (((char *)p) + n)


static char *out_of_mem = "Out of memory!";
static char *no_arena = "Arena not initialized";


#ifdef UNIX
char *malloc();
void  free();
#define primary_alloc malloc
#define primary_free(p, s) free(p)
#endif UNIX
#ifdef VMS
#define odd(x) (1 & x)
static char *
primary_alloc(n)
    int n;
{
    char *rslt;
    int nbytes={n};
    return odd(LIB$GET_VM(&nbytes, &rslt)) ? rslt : (char *)0;
}
#define primary_free(p, s) LIB$FREE_VM(p, &(s))
#undef odd
#endif VMS


#ifdef DEBUGA
static void
arena_address_check(arena, p)
struct arena *arena;
struct element *p;
{
    struct block *block = F(blocks, arena);
    while (block) {
        if ((char *)p >= offset(block, sizeof(struct block)) && 
	    (char *)p < offset(block, block->size)) return;
	block = FL(arena, block);
    }
    FAULT("Address check failed");
}


static void
check_old_block(marena, p)
struct main_arena *marena;
struct block *p;
{
    struct block *block = F(used_blocks, marena);
    while (block) {
        if (p == block) return;
	R_FNEXT(main_arena, block, used_blocks, marena);
    }
    FAULT("Main arena address check failed");
}
#else
#define arena_address_check(arena, p)
#define check_old_block(arena, p)
#endif DEBUGA


void
mem_main_arena(marena)
struct main_arena **marena;
{
    if (*marena) FAULT("Main arena already in use");
    *marena = (struct main_arena *)primary_alloc(sizeof(struct main_arena));
    if (!(*marena)) FAULT(out_of_mem);
    SH_NULL(free_blocks, (*marena));
    SH_NULL(used_blocks, (*marena));
}


void
mem_pool_main_arena(marena_by_ref)
struct main_arena **marena_by_ref;
{
    register struct main_arena *marena = *marena_by_ref;
    struct block *block;
    if (!marena) FAULT("Main arena not initialized");
    while (block = F(free_blocks, marena)) {
        SH_DR_PICK(main_arena, block, free_blocks, marena);
	if (block->allocated) primary_free((char *)block, block->allocated);
    }
    while (block = F(used_blocks, marena)) {
        SH_DR_PICK(main_arena, block, used_blocks, marena);
	if (block->allocated) primary_free((char *)block, block->allocated);
    }
    primary_free((char *)marena, sizeof(*marena));
    *marena_by_ref = 0;
}


#define new_block(marena, block, sz) {\
    block->size = sz;\
    block->allocated = 0;\
    DR_NULL(main_arena, block);\
    SL_NULL(arena, block);\
    SH_DR_FQUE(main_arena, block, free_blocks, marena);\
    }


#define free_block(marena, block) {\
    check_old_block(marena, block);\
    SH_DR_PICK(main_arena, block, used_blocks, marena);\
    SH_DR_FQUE(main_arena, block, free_blocks, marena);\
    }


static struct block *
get_block(marena, sz)
struct main_arena *marena;
unsigned sz;
{
    struct block *block = F(free_blocks, marena);
    unsigned size, allocsize;
    int safety;

    size = to_block_boundary(sz); as(size >= sz);
    if (size > sz && size - sz < sizeof(struct element)) 
	size += BLOCKSIZE;

    safety = 2; /* 2nd pass MUST win */
    while (safety--) {
	while (block) {
	    if (block->size >= size) {
		SH_DR_PICK(main_arena, block, free_blocks, marena);
		SH_DR_FQUE(main_arena, block, used_blocks, marena);
    
		if (block->size > size) {
		    struct block *remainder;
		    remainder = (struct block *)offset(block, size);
		    new_block(marena, remainder, block->size - size);
		    block->size = size;
		}
    
		block->free = size - sizeof(struct block);
		return block;
	    }
	    R_FNEXT(main_arena, block, free_blocks, marena);
	}
    
	allocsize = 1<<LOG2_SBRK_SIZE;
	if (size > allocsize) allocsize = size;
    
	while (!block && allocsize >= size) {
	    if (block = (struct block *)primary_alloc(allocsize)) {
		new_block(marena, block, allocsize);
		block->allocated = allocsize;
	    }
	    else allocsize = allocsize >> 1;
	}
        if (!block) FAULT(out_of_mem);
    }
    FAULT("Serious primary block insertion error");
    return (struct block *)0;  /* just for lint */
}


void
memarena(marena, arena_by_ref)
struct main_arena *marena;
struct arena **arena_by_ref;
{
    struct arena *arena;
    if (!marena) FAULT("Main arena not established");
    if (*arena_by_ref) FAULT("Arena already initialized");
    arena = *arena_by_ref = 
	(struct arena *)get_block(marena, sizeof(struct arena));
    arena->self.free = arena->self.size - sizeof(struct arena);
    arena->main_arena = marena;
    SH_NULL(elements, arena);
    SH_NULL(blocks, arena);
    SHR_NULL(pointer_blocks, arena);
    SH_SL_FQUE(arena, (struct block *)arena, blocks, arena);
}



void
mempool(arena_by_ref)
struct arena **arena_by_ref;
{
    register struct arena *arena = *arena_by_ref;
    struct main_arena *marena;
    struct block *block;
    if (!arena) FAULT(no_arena);
    marena = arena->main_arena;
    SH_SL_POP(arena, block, blocks, arena);
    while (block) {
	if (block != (struct block *)arena) free_block(marena, block);
	SH_SL_POP(arena, block, blocks, arena);
    }
    free_block(marena, (struct block *)arena);
    *arena_by_ref = 0;
}


#define free_element(arena, p, sz) {\
    (p)->size = sz;\
    SL_NULL(next, p);\
    arena_address_check(arena, p);\
    SH_SL_FQUE(next, p, elements, arena);\
}


#ifdef DEBUGSZ
#define size_check_adjust(size) size += 4
#define size_check_mark(el, size) {\
    unsigned long *ptr = (unsigned long *)el;\
    *ptr = size;\
    el = (struct element *)offset(el, 4);\
}
#define size_check(el, size) {\
    unsigned long *ptr;\
    size += 4;\
    el = (struct element *)offset(el, (-4));\
    ptr = (unsigned long *)el;\
    if (*ptr != size) FAULT("Flunked size check");\
}
#else
#define size_check_adjust(s)
#define size_check_mark(e,s)
#define size_check(e,s)
#endif DEBUGSZ
    

/*
    When single pointers are requested, they are handled as a special
    case -- they are allocated from a ring of pointer_blocks.  Pointer
    blocks that contain free space are kept at the head of the ring.
    Those that don't are kept at the rear.  If the head element 
    contains no space, then none of them do.

    The pointer_block elements are 'mem'ed from their arena, so they
    are automatically reclaimed if the arena is pooled.
*/


static char *
allocate_pointer(arena)
struct arena *arena;
{
    register struct pointer_block *pb;

#ifdef DEBUGP
    fprintf(stderr, "allocate_pointer(arena 0x%x)\n", arena);
#endif
    pb = SH_SR_HEAD(next, pointer_blocks, arena);
    if (pb && pb->unused) {
        register unsigned long mask, i, unused = pb->unused;
#ifdef DEBUGP
        fprintf(stderr, "use pb 0x%x unused 0x%x\n", pb, pb->unused);
#endif
        for (mask = 1, i = POINTERS_PER_PBLOCK;
	     i && !(unused & mask); 
	     i--, mask <<= 1) ;
	as(i);
	pb->unused = (unused &= ~mask);
#ifdef DEBUGP
        fprintf(stderr, "    mask 0x%x unused 0x%x allocated pointer 0x%x\n",
	    mask, pb->unused, (char *)(&(pb->pointers[i-1])));
#endif
	if (!unused) SH_SR_ROTATE(next, pointer_blocks, arena);
	return (char *)(&(pb->pointers[--i]));
	}

    as(sizeof(unsigned long) == 4);
    as(sizeof(char*) == 4);
    as(sizeof(struct pointer_block) == 34*sizeof(char*));

    pb = (struct pointer_block *)mem(arena, sizeof(struct pointer_block));
    SR_NULL(next, pb);
    SH_SR_FQUE(next, pb, pointer_blocks, arena);

    pb->unused = INITIAL_UNUSED; /* last element used, rest unused */
#ifdef DEBUGP
    fprintf(stderr, "new pb 0x%x allocated pointer 0x%x\n", pb,
	(char *)(&(pb->pointers[POINTERS_PER_PBLOCK-1])));
    
#endif
    return (char *)(&(pb->pointers[POINTERS_PER_PBLOCK-1]));
}


static void
free_allocated_pointer(arena, p)
struct arena *arena;
register char *p;
{
    register struct pointer_block *pb, *prev;

#ifdef DEBUGP
    fprintf(stderr, "free_pointer(arena 0x%x, pointer 0x%x)\n", arena, p);
#endif
    pb = SH_SR_HEAD(next, pointer_blocks, arena);  prev = 0;
    while (pb) {
#ifdef DEBUGP
        fprintf(stderr, "    pb 0x%x", pb);
#endif
	if ((unsigned long)(pb->pointers) <= (unsigned long)p &&
	    (char**)p - pb->pointers < POINTERS_PER_PBLOCK) {
	    register unsigned long mask = 1;
	    unsigned shift = POINTERS_PER_PBLOCK - 1 - ((char**)p - pb->pointers);
	    if (shift) mask <<= shift;
#ifdef DEBUGP
            fprintf(stderr, " shift = %d, mask = 0x%x\n", shift, mask);
#endif
	    pb->unused |= mask;
	    SH_SR_PICK(next, pb, prev, pointer_blocks, arena);
	    SH_SR_FQUE(next, pb, pointer_blocks, arena);
	    return;
	    }
#ifdef DEBUGP
        fprintf(stderr, "\n");
#endif
	prev = pb;
	pb = SH_SR_NEXT(next, pb, pointer_blocks, arena);
        }
    FAULT("Yeow -- severe problem with pointer allocation or its use");
}


void
memfree(arena,p,sz)
struct arena *arena;
char *p;
unsigned sz;
{
    struct element *element = (struct element *)p;
    unsigned size = to_word_boundary(sz);
    as(size >= sz);
    if (size <= sizeof(char*)) {
        free_allocated_pointer(arena, p);
	return;
	}
    if (size < sizeof(struct element)) 
        size = to_word_boundary(sizeof(struct element));
    size_check(element, size);
    free_element(arena, element, size);
}

/* Prevent paging through all spent blocks whenever current one is spent */
#define MAX_BLOCKS_TO_CHECK 3

    

char *
mem(arena, sz)
struct arena *arena;
unsigned sz;
{
    struct element *element, *prev;
    struct block *block;
    unsigned size = to_word_boundary(sz);
    int safety;

    if (!arena) FAULT(no_arena);
    as(size >= sz);
    if (size <= sizeof(char*)) return allocate_pointer(arena);
    if (size < sizeof(struct element)) 
        size = to_word_boundary(sizeof(struct element));

    size_check_adjust(size);

    element = F(elements, arena);  prev = 0;
    while (element) {
        if (element->size == size ||
	    element->size >= size + sizeof(struct element)) {

	    SH_SL_PICK(next, element, prev, elements, arena);

	    if (element->size > size) {
	        struct element *remainder;
		remainder = (struct element *)offset(element, size);
		free_element(arena, remainder, element->size - size);
	    }

            size_check_mark(element, size);
	    return (char *)element;
	}
	prev = element;
	element = FL(next, element);
    }

    safety = 2; /* 2nd pass MUST win */
    while (safety--) {
        int blocks_to_check;
	block = F(blocks, arena);  
	blocks_to_check = MAX_BLOCKS_TO_CHECK;
	while (block && blocks_to_check--) {
	    if (block->free == size ||
		block->free >= size + sizeof(struct element)) {
		struct element *val;
		val = (struct element *)offset(block, (block->size - block->free));
		block->free -= size;
		size_check_mark(val, size);
		return (char *)val;
	    }
	    block = FL(arena, block);
	}
    
	block = get_block(arena->main_arena, size + sizeof(struct block));
	SH_SL_FQUE(arena, block, blocks, arena);
    }
    FAULT("Serious block insertion error");
    return (char *)0; /* for lint */
}


char *
cmem(arena, count, sz)
struct arena *arena;
unsigned count, sz;
{
    WORD *gotten, *end;
    register WORD *p;
    unsigned size = to_word_boundary(sz);
    as(size >= sz);
    size *= count;
    gotten = (WORD *)mem(arena, size);
    end = (WORD *)offset(gotten, size);
    for (p = gotten; p < end ; p++) *p = 0;
    return (char *)gotten;
}


void
cmemfree(arena, p, count, size)
struct arena *arena;
char *p;
unsigned count, size;
{
    size = to_word_boundary(size);
    memfree(arena, p, size * count);
}


char *
rmem(arena, p, oldsize, newsize)
struct arena *arena;
char *p;
unsigned oldsize, newsize;
{
    WORD *gotten, *end;
    register WORD *source, *dest;
    struct element *element;

    oldsize = to_word_boundary(oldsize);
    newsize = to_word_boundary(newsize);
    
    if ((oldsize <= sizeof(char*)) || (newsize <= sizeof(char*))) {
        FAULT("Can't rmem() to or from sizeof(char*) -- not implemented");
	}

    if (oldsize < sizeof(struct element)) 
        oldsize = to_word_boundary(sizeof(struct element));
    if (newsize < sizeof(struct element)) 
        newsize = to_word_boundary(sizeof(struct element));

    if (oldsize == newsize) return p;

    if (newsize < oldsize && oldsize - newsize > sizeof(struct element)) {
	struct element *remainder = (struct element *)offset(p, newsize);
	free_element(arena, remainder, oldsize - newsize);
	return p;
    }

    dest = gotten = (WORD *)mem(arena, newsize);
    source = (WORD *)p;  
    if (newsize < oldsize) {
	end = (WORD *)offset(gotten, newsize);
	while (dest < end) *dest++ = *source++;
	}
    else {
	end = (WORD *)offset(p, oldsize);
	while (source < end) *dest++ = *source++;
	}
    element = (struct element *)p;

    size_check(element, oldsize);
    free_element(arena, element, oldsize);
    return (char *)gotten;
}


void
mem_report_usage(arena, size, free)
struct arena *arena;
unsigned *size, *free;
{
    struct block *block;
    struct element *element;
    *size = *free = 0;
    if (!arena) return;

    block = F(blocks, arena);
    while (block) {
        *size += block->size;
	*free += block->free;
	block = FL(arena, block);
    }

    element = F(elements, arena);
    while (element) {
        *free += element->size;
	element = FL(next, element);
    }
}


void
mem_report_main_usage(marena, size, free)
struct main_arena *marena;
unsigned *size, *free;
{
    struct block *block;
    *size = *free = 0;
    if (!marena) return;

    block = F(used_blocks, marena);
    while (block) {
        *size += block->size;
	R_FNEXT(main_arena, block, used_blocks, marena);
    }

    block = F(free_blocks, marena);
    while (block) {
        *size += block->size;
        *free += block->size;
	R_FNEXT(main_arena, block, free_blocks, marena);
    }

}
