// dwg_entry.C -- Drawing table entries
#include "Clib/heap.h"
#include "error.h"
#include "dwg_special.h"
#include "dwg_entry.h"
#include "special.h"
#include "table.h"
#include "lex.h"

#include "/home/andre/hoshah/c++/ds/ds_comp.h"
#include "/home/andre/hoshah/c++/ds/ext_ds.h"

extern ds_comp_env *CURR_DSENV_PTR;



const MAX_VERSION_NUM = 10000;	// per previous ValidCOMPILERs
const MAX_EXTENSION_SIZE = 64;	// Much larger than needed


erule_drawing_exception::erule_drawing_exception(const string& nm,
						 erule_extension* ex)
{
    AS(!this);
    this = (erule_drawing_exception*)erule_heap.mem(sizeof(*this));
    _name = nm;
    extension = ex;
}


erule_drawing_entry::erule_drawing_entry(const string& name)
{ 
    AS(!this);
    this = (erule_drawing_entry*)erule_heap.mem(sizeof(*this));
    AS(!versions.first());
    _name = name;  
    init_flag = expressions_read = FALSE;
}

void erule_drawing_entry::dump(ostream& f)
{
    f << name() NL;

    erule_drawing_version* v = versions.first();
    while (v) { v->dump(f);  v = versions.next(v); }
    erule_drawing_special* s = specials.first();
    while (s) { s->dump(f);  s = specials.next(s); }
}


/*
//init the entry with all the availiable versions, pages, and special files
//for files of type connectivity, from the data services
*/
void erule_drawing_entry::initialize_entry()
{
    DSScaldPath scpath;
    AS(!CURR_DSENV_PTR);
    if (CURR_DSENV_PTR->get_dwg(name(), scpath)) {
	erule_current_ds_compdwg_ptr = new ds_compdwg(scpath);

	AS(!erule_current_ds_compdwg_ptr);
	read_pages(*erule_current_ds_compdwg_ptr);
	read_special_files(*erule_current_ds_compdwg_ptr);
	
	//donot delete erule_current_ds_compdwg_ptr;will be deallocated at
	//end of drawing compile
	erule_current_ds_compdwg_ptr = 0;
    }
    init_flag = 1;
}

void erule_drawing_entry::read_pages(ds_compdwg &dsdwg_mod)
{
    const char* CONN_FILE="CN";

    AS(this);
    AS(erule_current_drawing);

#ifdef DEBUG
cerr << "Read pages of " << name() << " from " << dsdwg_mod.name() NL;
#endif


    gslist(ds_fileobjP) filelist;
    if (dsdwg_mod.get_filelist(CONN_FILE, filelist)) {
	gslist_iterator(ds_fileobjP) list_itr(filelist);
	ds_fileobj *dsc_ptr = 0;
	while (dsc_ptr = list_itr()) {
	    char extension[MAX_EXTENSION_SIZE];

#ifdef DEBUG
cerr << "    CN file " << dsc_ptr->full_ds_name() NL;
#endif
	    if (strlen(dsc_ptr->type())< MAX_EXTENSION_SIZE)
		strcpy(extension, dsc_ptr->type());
	    erule_current_extension = extension;
	    erule_drawing_version* vers = 
		enter_version(dsc_ptr->vers());
	    if (vers) {
		erule_drawing_extension* ext = 
		    vers->enter(dsc_ptr->type(), &dsdwg_mod);
		if (ext) ext->enter_page(dsc_ptr->page());
	    }
	}
    }
    erule_current_version = 0;
    erule_current_extension = 0;
    erule_current_page = 0;
    read_expressions();
}


void erule_drawing_entry::read_special_files(const ds_compdwg& dwg_mod)
{
    AS(erule_current_drawing);

    erule_special* special;
    for (special = erule_the_rule_table->first_special();  special;
	 special = erule_the_rule_table->next_special()) {
	    enter_special(special->name(), dwg_mod);
	}
}


void
erule_drawing_entry::enter_exception(const string& context, erule_extension* e)
{
    erule_drawing_exception* except = exceptions.find(context);
    if (except) {
	delete except->extension;
	except->extension = e;
    }
    else {
	except = new erule_drawing_exception(context, e);
	exceptions.insert(except);
    }
}


erule_drawing_extension*
erule_drawing_entry::select_from_single_version(erule_rule* rule)
{
    AS(is_single_versioned());
    return versions.first()->select(rule);
}


erule_drawing_extension*
erule_drawing_entry::select_extension(erule_rule* rule, 
				      boolean non_existence_is_not_error)
{
    /* 
	NOTE -- this currently allows "no selection expression TRUE" to pass
	without error if  there is a "special" model available.  
	This is probably the optimum behaviour.
    */
    boolean found_something = FALSE;
    erule_drawing_extension* a_default = 0;
    erule_drawing_extension* selected = 0;
    erule_drawing_version* vers = versions.first();
    while (vers) {
        erule_drawing_extension* extension = vers->select(rule);
	if (extension) {
	    found_something = TRUE;
	    if (extension->is_default()) {
		if (!a_default) a_default = extension;
		else selection_error(ERR_MULTIPLE_DEFAULT_VERSIONS, rule);
	    }
	    else if (extension->evaluate()) {
		if (!selected) selected = extension;
		else selection_error(ERR_MULTIPLE_TRUE_VERSIONS, rule);
	    }
	}
	vers = versions.next(vers);
    }
    if (!selected) selected = a_default;

    if (!selected && !non_existence_is_not_error)
	/*if (!found_in_some_directory)
	    selection_error(ERR_DRAWING_NOT_FOUND, rule);
        else*/ if (found_something)
	    selection_error(ERR_NO_TRUE_VERSION, rule);
	else selection_error(ERR_NO_ACCEPTABLE_EXTENSION, rule);

    return selected;
}


erule_drawing_special* erule_drawing_entry::
select_special(erule_rule* rule)
{
    erule_drawing_special* selected = 0;
    int selected_val = 0; // The value for "not in the rule"
    erule_drawing_special* spec;
    for (spec = specials.first(); spec; spec = specials.next(spec)) {
	int spec_val;
	if (spec_val = rule->ord(spec, selected_val)) {
	    selected = spec;
	    selected_val = spec_val;
	}
    }
    return selected;
}


erule_drawing_version* erule_drawing_entry::
enter_version(int num)
{
    if (num < 0 || num > MAX_VERSION_NUM) {
	erule_err.current_conn_file(ERR_VERSION_NUM);
        return 0;
    }

    erule_drawing_version* prev = 0;
    erule_drawing_version* vers = versions.first();
    while (vers) {
	if (vers->version_number() >= num) break;
	prev = vers;
	vers = versions.next(vers);
    }
    if (!vers || vers->version_number() != num) {
	vers = new erule_drawing_version(num);
	if (prev) versions.insert_after(vers, prev);
	else versions.insert_front(vers);
    }
    return vers;
}


erule_drawing_special* erule_drawing_entry::
enter_special(const string& nm, const ds_compdwg &dwg_mod)
{
    // NEED to check for error of multiple models from different dirs
    erule_drawing_special* model = specials.first();
    while (model) {
	if (model->name() == nm) break;
	model = specials.next(model);
    }
    ds_compdwg *dp = 0;
    if (model) dp = model->get_associated_ds_compdwg();
    if (dp) {
	String p = (dp->StartPath()).PathName();
	String r = (dwg_mod.StartPath()).PathName();
	if (p != r)
	    model->collision_error(dwg_mod);
    }
    else {
	model = new erule_drawing_special(nm, &dwg_mod);
	specials.insert_front(model);
    }
    return model;
}


void erule_drawing_entry::
selection_error(int err, erule_rule* rule)
{
    erule_err.message(err, this);

    switch(err) {
    case ERR_MULTIPLE_DEFAULT_VERSIONS:
    case ERR_MULTIPLE_TRUE_VERSIONS:
    case ERR_NO_TRUE_VERSION:
	erule_drawing_version* vers = versions.first();
	while (vers) {
	    erule_drawing_extension* extension = vers->select(rule);
	    if (extension) extension->error_dump_expression();
	    vers = versions.next(vers);
	}
	break;
    case ERR_DRAWING_NOT_FOUND:
    case ERR_NO_ACCEPTABLE_EXTENSION:
	break;
    default:
	erule_err.message(ERR_ASSERT);
	erule_err ERR_INDENT << 
	         "Don't know what else to print for previous message" ERR_NL;
	break;
    }
}


void erule_drawing_entry::read_expressions()
{
    if (expressions_read) return;
    expressions_read = TRUE;

    erule_drawing_version* vers;
    for (vers = versions.first(); vers; vers = versions.next(vers))
	vers->read_expressions(this);
}


/*
 * The remainder of this module processes the menu drawings that may still
 * be in existence.
 */


static erule_drawing_extension* menu_ext = 0;	// For error messages
static erule_drawing_page* menu_page = 0;


void erule_drawing_entry::
read_menu(erule_drawing_extension* menu)
{
    menu_ext = menu;
    for (menu_page = menu->first_page(); 
         menu_page; 
	 menu_page = menu->next_page(menu_page)) {

	const char* ds_filename = menu->connectivity_file(this, menu_page);
	if (!ds_filename) continue;	// Error already issued


        FILE *fp = dsfile_fopen(ds_filename, 0, 'r');
	AS(!fp);
	erule_lex lex(fp);
    
        if (!fp) {
	    erule_err.message(ERR_CANT_OPEN_CN);
	    erule_err ERR_INDENT << "DS Object=" << ds_filename ERR_NL;
	    continue;
         }

	if (!lex.scald_file_type()) continue;
	if (strcmp(lex.val(), "MACRO_DEFINITION")) {
	    // We COULD make this an assertion failure
	    erule_err.message(ERR_WRONG_FILETYPE);
	    erule_err ERR_INDENT << 
		      "Expected MACRO_DEFINITION containing MENU bodies" ERR_NL;
	    continue;
	}

	lex.next_token();
	if (lex == LEX_SEMI) lex.next_token();
	else lex.assert(35);

	if (lex != LEX_ID || strcmp(lex.val(), "MACRO")) {
	    lex.assert(28 /* Expected MACRO */);
	    return;
	}
	lex.next_token();
	while (lex == LEX_ID && !strcmp(lex.val(), "INVOKE")) {
	    lex.next_token();
	    if (lex != LEX_STRING) {
		lex.assert(26 /* missing macro name */);
		lex.skip_to_id("END_INVOKE");
	    }
	    else if (!strcmp(lex.val(), "MENU")) {
		lex.next_token();
		parse_menu_body(&lex);
	    }
	    else if (!strcmp(lex.val(), "DRAWING")) 
		lex.skip_to_id("END_INVOKE");	// Ignore DRAWING body
	    else {
		menu_error(78 /* not allowed */);
		erule_err ERR_INDENT << "Body name=" << lex.val() ERR_NL;
		lex.skip_to_id("END_INVOKE");
	    }

	    if (lex == LEX_ID && !strcmp(lex.val(), "BINDINGS")) {
		menu_error(136);
		erule_err ERR_INDENT << "Body name=MENU" ERR_NL;
		lex.skip_to_id("END_INVOKE");
	    }

	    if (lex == LEX_ID && !strcmp(lex.val(), "END_INVOKE"))
		lex.next_token();
	    else lex.assert(27);
	    if (lex == LEX_SEMI) lex.next_token();
	    else lex.assert(35);
	}
    }
}


void erule_drawing_entry::
parse_menu_body(erule_lex *lex)
{
    if (*lex != LEX_ID || strcmp(lex->val(), "PROPERTY")) return;
    lex->next_token();

    do {
	if (*lex != LEX_ID) {
	    lex->assert(20 /* unexpected symbol in property body */);
	    lex->skip(LEX_ID);
	    while (*lex == LEX_ID && strcmp(lex->val(), "BODY") && 
				     strcmp(lex->val(), "PIN") &&
				     strcmp(lex->val(), "PARAMETER") &&
				     strcmp(lex->val(), "END_PROPERTY"))
		lex->skip(LEX_ID);
	}
	else {
	    if (!strcmp(lex->val(), "BODY")) parse_menu_properties(lex);
	    else if (!strcmp(lex->val(), "END_PROPERTY")) break;
	    else if (!strcmp(lex->val(), "PIN")) {
		menu_error(125 /* not allowed */);
		lex->skip_past_id("END_PIN");
	    }
	    else if (!strcmp(lex->val(), "PARAMETER")) {
		menu_error(188 /* not allowed */);
		lex->skip_past_id("END_PARAMETER");
	    }
	    else {
		lex->assert(20 /* unexpected symbol in property body */);
		do {
		    lex->next_token();
		    lex->skip(LEX_ID);
		} while (*lex == LEX_ID && strcmp(lex->val(), "BODY") && 
					   strcmp(lex->val(), "PIN") &&
					   strcmp(lex->val(), "PARAMETER") &&
					   strcmp(lex->val(), "END_PROPERTY"));
	    }
	}
    } while (*lex && *lex != LEX_END);

    if (*lex == LEX_ID && !strcmp(lex->val(), "END_PROPERTY")) 
	lex->next_token();
    else lex->assert(21);
    if (*lex == LEX_SEMI) lex->next_token();
    else lex->assert(35);
}


void erule_drawing_entry::
set_expression_from_menu(int vers, const erule_expression& expression)
{
    if (vers == 1) { menu_error(129 /* vers 1 is menu */);  return; }

    erule_drawing_version* v = versions.first();
    for ( ; v; v = versions.next(v)) if (v->version_number() == vers) break;

    if (v) v->set_expression_from_menu(this, menu_ext, expression);
    else {
	menu_error(ERR_VERS_NOT_FOUND_IN_DIR);
	erule_err ERR_INDENT << "Unfound version=" <<
	          menu_ext->name() << "." << vers ERR_NL;
    }
}


void erule_drawing_entry::
menu_error(int num, const char* property)
{
    erule_err.message(num);
    erule_err ERR_INDENT << "Menu drawing=" << name() << 
              "." << menu_ext->name();
    erule_err << "." << menu_ext->version_number()  << 
                 "." << menu_page->page_number() ERR_NL;
    if (property) erule_err ERR_INDENT << "Property=" << property ERR_NL;
}


const MAX_MENU_ENTRY = 256;
gring_define(next,menu_entry);
static class menu_entry {
    friend class menu_table;
    gring_inst(next,menu_entry);
    int num;
    erule_expression expr;
    erule_expression version;
    boolean version_set, expr_set;
    menu_entry(int i, heap* h)
    { 
	this = this ? this : h->mem(sizeof(menu_entry));
	num = i;
	version_set = expr_set = FALSE;
    }
    ~menu_entry() { fault("Did not expect this destructor to be called"); }
public:
    void to_directory(erule_drawing_entry* drawing)
	{ drawing->set_expression_from_menu(version.evaluate(), expr); }
};
gring_inline(next,menu_entry);


static class menu_table : public gringroot(next,menu_entry) {
    heap h;
    menu_entry* enter(int);
    erule_drawing_entry* drawing;	// For errors
public:
    menu_table(erule_drawing_entry* d) { drawing = d; };
    ~menu_table();
    void add_version(int, const char*);
    void add_expr(int, const char*);
};


static menu_table::~menu_table()
{
    menu_entry* m = first();
    for ( ; m; m = next(m)) m->expr = m->version = 0;
    h.pool();
}


static menu_entry* menu_table::enter(int n)
{
    menu_entry* parent = 0;
    menu_entry* m = first();
    for ( ; m; m = next(m)) 
	if (m->num == n) return m;
	else if (m->num < n) parent = m;
	else break;

    m = new menu_entry(n, &h);
    if (parent) insert_after(m, parent);
    else insert_front(m);
    return m;
}


static void menu_table::add_version(int n, const char* prop_val)
{
    menu_entry* m = enter(n);
    if (!(m->version_set)) { m->version = prop_val;  m->version_set = TRUE; }
    else {
    	char pname[16];  sprintf(pname, "VERSION%d", n);
	drawing->menu_error(115, pname);
    }
}


static void menu_table::add_expr(int n, const char* prop_val)
{
    menu_entry* m = enter(n);
    if (!(m->expr_set)) { m->expr = prop_val;  m->expr_set = TRUE; }
    else {
    	char pname[16];  sprintf(pname, "EXPR%d", n);
	drawing->menu_error(82, pname);
    }
}


void erule_drawing_entry::
parse_menu_properties(erule_lex *lex)
{
    lex->next_token();			// Eat "BODY"
    menu_table menus(this);
    while (*lex == LEX_ID) {
	if (!strcmp(lex->val(), "END_BODY")) break;
	boolean do_expr, do_version;
	int num = 1;
	do_expr = do_version = FALSE;
	//if (sscanf(lex->val(), "EXPR%d", num) == 1) do_expr = TRUE;
	//else if (sscanf(lex->val(), "VERSION%d", num) == 1) do_version = TRUE;
	//else menu_error(80 , lex->val());
	if (num < 0 || num > MAX_MENU_ENTRY) 
	    menu_error(81, lex->val());

	lex->next_token();  if (*lex != LEX_EQ) lex->assert(178);
	lex->next_token();
	if (*lex != LEX_STRING) {
	    lex->assert(18);
	    do_expr = do_version = FALSE;
	}
	if (do_expr) menus.add_expr(num, lex->val());
	else if (do_version) menus.add_version(num, lex->val());
    }

    menu_entry* m = menus.first();
    for ( ; m; m = menus.next(m)) m->to_directory(this);

    if (*lex == LEX_ID /* implying "END_BODY" */) lex->next_token();
    else lex->assert(12);
    if (*lex == LEX_SEMI) lex->next_token();
    else lex->assert(35);
}


void erule_drawing_entry::push_exception(erule_rule* r, const string& context)
{ 
    erule_drawing_exception* ex;
    if (ex = exceptions.find(context)) r->push_exception(ex->extension);
    else if (ex = exceptions.find(0)) r->push_exception(ex->extension);
}


void erule_drawing_entry::pop_exception(erule_rule* r, string& context)
{
    erule_drawing_exception* ex;
    if (ex = exceptions.find(context)) r->pop_exception(ex->extension);
    else if (ex = exceptions.find(0)) r->pop_exception(ex->extension);
}


boolean erule_drawing_entry::is_single_versioned()
{
    erule_drawing_version* vers = versions.first();
    return (boolean)(vers && !versions.next(vers));  // type checking bug in cc
}
