#include "error.h"
#include "entry.h"
#include "lex.h"



void erule_entry::dump(ostream& f)
{
    if (!_rules_file) f << "{ Default rule }\n";
    else f << "{ from " << _rules_file << " }\n";
    erule_rule *r = rules.first();
    if (!r) f << "    { no rules !! }\n";
    for ( ; r; r = rules.next(r)) r->dump(f, _compile_type);
}


erule_entry::erule_entry(const string& compile_type)
{
    AS(!this);
    this = (erule_entry*)erule_heap.mem(sizeof(*this));

    _compile_type = compile_type;
    _rules_file = 0;

    rules.insert_front(new erule_rule(this));	// Start with default rule

    has_default = FALSE;
    has_automatic_true = TRUE;
}


// lex used only for error messages
erule_rule* erule_entry::append(const char* expression, erule_lex* lex)
{

    if (!expression) {
        if (has_automatic_true) 
	    { lex->error(ERR_MULTIPLE_TRUE_RULES);  return 0; }
        has_automatic_true = TRUE;
    }
    else if (!strcmp(expression, "DEFAULT")) {
        if (has_default) 
	    { lex->error(ERR_MULTIPLE_TRUE_RULES);  return 0; }
	has_default = TRUE;
    }
    erule_rule* rule = new erule_rule(expression);
    rules.insert_rear(rule);
    return rule;
}


void erule_entry::reset(const string& rules_file)
{
    erule_rule* rule;
    while (rule = rules.remove_first()) delete rule;
    _rules_file = rules_file;
    has_default = has_automatic_true = FALSE;
}


erule_rule* erule_entry::select()
{
    erule_rule* a_default = 0;
    erule_rule* applicable = 0;
    erule_rule* rule = rules.first();
    while (rule) {
        if (rule->is_default()) {
	    if (a_default) {
	        selection_error(ERR_MULTIPLE_DEFAULT_RULES);
		erule_err.message(ERR_ASSERT);
		erule_err ERR_INDENT << 
		    "Multiple defaults should be handled on input" ERR_NL;
		break;
    	    }
	    a_default = rule;
	}
	else if (rule->evaluate()) {
	    if (applicable) {
	        selection_error(ERR_MULTIPLE_TRUE_RULES);
		break;
	    }
	    applicable = rule;
	}
	rule = rules.next(rule);
    }

    if (!applicable && !a_default) 
	selection_error(ERR_NO_TRUE_RULES);
    return applicable ? applicable : a_default;
}


void erule_entry::selection_error(int err)
{
    AS(erule_current_drawing);
    if (!_rules_file) {
	erule_err.message(ERR_ASSERT);
	erule_err ERR_INDENT << 
	    "Default rules should not have the following error" ERR_NL;
    }

    erule_err.message(err, erule_current_drawing);

    switch(err) {
    case ERR_MULTIPLE_DEFAULT_RULES:
    case ERR_MULTIPLE_TRUE_RULES:
    case ERR_NO_TRUE_RULES:
	if (_rules_file)
	    erule_err ERR_INDENT << "Expansion rules file=" << _rules_file ERR_NL;
	erule_err ERR_INDENT << "Expressions:" ERR_NL;
	erule_rule* rule;
	for (rule = rules.first(); rule; rule = rules.next(rule))
	    erule_err ERR_INDENT << "    " << rule->expression() ERR_NL;
	break;
    default:
	erule_err.message(ERR_ASSERT);
	erule_err ERR_INDENT << 
	          "Don't know what else to print for previous message" ERR_NL;
	break;
    }
}


// If (!this) just check syntax.  Return TRUE iff syntax ok.
boolean erule_entry::parse(erule_lex* lex, const string& filename)
{
#ifdef DEBUG
cerr << "erule_entry::parse(0x" << hex((long)this) << ", " << filename << ")\n";
#endif
    if (this && filename != _rules_file) reset(filename);

    erule_rule* rule;
    if (*lex == LEX_STRING || *lex == LEX_ID) {
	rule = this ? append(lex->val(), lex) : 0;
	lex->next_token();
    }
    else rule = this ? append(0, lex) : 0;

    if (*lex != LEX_COLON) {
	lex->error(ERR_EXP_COLON);
	lex->skip_past(LEX_SEMI);
	if (rule) {
	    erule_rule* prev = 0;
	    erule_rule* r;
	    for (r = rules.first(); r && r != rule; r = rules.next(r)) 
		prev = r;
	    if (!r) fault("confused");
	    rules.remove_successor(prev);
	    delete rule;
	}
	return FALSE;
    }

    lex->next_token();

    return rule->parse(lex, this);
}
