/*
	cam.c  -  CAM handling routines and definitions
*/
#include	<types.h>
#include 	<krnl.h>

/* addresses of CAM/ACTEL registers */
#define	CAM_CMD			((volatile ushort*)0xB0000004)	/* CAM cmd port */
#define	CAM_DATA		((volatile ushort*)0xB0000000)	/* CAM data port */
#define	CAM_FLAG_REG	((volatile byte*)0xB0000008)	/* ACTEL flag reg */
#define	CAM_MODE_REG	((volatile byte*)0xB0000009)	/* ACTEL mode reg */
/* Bit definitions for CAM flag register */
#define	CF_BUSY			0x01		/* 1 = CAM busy (in use by H/W) */
#define	CM_EASET		0x04		/* 1 = allow A flag set on CAM match  */
#define CM_ECIPDE		0x08		/* bit for testing */
/* Bit definitions for CAM mode register */
#define	CM_ONLINE		0x01		/* 1 = CAM online for H/W use */
#define CM_ORDER		0x02		/* 1 = flip bit to ethernet order.  */
#define	CM_COPYSET		0x04		/* 1 = allow C flag set on CAM match */
#define	CM_ADDR16		0x08		/* 1 = 16-bit addr, 0 = 48-bit addr */
#define	CM_SRCINV		0x10		/* 1 = copy if src addr does NOT match */
#define	CM_SRCEN		0x20		/* 1 = copy frame on src addr match */
#define	CM_DSTINV		0x40		/* 1 = copy if dst addr does NOT match */
#define	CM_DSTEN		0x80		/* 1 = copy frame on dst addr match */

#define CM_STRIPOFF		0x02		/* 1 = Disable stripping */
#define CM_CAMOFF		0x01		/* 1 = Disable CAM */

#define	CAM_MODE		(CM_ONLINE|CM_ORDER|CM_COPYSET|CM_SRCEN|CM_DSTEN)	/* 0xA7 */
#define CAM_FLAGS 		(CM_EASET | CM_ECIPDE)								/* 0x0c */

/* Temporary Command Overrides */
#define	TCO_CT		0x0200		/* control reg */
#define	TCO_SC		0x0210		/* segment control reg */
#define	TCO_DS		0x0228		/* device select reg */
#define	TCO_PA		0x0208		/* page address reg */
/* Control register values */
#define	CAM_RESET	0x0000		/* reset the CAM */
#define	CT_CONFIG	0x8058		/* enable MF & FF, no scramble, 48 CAM bits,mask1,no inc/dec */
/* Device select register values */
#define	DEV_DS		0xffff		/* disable device select function */
/* Segment control register values */
#define	SC_CONFIG	0x39c9		/* dst: 01-11, load w/01, src: 01-11, load w/01 */
/* CAM instructions */
#define	SPS_HMA		0x0005		/* set persistent src: highest matching addr */
#define	SPD_CR		0x0100		/* set persistent dst: comparand reg */
#define	SPS_CR		0x0000		/* set persistent src: comparand reg */
#define	MOV_NF_CR_V	0x0334		/* move comparand to Nxt Free Addr, set valid */
#define	MOV_MR1_CR	0x0308		/* move comparand to mask reg 1 */
#define	VBC_ALM_E	0x043d		/* at all matching locs set empty */
#define	CAM_NOP		0x0300		/* nop */

#define	CAM_WAIT_LIM	10		/* max times thru loop waiting for CAM access */
#define	CAM_ENTRY_COUNT	1024	/* no. of entries in the CAM */


int freeCamCnt;			/* count of available entries in the CAM */


/***********************************************************************
	CamInit  -  Routine to initialize the CAM

	Description: CAM word length is 64 bits, but we only need 48 bits 
		for an address, so we partition the CAM word as 16 bits of RAM 
		(which we don't use) and 48 bits of CAM.  This also requires
		that we set the segment registers in the CAM to cycle from 01 to
		11 (because segment 00 is the 16-bit RAM segment that we dont use).

		We must set up the CAM with the comparand register as the
		'persistent destination' so that when the h/w latches the 48-bit
		network address and hands it to the CAM, it goes into the
		comparand register.

	Inputs: none

	Returns: none
***********************************************************************/
CamInit()
{
	register word saveMask;
	ushort junk;

	MaskAllInts(saveMask);
	*CAM_MODE_REG = (byte)(CAM_MODE & ~CM_ONLINE);		/* CPU wants the CAM */
	if( CamAccessible() )	/* if we got the CAM from the H/W */
	{
		 *CAM_CMD = 0x0000;		/* init the cam after pwr up */

#ifdef notdef
		*CAM_CMD = TCO_DS;		/* Temp Cmd Override: dev select reg */
		*CAM_CMD = DEV_DS;		/* disable device select function */
#endif

		*CAM_CMD = TCO_CT;		/* Temp Cmd Override: control reg */
		*CAM_CMD = CAM_RESET;	/* reset the CAM */

#ifdef notdef
		*CAM_CMD = TCO_PA;		/* Temp Cmd Override: page addr reg */
		*CAM_CMD = 0;			/* we are page 0 */
#endif

		*CAM_CMD = SPS_CR;		/* set persistent src: comparand reg */
		*CAM_CMD = SPD_CR;		/* set persistent dst: comparand reg */

		/* load mask reg 1 for comparisons on the 48 bits of CAM only (we 
		actually load the 'persistent dest' and xfer it to the mask reg) */
		*CAM_DATA = 0xffff;		/* force match on seg 0 (RAM) */
		*CAM_DATA = 0;		/* force match on seg 0 (RAM) */
		*CAM_DATA = 0;		/* force match on seg 0 (RAM) */
		*CAM_DATA = 0;		/* force match on seg 0 (RAM) */
		*CAM_CMD = MOV_MR1_CR;		/* move comparand reg to mask reg 1 */

		*CAM_CMD = TCO_CT;		/* Temp Cmd Override: control reg */
		*CAM_CMD = CT_CONFIG;		/* config CAM */
	
		*CAM_CMD = TCO_SC;		/* Temp Cmd Override: segment control reg */
		*CAM_CMD = SC_CONFIG;		/* config segment control */

		freeCamCnt = CAM_ENTRY_COUNT;	/* all entries free at the start */
		*CAM_FLAG_REG = (byte)CAM_FLAGS; 
		*CAM_MODE_REG = (byte)CAM_MODE;		/* set up the CAM mode register */
	}
	RestoreIntMask(saveMask);

} /* end CamInit() */



/***********************************************************************
	CamAdd  -  Routine to add an entry to the CAM

	Description: The CAM keeps track of the 'next free address', so to
		add an entry we just write it to the CAM comparand register
		(which is the 'persistent destination') and give the CAM a
		command to write the comparand register to the 'next free
		address' and mark that entry valid.

	Inputs: pointer to the 6-byte addr to be added to CAM

	Returns: 1 if OK, 0 if CAM already full or error
***********************************************************************/
CamAdd(netAddr)
ushort* netAddr;
{
	int ret;

	/* clear the 'online' bit in the CAM mode reg */
	*CAM_MODE_REG = (byte)(CAM_MODE & ~CM_ONLINE);		/* CPU wants the CAM */
	ret = 0;			/* in case of problem */
	if( freeCamCnt )		/* if there is an available CAM entry */
	{
		freeCamCnt--;		/* one less free entry */
		if( CamAccessible() )	/* if we got the CAM from the H/W */
		{
			*CAM_CMD = TCO_SC;		/* Temp Cmd Override: segment control reg */
			*CAM_CMD = SC_CONFIG;		/* reset segment counters */

			*CAM_CMD = CAM_NOP;			/* magic nop */
			*CAM_CMD = CAM_NOP;			/* magic nop */
			*CAM_DATA = netAddr[0];	/* move each word into comparand reg */
			*CAM_DATA = netAddr[1];
			*CAM_DATA = netAddr[2];
			*CAM_CMD = CAM_NOP;			/* magic nop */
			*CAM_CMD = CAM_NOP;			/* magic nop */
			*CAM_CMD = CAM_NOP;			/* magic nop */
			*CAM_CMD = MOV_NF_CR_V;		/* move it to next free slot in CAM */
			ret = 1;			/* all is well */
		}
	}
	*CAM_MODE_REG = (byte)CAM_MODE;	/* release CAM to the H/W */
	return(ret);
} /* end CamAdd() */




/***********************************************************************
	CamDel  -  Routine to delete an entry from the CAM

	Description: We write the 48-bit address to the comparand register
		(the 'persistent destination') and tell the CAM to find the
		matching entry in the CAM array and mark it empty.

	Inputs: pointer to the 6-byte addr to be deleted from CAM

	Returns: none
***********************************************************************/
CamDel(netAddr)
ushort* netAddr;
{
	
	/* clear the 'online' bit in the CAM mode reg */
	*CAM_MODE_REG = (byte)(CAM_MODE & ~CM_ONLINE);		/* CPU wants the CAM */
	++freeCamCnt;			/* one more free entry in the CAM */
	if( CamAccessible() )		/* if the H/W gave up the CAM */
	{
		*CAM_CMD = TCO_SC;		/* Temp Cmd Override: segment control reg */
		*CAM_CMD = SC_CONFIG;		/* reset segment counters */
		*CAM_DATA = *netAddr++;		/* move each word into comparand reg */
		*CAM_DATA = *netAddr++;
		*CAM_DATA = *netAddr++;
		*CAM_CMD = VBC_ALM_E;		/* mark matching entry empty */
	}
	*CAM_MODE_REG = (byte)CAM_MODE;	/* release CAM to the H/W */
} /* end CamDel() */



/***********************************************************************
	CamAccessible  -  Routine to wait for h/w to give CPU access to CAM

	Description: The CPU and the h/w can't both beat on the CAM at once,
		so if the CPU wants to read or write to the CAM we must lock
		the h/w out by asserting a bit in a control register, then wait
		for the h/w to give it up.

		This routine assumes that the caller asserts/de-asserts the h/w
		'lockout' bit.

	Inputs: none

	Returns: 1 if CPU gained access, 0 if not
***********************************************************************/
CamAccessible()
{
	int i;

	for(i=0; i<CAM_WAIT_LIM; i++)		/* we'll only wait so long */
		if( (*CAM_FLAG_REG & CF_BUSY) == 0 )	/* if h/w gave it up */
			return(1);					/* CPU has the CAM */
	printf("H/W won't release the CAM\n");	/* zzz */
#ifdef notdef
	ErrorFunction??(CAM_PROBLEM);		/* Houston, we have a problem */
#endif
	return(0);							/* indicate CAM problem */
} /* end CamAccessible() */


typedef union MyNid {
   ushort s[10][3];
   char   b[10][6];
} MYNID;
/***********************************************************************/


CamDisplay() {

	register word saveMask;
	int i, k;
	MYNID nid;

	k = (1024 - freeCamCnt) < 10 ? (1024 - freeCamCnt) : 10;

	MaskAllInts(saveMask);
	*CAM_MODE_REG = (byte)(CAM_MODE & ~CM_ONLINE);		/* CPU wants the CAM */
	if( CamAccessible() )		/* if the H/W gave up the CAM */
   {
		*CAM_CMD = TCO_CT;		/* Temp Cmd Override: control reg */
		*CAM_CMD = 0x8050;		/* config CAM */
		*CAM_CMD = 0x0004;        /* set mem @ AR as persistant source */
		*CAM_CMD = 0x220;		/* TCO: addr reg */
        *CAM_CMD = 0;		 /* AR = 0 */
	   for(i=0 ; i < k; i++) {
			  nid.s[i][0] = *CAM_DATA;
			  nid.s[i][1] = *CAM_DATA;
			  nid.s[i][2] = *CAM_DATA;
		}
		*CAM_CMD = SPS_CR;		/* set persistent src: comparand reg */
	}
	else {
			printf("\nCan't Get Cam.....\n");
	}
	*CAM_CMD = TCO_CT;		/* Temp Cmd Override: control reg */
	*CAM_CMD = CT_CONFIG;		/* config CAM */

	 *CAM_MODE_REG = (byte)CAM_MODE;	/* release CAM to the H/W */
	RestoreIntMask(saveMask);


	for(i=0; i<k; i++ ) {
		    printf("\nCam Addr: %02x%02x%02x%02x%02x%02x", nid.b[i][0], nid.b[i][1], nid.b[i][2], nid.b[i][3], nid.b[i][4], nid.b[i][5]);
	}
	printf("\nNum Addr In Cam: %d\n",1024 - freeCamCnt);
}
