/*****************************************************************************
*       Program Name:  nim960 bridge
*
*       Filename:      834filter.c
*
*       $Log:   /b/gregs/bridge/declike/cmd/834filter.c_v  $
 * 
 *    Rev 1.1   30 Jul 1993 13:41:28   franks
 * No change.
 * 
 *    Rev 1.0   04 May 1993 16:03:50   franks
 * Initial revision.
 * 
 *    Rev 1.4   04 May 1992 12:16:32   franks
 * No change
 * 
 *    Rev 1.3   17 Apr 1992 16:17:52   franks
 * No change.
 * 
 *    Rev 1.2   16 Apr 1992 14:47:00   franks
 * No change.
 * 
 *    Rev 1.1   14 Apr 1992 13:07:16   franks
 * No change.
 * 
 *    Rev 1.0   30 Mar 1992 17:48:06   pvcs
 * Initial revision.
*
*       Comments:      Developed for i960 platform.
*                      This file contains the routines necessary to support
*                      the protocol filter functions.
*
*
*       (C) Copyright 1991 Hughes LAN Systems
*       
*	Listing of routines in this file:
*
*		
******************************************************************************/

#include <target.h>
#include <bridges.h>
#include <filter.h>
#include <nvrecs.h>
#include <prcadr.h>
#include <prcctl.h>
#include <cpb.h>
#include <834error.h>

#define NUMPTYPES  (sizeof(ProtocolList)/sizeof(unsigned))

/* External Variable Declarations */
extern NVR_BSTATUS *BridgeStatus;
extern PRCCTL prc;

UDPDIR *FindUDP( char * );  

/* Global Variables */

extern char          *PFilTable; /* char == byte, 64K */
extern NVR_PROTOCOLS nvr_udpd;
extern UDPDIR        *udpdir;
extern uint          *PFilterList;

static unsigned ProtocolList[] = {
		IP_T,      ARP_T,
		REVARP_T,  DECNET1_T,
		DECNET2_T, DECNET3_T,
		DECNET4_T, DECNET5_T,
		DECNET6_T, DECNET7_T,
		DECNET8_T, DECLAT_T,
		DECLAVC_T, XNS1_T,
		XNS2_T,    XNS4_T,
		NOVELL_T,  BRID1_T,
		BRID2_T,   APPLE1_T,
		APPLE2_T,  APPLE3_T,
		APPLE4_T,  V2_L,
	    NOVELL_L,  TOP_L 
};

/*****************************************************************************/
/* WARNING: Do not make changes in the order of the protocols without making */
/* simultaneous changes to the defines in filter.h                  */

PROTCNTLREC Protocols[] = {
   {"ip"    , 1,  IP_P     , IP_T     , 0 },
   {"arp"   , 2,  ARP_P    , ARP_T    , 0 },
   {"revarp", 1,  REVARP_P , REVARP_T , 0 },
   {"top"	, 2,  TOP_P    , TOP_L    , 0 },
   {"v2"    , 1,  V2_P     , V2_L     , 0 },
   {"novell", 1,  NOVELL_P , NOVELL_T , NOVELL_L , 0 },
   {"lat"	, 1,  LAT_P    , DECLAT_T , DECLAVC_T, 0 },
   {"bridge", 1,  BRIDGE_P , BRID1_T  , BRID2_T  , 0 },
   {"tcp"   , 2,  TCP_P    , REVARP_T , ARP_T    , IP_T     , 0 },
   {"xns"   , 1,  XNS_P    , XNS1_T   , XNS2_T   , XNS4_T   , 0 },
   {"dec"   , 1,  DEC_P    , DECNET1_T, DECNET2_T, DECNET3_T, DECNET4_T,
                  DECNET5_T, DECNET6_T, DECNET7_T, DECNET8_T, 0 },
   {"apple" , 2,  APPLE_P  , APPLE1_T , APPLE2_T , APPLE3_T , APPLE4_T , 0 }
};
/******************************************************************************/


/*******************************************************************************
** Name: InitPFilTable()
**
** Description: Will initialize the PFilTabIdx entries to point the appropriate
**              protocol entry in the PFilTable. If the protocol type is not
**              supported then the PFilTabIdx entry is set to NULL.
**
** Return: None.
*******************************************************************************/

InitPFilTable()  {

	unsigned cnt, i;
	uint PortMask;
	uint ProtCnt;
	uint UserVal;
	PROTCNTLREC *plist   = Protocols;

    PFilTable = (char *)lmalloc(FILTABLESZ);      /* char == byte, 64K */
	PFilterList = &BridgeStatus->PFilterList[0];

	/* First initialize all entries to null */
	for( cnt=0; cnt < FILTABLESZ; cnt++ )
		PFilTable[cnt] = NULL;

	/* For each of the pre-defined protocol types */
	for( cnt=0; cnt < PROTNUM; cnt++ ) {
		
		PortMask = BridgeStatus->PFilterList[cnt];
		/* Is this protocol type blocked on any ports? */
		/* The port list is bit mapped. */
		if( PortMask ) {
			SetPortProt( (plist+cnt)->prots, PortMask, PTYPE);
		}
	}
	for( cnt = 0; cnt < MAXUSRTYPE; cnt++ ) {

		UserVal = (udpdir+cnt)->UProtTyp;

		/* 1815 is really 1518 because UserVal has been swapped */
		if( UserVal <= 0xffff && UserVal >= 1815 )
			SetPortProt( &UserVal, (udpdir+cnt)->PrtSet, UTYPE );
	}
	return(0);
}

/*******************************************************************************
** Name: Block
**
** Description: Will
**
** Return: None.
*******************************************************************************/

Block( CPB_P pcpb, uint PortMask ) {

	PRCCTL *prcctl = &prc;
	int retval;
	uint i;
	unsigned int cnt;
	unsigned int Index;
	char *Vector2 = &pcpb->cmd[pcpb->cl_idx];

	/* Check for correct num of args and process command */

	/* Debug command, display control structures */
	if( strncmp(Vector2 ,"debug", 5) == 0 ) {

		printf("\nPFilterList in BridgeStatus Structure..\n");
		for( i=0; i<PROTNUM; i++ ) 
			printf("Protocol: %x ---> %x\n", i, BridgeStatus->PFilterList[i]);

		printf("FilterMode: %x,   ProtFilter: %x\n", BridgeStatus->FilterMode,
													 BridgeStatus->ProtFilter);

		printf("\nPrcctl Struct\n");
		printf("ProtFltMode: %x,  ProtFlt: %x\n", prcctl->Prc_ProtFltMode,
											  	 prcctl->Prc_ProtFlt);
		getchar();
		for( i=0; i < 0xffff; i++)
			if( PFilTable[i] )
				printf("Type: %04x ----> Value: %x\n", i, PFilTable[i]);

		return(0);
	}
	else  {

		/* Check protocol type args */

		
		if (retval = isValidArg(pcpb, ADD, PortMask)) {
			return( retval );
		}
	

		/* Change filtering modes if necessary. */
		ChangeMode(PortMask, BLOCKMODE, '+');

		/* Set protocol blocking bits... */
		retval = Plus(pcpb, PortMask); 

		SaveBStatus();
		SaveProtocolTbl();
		return( retval );
	}
}


/*******************************************************************************
** Name: Permit
**
** Description: Will
**
** Return: None.
*******************************************************************************/

Permit( CPB_P pcpb, uint PortMask ) {

	int  retval;
	uint i;
	unsigned int cnt;
	unsigned int Index;
	char *Vector2 = &pcpb->cmd[pcpb->cl_idx];


	/* Check protocol type args */

	if (retval = isValidArg(pcpb, ADD, PortMask))
		return( retval );

	/* Change filtering modes if necessary. */

	ChangeMode(PortMask, PERMITMODE, '+');

	/* Set protocol blocking bits... */
	retval = Plus(pcpb, PortMask); 
	SaveBStatus();
	SaveProtocolTbl();
	return( retval );
}

/*****************************************************************************/

Plus( CPB_P pcpb, uint PortMask ) {

   PROTCNTLREC *plist   = Protocols;
   char	       *ProtTyp;
   unsigned	    UserVal;
   int	        PIndex;	  /* Set by ProtMatch, Idx to protocol in Protocols[] */
   int			length;
   int			cnt;

   while( pcpb->cl_idx < pcpb->cmdlen ) {

		/* Get protocol type and adjust cmd line pointer */
		length = pick_a_kw(&pcpb->cmd[pcpb->cl_idx],&ProtTyp,CASE_INSENSITIVE);
		pcpb->cl_idx += length;	

		ProtMatch( ProtTyp, &PIndex );
		switch ( PIndex )  {

			case IP_P:
			case ARP_P:
			case REVARP_P:
			case TOP_P:
			case V2_P:
			case NOVELL_P:
			case LAT_P:
			case BRIDGE_P:
			case XNS_P:
			case DEC_P:
			case APPLE_P:
			   /* Update protocol filter table */
			   SetPortProt( &((plist+PIndex)->prots), PortMask, PTYPE);
			   /* Update admin control structure */
               SetPortBit( (PFilterList+PIndex), PortMask);
			   break;

			case TCP_P:
			   /* Update protocol filter table */
			   SetPortProt( &((plist+PIndex)->prots), PortMask, PTYPE);
			   /* Update admin control structure */
               SetPortBit( (PFilterList+IP_P), PortMask);
               SetPortBit( (PFilterList+ARP_P), PortMask);
               SetPortBit( (PFilterList+REVARP_P), PortMask);
               break;

			default:

			   /* lookup the type in the UDP dir and if it exists */
			   /* get the hex type.                               */
			   UserVal = SetUsrTyp(ProtTyp, PortMask);

			   /* Set bit in protocol filter table */
			   SetPortProt( &UserVal, PortMask, UTYPE ); 
		}
   }
   ProtFilEnaDis( PortMask );
   return(0);
}



/*****************************************************************************/

Minus( CPB_P pcpb, uint PortMask )  {


   PROTCNTLREC *plist   = Protocols;
   char	       *ProtTyp;
   uint         UserVal;
   int	        PIndex;	  /* Set by ProtMatch, Idx to protocol in Protocols[] */
   int 			length;
   int			cnt;


	while( pcpb->cl_idx < pcpb->cmdlen ) {

		/* Get protocol type and adjust cmd line pointer */
		length = pick_a_kw(&pcpb->cmd[pcpb->cl_idx],&ProtTyp,CASE_INSENSITIVE);
		pcpb->cl_idx += length;	

		if(strncmp(ProtTyp, "ALL", 3) == 0) {

			ClearAllPProts( PortMask );
			ClearAllUProts( PortMask );
			pcpb->cl_idx = pcpb->cmdlen;
            ProtFilEnaDis( PortMask );
			return(0);
		}
	
		ProtMatch( ProtTyp, &PIndex );
		switch ( PIndex )  {

			case IP_P:
			case ARP_P:
			case REVARP_P:
			case TOP_P:
			case V2_P:
			case NOVELL_P:
			case LAT_P:
			case BRIDGE_P:
			case XNS_P:
			case DEC_P:
			case APPLE_P:
			   ClearPortProt( &((plist+PIndex)->prots), PortMask, PTYPE );
               ClearPortBit( (PFilterList+PIndex), PortMask );
			   break;

			case TCP_P:
			   ClearPortProt( &((plist+PIndex)->prots), PortMask, PTYPE );
               ClearPortBit( (PFilterList+IP_P), PortMask );
               ClearPortBit( (PFilterList+ARP_P), PortMask );
               ClearPortBit( (PFilterList+REVARP_P), PortMask );
			   break;

			default:
			   UserVal = RemoveUsrType( ProtTyp, PortMask);

			   /* Set bit in protocol filter table */
			   ClearPortProt( &UserVal, PortMask, UTYPE ); 
		}
   }
   ProtFilEnaDis( PortMask );
   return(0);
}

/*****************************************************************************/
/* Will determine the number of protocol types entered on the command line   */
/* *Pcnt will indicate the number of protocols entered on the cmd line and   */
/* Vcnt is set as an index pointing to the first protocol entered on the cmd */
/* line.                                                                     */
/*****************************************************************************/

CntProtList(char **list, char *Plist, int *Pcnt, int *Vcnt)  {

  *Vcnt = 0;

  while (*list != Plist && *Pcnt > 0)  {

	 list++;
	 (*Pcnt)--;
	 (*Vcnt)++;
  }
}

/*****************************************************************************/
/*									     									 */
/* Will search through the protocol control table "Protocols" looking for    */
/* the pre-defined protocol indicated by ProtStr. It will determine if the   */
/* minimum required characters needed for positive protocol identification   */
/* are present. If so then Index will be set to point at the protocols entry */
/* in "Protocols". This will give required information needed to set the     */
/* proper bits in the protocol filter table.                                 */
/*									     									 */
/*****************************************************************************/

ProtMatch( char *ProtStr , int *Index) {

   PROTCNTLREC       *pptr = Protocols;
   register    char  *sptr;
   register    char  *iptr;
   int		          cnt;
   int		          i;


   for (cnt=0; cnt < PROTNUM; cnt++, pptr++)  {

	 sptr = pptr->ProtName;
	 iptr = ProtStr;
	 i    = 0;

	 while (*sptr == tolower(*iptr) && *sptr != NULL )  {

		   sptr++;
		   iptr++;
		   i++;
	 }

	 /* if enough of a match, return the protocol idx in the protocol table */

	 if( i >= pptr->min  && (*iptr == SPACE || *iptr == NULL) ) {
		*Index = ( pptr->Index );
		return(0);
	 }
   }
   *Index = -1;             /* Set to invalid number */
   return (ERR_INV_ARG );
}


/*****************************************************************************/
/* Will set the protocol bits for ports indicated by PMask.		             */
/*****************************************************************************/

SetPortProt( unsigned *PType, unsigned PMask, int UType )  {

  register int Index;

  /* User type only affects one entry in protocol filter table */
  if( UType ) {

	*(PFilTable+(*PType)) |= PMask;
  }

  /* Pre-Defined type can affect several entries in protocol filter table */
  /* so set all of the protocols associated with the pre-defined type.    */
  else  {

	for (; *PType != 0; PType++)
		*(PFilTable+(*PType)) |= PMask;
  }
}

/*****************************************************************************/
/* Will clear the protocol bits for ports indicated by PMask.		         */
/*****************************************************************************/

ClearPortProt( uint *PType, uint PMask, int UType )  {

  register int Index;

  /* User type only affects one entry in protocol filter table */
  if( UType )  {

	 *(PFilTable+(*PType)) &= ~PMask;
  }

  /* Pre-Defined type can affect several entries in protocol filter table */
  else  {

	for (; *PType != 0; PType++)
		*(PFilTable+(*PType)) &= ~PMask;
  }
}


/*****************************************************************************/
/* Will pre-process the protocol argument list and verify correct spelling,  */
/* make sure user-defined types are > 0x05EF, do not exceed maximum number   */
/* of user-defined protocols per port if adding.	 						 */
/*****************************************************************************/

isValidArg( CPB_P pcpb, char Action, uint PortMask ) {

   PROTCNTLREC *plist   = Protocols;
   UDPDIR		*udp;
   char	       *ProtTyp;
   unsigned	    UserVal;
   uint			length;
   int	        PIndex;	  /* Set by ProtMatch, Idx to protocol in Protocols[] */
   int			cnt;
   int          Pcnt=0;   /* Cnt of user-def protocols entered */
   int			PortSelect = 1;
   uint         CmdLinePos = pcpb->cl_idx; /* save current cmdline position */

   while( pcpb->cl_idx < pcpb->cmdlen ) {

		/* Get protocol type and adjust cmd line pointer */
		length = pick_a_kw(&pcpb->cmd[pcpb->cl_idx],&ProtTyp,CASE_INSENSITIVE);
		pcpb->cl_idx += length;	

		if(strcmp(ProtTyp, "ALL") == 0) {
   			pcpb->cl_idx = CmdLinePos;
			return(0);
		}

		ProtMatch( ProtTyp, &PIndex );

		switch ( PIndex )  {

			case IP_P:
			case ARP_P:
			case REVARP_P:
			case TOP_P:
			case V2_P:
			case NOVELL_P:
			case LAT_P:
			case BRIDGE_P:
			case TCP_P:
			case XNS_P:
			case DEC_P:
			case APPLE_P:
				break;


			default:

				/* Make sure protocol type is defined */
				udp = FindUDP( ProtTyp );
				if( udp == (UDPDIR *)NULL ) 
					return( ERR_TYPE_NOTFOUND );
				else
		   			UserVal = udp->UProtTyp;


               /* Same as a pre-defined value? */
			   if ( isPreDef( UserVal ) ) {

					/* restore current cmdline position */
   					pcpb->cl_idx = CmdLinePos;
					/* return( ERR_TYPE_EXIST ); */
					return( 29 );
               }

		}
   }
   pcpb->cl_idx = CmdLinePos;  /* restore current cmdline position */
   return(0);
}

/*****************************************************************************/
/*									     */
/* Will search the table of pre-defined protocols to see if this the user    */
/* defined type matches a pre-defined type. If so then TRUE is returned.     */
/*									     */
/*****************************************************************************/

isPreDef( register unsigned Val )  {

	register int cnt;
	register unsigned *pl = ProtocolList;

	for (cnt=0; cnt < NUMPTYPES; cnt++)
		if ( *pl++ == Val )
			return( TRUE );

	return( FALSE );
}


/******************************************************************************/
						/* Start SetUsrType() */

SetUsrTyp(char *Type, char PMask)  {

	register int i, port;
    register char *PType;
	int PortSelect = 1;
	UDPDIR *udp;

	/* Find protocol type in UDP Dir */
	udp = FindUDP( Type );          
			
	/* If the protocol is in the UPD Dir then add port info */
	if( udp != (UDPDIR *)NULL )
		udp->PrtSet |= PMask;
	else
		return( ERR_TYPE_NOTFOUND );

	return( udp->UProtTyp );	
}


/*****************************************************************************/
/* Will search through the array of user defined protocol types for this     */
/* port to determine if this protocol type has already been defined by the   */
/* user. If yes => return pointer to type. If no => return NULL	 			 */
/*****************************************************************************/

UDPDIR *FindUDP( char *str )  {

	register int  cnt;
	register char *type;
	
	for( cnt=0; cnt<MAXUSRTYPE; cnt++ ) {

			type = udpdir[cnt].UProtStr;	
			if( strcmp( type, str ) == NULL)
					return( &udpdir[cnt] );
	}

	return((UDPDIR *)NULL);
}

/****************************************************************************/
/* Will determine if the protocol filtering mode is changing from block to  */
/* permit mode or visa versa. If so it will delete all of the protocol      */
/* types set by the previous mode and change  the port mode bit in the      */
/* BridgeStatus and prcctl structures. 										*/
/****************************************************************************/
					/* Start ChangeMode() */

ChangeMode(int Mask, int Mode, int Op)  {

	 PRCCTL *prcctl = &prc;
	 PROTCNTLREC  *plist     = Protocols;
	 unsigned int NewMask    = 0;
	 unsigned int PortSelect = 1;
	 unsigned int type;
			  int port;

	 /* Build a mask of ports listed on the command line which are in
	 ** the opposite mode. */

	 for( port=0; port<ACTUALMAXPORT; port++, PortSelect<<=1)
		  if( Mask & PortSelect )
			  if( (BridgeStatus->FilterMode & PortSelect) >> port != Mode )
						  NewMask |= PortSelect;

	 /* For ports which are changing mode, clear all pre-def protocol bits set
	 ** during that mode.  */

	 if( NewMask )  {

		/*
		for(type=0; type < PROTNUM; type++)  {
			   ClearPortProt( ((plist+type)->prots), NewMask , PTYPE );
               ClearPortBit( (PFilterList+type), NewMask );
        }
		*/

		/* Clear pre-defined protocol bits set during old mode. */
		ClearAllPProts( NewMask );

		/* Clear user-defined protocol bits set during old mode. */
		ClearAllUProts( NewMask ); 

		/* Change the FilterMode flag in the BridgeStatus 
		   record to the new mode */

		if ( Mode == BLOCKMODE ) {
			SetBitOrNot( &NewMask, SETBLMODE, &BridgeStatus->FilterMode );
			SetPrcBitOrNot( (byte *)&NewMask, SETBLMODE, 
							&prcctl->Prc_ProtFltMode );
		}
		else
        if ( Mode == PERMITMODE && Op != '-' ) {
			SetBitOrNot( &NewMask, SETPRMODE, &BridgeStatus->FilterMode );
			SetPrcBitOrNot( (byte *)&NewMask, SETPRMODE, 
							&prcctl->Prc_ProtFltMode  );
		}
	 }
}


/****************************************************************************/
/* Will search through the array of user defined protocol types and look    */
/* the first empty slot to place the new user defined type into.            */
/****************************************************************************/

GetEmptySlot( int *TypeList )  {

	int cnt;

	for( cnt=0; cnt<MAXUSRTYPE; cnt++ )
			if( *TypeList++ == NULL )
					return( cnt );

	return( NOTFOUND );
}

/****************************************************************************/

PrintType( char *str )  {

   printf("\nProtocol Type ( ");
   while( *str != SPACE && *str != '\0' && *str != TAB )
		  putchar(*str++);
   printf(" )\n\n");
}

/*****************************************************************************/
						/* Start ClearAllPProts() */

ClearAllPProts( uint Mask ) {
	
	PROTCNTLREC  *plist     = Protocols;
	uint type;

	for(type=0; type < PROTNUM; type++)  {
		   ClearPortProt( ((plist+type)->prots), Mask , PTYPE );
           ClearPortBit( (PFilterList+type), Mask );
    }
}


/*****************************************************************************/
/* Will clear the protocol filtering bits for all of the user-defined        */
/* protocols associated with each port, as indicated by the Mask.            */
/*****************************************************************************/
ClearAllUProts( uint Mask )  {

	UDPDIR *udp;
	uint cnt;

	/* For each entry in the UDP directory */
	for( cnt = 0; cnt < MAXUSRTYPE; cnt++ ) {

		udp = &udpdir[cnt]; /* udpdir is a Global Variable */

		/* If the UDP is set to affect ports listed on cmd line */
		if ( Mask & udp->PrtSet )  {

			/* Clear the protocol filtering bit */
			ClearPortProt( &(udp->UProtTyp), Mask, UTYPE);

	        /* Remove user-def entry from the port table of user-def types */
			RemoveUsrType( udp->UProtStr, Mask );
		}
	}
	return(0);
}

/*****************************************************************************/
/* Will remove a user-defined protocol type entry from a ports table of      */
/* user-defined protocols.													 */
/*****************************************************************************/

RemoveUsrType( char *Type, int PortMask )  {

	register int i, port;
    register char *PType;
	UDPDIR *udp;

	/* Find protocol type in UDP Dir */
	udp = FindUDP( Type );          
			
	/* If the protocol is in the UPD Dir then remove port info */
	if( udp != (UDPDIR *)NULL )
		udp->PrtSet &= ~PortMask;
	else
		return( ERR_TYPE_NOTFOUND );

	return( udp->UProtTyp );	
}

/*****************************************************************************/
/* Will return a string indicating the protocol filtering mode for a port:   */
/* "Block" / "Permit"                                                        */
/*****************************************************************************/

char *GetPFMode( uint PortMask )  {

     if( !(BridgeStatus->FilterMode & PortMask) )
        return( "Block" );
     else
        return( "Permit");
}

/*****************************************************************************/

PrintPDProt( unsigned PortMask )  {

    PROTCNTLREC *plist = Protocols;
    int retval;
    int cnt;

    retval = 0;
    for( cnt=0; cnt<PROTNUM; cnt++ )
         if( *(PFilterList+cnt) & PortMask )  {

             if( cnt == TCP_P )
				 ;
				 /*
                 retval += printf("ip arp revarp ");
				 */
             else
                 retval += printf("%s ", ((plist+cnt)->ProtName) );
         }
    return( retval );
}

/*****************************************************************************/

PrintUDProt( uint PortMask, int spaces )   {

    UDPDIR *udp = &udpdir[0];
    int retval;
    int tilNewLine;
    int NewLine;
    int cnt;

    retval     = FALSE;
    NewLine    = TRUE;
    spaces     = 71-spaces;
    tilNewLine = spaces/5;

    for( cnt = 0; cnt < MAXUSRTYPE; cnt++ )  {
        if( (udp+cnt)->PrtSet & PortMask ) {

            if( tilNewLine ) {

              retval = TRUE;
              printf("%s ", (udp+cnt)->UProtStr );
              tilNewLine--;
            }
            else
            if( NewLine ) {

              NewLine = FALSE;
              printf("\n        ");
              printf("%s ", (udp+cnt)->UProtStr );
            }
            else
              printf("%s ", (udp+cnt)->UProtStr );
        }
    }
    return( retval );
}

/*****************************************************************************/
						/* Start ProtFilEnaDis() */
/* Will mark protocol filtering enabled or disabled for a port if blocking */
/* or permitting protocol types.	*/

ProtFilEnaDis( uint PortMask )  {

	PRCCTL *prcctl = &prc;
	uint PortBit;

	for( PortBit=1; PortBit <= MAXPORTBIT; PortBit<<=1 ) {

		if( PortBit & PortMask )  {
			if( (isPDProt( PortBit )) || (isUDProt( PortBit )) ) {
			   SetPortBit( &(BridgeStatus->ProtFilter), PortBit);
               ByteSetPortBit( &(prcctl->Prc_ProtFlt), (byte)PortBit);
			}
			else  {
			   ClearPortBit( &(BridgeStatus->ProtFilter), PortBit);
			   ByteClearPortBit( &(prcctl->Prc_ProtFlt), (byte)PortBit);
			}
		}
	}
}

/* End ProtFilEnaDis() */


/*****************************************************************************/
/* Determine if there are any pre-defined protocols being either blocked or  */
/* permited. Upon finding any protocol marked as blocked or permit, return   */
/* TRUE. Otherwise if the entire table is searched without finding any       */
/* protocol types blocked, return FALSE.									 */
/*****************************************************************************/
						/*  Start isPDProt( uint ) */

isPDProt( unsigned PortMask )  {

    PROTCNTLREC *plist = Protocols;
    int cnt;

    for( cnt=0; cnt<PROTNUM; cnt++ )
         if( *(PFilterList+cnt) & PortMask )  {
			return(TRUE);
		 }
         
    return( FALSE );
}

/*****************************************************************************/
/* Determine if there are any user-defined protocols being  blocked or       */
/* permited. Upon finding any protocol marked as blocked or permit, return   */
/* TRUE. Otherwise if the entire table is searched without finding any       */
/* protocol types blocked, return FALSE.									 */
/*****************************************************************************/

isUDProt( uint PortMask )   {

	uint cnt;
    UDPDIR *udp = udpdir;

	for( cnt=0; cnt < MAXUSRTYPE; cnt++ )  {
		if( (udp+cnt)->PrtSet & PortMask )
			return( TRUE );
	}
	return( FALSE );
}

/*****************************************************************************/

UserProtsToDefault() {

	memset( &nvr_udpd, 0, sizeof(nvr_udpd));
	SaveProtocolTbl();
}

/*****************************************************************************/
                            /* BuildProtList() */

BuildProtList( char Buffer[PROTNUM+MAXUSRTYPE][9], uint *cnt, uint PortMask  ) {

    PROTCNTLREC *plist = Protocols;
	UDPDIR       *udp   = &udpdir[0];
	uint         i;

	*cnt = 0;

	/* Build the list of pre-defined protocols */
	for( i=0; i < PROTNUM; i++ ) {

		if( *(PFilterList+i) & PortMask ) {

			if( i == TCP_P ) {
				sprintf(&Buffer[(*cnt)++][0],"%s", "ip");
				sprintf(&Buffer[(*cnt)++][0],"%s", "arp");
				sprintf(&Buffer[(*cnt)++][0],"%s", "revarp");
            }
			else
				strcpy( &Buffer[(*cnt)++][0], ((plist+i)->ProtName) );
        }
    }

	/* Build the list of user-specified protocols */

    for( i=0; i < MAXUSRTYPE; i++ ) {
		if( (udp+i)->PrtSet & PortMask ) {
		    strcpy( Buffer[(*cnt)++], ((udp+i)->UProtStr) );
        }
    }
}
