/********************************************************************
	xlate.c  -  translation routines, buffer handling, etc. for
		fddi <--> ethernet translation.
********************************************************************/

#include <types.h>
#include "prcfrm.h"
#include <krnl.h>
#include <pkt.h>
#include <sncvar.h>
#include <xlate.h>
#include <prcadr.h>

PKT *xl_pktstart_addr;
extern PKT* xlateBufList;			/* translate buffer list head ptr */
extern PKT* F2EPktHead;
extern int F2EPktCnt;



/* Setup the location of the translation buffers in shared ram. */
/* They will be put in the 7k LBUF portion of the 2nd 64k block */
/* of shared ram.  FS - 3 May 1993                              */  

#define SHRAM1			0x50010000	/* The 2nd 64k block of shared ram */
#define SV_STRT_XLBUF   ( (SHRAM1) + (SV_TOTL_PCTL) + (SV_TOTL_LCTL) )  
#define SV_NMBR_XLBUF	80	

#define F2EBUFSIZE	16
#define F2EBUFNUM	150
#define F2EBUFSTART	0x500e0000

/**************************************************************************
				Ethernet to FDDI translation

                          +---------+--------+------+------------+
          Ethernet pkt    |   DST   |   SRC  | TYPE |  MAC DATA  |
                          +---------+--------+------+------------+
                    _ _ _/              _ _ /|      |            |
              _ _ _/              _ _ _/     |      |            |
  FDDI  _ _ _/              _ _ _/           |      |            |
   pkt /                   /                 |      |            |
   +--+---------+---------+------+----+------+------+------------+
   |FC|   DST   |   SRC   | LSAP | UI |  PID | TYPE |  LLC DATA  |
   +--+---------+---------+------+----+------+------+------------+
     1     6         6    |   2     1     3       2 |
                          |<----  SNAP header  ---->|

	where 
		- 'FC' is FDDI frame control byte
		- 'SNAP' is 0xaaaa
		- 'UI' is 0x03
		- 'PID' is 0x000000 for RFC1042 frames,  0x0000f8 for 'tunnel' frames
		- 'TYPE' is the ethernet protocol type


/********************************************************************
	GetXlateBuf - Get a translate buffer off the free list.
********************************************************************/
PKT* GetXlateBuf()			/* get a eth_to_fddi translate buffer */
{
   register word saveMask;
	register PKT* retval;

	MaskAllInts(saveMask);
	if( (retval = xlateBufList) != 0 )	{		/* if list head is non-null */
		xlateBufList = xlateBufList->pktDriverLink;		/* update head ptr */
	}
	RestoreIntMask(saveMask);
	return(retval);
} /* end GetXlateBuf() */


/********************************************************************
	FreetXlateBuf - Free a translate buffer to the free list.
********************************************************************/
void FreeXlateBuf(bp)			/* free a eth_to_fddi translate buffer */
register PKT* bp;
{
   register word saveMask;

	MaskAllInts(saveMask);
	bp->pktBufLink = (PKT *)NULL;
	bp->pktDriverLink = xlateBufList;		/* link new buff at head of list */
	xlateBufList = bp;							/* update list head ptr */

	RestoreIntMask(saveMask);
} /* end FreeXlateBuf() */

/********************************************************************
	ntohs_local() - Local copy to facilitate compiler inlining.	
********************************************************************/
ntohs_local(x)
int x;
{
	 return ((x<<8 & 0xff00) | (x>>8 & 0xFF));
}

/********************************************************************
	Get_F2E_Buf() - Routine will return a buffer of size F2EBUFSIZE
********************************************************************/
PKT *Get_F2E_Buf() {

	register word saveMask;
	register PKT *pp;

	MaskAllInts(saveMask);

	if( pp = F2EPktHead ) {
		F2EPktHead = (PKT *)pp->pktMsgHdr.mh_link;
	}

	RestoreIntMask(saveMask);
	return pp;
}

/********************************************************************
	Put_F2E_Buf() - Routine will put buffer back on stack.
********************************************************************/
void FreeF2EBuf( PKT *pp) {

	register word saveMask;

	MaskAllInts(saveMask);

	pp->pktBufLink = (PKT *)NULL;
	/* link new buff at head of list */
	pp->pktMsgHdr.mh_link = (MSGHDR *)F2EPktHead;
	F2EPktHead = pp;								/* update list head ptr */

	RestoreIntMask(saveMask);
}

/********************************************************************
	Translate_fddi_to_eth - translate an FDDI frame to the format
		needed to tx it out on an ethernet port.
********************************************************************/
Translate_fddi_to_eth(pp,xlt_flg)
PKT** pp;			/* PKT ptr for frame to be translated */
register int xlt_flg;
{
	register XFRMHDR* fp;		/* ptr to hdr of FDDI frame to be translated */
	register FRMHDR* xfp;   /* ptr to hdr of ETH xlation frame */
	register PKT*	xpp;		/* Translation buffer */
	register word* ap;		/* ptr to addrs in frame after translation shift */
	/* byte tmp_ln[2]; */
	register int i;
	register int sz;

	fp = (XFRMHDR*)((*pp)->pktDataPtr );		/* ptr to frame (after FC byte) */

	if( xlt_flg == ADR_ETH2_TYPE || xlt_flg != ADR_SNAP_1042_TYPE &&
		(fp->_s == 0xaaaa  &&  ((fp->_n) & 0x00ff) == 0x0003) && /* if SNAP UI */
		 (((fp->_a == PID_RFC1042)&&(fp->_p != AARP_PT))||(fp->_a == PID_TUNNEL))
	  ) {
		
			/* Squeeze out the SNAP subheader by
				shifting the ethernet addresses down to just before the 
				protocol type (the last 2 bytes of the SNAP subheader).
				Note 1: the source and destination areas of this copy
				overlap, so we copy from the end to prevent the data
				from getting clobbered. 
				Note 2: Since we must bit reverse each address byte,
				this is a good time to do it. */

         /* Since data ptr was incremented by one in the packet */
         /* processing code additional increment should be      */
         /* (DELTA_XL - 1)                                      */
			(*pp)->pktDataPtr += (DELTA_XL-1);		/* adjust data ptr */

			ap = (word*)(*pp)->pktDataPtr;			/* new dst addr */

			for(i=2; i>=0; i--)		/* for each word of addr (12 bytes) */
			{
				bitrev.w = *((word*)fp + i);		/* get 4 bytes */
				*((shrt*)ap + (i*2))     = bitrev.s[0];	/* put 4 at new loc */
				*((shrt*)ap + ((i*2)+1)) = bitrev.s[1];	/* put 4 at new loc */
			}

			/* adjust the data ptr and the data length in the PKT structure */
			(*pp)->pktDataSize  -= (DELTA_XL-1);		/* ditto... */
			(*pp)->pktTotalSize -= (DELTA_XL-1);		/* ditto... */

			/* Pad the frame to ethernet minimum if necessary */
			if( (*pp)->pktTotalSize < 60 ) {
				(*pp)->pktDataSize  = 60;	
				(*pp)->pktTotalSize = 60;
			}
	} /* END OF NON-802.3 FRAME TRANSLATION */
	else {	/* Translate LLC into an 802.3 style pkt. Need to insert a	*/
					/* len field just after the Source Mac Address				*/
				
		/* Get a pkt and buffer */
#if FS_PERFORM
		if(xpp = (PKT *)GetShramPktBuf(14))  {   /* DA+SA+LEN */         
#endif
		if( xpp = Get_F2E_Buf() ) {

			/* Copy the mac header information to the new pkt buffer */
			xfp = (FRMHDR*)(xpp->pktDataPtr );			/* ptr to frame hdr */

			/* Copy mac addrs (3 words) to xlate buffer (bit-reverse
			** each byte in the process).
			*/ 
			*((word*)&xfp->dl) = *((word*)fp);				/* get 4 bytes */
			*((word*)&xfp->dl + 1) = *((word*)fp + 1);	/* get 4 bytes */
			*((word*)&xfp->dl + 2) = *((word*)fp + 2);	/* get 4 bytes */

			/* Fill in the 802.3 len field in the new pkt buffer.
			** Get the len info from the original fddi pkt.
			*/       
#if FS_PERFORM
			tmp_ln[0]=(((*pp)->pktTotalSize-12)&0xff00) >> 8;/* 12 == DA+SA */
			tmp_ln[1]=(((*pp)->pktTotalSize-12)&0x00ff);     /* 12 == DA+SA */
			(xfp->ln) = *((shrt *)&tmp_ln[0]);
#endif
			(xfp->ln) = (shrt)ntohs_local(((*pp)->pktTotalSize)-12); 

			/* Setup some crutial info in the xlate pkt/buff */
			xpp->pktTotalSize = (*pp)->pktTotalSize+2;		/* Add in Len field */
			xpp->pktUseCount  = (*pp)->pktUseCount;
			/* xpp->pktRcvPort   = (*pp)->pktRcvPort; */
			xpp->pktXmtPort	= (*pp)->pktXmtPort;
			(*pp)->pktXmtPort = 0;
			/* xpp->pktDataSize  = 14; */ /**FS_PERFORM - Done at initialization **/

			/* Adjust the data ptr of the original pkt */
			(*pp)->pktDataPtr  += 12;			/* DA+SA , FC - Already done.*/
			(*pp)->pktDataSize -= 12;			/* DA+SA , FC - Already done.*/

			/* Pad the frame if necessary */
			if( (xpp)->pktTotalSize < 60 ) {
				(*pp)->pktDataSize += (60 - (xpp)->pktTotalSize);
				(xpp)->pktTotalSize = 60;
      	}
			/* Chain the old pkt/buff to the new one, such that
			** the new xlate pkt/buff gets transmitted first.
		 	*/ 
			xpp->pktBufLink = (*pp);	/* Chain buffers.								*/
			(*pp) = xpp;					/* Switch the order of transmission.	*/
		}
		else {
			(*pp) = (PKT *)0;
		}
	}
} /* end Translate_fddi_to_eth() */



/********************************************************************
	Translate_eth_to_fddi - Translate a frame rcvd on ethernet to 
		be transmitted on FDDI.

	This is made a little more complicated by the fact that this
	frame may also be transmitted onto ethernet, so we must leave 
	the original PKT and its associated buffer unchanged.  This
	is done with a structure called an XB (translate buffer).
	Each XB is a PKT, with its associated buffer, chained to
	another PKT (the 2nd PKT, which has no buffer of its own, acquires 
	ownership of the original ethernet buffer during the translation
	process).

	When the frame is transmitted on the FDDI side, the MAC
	addresses (and the 802.2 LLC header, if it must be added)
	are transmitted from the first PKT in the XB.  The rest of the
	frame is transmitted from the 2nd PKT in the XB (which points
	into the original ethernet buffer at the 'length/type' field).

	On the ethernet side the frame can then be transmitted from
	the original PKT and its associated buffer (even though that
	buffer is now 'owned' by the 2nd PKT of the XB).
********************************************************************/
PKT* Translate_eth_to_fddi(pp)
register PKT* pp;			/* PKT ptr for frame to be translated */
{ 
	register FRMHDR* fp;			/* ptr to hdr of frame to be translated */
	register FFHDR* xfp;
	register PKT* xpp;
	register PKT* xpp2;
	register int tmp_size;
	extern Spcl_free_pkt();
   extern PKT *FreePktHead;
   extern int FreeThisPktBuf();

	if( (xpp = GetXlateBuf()) == 0 )	{		/* get PKT ptr */
		return(0);					/* sorry, fresh out of PKTs */
	}
    
#ifdef FS_FIX
   if( (xpp2 = (PKT *)GetClonePkt(pp)) == 0) {
#endif

/***************** GetClonePkt() ***********/
  
#if FS_FIX
  if( (xpp2 = (PKT *)get_pkt()) != (PKT *)NULL ) {
#endif  
  if( (xpp2 = FreePktHead) != (PKT *)NULL) {
		FreePktHead = (PKT *)xpp2->pktMsgHdr.mh_link;

		xpp2->pktDriverFree = pp->pktDriverFree;
		xpp2->pktDriverInfo = pp->pktDriverInfo;
		xpp2->pktBufPtr     = pp->pktBufPtr;
		xpp2->pktBufLen     = pp->pktBufLen;
		xpp2->pktDataPtr    = pp->pktDataPtr; 
		xpp2->pktDataSize   = pp->pktDataSize;
		xpp2->pktDriverLink = pp->pktDriverLink;
		xpp2->pktFree       = (void *)FreeThisPktBuf;
		xpp2->pktUseCount   = 2;
		xpp2->pktXmtPort	  = 0;
   }
/***************** GetClonePkt() End ***********/
   else {
		(xpp->pktFree)(xpp);
		return(0);                 /* no more pkts */ 
    }

	/*
	* use pktBufPtr since pktDataPtr is 3 byte off -yke
	*/
	xfp = (FFHDR*)xpp->pktBufPtr;		/* get buffer ptr */
	fp = (FRMHDR*)pp->pktDataPtr;			/* ptr to original frame  */

	/* Copy eth addrs (3 words) to xlate buffer */ 
	*((word*)&xfp->dl) 	  = *((word*)fp);		/* get 4 bytes */
	*((word*)&xfp->dl + 1) = *((word*)fp + 1);		/* get 4 bytes */
	*((word*)&xfp->dl + 2) = *((word*)fp + 2);		/* get 4 bytes */

	tmp_size = ntohs_local(fp->ln); /* Number of non-padding bytes in data field */

	/* if the frame is 'eth' then we must add the 802.2 LLC header, else
		frame is 802.3 and we just add the 'frame ctrl' byte */
	if( tmp_size > 1518  || (tmp_size == 0))		/* if eth type, not len */
	{

		/* Mark the address record for the source address of this frame */
		((ADR *)pp->pktXlateSA)->xlt_flg = ADR_ETH2_TYPE;

		/* if the protocol type is AARP, translate to 'tunnel' format, else
		*	translate to 'RFC1042' format.  The two formats differ by just
		*	a single byte  -  the third byte of the PID in the SNAP
		*	subheader.  The 802.2 header and the first 6 bytes of the SNAP
		*	subheader were put into the translate buffer at init time.
		*	The last 2 bytes of the SNAP hdr (which are the
		*	same as the protocol type) are the first bytes that will
		*	be used in the original buffer. */

		xfp->_a = PID_RFC1042;		/* value for bytes 2, 3 of PID if '1042' */
		if( tmp_size == 0x80F3 )		/* if protocol type is AARP */
			xfp->_a = PID_TUNNEL;	/* bytes 2, 3 of PID for 'tunnel' */

		/* Finish up the 1st PKT in the XB.  Data transmitted from this
			buffer is the FC byte, the src and dest addresses, and
			the SNAP header */
		xpp->pktTotalSize = pp->pktTotalSize + DELTA_XL;  /* net len change */
		xpp->pktDataSize = 1+6+6+8-2;		/* fc+dst+src+len+SNAP-type */

		/* Adjust the data ptr and the data length from the original PKT struct */
		(char *)xpp2->pktDataPtr += 12;		/* moved 12 bytes to xlate buff  */
		xpp2->pktDataSize 		 -= 12;		/* ditto...                      */
		xpp2->pktTotalSize 		  = xpp2->pktDataSize;		/* ditto...       */

		/* Adjust the amount of data to be transmitted from the original pkt */
		pp->pktDataSize = 12;	/* Only transmit the mac header information    */
										/* via the original pkt. The rest of the data  */
										/* will be transmitted from xpp2.              */
	}
	else		/* 802.3 frame */
	{

		/* Mark the address record for the source address of this frame */
		if((fp->_s == 0xaaaa  &&  ((fp->_n) & 0x00ff) == 0x0003) && /* if SNAP UI */
			 (fp->_a == PID_RFC1042)) {
			((ADR *)pp->pktXlateSA)->xlt_flg = ADR_SNAP_1042_TYPE;
		}
		else
			((ADR *)pp->pktXlateSA)->xlt_flg = ADR_8023_TYPE;

		/* Finish up the 1st PKT in the XB.  Data transmitted from this */
		/*	buffer is the FC byte, and the src and dest addresses */
		/* Need to strip the 802.3 len field and padding bytes from the pkt */
		/* Use the 802.3 frame len so that the extra padding bytes are ignored	*/
      /* this will allow the remote bridge to correctly recalculate the 802.3 */
		/* frame length.																			*/

		xpp->pktTotalSize  = tmp_size+1+6+6; /* Unpadded frame len+FC+DA+SA */
		xpp2->pktTotalSize = tmp_size; /* Unpadded frame len */

		xpp->pktDataSize = 1+6+6;						/* fc+dst+src */

		/* Adjust the data ptr and the data length from the original PKT struct */
		(char *)xpp2->pktDataPtr	+= 14;		/* moved 14 bytes to xlate buff  */
		xpp2->pktDataSize 			-= 14 ;     /* moved 14 bytes to xlate buff  */
		/* Adjust the amount of data to be transmitted from the original pkt */
		pp->pktDataSize = 14;	/* Only transmit the mac header information    */
										/* via the original pkt. The rest of the data  */
										/* will be transmitted from xpp2.              */
	}

	/*  Use the 2nd PKT in the XB to point to the original buffer.  Data
		transmitted from this buffer is the length/type field and
	    everything after it. */

	/* Link the xpp pkt to the 2nd xlation pkt/buffer */
	xpp->pktBufLink = pp->pktBufLink = xpp2;
	xpp2->pktBufLink = 0;	/* i'm the last one -yke */
	pp->pktUseCount   &=  ~PKTBROUT; 
	pp->pktXmtPort	   &=  ~FDDI_PORT_M;

	/* take ownership of the original buffer the frame is in */
	pp->pktDriverFree = NULL;			/* orig. PKT no longer owns it */
	pp->pktFree			= (void *)Spcl_free_pkt;

	return(xpp);                            /* return XB ptr */

} /* end Translate_eth_to_fddi() */


/********************************************************************
	InitEFTranslation  -  Init buffers, etc. for the
		fddi <--> ethernet translation.

	Each 'xlate buffer' (XB) is actually a PKT, with its associated buffer,
	chained to another PKT (the 2nd PKT, which has no buffer of its own,
	gets temporary ownership of an ethernet buffer during the translation
	process). 

	The first PKT and its buffer are for the data (FDDI MAC and 802.2
	headers) that is pre-pended to a pkt during the ethernet-to-FDDI
	translation process.  The 2nd PKT will then point into the original 
	ethernet buffer at a point to pick up the rest of the pkt.
********************************************************************/
InitEFTranslation()			/* init the eth_to_fddi translation stuff */
{
	PKT* pp;
	FFHDR* bp;
	int i,j, tabval;

    /* ALLOC MEMORY OF THE XLATE PKT STRUCTS */

   xl_pktstart_addr = (PKT *)smalloc( sizeof(PKT) * SV_NMBR_XLBUF );
	if(xl_pktstart_addr == (PKT *)NULL) {
		printf("ERROR: \nCould Not Allocate XLB PKT STRUCT!\n");
		return(-1);
    }
 
	/* INIT THE TRANSLATION BUFFERS */

	bp = (FFHDR*)SV_STRT_XLBUF;		/* start of xlate buffer area in sh RAM */
	pp = (PKT*)xl_pktstart_addr;		/* start of PKTs for translate buffers  */
	xlateBufList = pp;					/* free list head points to first XB    */

	for(i=0; i<SV_NMBR_XLBUF; i++)	/* for each XB (PKT/buffer/PKT)         */
	{
		/* INIT BUFFER WITH A SNAP SUBHEADER FOR '1042' OR 'TUNNEL' FORMAT	 */
		bp->unused1	= FDDI_FC;
		bp->unused2	= FDDI_FC;
		bp->unused3	= FDDI_FC;
		bp->fc = FDDI_FC;							/* frame control byte         	 */
		bp->_s = 0xaaaa;							/* first 2 byes of SNAP hdr		 */
		bp->_n = 0x0003;							/* next 2 (byte reversed)     	 */

		/* INIT PKT FOR THIS BUFFER (first PKT) */
		pp->pktBufLen = sizeof(FFHDR);		/* buffer size                    */
		pp->pktBufPtr = (char *)(bp);			/* PKT points to start of buffer  */
		pp->pktDataPtr = (char *)&((bp)->fc);/* PKT points to FC in buffer     */
		pp->pktFree = FreeXlateBuf;			/* ptr to PKT/buffer free routine */
		pp->pktDriverFree = NULL;				/* dummy (null) free routine      */
		pp->pktUseCount = 0;

		pp->pktDriverLink = pp + 1;			/* point each to the next XB      */
		pp->pktBufLink = NULL;					/* for chaining pnt to nxt PKT    */

		bp++;											/* advance to next buffer         */
		pp++;											/* adv to next avail PKT          */
	}
	(--pp)->pktDriverLink = 0;					/* null terminate the list        */

	return(0);
} /* end InitEFTranslation() */

/*******************************************************************************/
InitFddi_to_Eth_Buffers() {

	char *bp = (char *)F2EBUFSTART;
	PKT *pp;
	int i;

   F2EPktHead = (PKT *)smalloc( sizeof(PKT) * F2EBUFNUM );
	if( F2EPktHead == (PKT *)NULL ) {
		printf("\nError: Could Not Allocate Fddi To Ethernet Xlation Packets.\n");
		return(1);
	}

#if 0
	bp = (char *)shmalloc( F2EBUFSIZE * F2EBUFNUM );
	if( bp == (char *)NULL ) {
		printf("\nError: Could Not Allocate Fddi To Ethernet Xlation Buffers.\n");
		return(1);
	}
#endif

	pp = F2EPktHead;
	for( i=0; i<F2EBUFNUM; i++ ) {

		/* INIT PKT FOR THIS BUFFER (first PKT) */
		pp->pktBufLen = F2EBUFSIZE;		/* buffer size                      */
		pp->pktBufPtr = (char *)(bp);			/* PKT points to start of buffer */
		pp->pktDataPtr = (char *)(bp);/* PKT points to FC in buffer          */
		pp->pktFree = FreeF2EBuf;			/* ptr to PKT/buffer free routine   */
		pp->pktDriverFree = NULL;				/* dummy (null) free routine     */
		pp->pktUseCount = 1;
		pp->pktDataSize  = F2EBUFSIZE-2; 
		pp->pktRcvPort   = FDDI_PORT_NUM;

		pp->pktDriverLink = NULL;
		pp->pktMsgHdr.mh_link = pp + 1; /* point each to the next XB*/
		pp->pktBufLink = NULL;					/* for chaining pnt to nxt PKT   */

		pp++;											/* adv to next avail PKT         */
		bp += F2EBUFSIZE; 						/* adv to next avail buffer		*/
	}
	(--pp)->pktMsgHdr.mh_link = 0;			/* null terminate the list       */
	return(0);
}
