/**************************************************************************
 *  Copyright (c) 1992 Hughes LAN Systems                                 *
 *  All rights Reserved                                                   *
 *                                                                        *
 *  This file contains unpublished proprietary source code of Hughes LAN  *
 *  Systems.                                                              *
 **************************************************************************
                                                                           
   This file contains the service routines for RFC1286-MIB
   It is directly derived from the MIB specification using the AWK program 
   'MIBSVC.AWK'
   Should significant changes be needed in this file that are NOT          
   reflected back into the MIB specification, then modify this header to   
   indicate that this is the case.                                         
                                                                           
 *************************************************************************/

#define RFC1286-MIB_SVC

#include <types.h> 
#include <target.h>
#include <krnl.h>
#include <bridges.h>
#include <dbd.h>
#include <snmp.h>
#include <asn1.h>
#include <syteksnm.h>
#include <prcctl.h>
#include <sncvar.h>
#include <stp.h>
#include <nvrecs.h>

/*
 *	The protocol specification for STP
 */
#define	STP_PROTOCOL_UNKNOWN	1
#define	STP_PROTOCOL_DECLB100	2
#define	STP_PROTOCOL_IEEE8021d	3

/**************************************************************************
*  Procedure    :   svc_dot1dStpProtocolSpecification
*  Path         :   1.3.6.1.2.1.17.2.1
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "An indication of what version of the Spanning
*                 Tree Protocol is being run.  The value
*                 'decLb100(2)' indicates the DEC LANbridge 100
*                 Spanning Tree protocol.  IEEE 802.1d
*                 implementations will return 'ieee8021d(3)'.  If
*                 future versions of the IEEE Spanning Tree Protocol
*                 are released that are incompatible with the
*                 current version a new value will be defined."
*
**************************************************************************/

int svc_dot1dStpProtocolSpecification(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) STP_PROTOCOL_IEEE8021d;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPriority
*  Path         :   1.3.6.1.2.1.17.2.2
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The value of the write-able portion of the Bridge
*                 ID, i.e., the first two octets of the (8 octet
*                 long) Bridge ID.  The other (last) 6 octets of the
*                 Bridge ID are given by the value of
*                 dot1dBaseBridgeAddress."
*
**************************************************************************/

int svc_dot1dStpPriority(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)BridgeStatus->BridgePriority;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		BridgeStatus->BridgePriority = *value;
		set_bridge_priority(*value);
		SaveBStatus();
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value < MIN_BRIDGE_PRIORITY ||
				*value > MAX_BRIDGE_PRIORITY)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpTimeSinceTopologyChange
*  Path         :   1.3.6.1.2.1.17.2.3
*  Access       :   RO
*  Syntax       :   TimeTicks VT_TIMETICKS
*  Description  :   
*                 "The time (in hundredths of a second) since the
*                 last time a topology change was detected by the
*                 bridge entity."
*
**************************************************************************/

int svc_dot1dStpTimeSinceTopologyChange(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) RealTimeTicks() - StpTopChgUpTime; 
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpTopChanges
*  Path         :   1.3.6.1.2.1.17.2.4
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                 "The total number of topology changes detected by
*                 this bridge since the management entity was last
*                 reset or initialized."
*
**************************************************************************/

int svc_dot1dStpTopChanges(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) StpTopChgDet;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpDesignatedRoot
*  Path         :   1.3.6.1.2.1.17.2.5
*  Access       :   RO
*  Syntax       :   BridgeId VT_STRING
*  Description  :   
*                 "The bridge identifier of the root of the spanning
*                 tree as determined by the Spanning Tree Protocol
*                 as executed by this node.  This value is used as
*                 the Root Identifier parameter in all Configuration
*                 Bridge PDUs originated by this node."
*
**************************************************************************/

int svc_dot1dStpDesignatedRoot(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	unsigned short	*p = (unsigned short *)value;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*p = htons(bridge_info.designated_root.priority);
			p++;
			ncopy(p, bridge_info.designated_root.stp_address);
			*lenp = sizeof(bridge_info.designated_root.priority) +
				sizeof(bridge_info.designated_root.stp_address);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpRootCost
*  Path         :   1.3.6.1.2.1.17.2.6
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The cost of the path to the root as seen from
*                 this bridge."
*
**************************************************************************/

int svc_dot1dStpRootCost(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) bridge_info.root_path_cost;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpRootPort
*  Path         :   1.3.6.1.2.1.17.2.7
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The port number of the port which offers the
*                 lowest cost path from this bridge to the root
*                 bridge."
*
**************************************************************************/

int svc_dot1dStpRootPort(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) bridge_info.root_port;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpMaxAge
*  Path         :   1.3.6.1.2.1.17.2.8
*  Access       :   RO
*  Syntax       :   Timeout VT_NUMBER
*  Description  :   
*                 "The maximum age of Spanning Tree Protocol
*                 information learned from the network on any port
*                 before it is discarded, in units of hundredths of
*                 a second.  This is the actual value that this
*                 bridge is currently using."
*
**************************************************************************/

int svc_dot1dStpMaxAge(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{	/*
		 	 *	we store the max_age in seconds.
			 */
			*value = (UINT_32_T) bridge_info.max_age * 100;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpHelloTime
*  Path         :   1.3.6.1.2.1.17.2.9
*  Access       :   RO
*  Syntax       :   Timeout VT_NUMBER
*  Description  :   
*                 "The amount of time between the transmission of
*                 Configuration bridge PDUs by this node on any port
*                 when it is the root of the spanning tree or trying
*                 to become so, in units of hundredths of a second.
*                 This is the actual value that this bridge is
*                 currently using."
*
**************************************************************************/

int svc_dot1dStpHelloTime(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
 			*value = (UINT_32_T) bridge_info.hello_time * 100 ;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpHoldTime
*  Path         :   1.3.6.1.2.1.17.2.10
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "This time value determines the interval length
*                 during which no more than two Configuration bridge
*                 PDUs shall be transmitted by this node, in units
*                 of hundredths of a second."
*
**************************************************************************/

int svc_dot1dStpHoldTime(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) bridge_info.hold_time * 100;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpForwardDelay
*  Path         :   1.3.6.1.2.1.17.2.11
*  Access       :   RO
*  Syntax       :   Timeout VT_NUMBER
*  Description  :   
*                 "This time value, measured in units of hundredths
*                 of a second, controls how fast a port changes its
*                 spanning state when moving towards the Forwarding
*                 state.  The value determines how long the port
*                 stays in a particular state before moving to the
*                 next state.  For example, how long a port stays in
*                 the Listening state when moving from Blocking to
*                 Learning.  This value is also used, when a
*                 topology change has been detected and is underway,
*                 to age all dynamic entries in the Forwarding
*                 Database.  [Note that this value is the one that
*                 this bridge is currently using, in contrast to
*                 dot1dStpBridgeForwardDelay which is the value that
*                 this bridge and all others would start using
*                 if/when this bridge were to become the root.]"
*
**************************************************************************/

int svc_dot1dStpForwardDelay(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) bridge_info.forward_delay * 100;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpBridgeMaxAge
*  Path         :   1.3.6.1.2.1.17.2.12
*  Access       :   RW
*  Syntax       :   Timeout VT_NUMBER
*  Description  :   
*                 "The value that all bridges use for MaxAge when
*                 this bridge is acting as the root.  Note that
*                 802.1d/D9 specifies that the range for this
*                 parameter is related to the value of
*                 dot1dStpBridgeHelloTime. The granularity of this
*                 timer is specified by 802.1d/D9 to be 1 second.
*                 An agent may return a badValue error if a set is
*                 attempted to a value which is not a whole number
*                 of seconds."
*
**************************************************************************/

int svc_dot1dStpBridgeMaxAge(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	max_age = 0;	/* in seconds	*/

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) bridge_info.bridge_max_age * 100;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		max_age = *value / 100;
		BridgeStatus->MaxAge = max_age;
		SetMaxAge(max_age);
		SaveBStatus();
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value != 0)
				max_age = *value / 100;
			/*
			 *	The granularity of max_age must be in 
			 *	1 second.
			 */
			if (*value == 0 || (*value % 100) != 0 || 
				max_age < MIN_MAX_AGE ||
				max_age > MAX_MAX_AGE)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpBridgeHelloTime
*  Path         :   1.3.6.1.2.1.17.2.13
*  Access       :   RW
*  Syntax       :   Timeout VT_NUMBER
*  Description  :   
*                 "The value that all bridges use for HelloTime when
*                 this bridge is acting as the root.  The
*                 granularity of this timer is specified by
*                 802.1d/D9 to be 1 second.  An agent may return a
*                 badValue error if a set is attempted to a value
*                 which is not a whole number of seconds."
*
**************************************************************************/

int svc_dot1dStpBridgeHelloTime(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	hello_time = 0;	/* in seconds	*/

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) bridge_info.bridge_hello_time * 100 ;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		hello_time = *value / 100;
		BridgeStatus->HelloTime = hello_time;
		SetHelloTime(hello_time);
		SaveBStatus();
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value != 0)
				hello_time = *value / 100;
			/*
			 *	The granularity of hello_time must be in 
			 *	1 second.
			 */
			if (*value == 0 || (*value % 100) != 0 || 
				hello_time < MIN_HELLO_TIME ||
				hello_time > MAX_HELLO_TIME)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpBridgeForwardDelay
*  Path         :   1.3.6.1.2.1.17.2.14
*  Access       :   RW
*  Syntax       :   Timeout VT_NUMBER
*  Description  :   
*                 "The value that all bridges use for ForwardDelay
*                 when this bridge is acting as the root.  Note that
*                 802.1d/D9 specifies that the range for this
*                 parameter is related to the value of
*                 dot1dStpBridgeMaxAge.  The granularity of this
*                 timer is specified by 802.1d/D9 to be 1 second.
*                 An agent may return a badValue error if a set is
*                 attempted to a value which is not a whole number
*                 of seconds."
*
**************************************************************************/

int svc_dot1dStpBridgeForwardDelay(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	forward_delay = 0;	/* in seconds	*/

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) bridge_info.bridge_forward_delay * 100;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		forward_delay = *value / 100;
		BridgeStatus->ForwardDelay = forward_delay;
		SetForwardDelay(forward_delay);
		SaveBStatus();
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value != 0)
				forward_delay = *value / 100;
			/*
			 *	The granularity of forward_delay must be in 
			 *	1 second.
			 */
			if (*value == 0 || (*value % 100) != 0 || 
				forward_delay < MIN_FORWARD_DELAY ||
				forward_delay > MAX_FORWARD_DELAY)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPort
*  Path         :   1.3.6.1.2.1.17.2.15.1.1
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The port number of the port for which this entry
*                 contains Spanning Tree Protocol management
*                 information."
*
**************************************************************************/

int svc_dot1dStpPort(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) *compl;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortPriority
*  Path         :   1.3.6.1.2.1.17.2.15.1.2
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The value of the priority field which is
*                 contained in the first (in network byte order)
*                 octet of the (2 octet long) Port ID.  The other
*                 octet of the Port ID is given by the value of
*                 dot1dStpPort."
*
**************************************************************************/

int svc_dot1dStpPortPriority(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
 	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) BridgeStatus->priority[*compl - 1];
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		BridgeStatus->priority[*compl - 1] = *value;
		set_port_priority(*compl, *value);
		SaveBStatus();
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value < MIN_PORT_PRIORITY ||
				*value > MAX_PORT_PRIORITY)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortState
*  Path         :   1.3.6.1.2.1.17.2.15.1.3
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The port's current state as defined by
*                 application of the Spanning Tree Protocol.  This
*                 state controls what action a port takes on
*                 reception of a frame.  If the bridge has detected
*                 a port that is malfunctioning it will place that
*                 port into the broken(6) state.  For ports which
*                 are disabled (see dot1dStpPortEnable), this object
*                 will have a value of disabled(1)."
*
**************************************************************************/

int svc_dot1dStpPortState(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	/*
	 *	STP port state to snmp port state
	 *	State		STP 	SNMP 
	 *	disabled	0	1
	 *	listening	1	3
	 *	learning	2	4
	 *	forwarding	3	5
	 *	blocking	4	2
	 *	broken		-	6
	 */
	static char	Stp2SnmpState[] = {1, 3, 4, 5, 2};

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) port_info[*compl].state;
			*value = (UINT_32_T)Stp2SnmpState[*value];
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortEnable
*  Path         :   1.3.6.1.2.1.17.2.15.1.4
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The enabled/disabled status of the port."
*
**************************************************************************/

int svc_dot1dStpPortEnable(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	PortMask;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = port_info[*compl].state == Disabled ? 2 : 1;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		PortMask = 1 << (*compl - 1);
		if (*value == 1)
		{
			/*
			 *	enable port
			 */
			BridgeStatus->PortState |= PortMask;
			prc.Prc_PortState |= PortMask;
			if (prc.Prc_StpMode == STP_MODE_ENABLE)
				enable_port(*compl);
		}
		else
		{
			/*
			 *	disable port.
			 */
			BridgeStatus->PortState &= ~PortMask;
			prc.Prc_PortState &= ~PortMask;
			if (prc.Prc_StpMode == STP_MODE_ENABLE)
				disable_port(*compl);
		}
		SaveBStatus();
		pre_proc_prcctl();
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (!(*value == 1 || *value == 2))
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortPathCost
*  Path         :   1.3.6.1.2.1.17.2.15.1.5
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The contribution of this port to the path cost of
*                 paths towards the spanning tree root which include
*                 this port."
*
**************************************************************************/

int svc_dot1dStpPortPathCost(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) port_info[*compl].path_cost;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		BridgeStatus->PathCost[*compl - 1] = *value;
		set_path_cost(*compl, *value);
		SaveBStatus();
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value < MIN_PATH_COST || *value > MAX_PATH_COST)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortDesignatedRoot
*  Path         :   1.3.6.1.2.1.17.2.15.1.6
*  Access       :   RO
*  Syntax       :   BridgeId VT_STRING
*  Description  :   
*                 "The unique Bridge Identifier of the Bridge
*                 recorded as the Root in the Configuration BPDUs
*                 transmitted by the Designated Bridge for the
*                 segment to which the port is attached."
*
**************************************************************************/

int svc_dot1dStpPortDesignatedRoot(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	unsigned short	*priority = (unsigned short *)value;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*priority = htons(port_info[*compl].designated_root.priority);
			priority++;
			ncopy(priority, port_info[*compl].designated_root.stp_address);
			*lenp = sizeof(port_info[*compl].designated_root.priority) +
				sizeof(port_info[*compl].designated_root.stp_address);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortDesignatedCost
*  Path         :   1.3.6.1.2.1.17.2.15.1.7
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                 "The path cost of the Designated Port of the
*                 segment connected to this port.  This value is
*                 compared to the Root Path Cost field in received
*                 bridge PDUs."
*
**************************************************************************/

int svc_dot1dStpPortDesignatedCost(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) port_info[*compl].designated_cost;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortDesignatedBridge
*  Path         :   1.3.6.1.2.1.17.2.15.1.8
*  Access       :   RO
*  Syntax       :   BridgeId VT_STRING
*  Description  :   
*                 "The Bridge Identifier of the bridge which this
*                 port considers to be the Designated Bridge for
*                 this port's segment."
*
**************************************************************************/

int svc_dot1dStpPortDesignatedBridge(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	unsigned short	*priority = (unsigned short *)value;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*priority = htons(port_info[*compl].designated_bridge.priority);
			priority++;
			ncopy(priority, port_info[*compl].designated_bridge.stp_address);
			*lenp = sizeof(port_info[*compl].designated_bridge.priority) +
				sizeof(port_info[*compl].designated_bridge.stp_address);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortDesignatedPort
*  Path         :   1.3.6.1.2.1.17.2.15.1.9
*  Access       :   RO
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                 "The Port Identifier of the port on the Designated
*                 Bridge for this port's segment."
*
**************************************************************************/

int svc_dot1dStpPortDesignatedPort(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = htons(port_info[*compl].port_id);
			*lenp = sizeof(port_info[*compl].port_id);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dot1dStpPortForwardTransitions
*  Path         :   1.3.6.1.2.1.17.2.15.1.10
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                 "The number of times this port has transitioned
*                 from the Learning state to the Forwarding state."
*
**************************************************************************/

int svc_dot1dStpPortForwardTransitions(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) StpForwardTransitions[*compl];
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

