/**			       
*
*	Program Name:	Stanley
*
*	Filename:	stp.c
*
*	$Log:   /b/gregs/bridge/stp/stp.c_v  $
 * 
 *    Rev 1.4   24 Nov 1993 08:40:02   gregs
 * Changed FlushArp() to fls_arp_cache() and added this routine
 * to the topology_changed_detect function.
 * 
 *    Rev 1.3   17 Nov 1993 12:43:44   gregs
 * Flush the arp cache whenever the bridge reconfigures its ports
 * 
 *    Rev 1.2   13 Aug 1993 11:58:46   vinay
 * trap id's for root and topology change were reversed, fixed it.
 * 
 *    Rev 1.1   09 Aug 1993 10:23:20   vinay
 * Added topology change and newroot traps
 * 
 *    Rev 1.0   30 Jul 1993 13:51:38   vinay
 * Initial revision.
 * 
 *    Rev 1.0   30 Mar 1992 17:40:36   pvcs
 * Initial revision.
*
*	Creation Date:	not known
*
*	Date:		12.4.91
*
*	Version:	1.3
*
*	Programmers:	K Kong
*
*	Modifications:	K Kong	9.3.91	1.1
*			Because of 960 alignment problem, we cannot 
*			operate the root_path_cost directly from
*			the configuration bpdu, (i.e. Config_bpdu).
*			We have to copy it out from the communication
*			buffer before we operate on it.
*
*			K Kong	9.9.91	1.2
*			When a change in the topology is detected, the 
*			bridge should use the value of Forward Delay to
*			ago out dynamic entries.  This is to ensure that
*			after forward delay time has elapsed while the
*			topology change flag is set in all configuration
*			messages received from the root, the only dynamic
*			entries remaining in the filtering database are those
*			that have been created or updated during that 
*			period.
*
*			K Kong	12.4.91	1.3
*			Move all globals declarations to stpglobals.c.
*
*	Comments:	This is the spanning tree protocol.  It is based
*			on the work done by Kevin Huynh on the 80386 based
*			bridges.
*			This module follows the algorithm given on the
*			IEEE std 802.1D-1990 Page 78 to 106.
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/


#include <types.h>
#include <target.h>
#include <krnl.h>
#include <stp.h>

extern void SetProcState(int port_no, int state);
void	set_port_priority(Int port_no, Int new_port_id);
void	set_path_cost(Int port_no, Int path_cost);
void	enable_port(Int port_no);

/****************************************************************************
*	Static Storage Allocation												*
*****************************************************************************/
static Config_bpdu	config_bpdu[NumberOfStpPorts + 1];
static Tcn_bpdu		tcn_bpdu[NumberOfStpPorts + 1];
static Timer		hello_timer;
static Timer		tcn_timer;
static Timer		topology_change_timer;
static Timer		message_age_timer[NumberOfStpPorts + 1];
static Timer		forward_delay_timer[NumberOfStpPorts + 1];
static Timer		hold_timer[NumberOfStpPorts + 1];


static Boolean	equal_test(Identifier *id1, Identifier *id2);
static Boolean	supersedes_port_info(Int port_no, Config_bpdu *config);
static void	record_config_information(Int port_no, Config_bpdu *config);
static void	record_config_timeout_values(Config_bpdu *config);
static void	reply(Int port_no);
static void	transmit_tcn(void);
static Boolean	port_test(Identifier *id3, Identifier *id4);
static void	root_selection(void);
static void	designated_port_selection(void);
static void	become_designated_port(Int port_no);
static void	make_forwarding(Int port_no);
static void	make_blocking(Int port_no);
static void	set_port_state(Int port_no, State state);
static void	topology_change_acknowledged(void);
static void	acknowledge_topology_change(Int port_no);
static void	hello_timer_expiry(void);
static void	message_age_timer_expiry(Int port_no);
static void	forward_delay_timer_expiry(Int port_no);
static Boolean	designated_for_some_port(void);
static void	tcn_timer_expiry(void);
static void	topology_change_timer_expiry(void);
static void	hold_timer_expiry(Int port_no);
static void	stop_hello_timer(void);
static Boolean	hello_timer_expired(void);
static void	start_tcn_timer(void);
static Boolean	tcn_timer_expired(void);
static void	start_topology_change_timer(void);
static Boolean	topology_change_timer_expired(void);
static void	start_message_age_timer();
static void	stop_message_age_timer(Int port_no);
static Boolean	message_age_timer_expired(Int port_no);
static void	start_forward_delay_timer(Int port_no);
static void	stop_forward_delay_timer(Int port_no);
static Boolean	forward_delay_timer_expired(Int port_no);
static void	start_hold_timer(Int port_no);
static void	stop_hold_timer(Int port_no);
static Boolean	hold_timer_expired(Int port_no);
static void	RootPathCostCopy(ushort *to, ushort *from);
static void	ntoh_config_bpdu(Config_bpdu *config);

extern TIMER TopologyChangeTimer;
extern TIMER NewRootTimer;

void transmit_config(port_no)
Int	port_no;

	{
	Cost	root_path_cost;

	if (hold_timer[port_no].active)
		{
		port_info[port_no].config_pending = True;
		return;
		}
	config_bpdu[port_no].type = Config_bpdu_type;
	config_bpdu[port_no].root_id_priority = 
	        	htons(bridge_info.designated_root.priority);
	ncopy(config_bpdu[port_no].root_id_stp_address, 
		bridge_info.designated_root.stp_address);
	/**** K Kong, alignment problem !!! 
	config_bpdu[port_no].root_path_cost = 
			bridge_info.root_path_cost;
	******/
	root_path_cost = htonl(bridge_info.root_path_cost);
	RootPathCostCopy(config_bpdu[port_no].root_path_cost,
		(unsigned short *)&root_path_cost);
	config_bpdu[port_no].bridge_id_priority =
			htons(bridge_info.bridge_id.priority);
	ncopy(config_bpdu[port_no].bridge_id_stp_address, 
		bridge_info.bridge_id.stp_address);
	config_bpdu[port_no].port_id = htons(port_info[port_no].port_id);
	if(root_bridge())
		{
		config_bpdu[port_no].message_age = Zero;
		}
	else
		{
		config_bpdu[port_no].message_age =
			message_age_timer[bridge_info.root_port].value + 
			Message_age_increment;
		}

	config_bpdu[port_no].message_age = 
		htons(config_bpdu[port_no].message_age * STP_TIMER_UNIT);
	config_bpdu[port_no].max_age = 
		htons(bridge_info.max_age * STP_TIMER_UNIT);
	config_bpdu[port_no].hello_time = 
		htons(bridge_info.hello_time * STP_TIMER_UNIT);
	config_bpdu[port_no].forward_delay = 
		htons(bridge_info.forward_delay * STP_TIMER_UNIT);
	config_bpdu[port_no].topology_change_flag = Zero;
	if (port_info[port_no].topology_change_acknowledge)
		config_bpdu[port_no].topology_change_flag |= acknowledge_true;
	else
		config_bpdu[port_no].topology_change_flag &= acknowledge_false;

	port_info[port_no].topology_change_acknowledge = False;
	if (bridge_info.topology_change)
		config_bpdu[port_no].topology_change_flag |= change_true;
	else
		config_bpdu[port_no].topology_change_flag &= change_false;
	StpConfigTxBPDU[port_no]++;

	/********
	printf("Sending config. bpdu on port %d\n", port_no);
	***********/
	SendFromSTP(port_no, &config_bpdu[port_no], CONFIG_BPDU_SIZE);
	port_info[port_no].config_pending = False;
	start_hold_timer(port_no);
	}

Boolean root_bridge()

	{
	return(equal_test(&bridge_info.designated_root, &bridge_info.bridge_id));
	}

Boolean designated_port(port_no)
Int	port_no;

	{
	return((equal_test(&port_info[port_no].designated_bridge,
		&bridge_info.bridge_id))	&&
		(port_info[port_no].designated_port == port_info[port_no].port_id));
	}


void configuration_update()

	{
	root_selection();
	designated_port_selection();
	}


void port_state_selection()

	{
	Int port_no;

	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		if(port_no == bridge_info.root_port)
			{
			port_info[port_no].config_pending = False;
			port_info[port_no].topology_change_acknowledge = False;
			make_forwarding(port_no);
			}
		else if(designated_port(port_no))
			{
			stop_message_age_timer(port_no);
			make_forwarding(port_no);
			}
		else
			{
			port_info[port_no].config_pending = False;
			port_info[port_no].topology_change_acknowledge = False;
			make_blocking(port_no);
			}
		}
	}



void topology_change_detection()

	{
	StpTopChgDet++;
	/***
	StpTopChgUpTime = RealTimeTicks() - StpTopChgUpTime;
	***/
	StpTopChgUpTime = RealTimeTicks();
	if(root_bridge())
		{
		bridge_info.topology_change = True;
		start_topology_change_timer();
		}
	else if(bridge_info.topology_change_detected == False)
		{
		transmit_tcn();
		start_tcn_timer();
		}
	bridge_info.topology_change_detected = True;
 	/*
	 *	K Kong	9.9.91
	 *	When a change in the topology is detected, bridges should
	 *	use the value of Forward Delay to age out dynamic entries.
	 */
	fls_arp_cache();
	SetAgeTimeToForwardDelay();
	}

void received_config_bpdu(port_no, config)
Int	port_no;
Config_bpdu	*config;

	{
	Boolean	root;

	StpConfigRxBPDU[port_no]++;

	if (port_info[port_no].state != Disabled)
		{
		ntoh_config_bpdu(config);
		root = root_bridge();
		if(supersedes_port_info(port_no, config))
			{
			record_config_information(port_no, config);
			configuration_update();
			port_state_selection();
			if ((!root_bridge()) && root)
				{
				stop_hello_timer();
				if(bridge_info.topology_change_detected)
					{
					stop_topology_change_timer();
					transmit_tcn();
					start_tcn_timer();
					}
				}
			if(port_no == bridge_info.root_port)
				{
				record_config_timeout_values(config);
				config_bpdu_generation();
				if(config->topology_change_flag & acknowledge_true)
					{
					topology_change_acknowledged();
					}
				/****************
				if(config->topology_change_acknowledgment)
					{
					topology_change_acknowledged();
					}
				***************/
				}
			}
		else if(designated_port(port_no))
			{
			reply(port_no);
			}
		}
	}


void received_tcn_bpdu(port_no, tcn)
Int	port_no;
Tcn_bpdu	*tcn;

	{
	StpTopChgRxBPDU[port_no]++;

	if(port_info[port_no].state != Disabled)
		{
		if(designated_port(port_no))
			{
			topology_change_detection();
			acknowledge_topology_change(port_no);
			}
		}
	}

/*
 * name		initialize_port
 *
 * synopsis	initialize_port(port_no)
 *		int	port_no; <<	to be initialized. port_no starts
 *					from 1.
 *
 * description	It initializes the port "port_no" at startup state.
 *
 * returns	nothing
 */

void initialize_port(port_no)
Int	port_no;

	{
	become_designated_port(port_no);
	set_port_state(port_no, Blocking);
	port_info[port_no].topology_change_acknowledge = False;
	port_info[port_no].config_pending = False;
	stop_message_age_timer(port_no);
	stop_forward_delay_timer(port_no);
	stop_hold_timer(port_no);
	}

void disable_port(port_no)
Int	port_no;

	{
	Boolean	root;

	LedsOff(1 << (port_no - 1));
	root = root_bridge();
	fls_arp_cache();
	become_designated_port(port_no);
	set_port_state(port_no, Disabled);
	port_info[port_no].topology_change_acknowledge = False;
	port_info[port_no].config_pending = False;
	stop_message_age_timer(port_no);
	stop_forward_delay_timer(port_no);
	configuration_update();
	port_state_selection();
	if((root_bridge()) && (!root))
		{
		bridge_info.max_age = bridge_info.bridge_max_age;
		bridge_info.hello_time = bridge_info.bridge_hello_time;
		bridge_info.forward_delay = bridge_info.bridge_forward_delay;
		topology_change_detection();
		stop_tcn_timer();
		config_bpdu_generation();
		start_hello_timer();
		}
	/*
	 *	If all the ports has been disabled and the age time has
	 *	been changed, then we have to restore the original
	 *	age time.
	 */
	if (IsAllPortDisabled())
		RestoreAgeTime();
	}

void start_hello_timer()

	{
	hello_timer.value = (StpTime) Zero;
	hello_timer.active = True;
	}

void stop_tcn_timer()

	{
	tcn_timer.active = False;
	}

void stop_topology_change_timer()

	{
	topology_change_timer.active = False;
	}
/*
 * name		tick	- the one second timer
 *
 * synopsis	tick(StpTimer)
 *		TIMER	*StpTimer; <<	the timer structure for this timer
 *					service routine
 *
 * description	This is the stp main timer service function.  The timer
 *		expires every second.  The function polls all other timing
 *		structure to see if they have expired. The appropriate 
 *		functions will be called to service the expiry of the timers.
 *		This main timer service function will be re-started before
 *		returning.
 *
 * returns	nothing
 */

void tick (StpTimer)
TIMER	*StpTimer;

	{
	Int	port_no;

	if (hello_timer_expired())
		{
		hello_timer_expiry();
		}
	if (tcn_timer_expired())
		{
		tcn_timer_expiry();
		}
	if (topology_change_timer_expired())
		{
		topology_change_timer_expiry();
		}

	for (port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		if (forward_delay_timer_expired(port_no))
			{
			forward_delay_timer_expiry(port_no);
			}
		if (message_age_timer_expired(port_no))
			{
			message_age_timer_expiry(port_no);
			}
		if (hold_timer_expired(port_no))
			{
			hold_timer_expiry(port_no);
			}
		}
	StartTimerCall(StpTimer, 100, tick, (int)StpTimer);	/* restart timer */
	}


void set_port_priority(port_no, new_port_id)
Int	port_no;
Int	new_port_id;

	{
	if(designated_port(port_no))
		{
		port_info[port_no].designated_port = (new_port_id << 8) | (port_no);
		}

	port_info[port_no].port_id = (new_port_id << 8) | (port_no);
	if ((equal_test(&bridge_info.bridge_id, 
		&port_info[port_no].designated_bridge)) &&
		(port_info[port_no].port_id < 
			port_info[port_no].designated_port))
		{
		become_designated_port(port_no);
		port_state_selection();
		}
	}


void set_path_cost(port_no, path_cost)
Int	port_no;
Int	path_cost;

	{
	port_info[port_no].path_cost = path_cost;
	configuration_update();
	port_state_selection();
	}

void enable_port(port_no)
Int	port_no;

	{
	char	buffer[80];

	LedsOn(1 << (port_no -1));
	fls_arp_cache();
	initialize_port(port_no);
	port_state_selection();
	}

static Boolean equal_test(id1, id2)
Identifier	*id1;
Identifier	*id2;

	{
	return((id1->priority == id2->priority) && 
		(ncompare(id1->stp_address, id2->stp_address)));
	}

/*
 * name		supersedes_port_info
 *
 * synopsis	Boolean supersedes_port_info(port_no, config)
 *		Int		port_no; << 	received bpdu from this port
 *		Config_bpdu	*config; <<	the configuration bpdu
 *
 * description	It checks if the configuration bpdu conveying protocol
 *		information that supersedes that already held, i.e., if
 *		
 *		The root id denotes a bridge of higher priority
 *		than recorded as the designated root, or
 *
 *		the root id is the same as the designated root, and the 
 *		root path cost is lower than that recorded as the designated
 *		cost for the port, or
 *
 *		the root id and root path cost are as recorded for the port, 
 *		and the bridge id denotes a bridge of higher priority than
 *		that recorded as the designated bridge for the port, or
 *
 *		the root id and root path cost are as recorded for the port,
 *		and the bridge id is the same as that recorded as the 
 *		designated bridge for the port, and either
 *		1	the bridge receiving the bpdu is not the designated
 *			bridge for the port, or
 *		2	the port id denotes a port of priority no less than
 *			that recorded as the designated port.
 *
 * returns 	TRUE	if the configuration bpdu conveying protocol
 *			information that supersedes that already held.
 *		FALSE	otherwise
 */

static Boolean supersedes_port_info(port_no, config)
Int		port_no;
Config_bpdu	*config;
{
	Cost	root_path_cost;

	/*
	 *	Copy the root path cost out from the packet before
	 *	doing any operation on it. This is to solve the 
	 *	alignment problem.
	 */
	RootPathCostCopy((ushort *)&root_path_cost, config->root_path_cost);
	return( (port_test((Identifier *)&config->root_id_priority, 
			&port_info[port_no].designated_root))
		    ||
		    ( (equal_test((Identifier *)&config->root_id_priority, 
		    		&port_info[port_no].designated_root))
		      &&
		      ( (root_path_cost < port_info[port_no].designated_cost)
			    ||
			    ( (root_path_cost == port_info[port_no].designated_cost)
			      &&
			      ( (port_test((Identifier *)&config->bridge_id_priority,
					&port_info[port_no].designated_bridge))
			        ||
			        ( (equal_test((Identifier *)&config->bridge_id_priority,
				                  &port_info[port_no].designated_bridge))
					  && 
			          ( ( (config->bridge_id_priority !=
					       bridge_info.bridge_id.priority)
						  ||
				          (!(ncompare(config->bridge_id_stp_address,
						     bridge_info.bridge_id.stp_address)))
					  )
					  ||
			          (config->port_id <= port_info[port_no].designated_port)
		  ) ) ) ) ) ) );
}


static void record_config_information(port_no, config)
Int		port_no;
Config_bpdu	*config;

	{
	port_info[port_no].designated_root.priority = config->root_id_priority;
	ncopy(port_info[port_no].designated_root.stp_address,
		config->root_id_stp_address);
	/*****
	port_info[port_no].designated_cost = config->root_path_cost;
	*******/
	RootPathCostCopy((ushort *)&port_info[port_no].designated_cost,
		config->root_path_cost);
	port_info[port_no].designated_bridge.priority = config->bridge_id_priority;
	ncopy(port_info[port_no].designated_bridge.stp_address,
		config->bridge_id_stp_address);
	port_info[port_no].designated_port = config->port_id;
	start_message_age_timer(port_no, config->message_age);
	}


static void record_config_timeout_values(config)
Config_bpdu	*config;

	{
	bridge_info.max_age = config->max_age;
	bridge_info.hello_time = config->hello_time;
	bridge_info.forward_delay = config->forward_delay;
	/*
	 *	K Kong	9.10.91
	 *	If there has been a change in active topology
	 *	in part of the bridged lan, the age time should be
	 *	reduced in order to limit the effects of temporary
	 *	isolation of end systems attached to teh bridged lan
	 *	brought about by the use of incorrect information in the
	 *	filter database.
	 */
	if (config->topology_change_flag & change_true)
		{
		bridge_info.topology_change = True;
		SetAgeTimeToForwardDelay();
		}
	else
		{
		bridge_info.topology_change = False;
		RestoreAgeTime();
		}
	}


static void reply(port_no)
Int	port_no;

	{
	transmit_config(port_no);
	}


static void transmit_tcn()

	{
	Int	port_no;

	port_no = bridge_info.root_port;
	tcn_bpdu[port_no].type = Tcn_bpdu_type;

	StpTopChgTxBPDU[port_no]++;
	/********
	printf("Sending tcn bpdu on port %d\n", port_no);
	************/
	SendFromSTP(port_no, &tcn_bpdu[bridge_info.root_port], TCN_BPDU_SIZE);
	}

/*
 * name		port_test
 *
 * synopsis	Boolean  port_test(id3, id4)
 *		Identifier	*id3; <<	1st port id to be compared
 *		Identifier	*id4; <<	2nd port id to be compared 
 *
 * description	It checks the relative priority of the ports id3 and id4.
 *		The higher priority id has the lower numerical value.
 *
 * returns	TRUE 	id3 has higher priority than id4
 *		FALSE	id4 has higher priority than id3
 */

static Boolean  port_test(id3, id4)
Identifier	*id3;
Identifier	*id4;

	{
	if  (id3->priority < id4->priority)
		return True;
	if  (id3->priority == id4->priority)
		if (memcmp(id3->stp_address, id4->stp_address, 6) < 0)
			return True;
	return False;
	}

/*
 * name		root_selection
 *
 * synopsis	void root_selection()
 *
 * description	To select the designated root and the root port, and to
 *		calculate the root path cost for this bridge.
 *		The root port is set to identify the prot which; among those
 *		that are not the designated port for the LAN to which they
 *		are attached, are not disabled, and have a designated root
 *		parameter of higher priority that the bridge's bridge id;
 *		1	has the highest priority root associated with it, i.e.
 *		recorded as the designated root for the port.
 *		2	of two or more ports with the highest priority
 *		designated root parameter, has the lowest root path
 *		cost associated with it, i.e., the lowest sum of the 
 *		designated cost and path cost parameters for any port, or
 *		3	of two of more ports with the highest priority
 *		designated root parameter and lowest value of associated
 *		root path cost, has the highest priority bridge id recorded
 *		as the designated bridge for the LAN to which the port is
 *		attached, or
 *		4	of two of more ports with the highest priority 
 *		designated root parameter, lowest value of associated root
 *		path cost, and highest priority designated bridge, has the
 *		highest priority port id recorded as the designated port for
 *		the LAN for which the prot is attached, or
 *		5	of tow of more ports with the highest priority
 *		designated root parameter, lowest value of associated root
 *		path cost, and highest priority designated bridge and
 *		designated port, has the highest priority port id.
 *		
 *		If there is no such port, the value of the root port 
 *		parameter is et to zero, and
 *		1	the designated root parameter held by the bridge is
 *		set to the bridge id parameter held for the bridge, and
 *		2	the value of the root path cost parameter held by
 *		the bridge is set to zero. Othereise, i.e., if one of the
 *		bridge ports has been identified as the root port, then
 *		1	the designated root parameter held by the bridge is
 *		set to the designated root parameter held for the root 
 *		port, and
 *		2	the value of the root path cost parmeter held by 
 *		the bridge is set to the value of the root path cost
 *		parameter associated with the root port, i.e., the sume of 
 *		the values of the path cost and the designated cost
 *		parameters recorded for the root port.
 *
 * returns	nothing
 */

static void root_selection()
{
	Int root_port;
	Int port_no;

	root_port = No_port;

	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
	{
		if( ( (!designated_port(port_no))
			  &&
			  (port_info[port_no].state != Disabled)
			  &&
		      (port_test(&port_info[port_no].designated_root,
			             &bridge_info.bridge_id))
			)
			&&
		    ( (root_port == No_port)
			  || 
		      (port_test(&port_info[port_no].designated_root,
			             &port_info[root_port].designated_root))
			  ||
			  ( (equal_test(&port_info[port_no].designated_root,
			                &port_info[root_port].designated_root)
			    )
				&&
			    ( ( (port_info[port_no].designated_cost +
			         port_info[port_no].path_cost) <
			        (port_info[root_port].designated_cost +
			         port_info[root_port].path_cost)
			      )
				  ||
			      ( ( (port_info[port_no].designated_cost +
			           port_info[port_no].path_cost) ==
				      (port_info[root_port].designated_cost +
				       port_info[root_port].path_cost)
			        )
					&&
			        ( (port_test(&port_info[port_no].designated_bridge,
						&port_info[root_port].designated_bridge)
					  )
					  ||
				      ( (equal_test(&port_info[port_no].designated_bridge,
				                    &port_info[root_port].designated_bridge)
						)
						&&
				        ( (port_info[port_no].designated_port <
				           port_info[root_port].designated_port)
						  ||
				          ( (port_info[port_no].designated_port ==
				             port_info[root_port].designated_port)
							&&
				            (port_info[port_no].port_id <
			                 port_info[root_port].port_id)
          ) ) ) ) ) ) ) ) )
		{
			root_port = port_no;
		}
	}

	bridge_info.root_port = root_port;

	if(root_port == No_port)
	{
		bridge_info.designated_root.priority = bridge_info.bridge_id.priority;

		ncopy(bridge_info.designated_root.stp_address,
			bridge_info.bridge_id.stp_address);

		bridge_info.root_path_cost = Zero;
	}
	else
	{
		bridge_info.designated_root.priority = 
			port_info[root_port].designated_root.priority;

		ncopy(bridge_info.designated_root.stp_address,
			port_info[root_port].designated_root.stp_address);

		bridge_info.root_path_cost = (port_info[root_port].designated_cost +
		                              port_info[root_port].path_cost);
	}
}


static void designated_port_selection()
	{
	
	Int	port_no;

	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		if(designated_port(port_no)
		   ||
           	( 
		     (port_info[port_no].designated_root.priority != 
		      bridge_info.designated_root.priority
			 )
			 ||
			 (!(ncompare(port_info[port_no].designated_root.stp_address, bridge_info.designated_root.stp_address))
			 )
		   )
		   ||
		   (bridge_info.root_path_cost <
		    port_info[port_no].designated_cost
		   )
		   ||
           ( (bridge_info.root_path_cost ==
			  port_info[port_no].designated_cost)
			 &&
			 ( (port_test(&bridge_info.bridge_id,
			              &port_info[port_no].designated_bridge)
			   )
			   ||
			   ((equal_test(&bridge_info.bridge_id,
			                &port_info[port_no].designated_bridge)
				)
				&&
				(port_info[port_no].port_id <=
				   port_info[port_no].designated_port)
		 ) ) ) )
			{
			become_designated_port(port_no);
			}
		}
	}

/*
 * name		become_designated_port
 *
 * synopsis	become_designated_port(port)
 *		int	port; <<	make this port to be the designated
 *					port.  port starts from 1.
 *
 * description	It sets up the "port" to be the designated port.
 *
 * returns	nothing.
 */

static void become_designated_port(port_no)
Int port_no;

	{
	port_info[port_no].designated_root.priority =
		bridge_info.designated_root.priority;
	ncopy(port_info[port_no].designated_root.stp_address, 
		bridge_info.designated_root.stp_address);
	port_info[port_no].designated_cost = bridge_info.root_path_cost;
	port_info[port_no].designated_bridge.priority =
		bridge_info.bridge_id.priority;
	ncopy(port_info[port_no].designated_bridge.stp_address, 
		bridge_info.bridge_id.stp_address);
	port_info[port_no].designated_port = port_info[port_no].port_id; 
	}

static void make_forwarding(port_no)
Int	port_no;

	{
	if(port_info[port_no].state == Blocking)
		{
		set_port_state(port_no, Listening);
		start_forward_delay_timer(port_no);
		}
	}


static void make_blocking(port_no)
Int	port_no;

	{
	if ((port_info[port_no].state != Disabled) &&
		(port_info[port_no].state != Blocking))
		{
		if((port_info[port_no].state == Forwarding) ||
			(port_info[port_no].state == STPLearning))
			{
			topology_change_detection();
			}
		set_port_state(port_no, Blocking);
		stop_forward_delay_timer(port_no);
		}
	}

/*
 * name		set_port_state
 *
 * synopsis	set_port_state(port_no, state)
 *		int	port_no; <<	which port, starting from 1
 *		int	state; <<	new state to be set.
 *					This must be one of the following
 *					-	Disabled
 *					-	Blocking
 *					-	STPLearning
 *					-	forwarding
 *					-	Listening
 *
 * description	It sets the "port_no" to the new state "state".
 *
 * returns	nothing
 */

static void set_port_state(port_no, state)
Int	port_no;
State	state;

	{
	if((port_info[port_no].state == STPLearning && state == Forwarding) ||
		(port_info[port_no].state == Forwarding && state == Blocking))
	{
		STP_alarm(2);	
	}
	port_info[port_no].state = state;

	/*
	 *	Set the state in the packet processing data area
	 */
	SetProcState(port_no, state);
	}

static void topology_change_acknowledged()

	{
	bridge_info.topology_change_detected = False;
	stop_tcn_timer();
	}


static void acknowledge_topology_change(port_no)
Int	port_no;

	{
	port_info[port_no].topology_change_acknowledge = True;
	transmit_config(port_no);
	}

static void hello_timer_expiry()

	{
	config_bpdu_generation();
	start_hello_timer();
	}


static void message_age_timer_expiry(port_no)
Int	port_no;

	{
	Boolean	root;

	root= root_bridge();
	become_designated_port(port_no);
	configuration_update();
	port_state_selection();
	if((root_bridge()) && (!root))
		{
		bridge_info.max_age = bridge_info.bridge_max_age;
		bridge_info.hello_time = bridge_info.bridge_hello_time;
		bridge_info.forward_delay = bridge_info.bridge_forward_delay;
		topology_change_detection();
		stop_tcn_timer();
		config_bpdu_generation();
		start_hello_timer();
		}
	}


static void forward_delay_timer_expiry(port_no)
Int	port_no;
	{
	if(port_info[port_no].state == Listening)
		{
		set_port_state(port_no, STPLearning);
		start_forward_delay_timer(port_no);
		}
	else if(port_info[port_no].state == STPLearning)
		{
		StpForwardTransitions[port_no]++;
		set_port_state(port_no, Forwarding);
		if(designated_for_some_port())
			{
			topology_change_detection();
			}
		}
	}


static Boolean  designated_for_some_port()
	{

	Int	port_no;

	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		if(equal_test(&port_info[port_no].designated_bridge,
		              &bridge_info.bridge_id))
			{
			return(True);
			}
		}
	return(False);
	}


static void tcn_timer_expiry()

	{
	transmit_tcn();
	start_tcn_timer();
	}


static void topology_change_timer_expiry()

	{
	bridge_info.topology_change_detected = False;
	bridge_info.topology_change = False;

	STP_alarm(1);
	}


static void hold_timer_expiry(port_no)
Int	port_no;

	{
	if(port_info[port_no].config_pending)
		{
		transmit_config(port_no);
		}
	}

static void stop_hello_timer()

	{
	hello_timer.active = False;
	}


static Boolean  hello_timer_expired()

	{
	if(hello_timer.active && 
		((++hello_timer.value) >= bridge_info.hello_time))
		{
		hello_timer.active = False;
		return(True);
		}
	return(False);
	}


static void start_tcn_timer()

	{
	tcn_timer.value = (StpTime) Zero;
	tcn_timer.active = True;
	}


static Boolean  tcn_timer_expired()

	{
	if(tcn_timer.active && 
	   ((++tcn_timer.value) >= bridge_info.bridge_hello_time))
		{	
		tcn_timer.active = False;
		return(True);
		}
	return(False);
	}


static void start_topology_change_timer()

	{
	topology_change_timer.value = (StpTime) Zero;
	topology_change_timer.active = True;
	}

static Boolean  topology_change_timer_expired()

	{
	if(topology_change_timer.active && 
		((++topology_change_timer.value) >= 
			bridge_info.topology_change_time))
		{
		topology_change_timer.active = False;
		return(True);
		}
	return(False);
	}


static void start_message_age_timer(port_no, message_age)
Int	port_no;
StpTime	message_age;

	{
	message_age_timer[port_no].value = message_age;
	message_age_timer[port_no].active = True;
	}


static void stop_message_age_timer(port_no)
Int	port_no;

	{
	message_age_timer[port_no].active = False;
	}


static Boolean  message_age_timer_expired(port_no)
Int	port_no;

	{
	if(message_age_timer[port_no].active && 
	  ((++message_age_timer[port_no].value) >= bridge_info.max_age))
		{
		message_age_timer[port_no].active = False;
		return(True);
		}
	return(False);
	}


static void start_forward_delay_timer(port_no)
Int	port_no;

	{
	forward_delay_timer[port_no].value = Zero;
	forward_delay_timer[port_no].active = True;
	}


static void stop_forward_delay_timer(port_no)
Int	port_no;

	{
	forward_delay_timer[port_no].active = False;
	}


static Boolean  forward_delay_timer_expired(port_no)
Int	port_no;

	{
	if(forward_delay_timer[port_no].active && 
		((++forward_delay_timer[port_no].value) >= 
			bridge_info.forward_delay))
		{
		forward_delay_timer[port_no].active = False;
		return(True);
		}
	return(False);
	}


static void start_hold_timer(port_no)
Int	port_no;

	{
	hold_timer[port_no].value = Zero;
	hold_timer[port_no].active = True;
	}


static void stop_hold_timer(port_no)
Int	port_no;

	{
	hold_timer[port_no].active = False;
	}


static Boolean  hold_timer_expired(port_no)
Int	port_no;

	{
	if(hold_timer[port_no].active && 
		((++hold_timer[port_no].value) >= bridge_info.hold_time))
		{
		hold_timer[port_no].active = False;
		return(True);
		}
	return(False);
	}

static void RootPathCostCopy(ushort *to, ushort *from)

	{
	*to++ = *from++;
	*to = *from;
	}
/*
 *	put the config bpdu in the host order and the times in seconds.
 */
static void ntoh_config_bpdu(Config_bpdu *config)

	{
	Cost	root_path_cost;

	config->root_id_priority = ntohs(config->root_id_priority);
	/*****	K Kong, alignment problem !!
	config->root_path_cost = ntohl(config->root_path_cost);
	******/
	RootPathCostCopy((ushort *)&root_path_cost, config->root_path_cost);
	root_path_cost = (Cost) ntohl(root_path_cost);
	RootPathCostCopy(config->root_path_cost, 
		(ushort *)&root_path_cost);
	config->bridge_id_priority = ntohs(config->bridge_id_priority);
	config->port_id = ntohs(config->port_id);
	config->message_age = 
		ntohs(config->message_age) / STP_TIMER_UNIT;
	config->max_age = 
		ntohs(config->max_age) / STP_TIMER_UNIT;
	config->hello_time = 
		ntohs(config->hello_time) / STP_TIMER_UNIT;
	config->forward_delay = 
		ntohs(config->forward_delay) / STP_TIMER_UNIT;
	}

STP_alarm(alarm)
int alarm;
{
	void SendTopologyChangeTrap();
	void SendNewRootTrap();

	if(alarm == 1)
	{
		StopTimer(&TopologyChangeTimer);
		StartTimerCall((TIMER *)&NewRootTimer,
				MAX_FORWARD_DELAY * 150,SendNewRootTrap,
				alarm);
	}
	else
	{
		StartTimerCall((TIMER *)&TopologyChangeTimer,
				MAX_FORWARD_DELAY * 150,SendTopologyChangeTrap,
				alarm);
	}
			
}

/*
**
**	SendTopologyChangeTrap(), gets invoked at the end of MAX_FORWARD_DELAY
**	time.  This will send a trap if the newroot trap is not sent.
**
*/
SendTopologyChangeTrap(alarm)
{
	send_stp_trap(alarm);
}
/*
**
**	SendNewRootTrap(), gets invoked at the end of MAX_FORWARD_DELAY
**	time.  This will send a trap if the newroot trap is not sent.
**
*/
SendNewRootTrap(alarm)
{
	StopTimer(&TopologyChangeTimer);
	send_stp_trap(alarm);
}
