#include <eeprecs.h>
#include <nvrecs.h>
#include <menuparse.h>
#include "fddihdr.h"
#include "smtmacro.h"
#include "edfddinvr.h"
#include "ctype.h"

extern int boot_smt_default(int, int);
extern int boot_edsmt(int, int);
extern int boot_edmac(int, int);
extern int boot_edpath(int, int);
extern int boot_edport(int, int);
extern int boot_edattach(int, int);
extern int fddinvr_update(int, int);
extern int fddiboot_exit(int, int);
extern int QueryMenuItem(char*, int, int);
extern int QueryMenu(uChar, char**, char *);
extern unsigned QueryUnsigned(char *, unsigned, unsigned, unsigned);

extern const char *edfddi_boot_help_table[];
extern const char *dfddi_boot_help_table[];

extern char *FDDIPath[];
extern char *FDDIPathAvail[];
extern char *SMTConfigCapability[];
extern char *PortFotx[];

#define STATUS_EDIT_FDDI           1
#define STATUS_DISPLAY_FDDI        2

static int edit_status;

const char
edfddi_help_all[] =
	".                - repeat previous command\n"
	"? [cmd]          - help info for optional cmd\n"
	"he[lp] [cmd]     - help info for optional cmd\n"
        "de[fault]        - set all of SMT attributes to default values\n"
	"sm[t]            - modify the FDDI SMT boot record\n"
	"ma[c]            - modify the FDDI MAC boot record\n"
	"pa[th]           - modify the FDDI PATH boot record\n"
	"po[rt]           - modify the FDDI PORT boot record\n"
	"at[tach]         - modify the FDDI ATTACHMENT boot record\n"
        "up[date]         - Update the FDDI NVRAM Record\n"
	"ex[it]           - exit the FDDI Boot Record editor\n"
	;

const char
dfddi_help_all[] =
	".                - repeat previous command\n"
	"? [cmd]          - help info for optional cmd\n"
	"he[lp] [cmd]     - help info for optional cmd\n"
	"sm[t]            - Display the FDDI SMT boot record\n"
	"ma[c]            - Display the FDDI MAC boot record\n"
	"pa[th]           - Display the FDDI PATH boot record\n"
	"po[rt]           - Display the FDDI PORT boot record\n"
	"at[tach]         - Display the FDDI ATTACHMENT boot record\n"
        "up[date]         - Update the FDDI NVRAM Record\n"
	"ex[it]           - exit the FDDI Boot Record editor\n"
	;

static const struct cmd 
   boot_edfddinvr_tbl[] = 
   {
     {"de",    boot_smt_default, 0, 0 },
     {"sm",    boot_edsmt,    (int) BOOT_EDIT, "s" },
     {"ma",    boot_edmac,    (int) BOOT_EDIT, "s" },
     {"pa",    boot_edpath,   (int) BOOT_EDIT, "s" },
     {"po",    boot_edport,   (int) BOOT_EDIT, "s" },
     {"at",    boot_edattach, (int) BOOT_EDIT, "s" },
     {"up",    fddinvr_update, 0, 0},
     {"ex",    fddiboot_exit, 0, 0 },
     {"?",     menuparse_help, (int) edfddi_boot_help_table, "s" },
     {"he",    menuparse_help, (int) edfddi_boot_help_table, "s" },
     { NULL, 0}
   };


static const struct cmd 
   boot_display_fddinvr_tbl[] = 
   {
     {"sm",    boot_edsmt,    (int) BOOT_DISPLAY, "s" },
     {"ma",    boot_edmac,    (int) BOOT_DISPLAY, "s" },
     {"pa",    boot_edpath,   (int) BOOT_DISPLAY, "s" },
     {"po",    boot_edport,   (int) BOOT_DISPLAY, "s" },
     {"at",    boot_edattach, (int) BOOT_DISPLAY, "s" },
     {"up",    fddinvr_update, 0, 0},
     {"ex",    fddiboot_exit, 0, 0 },
     {"?",     menuparse_help, (int) dfddi_boot_help_table, "s" },
     {"he",    menuparse_help, (int) dfddi_boot_help_table, "s" },
     { NULL, 0}
   };

static char *SMTConfigPolicy[CT_SMTCONFIG_POLICY]
             = { "None", "Use the Hold flag", "Use the CF_Wrap_AB flag",
		 "Use the Hold flag and the CF_Wrap_AB flag" };

static char *SMTConnectionPolicy[CT_CONNECTION_POLICY] 
             = { "A-A", "A-B", "A-S", "A-M", 
		 "B-A", "B-B", "B-S", "B-M", 
		 "S-A", "S-B", "S-S", "S-M", 
		 "M-A", "M-B", "M-S", "M-M" };

static char *FrameStatus[CT_FRAME_STATUS]
             = {"FSC-Type0: MAC repeats A/C indicators as received",
		"FSC-Type1: MAC sets C but not A on copying or forwarding",
		"FSC-Type2: NAC resets C and sets A on C set and A reset if frame is not copied" };

static char *DupAddrPolicy[CT_DUP_ADDR_POLICY]
             = {"address change actions",
                "JAM A actions", "JAM B actions", "leave ring actions" };
static char *PortConnectionPolicy[CT_PORT_CONNPOLICY]
             = { "PC-MAC-LCT", "PC-MAC-Loop", "PC-MAC-Placement" };

static char PortType[CT_PORT_TYPE] = "ABSM";
static char *PortTypeA[CT_PORT_TYPE] = { "A","" , "S", "M" };
static char *PortTypeB[CT_PORT_TYPE] = { "" , "B", "S", "M" };

static char *PolicyStatus1[CT_BILOGIC] = { "Disable", "Enable" };

static char *PolicyStatus2[CT_BILOGIC] = { "Accepted", "Rejected" };

static char *IsExisting[CT_BILOGIC] = { "not present", "present" };

static uInt32 Power2[CT_POWER2]
             = { 0x01,       0x02,       0x04,       0x08,
		 0x10,       0x20,       0x40,       0x80,
		 0x100,      0x200,      0x400,      0x800, 
		 0x1000,     0x2000,     0x4000,     0x8000, 
		 0x10000,    0x20000,    0x40000,    0x80000, 
		 0x100000,   0x200000,   0x400000,   0x800000, 
		 0x1000000,  0x2000000,  0x4000000,  0x8000000, 
		 0x10000000, 0x20000000, 0x40000000, 0x80000000 };
 
extern char config_buffer[TEMP_BUF_SIZE];

NVR_FDDI  my_nvr_fddi_rec;
NVR_FDDI *FddiRec = &my_nvr_fddi_rec;
byte nvr_fddi_dirty;

int log2 ( num )
uInt32 num;
{
   int i;

   if (num)
   {
      for (i=-1; num ; i++)
      {
	 num>>=1;
      }
      return i;
   }
   else
   {
      return 0;
   }
}

int SMTnvr_display (dummy1, argc)
int dummy1;
int argc;
{
   char buffer[100];
   uChar i, j, mask;

   printf("\n\n\t\t\tThe FDDI Concentrator SMT Boot Record\n\n");

   printf("%2d) Exit.\n", 1);

   printf("%2d) Manufacturer's OUI  = (%02x.%02x.%02x)\n",(BOOT_MASK & BOOT_FDDI_MANUOUI),
         NVR_FDDI_SMT.manufacturerData.OUI[0],
         NVR_FDDI_SMT.manufacturerData.OUI[1],
         NVR_FDDI_SMT.manufacturerData.OUI[2]);

   printf("%2d) Manufacturer's DATA =  (", (BOOT_MASK & BOOT_FDDI_MANUDATA));
   for (i=0; i < 29; i++)
   {
      printf("%c", NVR_FDDI_SMT.manufacturerData.data[i]);
   }

   printf(").\n%2d) User Defined Information = (", 
	  (BOOT_MASK & BOOT_FDDI_USERINFO));
   for (i=0; i < 32; i++)
   {
      printf("%c", NVR_FDDI_SMT.userData[i]);
   }

   printf(")\n%2d) Configuration Policy = (%s)\n", 
	  (BOOT_MASK & BOOT_FDDI_CONFPOLICY),
	  SMTConfigPolicy[NVR_FDDI_SMT.configPolicy]);

   printf("%2d) Connection Policy =", (BOOT_MASK & BOOT_FDDI_CONNPOLICY));
   printf("\n   ");
   for (i=j=0; i < CT_CONNECTION_POLICY ; i++)
   {
      if ( ++j > CON_POLICY_COLUMN)
      {
         printf("\n   ");
         j = 1;
      }
      printf("%s:(%s)   ",  SMTConnectionPolicy[i],
	     (NVR_FDDI_SMT.connectionPolicy & Power2[i])
	            ? PolicyStatus2[TRUE]: PolicyStatus2[FALSE] );
   }

   printf("\n%2d) Status Report Frame's Report Limit = (%d).\n", 
	   (BOOT_MASK & BOOT_FDDI_REPORTLIMIT),
	   NVR_FDDI_SMT.reportLimit);
   printf("%2d) The timer T-Notify value = (%d) second(s).\n", 
	   (BOOT_MASK & BOOT_FDDI_T_NOTIFY),
	   NVR_FDDI_SMT.T_Notify);
   return 0;
}


int MACnvr_display (dummy1, argc)
int dummy1;
int argc;
{
   uChar macID;
   uInt16 i;
   int item;

   macID = argc;

   printf("\n\n\t\tThe FDDI Concentrator MAC %d's Boot Record\n\n", macID+1);

   printf("%2d) Exit.\n", 1);

   printf("%2d) The greatest lower bound of T_MAX = (%d) microseconds.\n", 
	  (BOOT_MASK & BOOT_MAC_T_MAX_GLBOUND), 
	  MFddi2Usecs(-NVR_FDDI_MAC[macID].T_MaxGreatestLowerBound));
   printf("%2d) The greatest lower bound of TVX = (%d) microseconds.\n",
	  (BOOT_MASK & BOOT_MAC_TVX_GLBOUND), 
	  MFddi2Usecs(-NVR_FDDI_MAC[macID].TVXGreatestLowerBound));
   printf("%2d) The PATH requested = (%s) path.\n", 
	   (BOOT_MASK & BOOT_MAC_PATHREQ), 
	   FDDIPath[log2(NVR_FDDI_MAC[macID].pathRequested)]);

   printf("%2d) MAC Address = (%02x", 
	  (BOOT_MASK & BOOT_MAC_ADDRESS),
	  NVR_FDDI_MAC[macID].SMTAddress[0]);
   for (i=1; i < MAC_LONG_ADDRESS_LEN; i++)
   {
      printf(".%02x", NVR_FDDI_MAC[macID].SMTAddress[i]);
   }
   printf(")\n");

   printf("%2d) T-Req timer value = (%d) microseconds.\n", 
	  (BOOT_MASK & BOOT_MAC_T_REQ), 
	  MFddi2Usecs(-NVR_FDDI_MAC[macID].T_Req));
   printf("%2d) T-Max timer value = (%d) microseconds.\n", 
	  (BOOT_MASK & BOOT_MAC_T_MAX), 
	  MFddi2Usecs(-NVR_FDDI_MAC[macID].T_Max));
   printf("%2d) TVX timer value = (%d) microseconds.\n", 
	  (BOOT_MASK & BOOT_MAC_TVX), 
	  MFddi2Usecs(-NVR_FDDI_MAC[macID].TVXValue));
   printf("%2d) T-Min timer value = (%d) microseconds.\n", 
	  (BOOT_MASK & BOOT_MAC_T_MIN), 
	  MFddi2Usecs(-NVR_FDDI_MAC[macID].T_Min));
   printf("%2d) Frame Status = (%s).\n", 
	   (BOOT_MASK & BOOT_MAC_FRFAME_STATUS),
	   FrameStatus[log2(NVR_FDDI_MAC[macID].frameStatus)]);
   printf("%2d) RMT address duplicate policy1: Perform (%s).\n", 
	   (BOOT_MASK & BOOT_MAC_ADDRDUP1), 
	   DupAddrPolicy[NVR_FDDI_MAC[macID].xdiRMTDupPolicy1 - 1]);
   printf("%2d) RMT address duplicate policy2: Perform (%s).\n", 
	   (BOOT_MASK & BOOT_MAC_ADDRDUP2), 
	   DupAddrPolicy[NVR_FDDI_MAC[macID].xdiRMTDupPolicy2 - 1]);
   printf("%2d) Frame Error Threshold = (%d).\n",
	  (BOOT_MASK & BOOT_MAC_FERR_THRESHOLD),
	  NVR_FDDI_MAC[macID].FrameErrorThreshold);
   printf("%2d) Frame Not Copied Threshold = (%d).\n",
	  (BOOT_MASK & BOOT_MAC_NOT_COPIED_THRESHOLD),
	  NVR_FDDI_MAC[macID].NotCopiedThreshold);

   return 0;
}

int PATHnvr_display (dummy1, argc)
int dummy1;
int argc;
{
/*   
 *   This part will be used if only when different PATHes are implemented 
 *   in XLNT.
    uChar mapping[MAX_PATH_COUNT];
   uChar pathID;
   uInt16 mask;
   int item;

   for (i=0, mask=eep_mfg_rec.eep_fddi.smtPathAvailable; 
	(mask) && (i<MAX_PATH_COUNT) ; mask>>=1, i++)
   {
      if (mask & 0x0001)
      {
	 mapping[i] = TRUE;
         printf("\n%2d) %s : Available.", (i+1), FDDIPath[i]);
      }
      else
      {
	 mapping[i] = FALSE;
         printf("\n%2d) %s : NonAvailable.", (i+1), FDDIPath[i]);
      }
   }

   if ((item = QueryMenuItem("\nPlease enter the Path number", 1,  
	 i)) <= 0)
   {
      return -1;
   }

   if (!mapping[--item])
   {
      printf("\nPath chosen is not available on this Concentrator currently.\n");
      return -1;
   }
  pathID = (uChar) argc;
 *
 *
 */
   printf("\n\n\t\tThe FDDI Concentrator PATH Boot Record\n\n");

   printf("%2d) Exit.\n", 1);

   printf("%2d) The lower bound of TVX = (%d) microseconds.\n", 
	   (BOOT_MASK & BOOT_PATH_TVX_LBOUND), 
	  MFddi2Usecs(-NVR_FDDI_PATH[0].TVXLowerBound));
   printf("%2d) The lower bound of T-Max = (%d) microseconds.\n", 
	   (BOOT_MASK & BOOT_PATH_T_MAX_LBOUND), 
	  MFddi2Usecs(-NVR_FDDI_PATH[0].T_MAXLowerBound)); 
   printf("%2d) Trace_Max  = (%d) microseconds.\n", 
	   (BOOT_MASK & BOOT_PATH_TRACE_MAX), 
	  MFddi2Usecs(-NVR_FDDI_PATH[0].Trace_MaxExpiration)); 

   return 0;
}

int PORTnvr_display (dummy1, argc)
int dummy1;
int argc;
{
   uChar portID;
   int i;

   portID = (uChar) argc;

   printf("\n\n\t\tThe FDDI Concentrator PORT %d's Boot Record\n\n", portID+1);

   printf("%2d) Exit.\n", 1);

   printf("%2d) Port Type = (%c) port.\n", 
	  (BOOT_MASK & BOOT_PORT_TYPE),
	  PortType[NVR_FDDI_PORT[portID].PC_Type]);

   printf("%2d) Connection Policy =\n   ", (BOOT_MASK & BOOT_PORT_CONNPOLICY));

   for (i=0; i < CT_PORT_CONNPOLICY; i++)
   {
      printf("%s:(%s).  ", PortConnectionPolicy[i],
	  (NVR_FDDI_PORT[portID].connectionPolicies & Power2[i])?
	   PolicyStatus1[1] : PolicyStatus1[0]);
   }
   printf("\n");

   printf("%2d) Path requested = (%s).\n", 
	  (BOOT_MASK & BOOT_PORT_PATHREQ),
	   FDDIPath[log2(NVR_FDDI_PORT[portID].pathRequested)]);
   printf("%2d) Time value for MAC Local Loop,T_Next(9) "
	  "= (%d) microseconds.\n", 
	  (BOOT_MASK & BOOT_PORT_T_NEXT9),
	   MFddi2Usecs(NVR_FDDI_PORT[portID].MACLoop_Time));
   printf("%2d) The LER Cutoff rate = (%d)\n", 
	  (BOOT_MASK & BOOT_PORT_LER_CUTOFF),
	  NVR_FDDI_PORT[portID].LER_Cutoff);
   printf("%2d) The LER Alarm rate = (%d)\n", 
	  (BOOT_MASK & BOOT_PORT_LER_ALARM),
	  NVR_FDDI_PORT[portID].LER_Alarm);
   printf("%2d) TB_Max "
	  "= (%d) microseconds.\n", 
	  (BOOT_MASK & BOOT_PORT_TB_MAX),
	   MFddi2Usecs(-NVR_FDDI_PORT[portID].TB_Max));
   return 0;
}

int ATTACHnvr_display (dummy1, argc)
int dummy1;
int argc;
{
   uChar attachID, totalAttach;
   int item;

   attachID = (uChar) argc;

   printf("\n\n\tThe FDDI Concentrator Port %d's Attachment Boot Record\n\n", 
	  attachID+1);

   printf("%2d) Exit.\n", 1);

   printf("%2d) Optical Bypass Switch does (%s).\n", 
	  (BOOT_MASK & BOOT_ATTACH_OPTBYPASS),
	  IsExisting[(NVR_FDDI_ATTACHMENT[attachID].opticalBypassPresent)]);
   return item;
}

void UpdateFDDInvr( typeID, entity )
uInt16 typeID;
uChar entity;
{
   uInt16 i;
   uInt32 temp;
   int item;

   switch (typeID & BOOT_TYPE_MASK)
   {
   case BOOT_FDDI_SMT:
      switch (typeID)
      {
      case BOOT_FDDI_MANUOUI:
	 printf("\n*** Cannot be changed ***\n");
	 break;
	 nvr_fddi_dirty = TRUE;
	 printf("Enter 3 hex numbers of the Manufacturer's OUI (%02x.%02x.%02x) ===>",
		NVR_FDDI_SMT.manufacturerData.OUI[0],
		NVR_FDDI_SMT.manufacturerData.OUI[1],
		NVR_FDDI_SMT.manufacturerData.OUI[2]);
         MEMZERO(config_buffer,3);
	 gets(config_buffer);
	 MEMCOPY(NVR_FDDI_SMT.manufacturerData.OUI, config_buffer, 3);
         break;

      case BOOT_FDDI_MANUDATA:
	 nvr_fddi_dirty = TRUE;
	 printf("Enter 29 character of the Manufacturer's Data\n(");
	 for (i=0; i < 29; i++)
	 {
	    printf("%c", NVR_FDDI_SMT.manufacturerData.data[i]);
         }
         printf(") ===>");
         MEMZERO(config_buffer,29);
	 gets(config_buffer);
	 for (i=0; i < 29; i++)
	 {
	    NVR_FDDI_SMT.manufacturerData.data[i] = 
	      (isprint(config_buffer[i])) ? config_buffer[i] : '.';
	 }
	 
         break;

      case BOOT_FDDI_USERINFO:
	 nvr_fddi_dirty = TRUE;
	 printf("Enter 32 characters of the user defined information\n"
		"(");
	 for (i=0; i < 32; i++)
	 {
	    printf("%c", NVR_FDDI_SMT.userData[i]);
         }
         printf(") ===>");
         MEMZERO(config_buffer,32);
	 gets(config_buffer);
	 for (i=0; i < 29; i++)
	 {
	    NVR_FDDI_SMT.userData[i] = 
	      (isprint(config_buffer[i])) ? config_buffer[i] : '.';
	 }
         break;

      case BOOT_FDDI_CONFPOLICY:
	 sprintf(config_buffer, "Please enter the number you want to have (%d)",
		 NVR_FDDI_SMT.configPolicy+1);
	 item = QueryMenu(CT_SMTCONFIG_POLICY, SMTConfigPolicy, config_buffer);
	 if (item > 0)
	 {
	    NVR_FDDI_SMT.configPolicy = --item;
	    nvr_fddi_dirty = TRUE;
	 }
         break;

      case BOOT_FDDI_CONNPOLICY:
	 printf("SMT Connection Policy ( PC_Type-PC_Neighbor ):\n");
	 for (i=0; i < CT_CONNECTION_POLICY; i++)
	 {
	    printf("%2d) %s: %s.\n", i+1, 
		   SMTConnectionPolicy[i], 
		   ((NVR_FDDI_SMT.connectionPolicy & Power2[i])? 
		   PolicyStatus2[TRUE] : PolicyStatus2[FALSE]));
	 }
	 item = QueryMenuItem("\nPlease enter the number to toggle the "
			      "configuration", 1, CT_CONNECTION_POLICY);
	 if (item > 0)
	 {
	    NVR_FDDI_SMT.connectionPolicy ^= Power2[--item];
	    nvr_fddi_dirty = TRUE;
	 }
         break;

      case BOOT_FDDI_REPORTLIMIT:
	 NVR_FDDI_SMT.reportLimit = 
	   QueryUnsigned("Please enter the maximum number of Status Report "
			 "frames held in the queue ",
			 NVR_FDDI_SMT.reportLimit, 1, 0);
         break;

      case BOOT_FDDI_T_NOTIFY: 
	 NVR_FDDI_SMT.T_Notify= 
	   QueryUnsigned("Please enter the timeout value for timer T_Notify "
			 "in seconds",
			 NVR_FDDI_SMT.T_Notify, T_NOTIFY_MIN, T_NOTIFY_MAX);
	 break;

      default:
         printf("\nFDDI SMT Boot Record %d is not found.\n",
		(typeID & BOOT_MASK));
         break;
      }
      break;
   /*
    * MAC
    */
   case BOOT_FDDI_MAC:
      switch (typeID)
      {
      case BOOT_MAC_T_MAX_GLBOUND:
	 temp = NVR_FDDI_MAC[entity].T_MaxGreatestLowerBound;
 	 NVR_FDDI_MAC[entity].T_MaxGreatestLowerBound = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the greatest lower bound of T_MAX "
		  "in microseconds",
		  MFddi2Usecs(-NVR_FDDI_MAC[entity].T_MaxGreatestLowerBound),
		  MAX(MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Min), 
		      T_MAX_MIN(MFddi2Usecs(-NVR_FDDI_MAC[entity].TVXValue))),
		  MFddi2Usecs(-NVR_FDDI_PATH[0].T_MAXLowerBound)));
	 if (temp == NVR_FDDI_MAC[entity].T_MaxGreatestLowerBound)
	 {
	    break;
	 }
	 temp = -NVR_FDDI_MAC[entity].T_MaxGreatestLowerBound;
	 if (temp < (-NVR_FDDI_MAC[entity].T_Max))
	 {	
	    printf("Updating the MAC %d's T_Max.\n", i);
	    NVR_FDDI_MAC[entity].T_Max =
	       NVR_FDDI_MAC[entity].T_MaxGreatestLowerBound;
	 }
         break;

      case BOOT_MAC_TVX_GLBOUND:
	 temp = NVR_FDDI_MAC[entity].TVXGreatestLowerBound;
 	 NVR_FDDI_MAC[entity].TVXGreatestLowerBound = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the greatest lower bound of TVX "
		  "in microseconds",
		  MFddi2Usecs(-NVR_FDDI_MAC[entity].TVXGreatestLowerBound),
		  TVX_MIN,
		  MFddi2Usecs(-NVR_FDDI_PATH[0].TVXLowerBound)));
	 if (temp == NVR_FDDI_MAC[entity].TVXGreatestLowerBound)
	 {
	    break;
	 }
	 temp = -NVR_FDDI_MAC[entity].TVXGreatestLowerBound;
	 if (temp < (-NVR_FDDI_MAC[entity].TVXValue))
	 {	
	    printf("Updating the MAC %d's TVX.\n", i);
	    NVR_FDDI_MAC[entity].TVXValue =
	      NVR_FDDI_MAC[entity].TVXGreatestLowerBound;
	 }
         break;

      case BOOT_MAC_PATHREQ:
	 sprintf(config_buffer, "Please enter the number of the path you requested "
		 "(%d)", log2(NVR_FDDI_MAC[entity].pathRequested)+1);
	 item = QueryMenu(CT_PATH, FDDIPath, config_buffer);
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_MAC[entity].pathRequested = Power2[item-1];
	 }
         break;

      case BOOT_MAC_ADDRESS:
	 nvr_fddi_dirty = TRUE;
	 sprintf(config_buffer, "Enter the MAC %d's Address ", entity);
	 AskNid(config_buffer, &NVR_FDDI_MAC[entity].SMTAddress);
/*
	 switch (entity+1)
	 {
	    printf("Set MAC %d's address to the default value.\n");
            case PRIMARY_MAC:
                 MCopyAddress(NVR_FDDI_MAC[entity].SMTAddress, MyNid(PRIMARY_MAC+1));
		 break;
            case SECONDARY_MAC:
		 MCopyAddress(NVR_FDDI_MAC[entity].SMTAddress, MyNid(SECONDARY_MAC+1));
		 break;
            default:
	 }
*/
         break;

      case BOOT_MAC_T_REQ:
 	 NVR_FDDI_MAC[entity].T_Req = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the value of "
			  "the timer T_REQ in microseconds",
			  MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Req),
			  MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Min),
			 MIN(T_REQ_MAX((MFddi2Usecs(-NVR_FDDI_MAC[entity].TVXValue))), 
			     MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Max))));
         break;

      case BOOT_MAC_T_MAX:
 	 NVR_FDDI_MAC[entity].T_Max = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the value of "
		   "the timer T_MAX in microseconds",
		   MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Max),
		   MAX(MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Min), 
		       T_MAX_MIN(MFddi2Usecs(-NVR_FDDI_MAC[entity].TVXValue))),
		   MFddi2Usecs(-NVR_FDDI_MAC[entity].T_MaxGreatestLowerBound)));
         break;

      case BOOT_MAC_TVX:
 	 NVR_FDDI_MAC[entity].TVXValue = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the value of "
		   "the timer TVX in microseconds",
		   MFddi2Usecs(-NVR_FDDI_MAC[entity].TVXValue),
		   TVX_MIN,
		   MFddi2Usecs(-NVR_FDDI_MAC[entity].TVXGreatestLowerBound)));
         break;

      case BOOT_MAC_T_MIN:
 	 NVR_FDDI_MAC[entity].T_Min = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the value of "
		   "the timer T_MIN in microseconds",
		   MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Min),
		   0, MIN(T_MIN_MAX, MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Max))));
         break;

      case BOOT_MAC_FRFAME_STATUS:
	 for (i=0; i < CT_FRAME_STATUS; i++)
	 {
	    printf("%2d) %s ---> %s.\n", i, FrameStatus[i], 
		   ((NVR_FDDI_MAC[entity].frameStatus & Power2[i])?
		    PolicyStatus1[TRUE] : PolicyStatus1[FALSE]));
	 }
	 item = QueryMenuItem("\nPlease enter the number to toggle the "
			      "configuration", 1, CT_FRAME_STATUS);
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_MAC[entity].frameStatus ^= Power2[--item];
	 }
         break;

      case BOOT_MAC_ADDRDUP1:
	 sprintf(config_buffer, "Please choose the Policy 1 for duplicate MAC Address"
		 " (%d)", NVR_FDDI_MAC[entity].xdiRMTDupPolicy1);
	 item = QueryMenu(CT_DUP_ADDR_POLICY, DupAddrPolicy, config_buffer);
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_MAC[entity].xdiRMTDupPolicy1 = item;
	 }
         break;

      case BOOT_MAC_ADDRDUP2:
	 sprintf(config_buffer, "Please choose the Policy 2 for duplicate MAC Address"
		 " (%d)", NVR_FDDI_MAC[entity].xdiRMTDupPolicy2);
	 item = QueryMenu(CT_DUP_ADDR_POLICY, DupAddrPolicy, config_buffer);
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_MAC[entity].xdiRMTDupPolicy2 = item;
	 }
         break;

      case BOOT_MAC_FERR_THRESHOLD:
 	 NVR_FDDI_MAC[entity].FrameErrorThreshold = 
	   QueryUnsigned("Please enter the value of "
		"the Frame Error Threshold",
		NVR_FDDI_MAC[entity].FrameErrorThreshold,
		0,
		0);
	 break;
	 
      case BOOT_MAC_NOT_COPIED_THRESHOLD:
 	 NVR_FDDI_MAC[entity].NotCopiedThreshold = 
	   QueryUnsigned("Please enter the value of "
		"the Frame Not Copied Threshold",
		NVR_FDDI_MAC[entity].NotCopiedThreshold,
		0,
		0);
	 break;

      default:
         printf("\nFDDI MAC Boot Record %d is not found.\n",
		(typeID & BOOT_MASK));
         break;
      }
      break;

   /*
    * PATH
    */
   case BOOT_FDDI_PATH:
      switch (typeID)
      {
      case BOOT_PATH_TVX_LBOUND:
	 temp = NVR_FDDI_PATH[0].TVXLowerBound;
 	 NVR_FDDI_PATH[0].TVXLowerBound = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the lower bound of TVX for all "
		  "pathes in microseconds",
		  MFddi2Usecs(-NVR_FDDI_PATH[0].TVXLowerBound),
		  TVX_MIN,
		  TVX_MAX));
	 if (temp == NVR_FDDI_PATH[0].TVXLowerBound)
	 {
	    break;
	 }
	 temp = -NVR_FDDI_PATH[0].TVXLowerBound;
	 for (i=0; i < eep_mfg_rec.eep_fddi.smtMac_Ct; i++)
	 {
	    if (temp > (-NVR_FDDI_MAC[i].TVXGreatestLowerBound))
	    {
	       printf("Updating the MAC %d's TVXGreatestLowerBound.\n", i);
	       NVR_FDDI_MAC[i].TVXGreatestLowerBound =
		             NVR_FDDI_PATH[0].TVXLowerBound;
	       if (temp > (-NVR_FDDI_MAC[i].TVXValue))
	       {	
		  printf("Updating the MAC %d's TVX.\n", i);
		  NVR_FDDI_MAC[i].TVXValue =
		             NVR_FDDI_PATH[0].TVXLowerBound;
	       }
	    }
	 }
         break;

      case BOOT_PATH_T_MAX_LBOUND:
	 temp = NVR_FDDI_PATH[0].T_MAXLowerBound;
 	 NVR_FDDI_PATH[0].T_MAXLowerBound = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the lower bound of T_MAX for all "
		  "pathes in microseconds",
		  MFddi2Usecs(-NVR_FDDI_PATH[0].T_MAXLowerBound),
 		  MAX(MFddi2Usecs(-NVR_FDDI_MAC[entity].T_Min), 
		      T_MAX_MIN(MFddi2Usecs(-NVR_FDDI_MAC[entity].TVXValue))),
		  T_MAX_MAX));
	 if (temp == NVR_FDDI_PATH[0].TVXLowerBound)
	 {
	    break;
	 }
	 temp = -NVR_FDDI_PATH[0].T_MAXLowerBound;

	 for (i=0; i < eep_mfg_rec.eep_fddi.smtMac_Ct; i++)
	 {
	    if (temp < (-NVR_FDDI_MAC[i].T_MaxGreatestLowerBound))
	    {
	       printf("Updating the MAC %d's T_MaxGreatestLowerBound.\n", i);
	       NVR_FDDI_MAC[i].T_MaxGreatestLowerBound =
		             NVR_FDDI_PATH[0].T_MAXLowerBound;
	       if (temp < (-NVR_FDDI_MAC[i].T_Max))
	       {	
		  printf("Updating the MAC %d's T_Max.\n", i);
		  NVR_FDDI_MAC[i].T_Max =
		             NVR_FDDI_PATH[0].T_MAXLowerBound;
	       }
	    }
	 }
         break;

      case BOOT_PATH_TRACE_MAX:
 	 NVR_FDDI_PATH[0].Trace_MaxExpiration = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the value of "
		"the Trace_MaxExpiration in microseconds",
		MFddi2Usecs(-NVR_FDDI_PATH[0].Trace_MaxExpiration),
		TRACE_MAX_MIN,
		0));
	 break;

      default:
         printf("\nFDDI PATH Boot Record %d is not found.\n",
		(typeID & BOOT_MASK));
         break;
      }
      break;

   /*
    * PORT
    */
   case BOOT_FDDI_PORT:
      switch (typeID)
      {
      case BOOT_PORT_TYPE:
	 if (entity > PHY_B)
	 {
	    printf("This port must stay as a 'M' port.\n");
	    break;
	 }

	 sprintf(config_buffer, "Please enter the number of the port type you want"
		 " (%d)", NVR_FDDI_PORT[entity].PC_Type+1);
	 item = QueryMenu(CT_PORT_TYPE, 
			  ((entity == PHY_A)? PortTypeA : PortTypeB), 
			  config_buffer);
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_PORT[entity].PC_Type = --item;
	 }
         break;

      case BOOT_PORT_CONNPOLICY:
	 printf("\n");
	 for (i=0; i < CT_PORT_CONNPOLICY; i++)
	 {
	    printf("%2d) %s: %s.\n", i+1, PortConnectionPolicy[i],
		   (NVR_FDDI_PORT[entity].connectionPolicies & Power2[i])?
		   PolicyStatus1[1] : PolicyStatus1[0]);

	 }
	 item = QueryMenuItem("Please enter the number of policy to toggle its"
			      " configuration", 1, CT_PORT_CONNPOLICY);
	 
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_PORT[entity].connectionPolicies ^= Power2[--item];
	 }
         break;

      case BOOT_PORT_PATHREQ:
	 sprintf(config_buffer, "Please enter the number of the path you requested "
		 "(%d)", log2(NVR_FDDI_PORT[entity].pathRequested)+1);
	 item = QueryMenu(CT_PATH, FDDIPath, config_buffer);
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_PORT[entity].pathRequested = Power2[item-1];
	 }
         break;

      case BOOT_PORT_T_NEXT9:
 	 NVR_FDDI_PORT[entity].MACLoop_Time = MUsecs2Fddi(
	   QueryUnsigned("Please enter the value of timer T_Next(9) "
			 "in microseconds",
			 MFddi2Usecs(NVR_FDDI_PORT[entity].MACLoop_Time),
			 T_NEXT9_MIN, 0));
         break;

      case BOOT_PORT_LER_CUTOFF:
	 NVR_FDDI_PORT[entity].LER_Cutoff =  
	   QueryUnsigned("Please enter the LER cutoff estimate in seconds",
			 NVR_FDDI_PORT[entity].LER_Cutoff, LER_MIN, LER_MAX);
         break;

      case BOOT_PORT_LER_ALARM:
	 NVR_FDDI_PORT[entity].LER_Alarm =  
	   QueryUnsigned("Please enter the LER cutoff estimate in seconds",
			 NVR_FDDI_PORT[entity].LER_Alarm, 
			 NVR_FDDI_PORT[entity].LER_Cutoff, LER_MAX);
         break;

      case BOOT_PORT_TB_MAX:
 	 NVR_FDDI_PORT[entity].TB_Max = -MUsecs2Fddi(
	   QueryUnsigned("Please enter the value of "
		 "the timer TVX in microseconds",
		 MFddi2Usecs(-NVR_FDDI_PORT[entity].TB_Max),
		 TB_MAX_MIN,
		 0));
	 break;

      default:
         printf("\nFDDI PORT Boot Record %d is not found.\n",
		(typeID & BOOT_MASK));
         break;
      }
      break;

   /*
    * ATTACHMENT
    */
   case BOOT_FDDI_ATTACH:
      switch (typeID)
      {
      case BOOT_ATTACH_OPTBYPASS:
	 sprintf(config_buffer, "%2d) Optical bypass switch dose %s.\n"
		 "Please enter the number to toggle its status", 1,
		 IsExisting[NVR_FDDI_ATTACHMENT[entity].opticalBypassPresent]);
	 item = QueryMenuItem(config_buffer, 1, 1);
	 if (item > 0)
	 {
	    nvr_fddi_dirty = TRUE;
	    NVR_FDDI_ATTACHMENT[entity].opticalBypassPresent = 
	      (NVR_FDDI_ATTACHMENT[entity].opticalBypassPresent)? FALSE : TRUE;
	 }
         break;

      default:
         printf("\nFDDI ATTACHMENT Boot Record %d is not found.\n",
		(typeID & BOOT_MASK));
         break;
      }
      break;

   default:
      break;
   } /* Switch Type ID */
}

int boot_edsmt( edit_mode, argc)
int edit_mode;
int argc;
{
   int item=2;

   while (item!=1)
   {
      SMTnvr_display(edit_mode, argc);

      if (*(uChar*)edit_mode == *BOOT_DISPLAY)
      {
	 
         item = QueryMenuItem("", 0, 0);
         continue;
      }
   
      if ((item=QueryMenuItem("\nPlease enter the number of the item you want "
			     "to modify ( 1 to quit) ", 1,
			     BOOT_FDDI_MAXNUM+1)) <= 1)
      {
         continue;
      }

      item |= BOOT_FDDI_SMT;

      UpdateFDDInvr( (uInt16) item , (uChar) DONT_CARE);
   }
   return 0;
}

int boot_edmac( edit_mode, argc)
int edit_mode;
int argc;
{
   int item, macID;

   if (!eep_mfg_rec.eep_fddi.smtMac_Ct)
   {
      printf("\nNo MAC is available on this Concentrator.\n");
      return 0;
   }

   if ((item = QueryMenuItem("\nPlease enter the MAC number", 1,  
			eep_mfg_rec.eep_fddi.smtMac_Ct)) <= 0)
   {
      printf("MAC no is not correct.\n");
      return 0;
   }

   macID = (uChar) item - 1;
   item = 2;


   while (item!=1)
   {
      MACnvr_display(edit_mode, macID);
   
      if (*(uChar*)edit_mode == *BOOT_DISPLAY)
      {
         item = QueryMenuItem("", 0, 0);
         continue;
      }
   
      if ((item=QueryMenuItem("\nPlease enter the number of the item you "
			     "want to modify ( 1 to quit) ", 1,
			     BOOT_MAC_MAXNUM+1)) <= 1)
      {
         continue;
      }

      item |= BOOT_FDDI_MAC;

      UpdateFDDInvr( (uInt16) item, (uChar) macID);
   }
   return 0;
}

int boot_edpath( edit_mode, argc)
int edit_mode;
int argc;
{
   int item=2;

   while (item!=1)
   {
      PATHnvr_display(edit_mode, argc);
   
      if (*(uChar*)edit_mode == *BOOT_DISPLAY)
      {
         item = QueryMenuItem("", 0, 0);
         continue;
      }
   
      if ((item=QueryMenuItem("\nPlease enter the number of the item you "
			     "want to modify ( 1 to quit) ", 1,
			     BOOT_PATH_MAXNUM+1)) <= 1)
      {
         continue;
      }

      item |= BOOT_FDDI_PATH;

      UpdateFDDInvr( (uInt16) item, (uChar) DONT_CARE);
   }
   return 0;
}

int boot_edport( edit_mode, argc)
int edit_mode;
int argc;
{
   int item, mask;
   uChar portID, i, totalPHY;

   totalPHY =   eep_mfg_rec.eep_fddi.smtMaster_Ct
	      + eep_mfg_rec.eep_fddi.smtNonmaster_Ct;

   for (i=0; i < totalPHY; i++)
   {
      printf("%2d) Port %d's Type = %c port.\n", (i+1), i,
	     PortType[NVR_FDDI_PORT[i].PC_Type]);
   }
   
   if ((item = QueryMenuItem("\nPlease enter the Port number", 1,  
			     totalPHY)) <= 0)
   {
      printf("Port number is not correct.\n");
      return 0;
   }
   portID = item - 1;
   item = 2;

   while (item!=1)
   {
      PORTnvr_display(edit_mode, portID);
   
      if (*(uChar*)edit_mode == *BOOT_DISPLAY)
      {
         item = QueryMenuItem("", 0, 0);
         continue;
      }
   
      if ((item=QueryMenuItem("\nPlease enter the number of the item you "
			     "want to modify ( 1 to quit) ", 1,
			     BOOT_PORT_MAXNUM+1)) <= 1)
      {
         continue;
      }

      item |= BOOT_FDDI_PORT;

      UpdateFDDInvr( (uInt16) item, (uChar) portID );
   }
   return 0;
}

int boot_edattach( edit_mode, argc)
int edit_mode;
int argc;
{
   int item, attachID;
   uChar totalAttach;

   totalAttach =   eep_mfg_rec.eep_fddi.smtMaster_Ct
	      + eep_mfg_rec.eep_fddi.smtNonmaster_Ct;

   if ((item = QueryMenuItem("\nPlease enter the port number attached "
			     "by the queried attachment", 1,  
			     totalAttach)) <= 0)
   {
      printf("The Attach number is not correct.\n");
      return 0;
   }
   attachID = item - 1;
   item = 2;

   while (item!=1)
   {
      ATTACHnvr_display(edit_mode, attachID);
   
      if (*(uChar*)edit_mode == *BOOT_DISPLAY)
      {
         item = QueryMenuItem("", 0, 0);
         continue;
      }
   
      if ((item=QueryMenuItem("\nPlease enter the number of the item you "
			     "want to modify ( 1 to quit) ", 1,
			     BOOT_ATTACH_MAXNUM+1)) <= 1)
      {
         continue;
      }

      item |= BOOT_FDDI_ATTACH;

      UpdateFDDInvr( (uInt16) item, (uChar) attachID );
   }
   return 0;
}

int boot_smt_default (dummy1, argc)
int dummy1;
int argc;
{
   printf("\nSet SMT attributes to default values\n\n");
   nvr_fddi_dirty = TRUE;
   setNVRAM_FDDIDefaults (FddiRec);
   return 0;
}

int boot_fddi_edit(dummy1, argc)
int dummy1;
int argc;
{
   GetFDDIRec(FddiRec);
   nvr_fddi_dirty = FALSE;
   edit_status = STATUS_EDIT_FDDI;
   MenuParse("\n\t\t FDDI Concentrator Boot Record Editor\n"
	     "\n\t\t Copyright (c) 1992\n\n", 
	     boot_edfddinvr_tbl);
   return 0;
}

int boot_fddi_display(dummy1, argc)
int dummy1;
int argc;
{
   edit_status = STATUS_DISPLAY_FDDI;
   MenuParse("\n\t\t FDDI Concentrator Boot Record Displayer\n"
	     "\n\t\t Copyright (c) 1992\n\n", 
	     boot_display_fddinvr_tbl);
   return 0;
}

int fddinvr_update(dummy1, argc)
{
   if (nvr_fddi_dirty)
   {
      PutFDDIRec(FddiRec);
      memcpy(&nvr_fddi_rec, &my_nvr_fddi_rec, sizeof(NVR_FDDI));
      nvr_fddi_dirty = FALSE;
      printf("\t\tFDDI Record is updated.\n");
   }
   else
   {
      printf("\t\tFDDI Record is not changed.\n");
   }
   return 0;
}


int fddiboot_exit(dummy1, argc)
{
   if (edit_status == STATUS_EDIT_FDDI)
   {
      if (nvr_fddi_dirty)
      {
         if (AskYesNo("Do you want to update the FDDI NVRAM Record?([y]/n)"))
	 {
	    printf("Updating the FDDI NVRAM Record.\n");
	    fddinvr_update(0,0);
	 }
	 else
	 {
	    printf("Discard changes made in the FDDI NVRAM Record.\n");
	 }
      }
      printf("\nLeave the FDDI boot record editor.\n\n");
   }
   else
   {
      printf("\nLeave the FDDI boot record displayer.\n\n");
   }
   return 1;
}


