/**			       
*
*	Program Name:	nim960 bridge
*
*	Filename:	tftpmain.c
*
*	$Log:   /b/gregs/bridge/tftpboot/tftpmain.c_v  $
 * 
 *    Rev 1.4   12 Oct 1993 09:16:40   franks
 * No change.
 * 
 *    Rev 1.3   29 Sep 1993 09:37:52   franks
 * No change.
 * 
 *    Rev 1.2   10 Sep 1993 15:14:06   franks
 * No change.
 * 
 *    Rev 1.1   08 Sep 1993 10:26:02   franks
 * No change.
 * 
 *    Rev 1.0   30 Jul 1993 13:27:38   franks
 * Initial revision.
 * 
 *    Rev 1.7   14 May 1992 19:30:36   kwok
 * Set the BurnFlash flag to be 1 if the flash has been burned.
 * 
 *    Rev 1.6   14 May 1992 10:59:32   kwok
 * Calling the function flush() to force output from printf rather than
 * modifing fault_cnt.
 * 
 *    Rev 1.5   13 May 1992 16:44:50   kwok
 * No change.
 * 
 *    Rev 1.4   13 May 1992 16:38:18   kwok
 * No change.
 * 
 *    Rev 1.3   13 May 1992 10:02:50   kwok
 * 
 *    Rev 1.2   11 May 1992 11:02:16   kwok
 * Do not disable interrupt before calling reset().
 * 
 *    Rev 1.1   10 Apr 1992 16:55:32   kwok
 * Read the Debug DIP rather than getting it from the internal
 * SYS data structure.
 * 
 *    Rev 1.0   30 Mar 1992 17:43:58   pvcs
 * Initial revision.
*
*	Creation Date:	2.13.91
*
*	Date:		
*
*	Version:	1.0
*
*	Programmers:	K Kong
*
*	Modifications:
*
*	Comments:	The tftp loader programme.
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/

#include <krnl.h>
#include <types.h>
#include <dips.h>
#include <sys.h>
#include <eeprecs.h>
#include <led.h>
#include <netbuf.h>
#include <error.h>
#include <tcpip.h>
#include <nim960h.h>
#include <udp.h>
#include <tftpboot.h>
#include <tftp.h>
#include <nvrecs.h>
#include <log.h>


#define	ENTRY_FROM_TFTP_MAGIC	0x54465450	/* "TFTP"	*/

extern void	*lmalloc(int);
extern int	TRxPacket(); 
byte	tolower();
int	BootImage(int);
int	IsSlip(void);
int	SlipBaud(void);

void	GetBootRecord(BOOT_BLOCK *boot);
int	CheckSumOk(ushort *buffer, long size);
int	ProgramFlash(void *feprom, byte *data, ulong length);
void	JumpToNewProgram(unsigned *);
void	send_sysctl();


static BOOT_BLOCK	boot_data;
static SaveFlashBurnStatus();
static CopyToFlash(byte *data, int length);

/*
 * name		tftpmain	- The entry for tftp loader
 *
 * synopsis	tftpmain(int mode)
 *		int	mode;	<< which mode tftp should be in.
 *				1	TFTP_MODE_AUTO
 *				2	TFTP_MODE_REMOTE
 *				3	TFTP_MODE_MANUAL
 *
 * description	This is the tftp loader.  It begines by loading all
 *		parameters relevent to the load.  It supports three
 *		parameterization methods: 
 *		1	automatic mode
 *		It uses the parameters stored in eeprom.
 *		After loading, the image is executed. The image
 *		is not burned into flash eprom. This mode always 
 *		runs over the ethernet only. This mode is intended for use
 *		in the development lab.
 *
 *		2	remote mode
 *		It permits software updates to be initiated from a 9100.
 *		Its gathers parameters from eeprom. After loading, the 
 *		image is burned into flash eprom. The image is then executed.
 *		This mode always runs over the ethernets only.
 *
 *		3	manual mode
 *		The load parameters are gathered from eeprom. The user
 *		will be prompted to specify any changes to the eeprom
 *		load parameters, whether to load over the ethernets or
 *		over the uart, and whether or not to burn the image
 *		into flash eprom.
 *
 * returns	In automatic mode
 *			abort if fails to load the image file.
 *			or jump to new image and never return.
 *
 *		In Remote mode
 *			aborts if fails to load the image file.
 *			or jump to new image and never return.
 *
 *		In manual mode
 *			start again if fails to load the image file
 *			or jump to the new image and never return.
 */

tftpmain(int mode)

	{
	for (;;)
		{
		set_leds(LED_BI, LED_RDY | LED_ACT, LED_RDY_GRN | LED_ACT_RED);
		BootImage(mode);
		/*
		 *	If Tftp fails:-
		 *	returns to the parameter entry stage 
		 *	in manual mode	or
		 *	aborts in automatic mode 
		 */
		if (mode != TFTP_MODE_MANUAL)
			{
			return;
			}
		}
	}

BootImage(int mode)

	{
	ulong	FileLength;
/*	NIM960_HDR	*header;*/
	int	BurnFlash = 0;
	int	RetCode;

	/*	clear record	*/
	memset(&boot_data, 0, sizeof(boot_data));
	boot_data.mode = mode;
	GetBootRecord(&boot_data);
	if (boot_data.IsSlip)
		{
		if (SlipPortTest() == FALSE)
			return 0;
		}
	if (TftpInit() == FALSE)
		return 0;
	FileLength = TftpLoadFile(boot_data.ServerIP, boot_data.FileName,
			"octet", TRxPacket); 
	if (FileLength == 0)
		return 0;
	printf("\nFile loaded\n");
	if (CheckImage(boot_data.LoadAddress, FileLength) == 0)
		{
		return 0;
		}
	/*
	 *	We always program the flash eprom in the 
	 *	remote mode.
	 *	or if the debug switch is off, we will program
	 *	the flash.
	 */
	if (boot_data.mode == TFTP_MODE_REMOTE || !(Dip_read() & DIP_DEBUG_ON))
		{
		printf("File is being burned to the flash proms\n");
		BurnFlash = 1;
		flush();
		if (CopyToFlash(boot_data.LoadAddress, FileLength) == 0)
			return 0;
		}
	/*
	 *	or we ask the user if he wants to program the 
	 *	flash eprom.
	 */
	else if (boot_data.mode == TFTP_MODE_MANUAL)
		{
		if (AskYesNo("Do you want to program the flash eprom ([y]/n)"))
			{
			flush();
			BurnFlash = 1;
			if (CopyToFlash(boot_data.LoadAddress, FileLength) == 0)
				return 0;
			}
		}
	if (BurnFlash)
		{
		printf("\nFile has been burned into the flash proms successfully\n");
		flush();
		SaveFlashBurnStatus();
		}
#ifdef FSFIX
	/*
	 *	This is a software reset.
	 */
	nvr_sys.nvr_reset = 0xff;
	if (RetCode = Nvram_Updt(NVR_SYS_ADDR, &nvr_sys, NVR_SYS_SIZE))
		{
		printf("Error: (%d) Cannot save SYS REC to NVRAM.\n", RetCode);
		}
	/*
	 *	Save the reason why the reset - software update.
	 *	We have to load the Reset log record from
	 *	nvram first.
	 */
	RecordRst(RSTMSG10);
#endif

	/*
	 *	If we haven't burn the flash, just relocate the
	 *	new program and pass control to it.
	 *	Copy the programme loader to instruction cache.
	 *	sysctl - type is 0x02, configure cache
	 *		- mode is 0x06, load and lock 512 bytes
	 */
	/*
	 *	We use the Port 1 MAC address to make the
	 *	bootp request. The bootp server must have put
	 *	my ip, my port 1 MAC address pair into its
	 *	ARP table.  The server will put my Port 1 MAC address
	 *	as the destination address when it sends packet
	 *	to me. If the host is not on my Port 1, 
	 *	then I am not going to accept any packet 
	 *	sent to me by the server because it does not match
	 *	my MAC address on the receiving port. 
	 *	To solve the problem, I am going to broadcast an	
	 *	ARP reply to flush the server's arp table such
	 *	that the server will send it's next packet to my
	 *	port "n" MAC address, where n is the port number 
	 *	I can reach the server.
	 */	
	BroadcastArpReply(_initp->in_me);
	TimeDelay10Ms(70);	/* wait a bit for the BroadcastArp 
				 * packets to go out 
				 */
	flush();
	if (BurnFlash)
		{
		/*
		 *	re-boot the whole system in order to jump 
		 *	to the new program. "reset()" will close
		 *	any active telnet session or the adminbus
		 *	virtual console.
		 */
		reset();
		}
	/*
	 *	close the telnet or the admin bus console
	 *	session in case it is active and tell the SYSCARD
	 *	that I am going to reset..
	 */

	CloseTelnet();
	AdmConClose(0);
	AdmFuncardReset();
	stop_timers();
	flush();
	Di();
	send_sysctl(0x0206, JumpToNewProgram, 0);
	JumpToNewProgram((unsigned *)boot_data.LoadAddress);
	}

/*
 * name		GetBootRecord	- manually or from eeprom
 *
 * synopsis	GetBootRecord(boot)
 *		BOOT_BLOCK	*boot; >>	to be initialized
 *
 * description	If the Automatic DIP switch is on, then the parameters
 *		such as my ip address, server ip address, file to be
 *		loaded ...etc, required by tftp will be from the eeprom. 
 *		Otherwise, the user is prompted to enter those
 *		parameters from the console.
 *
 * returns	nothing
 */

void GetBootRecord(boot)
BOOT_BLOCK *boot;

	{
	int	confirmed = 0;
	int	yes = 0;
	int	EepromDataOk = 1;	/* assume eeprom is ok	*/
	EEP_MFG	*mfg = &eep_mfg_rec;	/* Manufacture data		*/
	EEP_BOOT	*bootrec = &eep_boot_rec;/* boot record		*/

	boot->DramSize = mfg->eep_dram_size;
	boot->SramSize = mfg->eep_sram_size;
	if (bootrec->eep_marker  != EEP_MARKER) 
		{
		EepromDataOk = 0;
		}
	else
		{
		boot->MyIP = bootrec->eep_IP;
		boot->ServerIP = bootrec->eep_ServerIP;
		boot->NetMask = bootrec->eep_NetMask;
		boot->RouterIP = bootrec->eep_RouterIP[0];
		strcpy(boot->FileName, bootrec->eep_image_file);
		}
	/*
	 *	If we are in automatic mode and the boot record in
	 *	the eeprom is ok, we are done.
	 */
	if ((boot->mode == TFTP_MODE_AUTO || boot->mode == TFTP_MODE_REMOTE)
		&& EepromDataOk)
		return;
	/*
	 *	If the boot record is good and we are in
	 *	manual mode, we ask the user if he wants to use
	 *	these parameters rather than to enter these data
	 */
	if (EepromDataOk)
		{
		yes = UseBootRecord(boot);
		}
	if (!yes)
		{
		/*
		 *	No, the user does not want to use the boot record 
		 *	stored in the eeprom or the boot record is bad.
		 *	We have to ask the user to enter boot record.
		 */
		EnterBootRecord(boot);
		bootrec->eep_IP = boot->MyIP;
		bootrec->eep_ServerIP = boot->ServerIP;
		bootrec->eep_NetMask = boot->NetMask;
		if ((boot->MyIP & boot->NetMask) != 
			(boot->ServerIP & boot->NetMask))
			bootrec->eep_RouterIP[0] = boot->RouterIP;
		strcpy(bootrec->eep_image_file, boot->FileName);
		/*
		 *	Save new records to eeprom
		 */
		if (AskYesNo("Do you want to save these records in EEPROM ([y]/n)"))
			{
			SaveEepBoot();
			}

		}
	/* 
	 *#####	return for now
	 */
	return;
	while (confirmed == 0)
		{
		yes = AskYesNo("Load over ethernet ([y]/n)");
		boot->IsSlip = !yes;
		if (boot->IsSlip != 0)
			{
			/*
			 *	Ask for the baud rate
			 */
			boot->baud = AskForBaud();
			}
		confirmed = AskYesNo("Confirm ([y]/n)");
		}
	}
/*
 * name		SetLoadAddress
 *
 * synopsis	SetLoadAddress(header, LoadAddress)
 *		NIM960_HDR	*header;<< nim960 header
 *		byte	**LoadAddress; >> where to load the file
 *
 * description	If the nim960 signature is OK, it set the load address
 *		to be the TextLoadAddress in the nim960 header.
 *		It then checks if there are enough ram space to
 *		load the file. 
 *		This function is called when the first tftp packet
 *		has been received.
 *
 * returns	TRUE	file header is ok and address has been set
 *		FALSE	error
 */

SetLoadAddress(NIM960_HDR *header, byte **LoadAddress)

	{
	ulong	TextLoadAddress;
	ulong	FileLength;

	/*
	 *	Check if file is a NIM960 file.
	 */
	if (header->signature != NIM960_MAGIC)
		{
		printf("Error: Invalid file format\n");
		return FALSE;
		}
	LongCopy(&TextLoadAddress, &header->TextLoadAddress);
	LongCopy(&FileLength, &header->FileLength);
	/*
	 *	If the loading address in not in 
	 *	the the SRAM or DRAM address space,
	 *	we cannot load the file at all.
	 */
	if (!AddressInDram(TextLoadAddress) &&
		!AddressInSram(TextLoadAddress))
		{
		printf("Load address (%X) not in RAM \n,", 
			TextLoadAddress);
		return FALSE;
		}
	if ((*LoadAddress = lmalloc(FileLength)) == NULL)
		{
		printf(" File too big %d\n", FileLength);
		return FALSE;
		}
	boot_data.LoadAddress = (byte *)*LoadAddress;
	return TRUE;
	}

/*
 *	Test if address is in the Dram address space
 *	returns TRUE if yes
 *		FALSE	otherwise
 */
AddressInDram(byte *address)

	{
	return (((byte *)DRAM + boot_data.DramSize) >= address &&
		(address >= (byte *)DRAM));
	}
/*
 *	Test if address is in the Sram address space
 *	It is assumed that the Dram starts right after the Sram.
 *	returns TRUE if yes
 *		FALSE	otherwise
 */
AddressInSram(byte *address)

	{
	return (((byte *)SRAM + boot_data.SramSize) >= address &&
		(address >= (byte *)SRAM));
	}


/*
 *	Are we on SLIP ?
 */
IsSlip()
	{
	return boot_data.IsSlip;
	}
/*
 *	Return the baud rate for SLIP
 */
SlipBaud()
	{
	return boot_data.baud;
	}
/*
 * name		JumpToNewProgram
 *
 * synopsis	JumoToNewProgram(address)
 *		unsigned 	*address; <<	the image is in here
 *
 * description	This function will re-locate the image at "address" to
 *		its run time address.  The run time address is the same
 *		as the program currently running.  It overwrites the 
 *		running program.  This function must be copied into
 *		the instruction cache first by doing a SYSCTL before
 *		control is passed here.
 *
 * returns	it will not return
 */

void JumpToNewProgram(unsigned *address)

	{
	NIM960_HDR	*header = (NIM960_HDR *)address;
	unsigned	*text;	/*	where to load the program	*/
	int	length;		/* 	of the program 			*/

	text = (unsigned *)header->TextLoadAddress;
	length = (header->FileLength + sizeof(unsigned) - 1);
	length /= sizeof(unsigned);	/* length in number of unsigned	*/
	while(length--)
		*text++ = *address++;
	/*
	 *	Tell the new program that control is passed
	 *	from the tftp loader.
	 */
	(*((int (*)())header->Entry))(ENTRY_FROM_TFTP_MAGIC); 
	}

LongCopy(ushort *to, ushort *from)
	
	{
	*to++ = *from++;
	*to = *from;
	}
/*
 *	Save the date and time the flash is programed.
 */
static SaveFlashBurnStatus()

	{

	char	buffer[10];
	uint	year;
	uint	month;
	uint	date;
	uint	hour;
	uint	minute;
	uint	second;

	GetDate(&date, &month, &year);
	if (year > 1900)
		year -=  1900;
	sprintf(buffer, "%02d-%02d-%02d", month, date, year);
	memcpy(eep_boot_rec.eep_BurnDate, buffer, 8);
	GetTime(&hour, &minute, &second);
	sprintf(buffer, "%02d:%02d:%02d", hour, minute, second);
	memcpy(eep_boot_rec.eep_BurnTime, buffer, 8);
	eep_boot_rec.eep_BurnCycles++;
	SaveEepBoot();
	}

/*
 *	Burn the file into the flash eproms.
 */
static CopyToFlash(byte *data, int length)

	{
	int	left;	/* to be burned	next	*/
	int	size;	/* to be burned now	*/
	int	ret;

	if (length > eep_mfg_rec.eep_fprom_size)
		{
		printf("Error: file size (%d) is bigger than the flash size (%d)\n",
			length, eep_mfg_rec.eep_fprom_size);
		return FALSE;
		}

	if (length > MIN_FLASH_SIZE)
		{
		size = MIN_FLASH_SIZE;;
		left = length - size;
		}
	else
		{
		size = length;
		left = 0;
		}
	if ((ret = ProgramFlash((void *)FEPROM, data, size)) == FALSE)
		return ret;
	if (left != 0)
		ret = ProgramFlash((void *)FEPROM + size, data + size, left);
	return ret;			
	}

