/***************************************************************************
 *  File    drv_timer.c
 *
 *  Description
 *      This file programs the Intel 82C54 chip counter 2. Counter 2 is
 *      dedicated to the CSP module.
 *
 *  Copyright (c) 1992, 1993 Hughes Lan Systems
 *
 *  Author:     Jia-wei Jang
 *
 *  $Log:   /user/pvcs/fddicon/drv/src/drv_timer.c_v  
 * 
 *    Rev 1.19   12 Oct 1993 11:07:02   gregs
 * No change.
 * 
 *    Rev 1.18   29 Sep 1993 13:42:24   gregs
 * No change.
 * 
 *    Rev 1.17   13 Sep 1993 13:50:04   gregs
 * No change.
 * 
 *    Rev 1.16   21 Jun 1993 17:35:36   jang
 * changed fffe to 7ffe
 * 
 *    Rev 1.15   13 Apr 1993 15:07:44   jang
 * changed all the read/write timer to store_byte/load_byte()
 * 
 *    Rev 1.14   13 Apr 1993 14:06:32   jang
 * 
 *    Rev 1.13   09 Apr 1993 07:14:24   gregs
 * timer 3 int. vector for fddi-eth bridge.
 * 
 *    Rev 1.12   06 Jan 1993 11:39:40   jlin
 * for FDDI release 0.00A
 * 
 *    Rev 1.11   17 Dec 1992 14:24:32   jang
 * fix a compiling error by adding ProcState pState to DRV_Init()
 *************************************************************************/
#include "fddi.h"
#include "cspmacro.h"
#include <drv_timer.h> 

#define TIMER2_MAX          0x7ffe


/*
 *------------------------------------------------------------------------
 *      Exported Variables
 *------------------------------------------------------------------------
 */
volatile DRV_TIMER *timer_2;


/*
 *------------------------------------------------------------------------
 *      Local Variables
 *------------------------------------------------------------------------
 */
static uint32 remain_ticks; /* timer2 cah hold max. 65536 ticks. So
                                the max interval it can hold is 32768 usec.
                                This variable used to stored the value
                                more than 32768 usec */


/*************************************************************************
 *  Function    DRV_Timer3Init
 *
 *  Description
 *      This function initializes Timer 2
 *
 *  Parameter   none
 *
 *  Return: void
 ************************************************************************/
void  DRV_Timer3Init ()
{
    extern Timer3Isr();
    long s;
    register byte control;

    timer_2 = (DRV_TIMER *)TIMERIO;

    /* stop the timer */
    control =  TIME_CW_COUNT2 | TIME_CW_LSB | TIME_CW_MODE4;

    s = crit_on();
    store_byte(control,&(timer_2->ctrl_reg));
    store_byte(0,&(timer_2->counter_2));
    crit_off(s);

    remain_ticks = 0;
    /* install the vector table */
#ifdef __FEBRIDGE
    init_hw_int(3,66,Timer3Isr);
#else
    init_hw_int(6,114,Timer3Isr);
#endif
}

/*************************************************************************
 *  Function    SetSystemTimer
 *
 *  Description
 *      This function enables the CSP timer (timer 2). The name is given
 *      because SMT core uses it.
 *
 *  Parameter:
 *      uint32  interval - # of usec to count
 *
 *  Return: void
 ************************************************************************/
void DRV_SetSystemTimer (interval)
uint32 interval;
{
  long s;
  register byte control;

  s = crit_on();
#if 0
  printf("set timer 3: interval %lx\n",interval);
#endif
  if (interval > TIMER2_MAX) {
    remain_ticks = interval - TIMER2_MAX;
    interval = TIMER2_MAX;
  }
  else {
    remain_ticks = 0;
  }

  interval *= DRV_1uSEC_TICKS;

  control = TIME_CW_COUNT2 | TIME_CW_WORD | TIME_CW_MODE4;
  store_byte(control,&(timer_2->ctrl_reg));
  store_byte(interval & 0xff,&(timer_2->counter_2));
  store_byte((interval >> 8) & 0xff,&(timer_2->counter_2));

  crit_off(s);
}

/*************************************************************************
 *  Function    ReadSystemTimer
 *
 *  Description
 *      This function returns the remaining timer 2 in micro seconds.
 *
 *  Parameter:      none
 *
 *  Return: uint32 usec 
 ************************************************************************/
uint32 DRV_ReadSystemTimer ()
{
    uint32 time, time_hi;
    long s;
    register byte control;

    s = crit_on();
    control = TIME_CW_COUNT2 | TIME_CW_LATCH;
    store_byte(control,&(timer_2->ctrl_reg));
    time = (uint32) load_byte(&(timer_2->counter_2)) & 0x0ff;

    time_hi = (uint32) load_byte(&(timer_2->counter_2)) & 0x0ff;
    time |= time_hi << 8;
    crit_off(s);
    time = time / DRV_1uSEC_TICKS + remain_ticks;
#if 0
 	printf("read from timer %lx\n",time);
#endif    
    return (time);
}
 
/***************************************************************************
 *  Function    DRV_Timer2ISR
 *
 *  Description
 *      This is the timer 2 ISR.
 *
 *  Parameter:      none
 *
 *  Return: void
 **************************************************************************/
void DRV_Counter2ISR ()
{
    extern ServiceCSPTimer();

    if (remain_ticks == 0) {
#if 0
    printf("interrupt occur, after ServiceCSPTimer()\n");
#endif
        ServiceCSPTimer();

    }   
    else {
#if 1
	SetSystemTimer(remain_ticks);       
#else
	/* this is used for testing */
        isr_set_timer(remain_ticks);
#endif
    }
}
 
static int isr_set_timer (ticks)
uint32 ticks;
{
  printf("isr is setting the timer --> %lx\n",ticks);
  SetSystemTimer(ticks);
}

/*
 *--------------------------------------------------------------------------
 *      Diagnostics Functions
 *--------------------------------------------------------------------------
 */

static bool test_smt_timer_flag = 0;
 
/*************************************************************************
 *  Function test_smt_timer
 *
 *  Description
 *     this function tests timer 3
 *
 *  Parameters:
 *      uint32 act - 1: start timer; 2 - stop timer
 *      uint32 num - dummy
 *
 *  Return:
 *************************************************************************/
int test_smt_timer (act,num)
uint32 act;
uint32 num;
{
    uint32 remain;
    int i;

    if (!test_smt_timer_flag) {
        test_smt_timer_flag = 1;
        DRV_Timer3Init();
    }

    switch (act) {
        case 1:     /* set # of usec to count */
            DRV_SetSystemTimer(100000);
            break;
        case 2:
            remain = DRV_ReadSystemTimer();
/*
            printf("the SMT timer reading is %ld\n",remain);
*/
            break;
    }       /* switch */
} 
