/***********************************************************************
 *  File    phy_isr.c
 *
 *  Description
 *          This module processes PLAYERs interrupts
 *
 *  Copyright (c) 1992-93 Hughes Lan Systems
 *
 *  Author: Jia-wei Jang
 *
 *  $Log:   /usr/shasta/pvcs/fddicon/drv/src/phy_isr.c_v  
 * 
 *    Rev 1.23   13 Oct 1993 10:52:56   jlin
 * Fix the LEM problem for bridge inside LEMT_ISR() by replacing PHY_MASTER_PORTS
 * to FDDI_NONMASTER_CT
 * 
 *    Rev 1.22   12 Oct 1993 11:08:30   gregs
 * No change.
 * 
 *    Rev 1.21   29 Sep 1993 13:42:30   gregs
 * No change.
 * 
 *    Rev 1.20   13 Sep 1993 13:51:14   gregs
 * No change.
 * 
 *    Rev 1.19   02 Aug 1993 16:09:12   gregs
 * Added FEBRIDGE ifdef
 * 
 *    Rev 1.18   13 Jul 1993 11:30:10   jlin
 * Allow the SMT to enable and disable SILS interrupt
 * 
 *    Rev 1.17   07 Jul 1993 18:47:50   jlin
 * Delete those PHY_MASTER_PORTS checking statements
 * 
 *    Rev 1.16   02 Jul 1993 11:49:04   jlin
 * Real working version for fixing PHY intensive interrupts and ANTC bug
 * 
 *    Rev 1.15   02 Jul 1993 10:54:06   jlin
 * Fix the intensive interrupts and ANTC bug
 * 
 *    Rev 1.14   30 Jun 1993 16:28:40   jang
 * took out NT, QLS and NSD from isr
 * 
 *    Rev 1.13   22 Apr 1993 17:45:14   jang
 * disable interrupt after we recieve the event
 * 
 *    Rev 1.12   09 Apr 1993 14:34:06   jang
 * load noise threshold register before enable interrupt
 * 
 *    Rev 1.11   15 Mar 1993 18:33:32   jang
 * fixed the bug of mask off ICMR when LEM threshold readhed
 * 
 *    Rev 1.10   07 Jan 1993 18:57:24   jang
 * modified to distinguish the backplane ports and the normal ports
 *********************************************************************/


#include <fddi.h>
#include <drv.h>
#include <phy.h>

#define PHY_DEBUG    0

/*
 *------------------------------------------------------------------------
 *          Exported variables
 *------------------------------------------------------------------------
*/
uint16 previousLS[PHY_MAX_PORTS];
uChar  SILSmask=0;   /* jlin: 07/13/93 */

/*
 *------------------------------------------------------------------------
 *          Imported variables
 *------------------------------------------------------------------------
*/
extern PLAYER_TYPE phys[];




/*
 *------------------------------------------------------------------------
 *          Function prototypes
 *------------------------------------------------------------------------
*/

#ifdef DO_LINT

#else
static void LEMT_isr();
static void RxConditionISR();
#endif


/**************************************************************************
 *  Function    PHY_Isr
 *
 *  Description
 *      This function processes all the PLAYER's interrupt.
 *
 *  Parameter:
 *      uint32 phy_num
 *
 *  Return: void
 ************************************************************************/
void PHY_Isr (phy_num)
uint32 phy_num;
{
    byte icr;
    volatile PHY_REG_TYPE *r_ptr;

    r_ptr = phys[phy_num].base_addr;
    icr = r_ptr->icr;
    if (icr == 0) {
      return;
    }

    /*----------------------------------------------------------------------
        RCA & RCB are based on 'cspintif.c' and 'cspsysif.c'. It is because
        SMT treats line state in a very strange way.
    -----------------------------------------------------------------------*/
    if (icr & (PHY_ICR_RCA | PHY_ICR_RCB)) {
        RxConditionISR(phy_num);
    }
    if ((icr & PHY_ICR_LEMT) && phys[phy_num].lem_int) {
        LEMT_isr(phy_num);
    }
#if 0
	/* we don't have parity error and control bus error checked */
    if (icr & PHY_ICR_DPE) {

    }
    if (icr & PHY_ICR_CPE) {

    }
#endif
    /* clear the ICR bits */
    PHY_SetIcr(r_ptr,icr);
}
 
/*
 *--------------------------------------------------------------------
 *          Local Functions
 *--------------------------------------------------------------------
 */

/****************************************************************************
 *  Function     LEMT_isr
 *
 *  Description
 *      This function process Link Error
 *
 *  Parameter
 *      uint32 phy_num
 *
 *  Return: void
 ***************************************************************************/
static void LEMT_isr (phy_num)
uint32 phy_num;
{
    PHY_CTRL_TYPE *i_ptr;
    PHY_REG_TYPE *r_ptr;

    i_ptr = phys[phy_num].image_ptr;
    r_ptr = phys[phy_num].base_addr;

    phys[phy_num].lem_count += i_ptr->letr;
    if (phys[phy_num].LerCutoff == 0) {
        /* there are 2 ways to handle this. The driver can either mask
	   off the lert interrupt, or reload the lert value. Which
	   way to do is depending on how the SMT handles when it 
	   receives the signal.
        */
      r_ptr->icmr &= ~PHY_ICMR_LEMTM;
#if PHY_DEBUG
	  printf("Send port %d LEMT to SMT\n",phy_num);
#endif
      /* send signal to SMT */
#ifdef __FEBRIDGE
	  if( phy_num < FDDI_NONMASTER_CT)
#endif
          CSP_SMTSendSignal(SIG_LEM_THRESHOLD,phy_num,0);
    }
    else {
        phys[phy_num].LerCutoff -= i_ptr->letr;
        PHY_SetLerCutoff(phy_num,phys[phy_num].LerCutoff);
    }
}

/****************************************************************************
 *  Function    RxConditionISR
 *
 *  Description
 *      This function processes the received condition interrupt. This routine
 *      is based on the XINT cspsysif.c and  cspintif.c due to the reason
 *      that the SMT module has a very special way to process the received
 *      line state. THIS FUNCTION NEEDS TO BE MODIFIED IF NECESSARY.
 *
 *  Parameter:
 *      uint32 phy_num
 *
 *  Return: void
 ****************************************************************************/
static void RxConditionISR (phy_num)
uint32 phy_num;
{
  volatile PHY_REG_TYPE *r_ptr;
  byte crsr, rcra, rcrb;
  uint16 line_state;
#if PHY_DEBUG
  byte debug_rcra,debug_rcrb,debug_crsr;
#endif
  r_ptr = phys[phy_num].base_addr;

  rcra = r_ptr->rcra;
  rcrb = r_ptr->rcrb;
#if PHY_DEBUG
  debug_rcra = rcra;
  debug_rcrb = rcrb;
#endif

  if (rcra & PHY_RCRA_NT) {
    PHY_SetNoiseThresh(r_ptr);
	rcra &= ~PHY_RCRA_NT;
    phys[phy_num].noise_count++;
  }

  PHY_ClearRCRA(r_ptr);
  PHY_ClearRCRB(r_ptr);
  /* sometimes the interrupt has been processed at last time */
  if ((rcra & r_ptr->rcmra == 0) && (rcrb & r_ptr->rcmrb == 0)) {
    return;
  }

#if 0
  r_ptr->rcmra = ~rcra & (PHY_RCRA_HLS | PHY_RCRA_MLS | PHY_RCRA_QLS);
#endif

  crsr = r_ptr->crsr & PHY_CRSR_LS_MASK;
#if PHY_DEBUG
  debug_crsr = crsr;
#endif

#if 0
  /* if rx QLS or NSD, remove interrupts mask from RCMRA */
  if (crsr == PHY_CRSR_NSD || crsr == PHY_CRSR_QLS)
    r_ptr->rcmra &= ~(PHY_RCMRA_QLSM | PHY_RCMRA_NSDM | PHY_RCMRA_NTM);
  else {
    /* load noise threshold registers before enable NT interrupt */
    PHY_SetNoiseThresh(r_ptr);
    r_ptr->rcmra |= PHY_RCMRA_QLSM | PHY_RCMRA_NSDM;
  }
#endif


  if (rcrb & PHY_RCRB_EBOU)   /* elasticity buffer overflow/underflow */ {
    phys[phy_num].BufErrCount++;
#if PHY_DEBUG
    printf("port %d buf overflow/underflow\n",phy_num);
#endif
  }

  if (rcra & PHY_RCRA_NSD)
    rcra |= PHY_RCRA_QLS;
  rcra &= (PHY_RCRA_QLS | PHY_RCRA_HLS | PHY_RCRA_MLS | PHY_RCRA_NT);
  rcrb &= PHY_RCRB_SILS;

  line_state = rcra | rcrb | previousLS[phy_num];

  /* store the previous line state */
  crsr = r_ptr->crsr & PHY_CRSR_LS_MASK;
#if 0
  if (crsr == PHY_CRSR_ILS)   /* check for SILS */
    crsr = r_ptr->crsr & PHY_CRSR_LS_MASK;
#endif

  switch (crsr) {
  case PHY_CRSR_ILS:
    previousLS[phy_num] = SILS;
    r_ptr->rcmrb &= ~(PHY_RCMRB_SILSM);
    r_ptr->rcmra |= PHY_RCMRA_NSDM | PHY_RCMRA_QLSM | PHY_RCMRA_HLSM |
                    PHY_RCMRA_MLSM;
    break;
            
  case PHY_CRSR_MLS:
    previousLS[phy_num] = MLS;
    r_ptr->rcmra &= ~PHY_RCMRA_MLSM;
    r_ptr->rcmrb |= SILSmask;
    r_ptr->rcmra |= PHY_RCMRA_NSDM | PHY_RCMRA_QLSM | PHY_RCMRA_HLSM;
    break;
            
  case PHY_CRSR_HLS:
    previousLS[phy_num] = HLS;
    r_ptr->rcmra &= ~PHY_RCMRA_HLSM;
    r_ptr->rcmrb |= SILSmask;
    r_ptr->rcmra |= PHY_RCMRA_NSDM | PHY_RCMRA_QLSM | PHY_RCMRA_MLSM;
    break;
            
  case PHY_CRSR_NSD:
  case PHY_CRSR_QLS:
    previousLS[phy_num] = QLS;
    r_ptr->rcmra &= ~(PHY_RCMRA_QLSM | PHY_RCMRA_NSDM);
    r_ptr->rcmrb |= SILSmask;
    r_ptr->rcmra |= PHY_RCMRA_HLSM | PHY_RCMRA_MLSM;
    break;
    
  default:
    previousLS[phy_num] = 0;
    break;
  }       /* switch */
  line_state |= previousLS[phy_num];  /* finally add in current line state */

  if (line_state & NLS) {
      CSP_SMTSendSignal(SIG_NOISE_EVENT,phy_num,line_state);
#if PHY_DEBUG
      printf("Send port %d Noise Event\n",phy_num);
#endif
    line_state &= ~NLS;
  }
    
  if (line_state) {
#if PHY_DEBUG
    printf("Port %d line st %x, rcra %x, rcrb %x, crsr %x, noise %d, ntr %x\n",phy_num,line_state,debug_rcra,debug_rcrb,debug_crsr,phys[phy_num].noise_count,r_ptr->ntr);
#endif
    CSP_SMTSendSignal(SIG_LINE_STATE,phy_num,line_state);
  }
}



