/*********************************************************************
	Management Application Module
	
	Frame Browsing Routines
	
	File:		mapbuf.c
	Created:	03/14/91

	SID:		1.2
	Last Modified:	5/2/91

	Copyright 1990,1991 XLNT Designs, Inc.

	This module provides the frame browsing routines when collecting
	SMT frames.

	Modification History:
*********************************************************************/

#include        "smtdefs.h"
#include        "smttypes.h"
#include        "smtmsg.h"
#include        "fddihdr.h"
#include        "fbmframe.h"
#include        "fbmmacro.h"
#include        "mibdefs.h"
#include	"maphdr.h"
#include	"mapglbl.h"


/*********************************************************************
	External Functions
*********************************************************************/

extern	void	FreeMAPMessage ();
extern	void	MarkMAPMessage ();
extern	void	DoDump ();
extern	uInt16	ntohs ();
extern	void	HiliteParam ();
extern	void	DisplayParam ();
extern	void	UnhiliteParam ();
extern	void	SendMAPMessage ();


/*********************************************************************
	Defined Values
*********************************************************************/

/*
*	Max # of frames in buffer.
*/
#define	MAX_BUF_QUEUE	200

/*
*	Key selections.
*/
#define NEXT_FRAME_KEY          'n'
#define PREV_FRAME_KEY          'p'
#define NEXT_FIELD_KEY          '>'
#define PREV_FIELD_KEY          '<'
#define FIRST_FRAME_KEY         'f'
#define LAST_FRAME_KEY          'l'

/*
*	Screen description.
*/
#define	NUM_BYTES		(FRAME_LINES * 16)	/* # bytes visible */
#define	PARAMLEN(p)		(ntohs (((ParamHdrType *) p)->length) + sizeof (ParamHdrType))

/*********************************************************************
	Global Variables
*********************************************************************/

/*
*	Flag is set when buffer is being displayed. This locks out
*	wrap-around feature while display is on.
*/
static	Flag		displayOn;

static	Int16		bufqHead,
			bufqTail,
			displayPtr;

/*
*	Frames are reported from FSP in messages. Store messages in
*	list.
*/
static SMTMessage *bufq[MAX_BUF_QUEUE];

/*
*	Screen descriptors.
*/
int	startPos,		/* starting position of frame on screen */
	paramPos;		/* starting position of param in frame */


/*********************************************************************
	Queue Macros
*********************************************************************/

/*
*	The NextQueueEntry(n) macro returns the next index with a
*	wrap to 0 if the array size will be exceeded.
*/
#define	NextQueueEntry(n)	((n == MAX_BUF_QUEUE - 1) ? 0 : n + 1)

/*
*	The PrevQueueEntry(n) macro returns the previous index with a
*	wrap to the top of the array.
*/
#define	PrevQueueEntry(n)	((n == 0) ? MAX_BUF_QUEUE - 1 : n - 1)


/*********************************************************************
	Buffer Management Routines
*********************************************************************/

void
StoreFrame (msg)
	SMTMessage	*msg;
/*********************************************************************
Function:	Place a frame in the queue.
Parameters:	msg	= pointer to an SMT message buffer.
Input:		Uses buffer queue.
Output:		Adds message to buffer queue.
Return:		None.
*********************************************************************/
{
ProcState	pState;

	MDisableInterrupts (&pState);

	/* 
	*	If queue is full and not displaying, overwrite
	*	first frame. Otherwise drop frame.
	*/
	if (NextQueueEntry (bufqTail) == bufqHead)
	{
		if (displayOn)
		{
			MRestoreInterrupts (&pState);
			return;
		}

		else
		{
			FreeMAPMessage (bufq[bufqHead]);
			bufqHead = NextQueueEntry (bufqHead);
		}
	}

	bufq[bufqTail] = msg;
	bufqTail = NextQueueEntry (bufqTail);
	/* mark message in use */
	MarkMAPMessage (msg);

	MRestoreInterrupts (&pState);
	return;
}

void
EmptyFrameBuffer ()
/*********************************************************************
Function:	Remove all frames frome buffer.
Parameters:	None.
Input:		Uses bufqTail, bufqHead, and bufq.
Output:		Message space freed.
Return:		None.
*********************************************************************/
{
	while (bufqHead != bufqTail)
	{
		FreeMAPMessage (bufq[bufqHead]);
		bufqHead = NextQueueEntry (bufqHead);
	}

	bufqHead = bufqTail = 0;

	return;
}

uInt32
InitFrameBuffer ()
/*********************************************************************
Function:	Initialize system for sending and receiving SMT messages.
Parameters:	None.
Input:		None.
Output:		Set system message processing.
Return:		0 if successful, error number if failed.
*********************************************************************/
{
	bufqHead = bufqTail = 0;
	return (0);
}


/*********************************************************************
	Buffer Display Routines
*********************************************************************/

extern	WINDOW	*inputwin, *eventwin, 
		*statwin, *workwin, 
		*framewin, *parsewin;

void
ShowFrameDump (index, startPos)
        Int16             index;
	int		startPos;
/*********************************************************************
Function:	Dump a frame to the frame display window.
Parameter:	msg		= frame received message.
		startPos	= starting position in frame.
Input:		None.
Output:		None.
Return:		None
*********************************************************************/
{
RecFrameBuf	*recBuf;
SMTFrameHdrType	*frameHdr;
SMTMessage	*msg;
uInt16		limit;
Int16		refNum, total;

	/* get buffers */
	msg = bufq[index];
	recBuf = (RecFrameBuf *) msg->p1.recFrameBuf;
	frameHdr = (SMTFrameHdrType *) msg->p2;

	total = bufqTail - bufqHead;
	if (total < 0)
		total += MAX_BUF_QUEUE;
	refNum = index - bufqHead;
	if (refNum < 0)
		refNum += MAX_BUF_QUEUE;
	refNum++;

	/* do the dump */
	werase (framewin);
	wmove (framewin, 0, 0);
	wprintw (framewin, 
		"Frame received on MAC %d. Length = %d. Indicators = %c%c%c",
		msg->entity, recBuf->frameLen,
		(recBuf->EACbits & E_Indicator) ? 'E' : 'e',
		(recBuf->EACbits & A_Indicator) ? 'A' : 'a',
		(recBuf->EACbits & C_Indicator) ? 'C' : 'c');
	wmove (framewin, 0, 62);
	wprintw (framewin, "Frame %3d of %3d", refNum, total);
	wmove (framewin, 1, 0);

	limit = recBuf->frameLen - startPos;
	if (limit > NUM_BYTES) 
		limit = NUM_BYTES;
	DoDump (&frameHdr->macHdr.FC + startPos, limit, framewin);

	return;
}

uChar *
NextParam (msg, curParam)
	SMTMessage	*msg;
	uChar		*curParam;
/*********************************************************************
Function:	Find next parameter in frame.
Parameters:	msg		= frame message.
		curParam	= current parameter pointer.
Input:		None.
Output:		None.
Return:		Pointer to next parameter in frame. If no more 
		parameters, current one is returned.
*********************************************************************/
{
SMTFrameHdrType	*frameHdr;
ParamHdrType	*phdr;
uChar		*newParam;
uInt16		count;

	/*
	*	Get frame values.
	*/
	frameHdr = (SMTFrameHdrType *) msg->p2;

	/*
	*	Get first param in frame.
	*/
	newParam = (uChar *) (frameHdr + 1);

	/*
	*	If frame header, then get first param.
	*	Otherwise, get next param.
	*/
	if (curParam == (uChar *) frameHdr)
	{
		/* if this frame does not have any parameters */
		if (ntohs (frameHdr->smtHdr.InfoField_Length) == 0)
			/* then return header */
			newParam = curParam;
	}

	else
	{
		count = (uInt16) (curParam - newParam);
		phdr = (ParamHdrType *) curParam;
		count = count + ntohs (phdr->length) + sizeof (ParamHdrType);
		if (count < ntohs (frameHdr->smtHdr.InfoField_Length))
			/* get next parameter */
			newParam = curParam + ntohs (phdr->length)
				+ sizeof (ParamHdrType);
		else
			/* no more parameters */
			newParam = curParam;
	}

	return (newParam);
}

uChar *
PrevParam (msg, curParam)
	SMTMessage	*msg;
	uChar		*curParam;
/*********************************************************************
Function:	Find previous parameter in frame.
Parameters:	msg		= frame message.
		curParam	= current parameter pointer.
Input:		None.
Output:		None.
Return:		Pointer to previoius parameter in frame. If no more 
		parameters, frame header is returned.
*********************************************************************/
{
SMTFrameHdrType	*frameHdr;
ParamHdrType	*phdr;
uChar		*newParam, *nextParam;

	/*
	*	Get frame values.
	*/
	frameHdr = (SMTFrameHdrType *) msg->p2;

	/*
	*	Get first param in frame.
	*/
	newParam = (uChar *) (frameHdr + 1);

	/*
	*	If frame header, then no previous parameters.
	*	Otherwise, get previous param.
	*/
	if (curParam == (uChar *) frameHdr)
	{
		/* return header */
		newParam = curParam;
	}

	else
	{
		/* if first parameter */
		if (newParam == curParam)
			/* move to header */
			newParam = (uChar *) frameHdr;

		else
		{
			/* loop through frame to find previous param */
			nextParam = newParam;
			while (nextParam != curParam)
			{
				newParam = nextParam;
				phdr = (ParamHdrType *) newParam;
				nextParam += (ntohs (phdr->length)
					+ sizeof (ParamHdrType));
			}
		}
	}

	return (newParam);
}

void
DoFrameDisplay ()
/*********************************************************************
Function:	Entry point to the frame buffer display.
Parameters:	None.
Input:		Uses window pointers.
Output:		None.
Return:		None.
*********************************************************************/
{
int	c;
uChar	*curParam;
int	oldstatwin, oldworkwin, oldinputwin, oldeventwin;
extern int	WindowVisible ();

	/* if queue is empty, ignore action */
	if (bufqTail == bufqHead)
	{
		wprintw (inputwin, "Frame Buffer is empty.\n");
		return;
	}

	/* set display buffer on */
	displayOn = SET;

	/* prepare windows */
	oldstatwin = WindowVisible (STATUS_WIN, FALSE);
	oldworkwin = WindowVisible (WORK_WIN, FALSE);
	oldinputwin = WindowVisible (INPUT_WIN, FALSE);
	oldeventwin = WindowVisible (EVENT_WIN, FALSE);

	WindowVisible (FRAME_BUF_WIN, TRUE);
	WindowVisible (FRAME_PARSE_WIN, TRUE);
	werase (framewin);
	werase (parsewin);

	/* set first frame of buffer */
	displayPtr = bufqHead;

	/* set current parameter to start of frame */
	curParam = (uChar *) bufq[displayPtr]->p2;
	startPos = paramPos = 0;

	/* display first frame */
	ShowFrameDump (displayPtr, startPos);
	HiliteParam (bufq[displayPtr], curParam, startPos);
	DisplayParam (bufq[displayPtr], curParam);

	c = 0;
	while (c != CHAR_RETURN)
	{
		wstandout (parsewin);
		mvwprintw (parsewin, 0, 8,
			"Frame: %c=prev, %c=next, %c=first, %c=last     "
			"Field: %c=prev, %c=next",
			PREV_FRAME_KEY, NEXT_FRAME_KEY,
			FIRST_FRAME_KEY, LAST_FRAME_KEY,
			PREV_FIELD_KEY, NEXT_FIELD_KEY);
		wstandend (parsewin);
		c = WGETCH (inputwin);
		if (c >= 'A' && c <= 'Z')
			c = tolower (c);

		/* process key */
		switch (c)
		{
		case NEXT_FRAME_KEY:
			if (NextQueueEntry (displayPtr) != bufqTail)
			{
				UnhiliteParam (bufq[displayPtr], curParam, 
					startPos);
				displayPtr = NextQueueEntry (displayPtr);
				startPos = paramPos = 0;
				ShowFrameDump (displayPtr, startPos);
				curParam = (uChar *) bufq[displayPtr]->p2;
				HiliteParam (bufq[displayPtr], curParam, 
					startPos);
				DisplayParam (bufq[displayPtr], curParam);
			}
			break;

		case PREV_FRAME_KEY:
			if (displayPtr != bufqHead)
			{
				UnhiliteParam (bufq[displayPtr], curParam, 
					startPos);
				displayPtr = PrevQueueEntry (displayPtr);
				startPos = paramPos = 0;
				ShowFrameDump (displayPtr, startPos);
				curParam = (uChar *) bufq[displayPtr]->p2;
				HiliteParam (bufq[displayPtr], curParam, 
					startPos);
				DisplayParam (bufq[displayPtr], curParam);
			}
			break;

		case NEXT_FIELD_KEY:
			UnhiliteParam (bufq[displayPtr], curParam, startPos);
			curParam = NextParam (bufq[displayPtr], curParam);
			paramPos = (int) (curParam - bufq[displayPtr]->p2);
			paramPos -= FC_PAD_SIZE;
			if (((paramPos + PARAMLEN(curParam)) - startPos)
				> NUM_BYTES)
			{
				startPos = paramPos;
				ShowFrameDump (displayPtr, startPos);
			}
			HiliteParam (bufq[displayPtr], curParam, startPos);
			DisplayParam (bufq[displayPtr], curParam);
			break;

		case PREV_FIELD_KEY:
			UnhiliteParam (bufq[displayPtr], curParam, startPos);
			curParam = PrevParam (bufq[displayPtr], curParam);
			paramPos = (int) (curParam - bufq[displayPtr]->p2);
			if (paramPos > FC_PAD_SIZE)
				paramPos -= FC_PAD_SIZE;
			if (paramPos < startPos)
			{
				startPos = paramPos;
				ShowFrameDump (displayPtr, startPos);
			}
			HiliteParam (bufq[displayPtr], curParam, startPos);
			DisplayParam (bufq[displayPtr], curParam);
			break;

		case FIRST_FRAME_KEY:
			UnhiliteParam (bufq[displayPtr], curParam, startPos);
			displayPtr = bufqHead;
			startPos = paramPos = 0;
			ShowFrameDump (displayPtr, startPos);
			curParam = (uChar *) bufq[displayPtr]->p2;
			HiliteParam (bufq[displayPtr], curParam, startPos);
			DisplayParam (bufq[displayPtr], curParam);
			break;

		case LAST_FRAME_KEY:
			UnhiliteParam (bufq[displayPtr], curParam, startPos);
			displayPtr = PrevQueueEntry (bufqTail);
			startPos = paramPos = 0;
			ShowFrameDump (displayPtr, startPos);
			curParam = (uChar *) bufq[displayPtr]->p2;
			HiliteParam (bufq[displayPtr], curParam, startPos);
			DisplayParam (bufq[displayPtr], curParam);
			break;

		default:
			break;
		}
	}

	displayOn = CLEAR;

	werase (framewin);
	werase (parsewin);
	WindowVisible (FRAME_BUF_WIN, FALSE);
	WindowVisible (FRAME_PARSE_WIN, FALSE);

	WindowVisible (STATUS_WIN, oldstatwin);
	WindowVisible (WORK_WIN, oldworkwin);
	WindowVisible (INPUT_WIN, oldinputwin);
	WindowVisible (EVENT_WIN, oldeventwin);
	touchwin (statwin);
	touchwin (workwin);
	touchwin (inputwin);
	touchwin (eventwin);

	return;
}

void
DoFrameTrace ()
/*********************************************************************
Function:	Toggle frame trace flag.
Parameters:	None.
Input:		Uses global frameTracing.
Output:		None.
Return:		None.
*********************************************************************/
{
SMTMessage	msg;

	frameTracing = !frameTracing;

	wprintw (inputwin, "Frame tracing is now %s\n",
		frameTracing ? "ON" : "OFF" );

	/* change trace status in FBM */
	msg.destination = FBM_MSG_ID;
	msg.source = MAP_MSG_ID;
	msg.type = FBM_ACTION_FRAME_TRACE;
	msg.typeInfo = 0;
	msg.localID = 0;
	msg.len1 = 0;
	msg.len2 = 0;
	msg.data.b8 = (frameTracing != 0);
	SendMAPMessage (&msg);

	return;
}
