/*********************************************************************
	Management Application Module
	
	Message Interface Routines
	
	File:		msgifmap.c
	Created:	12/01/89

	SID:		1.2
	Last Modified:	5/7/91

	Copyright 1990,1991 XLNT Designs, Inc.

	This module provides the interface routines to handle message
	passing between MAP and the other processes in the SMT system.

	The implementation shown here demonstrates the use of UNIX V.3
	message queues.

	Modification History:
	910108-001	LJP
		PostFBMQueue() now returns an error code when the
		FBM work queue is full.
	910108-002	LJP
		StoreMAPMessage() now checks if the message queue is full.
		If it is full, then the message is dropped.
	910124-002	LJP
		Do not allocate memory with interrupts disabled.
	05/03/91	LJP
		Changed receive message handling scheme to have a static
		array for message to come in to. Also uses MarkFrameBuffer()
		to mark frames in use.
*********************************************************************/
#include        "smtdefs.h"
#include        "smttypes.h"
#include        "smtmsg.h"
#include        "fddihdr.h"
#include        "fbmframe.h"
#include        "smtmacro.h"
#include        "fbmmacro.h"
#include        "mibdefs.h"

#include	"fbmhdr.h"


/*********************************************************************
	Message Interface External Functions
*********************************************************************/

extern	void	ReceiveCSPMessage ();
extern	void	ProcessMIBMessage ();
extern	void	ProcessMAPMessage ();
extern	uInt32	PostFBMQueue ();	/* 910108-001 LJP returns a value */
extern	void	MarkFrameBuffer ();
extern  SMTStationIdType *ThisStationId;


/*********************************************************************
	Message Interface Defined Values
*********************************************************************/

/*
*	Max # of messages in message queue.
*/
#define	MAX_MSG_QUEUE	50

/*********************************************************************
	MAP Interface Global Variables
*********************************************************************/

static	Int16		msgqHead,
			msgqTail;

typedef	struct	MessageBufStruct {
	uInt32		busy;
	SMTMessage	p;
	char		data[200];
} MessageBuf;

static	MessageBuf	msgList[MAX_MSG_QUEUE];
static	uChar		msgq[MAX_MSG_QUEUE];


/*********************************************************************
	Queue Macros
*********************************************************************/

/*
*	910108-002	LJP
*	The NextQueueEntry(n) macro returns the next index with a
*	wrap to 0 if the array size will be exceeded.
*/
#define	NextQueueEntry(n)	((n == MAX_MSG_QUEUE - 1) ? 0 : n + 1)


/*********************************************************************
	Message Interface Routines
*********************************************************************/

void
SendMAPMessage (msg)
	SMTMessage	*msg;
/*********************************************************************
Function:	Send a message from MAP to another process.
Parameters:	msg	= address of buffer containing message.
Input:		Uses msqid.
		msg	= contains SMT message structure that defines the
				message to send.
Output:		None.
Return:		None.
*********************************************************************/
{
ProcState	pState;

	switch (msg->destination)
	{
	case MIB_MSG_ID:
		ProcessMIBMessage (msg);
		break;

	case CSP_MSG_ID:
		ReceiveCSPMessage (msg);
		break;

	case FBM_MSG_ID:
#if 0
{
uInt16		len;
SMTMessage	*mp;
		len = sizeof (SMTMessage) + msg->len1 + msg->len2;
		mp = (SMTMessage *) GETMEMORY (len);
		if (!mp)
			break;
		MEMCOPY ((uChar *) mp, (uChar *) msg, sizeof (SMTMessage));
		mp->p1.ptr = (uChar *) (mp + 1);
		if (msg->len1)
			MEMCOPY (mp->p1.ptr, msg->p1.ptr, msg->len1);
		mp->p2 = mp->p1.ptr + msg->len1;
		if (msg->len2)
			MEMCOPY (mp->p2, msg->p2, msg->len2);
		/* 
		*	910108-001	LJP
		*	If queue is full, drop message
		*/
		if (PostFBMQueue (FBM_Q_MESSAGE, (uChar *) mp, 0, 0, 0)
				== EFBM_Q_FULL)
			FREEMEMORY (mp);
}
#else
		MAPDPT("$MAP: Send message to FBM$...");
		ProcessFBMMessage (msg);
#endif
		break;
	}

	return;
}

void
StoreMAPMessage (msg)
	SMTMessage	*msg;
/*********************************************************************
Function:	Receive a message from another process and hold it for
		later processing.
Parameters:	msg	= pointer to an SMT message buffer.
Input:		Uses message queue.
Output:		Adds message to message queue.
Return:		None.
*********************************************************************/
{
uChar		index;
uInt16		len;
ProcState	pState;

	MDisableInterrupts (&pState);

	/* 
	*	910108-002	LJP
	*	If queue is full, ignore message.
	*/
	if (NextQueueEntry (msgqTail) == msgqHead)
	{
		MRestoreInterrupts (&pState);
		return;
	}

	/* find an empty space to store message */
	for (index = 0; index < MAX_MSG_QUEUE; index++)
		if (msgList[index].busy == 0)
			break;

	/* list is full */
	if (index >= MAX_MSG_QUEUE)
	{	
		MRestoreInterrupts (&pState);
		return;
	}


	/* get base message size */
	len = sizeof (uInt32) + sizeof (SMTMessage) + msg->len1;

	/*
	*	For messages containing frames,
	*	mark frame buffer in use.
	*/
	if (msg->type == FBM_EVENT_FRAME_TRACE ||
			msg->type == FBM_EVENT_FRAME_RECEIVED)
		MarkFrameBuffer (msg->p2);
	else
		/* add in rest of data length */
		len += msg->len2;

	if (len > sizeof (MessageBuf))
	{
		MRestoreInterrupts (&pState);
		return;
	}

	/* copy message */
	MEMCOPY (&msgList[index].p, msg, sizeof (SMTMessage));

	/* copy first buffer */
	if (msg->len1)
		MEMCOPY (msgList[index].data, msg->p1.ptr, msg->len1);

	/* copy second buffer */
	if ((msg->len2) && !(msg->type == FBM_EVENT_FRAME_TRACE ||
			msg->type == FBM_EVENT_FRAME_RECEIVED))
		MEMCOPY (&msgList[index].data[msg->len1], msg->p2, msg->len2);

	msgList[index].busy = 1;

	msgq[msgqTail] = index;
	msgqTail = NextQueueEntry (msgqTail);

	MRestoreInterrupts (&pState);

	return;
}

void
FreeMAPMessage (msg)
	SMTMessage	*msg;
/*********************************************************************
Function:	Release message data space.
Parameters:	msg	= address of an SMT message buffer.
Input:		None.
Output:		Message space freed.
Return:		None.
*********************************************************************/
{
MessageBuf	*mptr;

	mptr = (MessageBuf *) (((uChar *) msg) - sizeof (uInt32));
	if (mptr->busy > 0)
	{
		mptr->busy--;
		if (mptr->busy == 0 && (msg->type == FBM_EVENT_FRAME_TRACE ||
				msg->type == FBM_EVENT_FRAME_RECEIVED))
			FreeFrameBuffer (msg->p2);
	}

	return;
}

void
MarkMAPMessage (msg)
	SMTMessage	*msg;
/*********************************************************************
Function:	Mark message space in use.
Parameters:	msg	= address of an SMT message buffer.
Input:		None.
Output:		Message space freed.
Return:		None.
*********************************************************************/
{
MessageBuf	*mptr;

	mptr = (MessageBuf *) (((uChar *) msg) - sizeof (uInt32));
	mptr->busy++;
	return;
}

uInt32
ReceiveMAPMessage (msg)
	SMTMessage	**msg;
/*********************************************************************
Function:	Receive a message from another process.
Parameters:	msg	= address of a pointer to an SMT message buffer.
Input:		None.
Output:		msg	= set to point to a buffer containing the
				received message.
Return:		0	message received successfully.
		-1	if no message or error occurred.
Note:		This function will not block.
		It is the responsibility of the calling procedure to
		release the memory allocated for the message buffer.
*********************************************************************/
{
ProcState	pState;

	MDisableInterrupts (&pState);

	/*
	*	Get next message.
	*/
	if (msgqHead == msgqTail)
	{
		MRestoreInterrupts (&pState);
		return (-1);
	}

	*msg = &msgList[msgq[msgqHead]].p;
	
	/*
	*	910108-002	LJP
	*	Use NextQueueEntry() macro.
	*/
	msgqHead = NextQueueEntry (msgqHead);

	MRestoreInterrupts (&pState);
	return (0);
}

uInt32
InitMAPMessage ()
/*********************************************************************
Function:	Initialize system for sending and receiving SMT messages.
Parameters:	None.
Input:		None.
Output:		Set system message processing.
Return:		0 if successful, error number if failed.
*********************************************************************/
{
uChar	i;

	msgqHead = msgqTail = 0;
	for (i = 0; i < MAX_MSG_QUEUE; i++)
	{
		msgq[i] = 0;
		msgList[i].busy = 0;
	}

	return (0);
}

/*********************************************************************
        MIB Interface Routines

        The MIB interface routines are placed with the message
        interface routines because the MIB interface may require
        strong support from the message passing facilities provided
        by the system. In cases where this process is separate and
        unique from the MIB process, the message passing facilities
        will implement some form of interprocess communication.
                The strong support will be required when this process
        is waiting for a response from the MIB. In this case, any other
        messages will need to be held off or enqueued for storage
        until the MIB response is received.
                MAP does not set MIB attributes through the standard
        change interface. Therefore, only the get interface needs to
        be implemented.
*********************************************************************/

uInt32
MAPGetMIBAttr (bufSize, bufData, setCount)
        uInt16          bufSize;
        uChar           *bufData;
        SetCountType    *setCount;
/*********************************************************************
Function:       Get information from the MIB. This function provides
                MAP with access to the MIB. In cases where the MIB and
                MAP reside in the same executable image, this function
                can simply call GetMIBAttr(). If MAP and MIB are separated,
                this function will perform the necessary communication
                functions to access the MIB and wait for a reply.
Parameter:      bufSize         = maximum length of buffer bufData
                                        measured in chars.
                bufData         = pointer to buffer containing the
                                        requested attribute information
                                        and used to hold attribute value
                                        returned.
                setCount        = pointer to buffer to hold current value
                                        of the fddiSMTSetCountAttribute
                                        (may be NULL if current SetCount
                                        is not required).
Input:          bufData         = contains the requested attribute ID and
                                        object index (if required) using
                                        the MIB TLV encoding for a Get
                                        Request (see SMT standard).
Output:         bufData         = holds the attribute value requested
                                        using the MIB TLV encoding specified
                                        in the SMT standard.
                setCount        = if not NULL, this buffer gets set to the
                                        current value of fddiSMTSetCount,
                                        also using the MIB TLV encoding.
Return:         None.
Note:           The operation of this function must cause MAP to block
                (or effectively stop processing) until the response
                is received from the MIB. This routine is called
                throughout the protocol processing of MAP.
*********************************************************************/
{
extern  uInt32  GetMIBAttr ();

	return (GetMIBAttr (bufSize, bufData, setCount));
}

uInt32
MAPChangeMIBAttr (bufSize, bufData, setCount)
        uInt16          bufSize;
        uChar           *bufData;
        SetCountType    *setCount;
/*********************************************************************
Function:       Change information in the MIB. This function provides
                MAP with access to the MIB. In cases where the MIB and
                MAP reside in the same executable image, this function
		can simply call ChangeMIBAttr(). If MAP and MIB are separate,
                this function will perform the necessary communication
                functions to access the MIB and wait for a reply.
Parameter:      bufSize         = maximum length of buffer bufData
                                        measured in chars.
                bufData         = pointer to buffer containing the
					new attribute information
                                        and used to hold attribute value
                                        returned.
                setCount        = pointer to buffer to hold current value
                                        of the fddiSMTSetCountAttribute
Input:          bufData         = contains the new attribute value using
                                        the MIB TLV encoding for a Get
                                        Request (see SMT standard).
Output:         bufData         = holds the attribute value using the MIB
					TLV encoding specified
                                        in the SMT standard.
		setCount        = set to the current value of
					fddiSMTSetCount after the change
					is attempted also using the MIB
					TLV encoding.
Return:         Result code as specified by the Parameter Management
		Protocol:
		RC_SUCCESS
		RC_BAD_SET_COUNT
		RC_READ_ONLY
		RC_NO_PARAM
		RC_OUT_OF_RANGE
		RC_ILLEGAL_PARAMETER
Note:           The operation of this function must cause MAP to block
                (or effectively stop processing) until the response
                is received from the MIB. This routine is called
                throughout the protocol processing of MAP.
*********************************************************************/
{
extern  uInt32  ChangeMIBAttr ();

	return (ChangeMIBAttr (bufSize, bufData, setCount, ThisStationId));
}


