#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) cspecm.c V6.2.3:cs.622c:5:5 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module
	
	ECM State Machine
	
	File:		cspecm.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.622c	08/02/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module implements the ECM state machine listed in the
	ANSI X3T9.5 standard.
		
	This file consists of two parts. The first part consists of
	a set of functions that perform the various actions associated
	with a transition into a state. The second part implements the
	ECM state machine.

	Modification History:

	*** Updated to SMT 6.2 ***

*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"cspmacro.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include	"mibdefs.h"
#include        "dips.h"     /* jlin */


#ifdef __ECM_DEBUG 
#   define ECMDPT(print_statement)   printf(print_statement);
#else
#   define ECMDPT(print_statement)    
#endif 

/*********************************************************************
	ECM External Functions
*********************************************************************/

extern	void	SendCSPEvent ();
extern	void	SMTSendSignal ();
extern	void	SetCSPTimer ();
extern	Flag	StationPathTest ();
extern	void	BypassRequest ();
extern	uInt16	ReadLineState ();
extern  uChar   ECM_Active;

/*********************************************************************
	Support Functions
*********************************************************************/

static void
SetRE_Flag (state)
	Flag	state;
/*********************************************************************
Function:	Set the RE_Flag to the given state and notify other
		state machines of the change in the RE_Flag.
Parameters:	state	= state to set RE_Flag to.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
uInt16	i;

	ecmData.RE_Flag = state;
	SendCSPEvent (fddiSMTHoldState, 0);

	/* notify attach ports */
	for (i = 0; i < stationData.attachCount; i++)
		SMTSendSignal (SIG_PC_RE_Flag, i, (uInt32) 0);

	/* notify MACs */
	for (i = 0; i < stationData.macCount; i++)
		SMTSendSignal (SIG_RM_RE_Flag, i, (uInt32) 0);

	return;
}

static void
CSPGetUpstreamID (entity, type, uentity, utype)
	uInt16	entity;
	uInt16	type;
	uInt16	*uentity;
	uInt16	*utype;
/*********************************************************************
Function:	Get the entity upstream within the station of the given
		entity.
Parameters:	entity	= CSP index of starting entity.
		type	= PHY_TYPE for PHY, or MAC_TYPE for MAC.
		uentity	= buffer for upstream entity.
		utype	= buffer for upstream entity type.
Input:		None.
Output:		uentity	= CSP index of upstream entity.
		utype	= type of upstream entity.
Return:		None.
Modification History:
*********************************************************************/
{
Flag	foundit = FALSE;
uInt16	currentEntity;
uInt16	currentType;
uInt16	currentPath;

	/*
	*	Initialize return values to current entity.
	*/
	currentEntity = entity;
	currentType = type;

	/*
	*	Set current path to follow.
	*/
	if (type == MAC_TYPE)
		currentPath = macData[entity].pathsRequested;
	else
	{
		/*
		*	Select input path based on configuration.
		*/
		switch (cemData[currentEntity].ceState)
		{
		case CE_INSERT_P:
		case CE_INSERT_X:
			currentPath = PA_PRIMARY;
			break;

		case CE_INSERT_S:
			currentPath = PA_SECONDARY;
			break;

		default:
			currentPath = phyData[entity].pathsRequested;
			break;
		}
	}

	/*
	*	Find upstream entity.
	*/
	do
	{
		/*
		*	Move to next entity in station.
		*/
		switch (currentType)
		{
		case MAC_TYPE:
			/*
			*	If first MAC, then skip to last port.
			*/
			if (currentEntity == MAC_1)
			{
				if (stationData.masterCount > 0)
					currentEntity
						= stationData.phyCount - 1;
				else
					/* use A for A or S */
					currentEntity = PHY_A;
				currentType = PHY_TYPE;
			}
			else
				/*
				*	Otherwise just move to next MAC.
				*/
				currentEntity--;
			break;

		case PHY_TYPE:
			/* S or B port is last port in path */
			if ((phyData[currentEntity].PC_Type == PC_Type_B)
				|| (phyData[currentEntity].PC_Type == PC_Type_S))
			{
				currentEntity = stationData.macCount - 1;
				currentType = MAC_TYPE;
			}
			else
			{
				/*
				*	A or M moves to next PHY.
				*/
				if (phyData[currentEntity].PC_Type
						== PC_Type_A)
					/* A port move to B */
					currentEntity = PHY_B;
				else
				{
					/* M port goes to A before B */
					currentEntity--;
					if (phyData[currentEntity].PC_Type
							== PC_Type_B)
						currentEntity = PHY_A;
				}
			}
			break;
		}

		/*
		*	Check if this entity is connected on this path.
		*/
		switch (currentType)
		{
		case MAC_TYPE:
			if (macData[currentEntity].operational
				&& rmtData[currentEntity].RM_Join
				&& (macData[currentEntity].pathsRequested
					== currentPath))
			{
				/* this is it, exit loop */
				foundit = TRUE;
			}
			else
			{
			   if (get_debug() == DEBUG_ON)
			      printf("ECM: MAC %d is not available on %s path.\n",
				     currentEntity, (currentPath == PA_PRIMARY)
				     ? "Primary" : "Secondary");
			}
			break;

		case PHY_TYPE:
			if (phyData[currentEntity].operational
				&& cemData[currentEntity].CF_Join)
			{
				/*
				*	Check configuration to see if PHY
				*	is on the path.
				*/
				switch (cemData[currentEntity].ceState)
				{
				case CE_INSERT_P:
					foundit = (currentPath == PA_PRIMARY);
					break;

				case CE_INSERT_S:
					foundit = (currentPath == PA_SECONDARY);
					break;

				case CE_INSERT_X:
					if (currentPath == PA_SECONDARY)
						foundit = SET;
					else
						/*
						*	If this is not it,
						*	then the path switches.
						*/
						currentPath = PA_SECONDARY;
					break;
				}
			}
			else 
			{
			   if (get_debug() == DEBUG_ON)
			      printf("ECM: PHY %d is not available on %s path.\n",
				     currentEntity, (currentPath == PA_PRIMARY)
				     ? "Primary" : "Secondary");
			}
		}
	} while (!foundit);

	*uentity = currentEntity;
	*utype = currentType;

	return;
}


/*********************************************************************
	State Transition Functions
*********************************************************************/

static void
Out_Actions ()
/*********************************************************************
Function:	Process entry to EC0:OUT.
Parameters:	None.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
	/* set current ECM state */
        ECMDPT("Out_Action..");
	ecmData.ecState = EC_OUT;
	SendCSPEvent (fddiSMTECMState, 0);
#if 1    /* jlin: Should disable the brip. 06/17/93 */
        DisableFDDIBrip();
#endif

	return;
}

static void
In_Actions ()
/*********************************************************************
Function:	Process entry to EC1:IN.
Parameters:	None.
Input:		Uses ecmData & stationData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
uInt16	i;				/* loop counter */

        ECMDPT("In_Action.");
	/* set current ECM state */
	ecmData.ecState = EC_IN;
	SendCSPEvent (fddiSMTECMState, 0);

	/* set RE_Flag */
	SetRE_Flag (SET);

	/*
	*	CLEAR *:Trace_Prop
	*	Note: Since Trace_Prop is used as a signal to ECM it does
	*	not need to be explicitly cleared as described in SMT.
	*/

	/* send *:PC_Start */
	for (i = 0; i < stationData.phyCount; i++)
		if (phyData[i].operational)
			SMTSendSignal (SIG_PC_Start, i, (uInt32) 0);

#if 1    /* jlin: Should enable the brip. 06/17/93 */
        EnableFDDIBrip();
#endif
	return;
}

static void
Trace_Actions ()
/*********************************************************************
Function:	Process entry to EC2:TRACE
Parameters:	None.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
	/* set current ECM state */
        ECMDPT("Trace_Action.");
	ecmData.ecState = EC_TRACE;
	SendCSPEvent (fddiSMTECMState, 0);
	
	/* reset TEC */
	SetCSPTimer (stationData.Trace_Max, ECM_SIGNALS, 0, &ecmData.TEC);
	
	return;
}

static void
Leave_Actions ()
/*********************************************************************
Function:	Process entry to EC3:LEAVE
Parameters:	None.
Input:		Uses ecmData & stationData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
Int16	i;				/* loop counter */

        ECMDPT("Leave_Action.");
	/* set current ECM state */
	ecmData.ecState = EC_LEAVE;
	SendCSPEvent (fddiSMTECMState, 0);

#if 1    /* jlin: Should disable the brip. 06/17/93 */
        DisableFDDIBrip();
#endif
	
	/* send *:PC_Stop */
	for (i = 0; i < stationData.phyCount; i++)
		SMTSendSignal (SIG_PC_Stop, i, (uInt32) 0);
		
	/* reset TEC */
	SetCSPTimer (TD_Min, ECM_SIGNALS, 0, &ecmData.TEC);
	
	return;
}

static void
Path_Test_Actions ()
/*********************************************************************
Function:	Process entry to EC4:PATH_TEST.
Parameters:	None.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
         ECMDPT("Path_Test_Action.");
	/* set current ECM state */
	ecmData.ecState = EC_PATH_TEST;
	SendCSPEvent (fddiSMTECMState, 0);
	
	/* set Path_Test = Testing */
	ecmData.Path_Test = PT_Testing;
	
	/* perform path test */
	if (StationPathTest ())
		/* path test passed */
		SMTSendSignal (SIG_Path_Test, 0, (uInt32) PT_Passed);
	else
		/* path test failed */
		SMTSendSignal (SIG_Path_Test, 0, (uInt32) PT_Failed);
		
	return;
}

static void
Insert_Actions ()
/*********************************************************************
Function:	Process entry to EC5:INSERT.
Parameters:	None.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
        ECMDPT("Insert_Action.");
	/* set current ECM state */
	ecmData.ecState = EC_INSERT;
	SendCSPEvent (fddiSMTECMState, 0);
	
	/* SM_PM_BYPASS.request(Insert) */
	BypassRequest (INSERTED);

	/* reset TEC */
	SetCSPTimer (stationData.I_Max, ECM_SIGNALS, 0, &ecmData.TEC);

	return;
}

static void
Check_Actions ()
/*********************************************************************
Function:	Process entry to EC6:CHECK.
Parameters:	None.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
        ECMDPT("Check_Action.");
	/* set current ECM state */
	ecmData.ecState = EC_CHECK;
	SendCSPEvent (fddiSMTECMState, 0);

	/* reset TEC */
	SetCSPTimer ((uInt32) 0, 0, 0, &ecmData.TEC);

	/*
	*	Check attach ports for stuck bypass
	*	condition.
	*/
	if (stationData.attachCount == 2)
	{
		/* get current line state */
		pcmData[PHY_A].PC_LS = ReadLineState (PHY_A);
		pcmData[PHY_B].PC_LS = ReadLineState (PHY_B);

		if ((pcmData[PHY_A].PC_LS == QLS
				|| pcmData[PHY_A].PC_LS == HLS)
			&& (pcmData[PHY_B].PC_LS == QLS
				|| pcmData[PHY_B].PC_LS == HLS))
		{
			/* EC(61) */
			ecmData.SB_Flag = CLEAR;
			SendCSPEvent (xdiSMTSB_Flag, 0);
			In_Actions ();
		}
		else if (!ecmData.SB_Flag
			&& (((pcmData[PHY_A].PC_LS == ILS)
				&& (pcmData[PHY_B].PC_LS == QLS))
			|| ((pcmData[PHY_A].PC_LS == QLS)
				&& (pcmData[PHY_B].PC_LS == ILS))))
		{
			/* EC(66) */
			ecmData.SB_Flag = SET;
			SendCSPEvent (xdiSMTSB_Flag, 0);
		}
	}
	else
	{
		if (pcmData[PHY_S].PC_LS == QLS
			|| pcmData[PHY_S].PC_LS == HLS)
		{
			/* EC(61) */
			ecmData.SB_Flag = CLEAR;
			SendCSPEvent (xdiSMTSB_Flag, 0);
			In_Actions ();
		}
		else if (!ecmData.SB_Flag
			&& (pcmData[PHY_S].PC_LS == ILS))
		{
			/* EC(66) */
			ecmData.SB_Flag = SET;
			SendCSPEvent (xdiSMTSB_Flag, 0);
		}
	}

	return;
}

static void
Deinsert_Actions ()
/*********************************************************************
Function:	Process entry to EC7:DEINSERT.
Parameters:	None.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
        ECMDPT("Deinsert_Action.");
	/* set current ECM state */
	ecmData.ecState = EC_DEINSERT;
	SendCSPEvent (fddiSMTECMState, 0);

	/* SM_PM_BYPASS.request(Insert) */
	BypassRequest (DEINSERTED);

	/* reset TEC */
	SetCSPTimer (stationData.I_Max, ECM_SIGNALS, 0, &ecmData.TEC);

	return;
}

static void
Prop_Actions (entity, type)
	uInt16	entity;
	uInt32	type;
/*********************************************************************
Function:	Process actions associated with a Trace_Prop signal.
Parameters:	entity	= CSP ID of entity issuing Trace_Prop signal.
		type	= PHY_TYPE or MAC_TYPE indicating type of entity.
Input:		None.
Output:		None.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
uInt16	nextEntity, nextType;		/* next trace entity indices */

        ECMDPT("Prop_Action.");
	/* Get entity on input */
	CSPGetUpstreamID (entity, (uInt16) type, &nextEntity, &nextType);

	/* if entity with output to PHY is a MAC */
	if (nextType == MAC_TYPE)
	{
		/* SIGNAL Trace_Terminated */
	        if (get_debug() == DEBUG_ON)
		   printf("ECM: Trace Terminated on MAC %d.\n", nextEntity);
		rmtData[nextEntity].traceStatus = Trace_Terminated;
		SendCSPEvent (fddiPATHClassPATHTraceStatus, nextEntity);

		/* set Path_Test = Pending */
		SMTSendSignal (SIG_Path_Test, 0, (uInt32) PT_Pending);
	}
	else
	{
		/* SIGNAL Trace_Propagated */
		if (phyData[nextEntity].pathsRequested & PA_PRIMARY)
		{
			rmtData[PRIMARY_MAC].traceStatus = Trace_Propagated;
			SendCSPEvent (fddiPATHClassPATHTraceStatus,
				PRIMARY_MAC);
		}
		else
		{
			rmtData[SECONDARY_MAC].traceStatus = Trace_Propagated;
			SendCSPEvent (fddiPATHClassPATHTraceStatus,
				SECONDARY_MAC);
		}

		/* propagate trace */
		if (get_debug() == DEBUG_ON)
		    printf("ECM: Trace propagated on PHY %d\n", nextEntity);
		SMTSendSignal (SIG_PC_Trace, nextEntity, (uInt32) nextType);
	}

	return;
}


/*********************************************************************
	ECM State Machine
*********************************************************************/

void
SignalECM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process an ECM signal.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses ecmData.
Output:		Changes ecmData.
Return:		No value returned.
Notes:		The value of Path_Test may be set at any number of places
		in ECM or PCM. To facilitate the detection of changes in
		Path_Test outside of the state machine, the signal
		SIG_Path_Test is used. However, the ECM state machine may
		also set Path_Test during transitions. When Path_Test is
		set during a transition, the variable is set immediately
		and no signal is made to ECM.
Modification History:
*********************************************************************/
{
	/*
	*	First process any changes to Path_Test.
	*/
	if (sigType == SIG_Path_Test)
		ecmData.Path_Test = (uChar) sigData;

	/*
	*	If timer has expired, then verify this timer is the current
	*	event being timed.
	*/
	if (sigType == SIG_EC_Timer)
	{
		if (ecmData.TEC != sigData)
			return;			/* not current timer */
		else
			ecmData.TEC = 0;	/* mark timer as expired */
		ECMDPT("\nEC_Timer timeout");
	}

	/*
	*	Select ECM state for processing.
	*/
	switch (ecmData.ecState)
	{
	/*
	*	EC0:OUT
	*/
	case EC_OUT:
		ECMDPT("\nEC0_OUT: ");
		/*
		*	In this state, only one signal will case processing to 
		*	occur - Connect. This will process only when Path_Test
		*	is Passed.
		*/
		if ((sigType == SIG_Connect) 
			&& (ecmData.Path_Test == PT_Passed))
		{
			/* check station has a bypass switch */
			if (stationData.Bypass)
				/* EC(05) */
				Insert_Actions ();
			else
				/* EC(01) */
				In_Actions ();
		}	
		break;

	/*
	*	EC1:IN
	*/
	case EC_IN:
		ECMDPT("\nEC1_IN: ");
		/*
		*	Select action on signal.
		*/
		switch (sigType)
		{		
		/*
		*	This case implements the optional hold policy.
		*/
		case SIG_EC_NO_Flag:
			/* EC(11a) */
			if (ecmData.RE_Flag
				&& ecmData.Hold
				&& !rmtData[PRIMARY_MAC].NO_Flag
				&& !rmtData[SECONDARY_MAC].NO_Flag
				&& (stationData.cfState == CF_THRU))
			{
			        ECMDPT("Recovery Disable.");
				/* clear RE_Flag */
				SetRE_Flag (CLEAR);
			}

			/* EC(11b) */
			else if (!ecmData.RE_Flag
				&& (!ecmData.Hold
					|| (rmtData[PRIMARY_MAC].NO_Flag
					    && rmtData[SECONDARY_MAC].NO_Flag)
					|| (stationData.cfState != CF_THRU)))
			{
				/* set RE_Flag */
			        ECMDPT("Recovery Enable.");
				SetRE_Flag (SET);
			}
			break;

		/* EC(12) */
		case SIG_Trace_Prop:
			Prop_Actions (sigEntity, sigData);
			Trace_Actions ();
			break;

		/* EC(13) */
		case SIG_Disconnect:
			Leave_Actions ();
			break;
		}
		break;

	/*
	*	EC2:TRACE
	*/
	case EC_TRACE:
		ECMDPT("\nEC2_TRACE: ");
		switch (sigType)
		{
		/* EC(22) */
		case SIG_Trace_Prop:
			/* EC(22) */
			Prop_Actions (sigEntity, sigData);
			break;

		/* EC(23a) */
		case SIG_Disconnect:
			ecmData.Path_Test = PT_Exiting;
			Leave_Actions ();
			break;

		/* EC(23b) */
		case SIG_Path_Test:
			if (ecmData.Path_Test == PT_Pending)
				Leave_Actions ();
			break;

		/* EC(23c) */
		case SIG_EC_Timer:
			ecmData.Path_Test = PT_Pending;
			Leave_Actions ();
			break;
		}
		break;

	/*
	*	EC3:LEAVE
	*/
	case EC_LEAVE:
		ECMDPT("\nEC3_LEAVE: ");
		switch (sigType)
		{
		case SIG_EC_Timer:
			if (ecmData.Path_Test == PT_Pending)
				/* EC(34) */
				Path_Test_Actions ();
			else
				if (stationData.Bypass)
					/* EC(37) */
					Deinsert_Actions ();
				else
					/* EC(30) */
					Out_Actions ();
			break;

		/* EC(31) */
		case SIG_Connect:
			if (ecmData.Path_Test == PT_Passed)
				In_Actions ();
			break;

		/* EC(33) */
		case SIG_Disconnect:
			if (ecmData.Path_Test == PT_Pending)
				ecmData.Path_Test = PT_Exiting;
			break;
		}
		break;

	/*
	*	EC4:PATH_TEST
	*/
	case EC_PATH_TEST:
		ECMDPT("\nEC4_PATH_TEST: ");
		switch (sigType)
		{
		case SIG_Path_Test:
			if (ecmData.Path_Test == PT_Failed)
			{
			        ECMDPT("Path_Test_Failed.");
				if (stationData.Bypass)
					/* EC(47a) */
					Deinsert_Actions ();
				else
					/* EC(40a) */
					Out_Actions ();
			}
			else if (ecmData.Path_Test == PT_Passed)
				/* EC(41) */
				In_Actions ();
			break;

		case SIG_Disconnect:
			if (stationData.Bypass)
				/* EC(47b) */
				Deinsert_Actions ();
			else
				/* EC(40b) */
				Out_Actions ();
			break;
		}
		break;

	/*
	*	EC5:INSERT
	*/
	case EC_INSERT:
		ECMDPT("\nEC5_INSERT: ");
		switch (sigType)
		{
		/* EC(56) */
		case SIG_EC_Timer:
			Check_Actions ();
			break;

		/* EC(57) */
		case SIG_Disconnect:
			Deinsert_Actions ();
			break;
		}
		break;

	/*
	*	EC6:CHECK
	*/
	case EC_CHECK:
		ECMDPT("\nEC6_CHECK: ");
		switch (sigType)
		{
		case SIG_EC_Test_BP:
			/*
			*	Check attach ports for stuck bypass
			*	condition.
			*/
			if (stationData.attachCount == 2)
			{
				if ((pcmData[PHY_A].PC_LS & (QLS | HLS))
					&& (pcmData[PHY_B].PC_LS & (QLS | HLS)))
				{
					/* EC(61) */
					ecmData.SB_Flag = CLEAR;
					SendCSPEvent (xdiSMTSB_Flag, 0);
					In_Actions ();
				}
				else if (!ecmData.SB_Flag
					&& (((pcmData[PHY_A].PC_LS & ILS)
						&& (pcmData[PHY_B].PC_LS & QLS))
					|| ((pcmData[PHY_A].PC_LS & QLS)
						&& (pcmData[PHY_B].PC_LS & ILS))))
				{
					/* EC(66) */
					ecmData.SB_Flag = SET;
					SendCSPEvent (xdiSMTSB_Flag, 0);
				}
			}
			else
			{
				if (pcmData[PHY_S].PC_LS & (QLS | HLS))
				{
					/* EC(61) */
					ecmData.SB_Flag = CLEAR;
					SendCSPEvent (xdiSMTSB_Flag, 0);
					In_Actions ();
				}
				else if (!ecmData.SB_Flag
					&& (pcmData[PHY_S].PC_LS & ILS))
				{
					/* EC(66) */
					ecmData.SB_Flag = SET;
					SendCSPEvent (xdiSMTSB_Flag, 0);
				}
			}
			break;

		/* EC(67) */
		case SIG_Disconnect:
			Deinsert_Actions ();
			break;
		}
		break;

	/*
	*	EC7:DEINSERT
	*/
	case EC_DEINSERT:
		ECMDPT("\nEC7_DEINSERT: ");
		switch (sigType)
		{
		/* EC(70) */
		case SIG_EC_Timer:
			Out_Actions ();
			break;
			
		/* EC(75) */
		case SIG_Connect:
			if (ecmData.Path_Test == PT_Passed)
				Insert_Actions ();
			break;
		}
		break;
	}

	return;
}
