#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) cspmsg.c V6.2.3:cs.911022:8:8 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module
	
	Message Processing Routines
	
	File:		cspmsg.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911022	10/28/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module provides the message processing routines for
	handling messages received by CSP.

	Modification History:

	*** Updated to SMT 6.2 ***

	910220-001	LJP
		Maintenance line states are reported from the MIB using
		SMT line state values. The fddiPORTMaintLineState in
		ProcessCSPMIBChange() and fddiPORTAction for maintPORT 
		in ProcessCSPMessage() did not convert to hardware line
		state values for the SendLineState() routine. This fix
		incorporates that translation.
	910326-002	LJP
		After setting the new maintenance line state, a signal
		must be sent to PCM to cause the new line state to
		be used.
	910807-003	LJP
		On change to fddiPORTPathsRequested, issue PC_Start
		signal to restart connection.
	911025-001	LJP
		On change to fddiSMTConfigPolicy, issue PC_Start to A if
		either A or B is in Tree mode.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"fddihdr.h"
#include	"smtmacro.h"
#include	"smtmsg.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include	"mibdefs.h"

/*********************************************************************
	External Functions
*********************************************************************/

extern	uInt32	InitCSPMIBData ();
extern	void	SendCSPMessage ();
extern	void	SMTSendSignal ();
extern	uInt32	LocalCSPMessages ();
extern	Flag	Allow_Connection ();

/*********************************************************************
	Message Processing Routines
*********************************************************************/

void
SendCSPEvent (eventType, entity)
	uInt16	eventType;
	uInt16	entity;
/*********************************************************************
Function:	Send a CSP event message.
Parameters:	eventType	= type of event record.
		entity		= PHY or MAC index if needed.
Input:		Uses CSP global data.
		entity	= PHY or MAC index or 0 if not needed.
Output:		None.
Return:		None.
Notes:		This function takes the global data based on the event
		type to reduce processing time.
Modification History:
*********************************************************************/
{
SMTMessage	eventMsg;		/* message to send */

	/*
	*	Create header.
	*/
	eventMsg.source = CSP_MSG_ID;
	eventMsg.destination = MIB_MSG_ID;
	eventMsg.type = CSP_EVENT_NOTIFY_MIB;
	eventMsg.typeInfo = eventType;
	eventMsg.entity = entity;
	eventMsg.localID = 0;
	eventMsg.len1 = 0;
	eventMsg.len2 = 0;

	/*
	*	Fill in data.
	*/
	switch (eventType)
	{
	/*
	*	Update MIB information.
	*/
	case fddiSMTECMState:
		eventMsg.data.b8 = ecmData.ecState;
		break;

	case fddiSMTCF_State:
		eventMsg.data.b8 = stationData.cfState;
		break;

	case fddiSMTHoldState:
		eventMsg.data.b16 = 0;

		/* determine hold state */
		if (ecmData.RE_Flag)
			eventMsg.data.b16 = HS_Not_Holding;
		else
		{
			/* check which (or both) ring is holding */
			if (!rmtData[PRIMARY_MAC].NO_Flag)
				eventMsg.data.b16 |= HS_Holding_Prm;
			if (!rmtData[SECONDARY_MAC].NO_Flag)
				eventMsg.data.b16 |= HS_Holding_Sec;
		}
		break;

	case fddiMACCurrentPath:
		if (rmtData[entity].RM_Loop)
			eventMsg.data.b16 = PA_LOCAL;
		else if (rmtData[entity].RM_Join)
			eventMsg.data.b16 = macData[entity].pathsRequested;
		else
			eventMsg.data.b16 = PA_ISOLATED;
		break;

	case fddiMACRMTState:
		eventMsg.data.b8 = rmtData[entity].rmState;
		break;

	case fddiMACDa_Flag:
		eventMsg.data.b8 = rmtData[entity].DA_Flag;
		break;

	case fddiPATHClassPATHTraceStatus:
		eventMsg.data.b16 = rmtData[entity].traceStatus;
		break;

	case fddiPATHClassPATHStatus:
		eventMsg.data.b16 = 0;
		break;

	case fddiPORTPC_Neighbor:
		eventMsg.data.b8 = phyData[entity].PC_Neighbor;
		break;

	case fddiPORTRemoteMACIndicated:
		eventMsg.data.b8
			= (MReadBit (pcmData[entity].R_Val, 9)) ? SET : CLEAR;
		break;

	case fddiPORTCE_State:
		eventMsg.data.b8 = cemData[entity].ceState;
		break;

	case fddiPORTLCTFail_Ct:
		eventMsg.data.b32 = phyData[entity].PC_LCT_Fail;
		break;

	case fddiPORTLer_Estimate:
		eventMsg.data.b8 = lemData[entity].estimate;
		break;

	case fddiPORTLem_Reject_Ct:
		eventMsg.data.b32 = lemData[entity].LEM_Reject_Ct;
		break;

	case fddiPORTLem_Ct:
		eventMsg.data.b32 = lemData[entity].lemCount;
		break;

	case fddiPORTPCMState:
		eventMsg.data.b8 = pcmData[entity].pcState;
		break;

	case fddiPORTPC_Withhold:
		eventMsg.data.b8 = phyData[entity].PC_Withhold;
		break;

	case fddiPORTBS_Flag:
		eventMsg.data.b8 = pcmData[entity].BS_Flag;
		break;

	case xdiSMTSB_Flag:
		eventMsg.data.b8 = ecmData.SB_Flag;
		break;

	case xdiMACDataAddress:
		/*
		*	RMT caused address to change.
		*/
		MCopyAddress (eventMsg.data.addr, macData[entity].dataAddress);
		break;

	case fddiMACT_Req:
		/* convert to FddiTimerTwosComplement */
		eventMsg.data.b32 = -MUsecs2Fddi (Jam_T_Req_Value);
		break;

	case fddiSMTConfigurationChgEvent:
		/*
		*	No data required.
		*/
		break;

	default:
		return;
	}

	/*
	*	Send message.
	*/
	SendCSPMessage (&eventMsg);

	return;
}

void
ProcessCSPMIBChange (smtmsg)
	SMTMessage	*smtmsg;
/*********************************************************************
Function:	Process change from the MIB.
Parameters:	smtmsg	= address of message buffer.
Input:		smtmsg	= contains message to process.
Output:		Changes global data and/or posts signals to the signal
		queue depending upon the message.
Return:		None.
Notes:		None.
Modification History:
	910220-001	LJP
*********************************************************************/
{
Int16		i;			/* loop counter */

	switch (smtmsg->typeInfo)
	{
	case fddiSMTConfigPolicy:
#ifdef NO_THRU_B
		/*
		*	If Thru_B is changed, then notify CFM.
		*/
		if ((stationData.configPolicy ^ (*(uChar *) smtmsg->data))
			& Configuration_Thru_B)
				SMTSendSignal (SIG_CF_Thru_B, 0, (uInt32) 0);
#endif
		/*
		*	If Hold policy changed, notify ECM.
		*/

		/*
		*	If CF_Wrap_AB changed, notify CFM.
		*	Removed notificat to CFM (see below).
		*/
		if ((stationData.configPolicy ^ smtmsg->data.b16)
				& Config_CF_Wrap_AB)
		{
			/*
			*	911025-001	LJP
			*	Removed notification to CFM.
			*	Send SIG_PC_Start to B if A or B
			*	is in Tree mode on DAS.
			*/
                        if ((stationData.attachCount == 2)
				&& ((phyData[PHY_A].PC_Mode == PC_Mode_Tree)
                        	  || (phyData[PHY_B].PC_Mode == PC_Mode_Tree)))
				SMTSendSignal (SIG_PC_Start, PHY_B, (uInt32) 0);
		}

		/*
		*	Set new value.
		*/
		stationData.configPolicy = smtmsg->data.b16;
		break;

	case fddiSMTConnectionPolicy:
		stationData.connectionPolicy = smtmsg->data.b16;
		/*
		*	Verify that all connections are valid.
		*/
		for (i = 0; i < stationData.phyCount; i++)
		{
			if (pcmData[i].pcState == PC_ACTIVE)
				if (!Allow_Connection (i))
					SMTSendSignal (SIG_PC_Start, i,
						(uInt32) 0);
		}
		break;

	case fddiMACUpstreamNbr:
		MCopyAddress (macData[smtmsg->entity].upstreamNbr,
			smtmsg->data.addr);
		break;

	case fddiMACDup_Addr_Test:
		/*
		*	FBM has had a change in the duplicate address
		*	detection status.
		*/
		SMTSendSignal (SIG_Dup_Addr, smtmsg->entity,
			(uInt32) smtmsg->data.b16);
		break;

	case fddiMACPathsRequested:
		break;

	case fddiPATHClassTrace_MaxExpiration:
		stationData.Trace_Max = smtmsg->data.b32;
		break;

	case fddiPATHClassT_MaxLowerBound:
		/*
		*	Since the PATHClass objects are currently
		*	implemented for the entire station. Both primary
		*	and secondary MACs get set to this new value.
		*/
		macData[PRIMARY_MAC].RM34cTime =
			MFddi2Usecs (-smtmsg->data.b32);
		if (macData[PRIMARY_MAC].RM34cTime > T_RM34c)
			macData[PRIMARY_MAC].RM34cTime -= T_RM34c;
		else
			macData[PRIMARY_MAC].RM34cTime = 0;
		if (stationData.macCount >= 2)
		{
		   macData[SECONDARY_MAC].RM34cTime =
			     macData[PRIMARY_MAC].RM34cTime;
		}
		break;

	case fddiPATHClassPATHT_Rmode:
		/* convert from FddiTime */
		stationData.T_Rmode = MFddi2Usecs (smtmsg->data.b32);
		break;

	case fddiPORTConnectionPolicies:
		phyData[smtmsg->entity].connectionPolicies = smtmsg->data.b8;
		break;

	case fddiPORTPathsRequested:
		/*
		*	910807-003	LJP
		*	If new path requested, restart connection.
		*/
		if (phyData[smtmsg->entity].pathsRequested != smtmsg->data.b8)
		{
			phyData[smtmsg->entity].pathsRequested
				= smtmsg->data.b8;
			SMTSendSignal (SIG_PC_Start, smtmsg->entity, (uInt32) 0);
		}
		break;

	case fddiPORTMACLoop_Time:
		/* convert from FddiTime */
		phyData[smtmsg->entity].MACLoop_Time =
			MFddi2Usecs (smtmsg->data.b32);
		break;

	case fddiPORTMaintLineState:
		/*
		*	910220-001 LJP
		*	Convert from SMT to CSP line state value.
		*/
		switch (smtmsg->data.b8)
		{
		case Maint_QLS:
			pcmData[smtmsg->entity].Maint_LS = QLS;
			break;

		case Maint_ILS:
			pcmData[smtmsg->entity].Maint_LS = ILS;
			break;

		case Maint_MLS:
			pcmData[smtmsg->entity].Maint_LS = MLS;
			break;

		case Maint_HLS:
			pcmData[smtmsg->entity].Maint_LS = HLS;
			break;

		case Maint_PDR:
			pcmData[smtmsg->entity].Maint_LS = PDRLS;
			break;
		}

		/*
		*	910326-002	LJP
		*	Post signal to PCM to affect change.
		*/
		SMTSendSignal (SIG_PC_Maint, smtmsg->entity, 
			(uInt32) pcmData[smtmsg->entity].Maint_LS);
		break;

	case fddiPORTTB_Max:
		/* convert from FddiTimerTwosComplement */
		phyData[smtmsg->entity].TB_Max =
			MFddi2Usecs (-smtmsg->data.b32);
		break;

	case fddiPORTLer_Cutoff:
		lemData[smtmsg->entity].cutoffRate = smtmsg->data.b8;
		break;

	case fddiATTACHMENTI_MaxExpiration:
		/* convert from FddiTimerTwosComplement */
		stationData.I_Max = MFddi2Usecs (-smtmsg->data.b32);
		break;

	case xdiSMTBothWrapPolicy:
		stationData.bothWrapPolicy = smtmsg->data.b8;
		break;

	case xdiMACRMTDupPolicy1:
		macData[smtmsg->entity].dupPolicy1 = smtmsg->data.b8;
		break;

	case xdiMACRMTDupPolicy2:
		macData[smtmsg->entity].dupPolicy2 = smtmsg->data.b8;
		break;

	case xdiMACDataAddress:
		MCopyAddress (macData[smtmsg->entity].dataAddress,
			smtmsg->data.addr);
		break;
	}

	return;
}

void
ProcessCSPMessage (smtmsg)
	SMTMessage	*smtmsg;
/*********************************************************************
Function:	Process an SMT message.
Parameters:	smtmsg	= address of message buffer.
Input:		smtmsg	= contains message to process.
Output:		Changes global data and/or posts signals to the signal
		queue depending upon the message.
Return:		None.
Notes:		None.
Modification History:
	910220-001	LJP
*********************************************************************/
{
	/*
	*	Select message type.
	*/
	switch (smtmsg->type)
	{
	/*
	*	Event messages.
	*/
	case MIB_EVENT_NOTIFY_CHANGE:
		ProcessCSPMIBChange (smtmsg);
		break;

	/*
	*	Action messages.
	*/
	case CSP_ACTION_FDDI_SMT:
		/*
		*	fddiSMTStationAction
		*/
		switch (smtmsg->typeInfo)
		{
		case StationAction_Connect:
			SMTSendSignal (SIG_Connect, 0, (uInt32) 0);
			break;

		case StationAction_Disconnect:
			SMTSendSignal (SIG_Disconnect, 0, (uInt32) 0);
			break;

		case StationAction_Path_Test:
		case StationAction_Self_Test:
			break;
		}
		break;

	case CSP_ACTION_FDDI_MAC:
		/*
		*	fddiMACAction
		*/
		break;

	case CSP_ACTION_FDDI_PORT:
		/*
		*	fddiPORTAction
		*/
		switch (smtmsg->typeInfo)
		{
		case PORTAction_Maint:
			/* Take maintenance line state from data */
			/*
			*	910220-001 LJP
			*	Convert from SMT to CSP line state value.
			*/
			switch (smtmsg->data.b8)
			{
			case Maint_QLS:
				pcmData[smtmsg->entity].Maint_LS = QLS;
				break;

			case Maint_ILS:
				pcmData[smtmsg->entity].Maint_LS = ILS;
				break;

			case Maint_MLS:
				pcmData[smtmsg->entity].Maint_LS = MLS;
				break;

			case Maint_HLS:
				pcmData[smtmsg->entity].Maint_LS = HLS;
				break;

			case Maint_PDR:
				pcmData[smtmsg->entity].Maint_LS = PDRLS;
				break;
			}

			SMTSendSignal (SIG_PC_Maint, smtmsg->entity,
				pcmData[smtmsg->entity].Maint_LS);
			break;

		case PORTAction_Enable:
			SMTSendSignal (SIG_PC_Enable, smtmsg->entity, (uInt32) 0);
			break;

		case PORTAction_Disable:
			SMTSendSignal (SIG_PC_Disable, smtmsg->entity, (uInt32) 0);
			break;

		case PORTAction_Start:
			SMTSendSignal (SIG_PC_Start, smtmsg->entity, (uInt32) 0);
			break;

		case PORTAction_Stop:
			SMTSendSignal (SIG_PC_Stop, smtmsg->entity, (uInt32) 0);
			break;
		}
		break;


	/*
	*	Response messages.
	*/

	/*
	*	Local messages.
	*/
	default:
		LocalCSPMessages (smtmsg);
		break;
	}

	return;
}

