#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) csppcm.c V6.2.3:cs.911031:8:8 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module

	PCM State Machine

	File:		csppcm.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911031	10/31/91

	Copyright 1990,1991 XLNT Designs, Inc.

	This module implements the PCM state machine listed in the
	ANSI X3T9.5 standard.

	Modification History:

	*** Updated to SMT 6.2 ***
	910122-001	LJP
		Added code to turn off SILS detect during LCT and MAC
		loopback. Turned off in PC(44a) for n=6 or n=8 and 
		turned on in PC(55a) or PC(55b).
	910503-001	LJP
		In SignalPCM(), the SIG_PC_Timer signal checks which
		timer receives the signal and clears that timer's ID.
		The code was clearing the TPC ID no matter what timer
		was being signalled. This has been corrected.
	910605-004	LJP
		Clear PC_Neighbor when PCM enters OFF or BREAK.
	910627-001	LJP
		Need to check for MLS received after TL_Min time in
		PC(88b1). This assures the code that if Trace is
		being propagated into this station, it is detected.
	910920-001	LJP
		Made SILS and ILS transitions consistent. All ILS
		transitions are now based on SILS since SILS implies
		ILS and SILS generates an interrupt.
	911023-001	LJP
		Corrected enable/disable of SILS detection. Now, SILS
		detection is disabled on all PC(44a) transitions and is
		enabled at the end of Signal_Actions().
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"cspmacro.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include	"mibdefs.h"

#ifdef __PCM_DEBUG 
#   define PCMDPT(print_statement)   if (sigEntity==2) printf(print_statement);

#   define PCMDPT1(print_statement,a) if (sigEntity==2) printf(print_statement,a);
#else
#   define PCMDPT(print_statement)   
#   define PCMDPT1(print_statement,a) 
#endif 


/*********************************************************************
	PCM State Machine State Entry Functions
*********************************************************************/

extern	void	SetCSPTimer ();
extern	void	TransmitPMD ();
extern	void	SendLineState ();
extern	void	SMTSendSignal ();
extern	void	SendCSPEvent ();
extern	uInt16	ReadLineState ();
extern	void	PC_TCode_Actions ();
extern	void	PC_RCode_Actions ();
extern	void	SetSILSDetect ();
extern  void    StopLCT();                   /* Advisory 6.2.3-018 */


/*********************************************************************
	PCM State Machine State Entry Functions
*********************************************************************/

static void
Off_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into OFF.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Off_Entry.", sigEntity);
	/* reset TPC */
	SetCSPTimer ((uTime) 0, 0, 0, &pcmData[sigEntity].TPC);

	return;
}

static void
Break_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into BREAK.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
uInt32	tval = 0;		/* timeout value */

        PCMDPT1("Port%d:Break_Entry.", sigEntity);
	/* determine timer value based on flag settings */
	if (!pcmData[sigEntity].LS_Flag)
		/*
		*	First time through so wait for TB_Min.
		*/
		tval = TB_Min;

	else if (!pcmData[sigEntity].BS_Flag)
		/*
		*	Stuck in break not set yet.
		*/
		tval = phyData[sigEntity].TB_Max;

	SetCSPTimer (tval, PCM_SIGNALS, sigEntity, &pcmData[sigEntity].TPC);

	return;
}

static void
Trace_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into TRACE.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Trace_Entry.", sigEntity);
	return;
}

static void
Connect_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into CONNECT.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Connect_Entry.", sigEntity);
	/*
	*	Set TPC
	*/
	if (pcmData[sigEntity].LS_Flag)
		SetCSPTimer (C_Min, PCM_SIGNALS, sigEntity,
			&pcmData[sigEntity].TPC);

	return;
}

static void
Next_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into NEXT.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
uInt32	tval = T_Out;		/* timeout value */

        PCMDPT1("Port%d:Next_Entry.", sigEntity);
	/*
	*	Set TPC
	*/
	if (pcmData[sigEntity].LS_Flag)
	{
		if (!pcmData[sigEntity].RC_Flag)
			tval = TL_Min;

		else if (pcmData[sigEntity].TD_Flag
			&& !pcmData[sigEntity].TC_Flag)
		{
			if (pcmData[sigEntity].n == 7)
				tval = pcmData[sigEntity].LC_Test;

			else
				tval = T_Next_9;
		}
	}

	SetCSPTimer (tval, PCM_SIGNALS, sigEntity, &pcmData[sigEntity].TPC);

	return;
}

static void
Signal_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into SIGNAL.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
uInt32	tval;			/* timeout value */

        PCMDPT1("Port%d:Signal_Entry.", sigEntity);
	/*
	*	Set TPC
	*/
	if (pcmData[sigEntity].LS_Flag)
		tval = TL_Min;
	else
		tval = T_Out;

	SetCSPTimer (tval, PCM_SIGNALS, sigEntity, &pcmData[sigEntity].TPC);

	return;
}

static void
Join_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into JOIN.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
uInt32	tval;			/* timeout value */

        PCMDPT1("Port%d:Join_Entry.", sigEntity);
	/*
	*	Set TPC
	*/
	if (pcmData[sigEntity].LS_Flag)
		tval = TL_Min;
	else
		tval = T_Out;

	SetCSPTimer (tval, PCM_SIGNALS, sigEntity, &pcmData[sigEntity].TPC);
	return;
}

static void
Verify_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into VERIFY.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
uInt32	tval;			/* timeout value */

        PCMDPT1("Port%d:Verify_Entry.", sigEntity);
	/*
	*	Set TPC
	*/
	if (pcmData[sigEntity].LS_Flag)
		tval = TL_Min;

	else
		tval = T_Out;

	SetCSPTimer (tval, PCM_SIGNALS, sigEntity, &pcmData[sigEntity].TPC);

	return;
}

static void
Active_Entry (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform actions for a transition into ACTIVE.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:		
Modification History:
*********************************************************************/
{
uInt32	tval;			/* timeout value */

        PCMDPT1("Port%d:Active_Entry.", sigEntity);
	/*
	*	Set TPC
	*/
	if ((pcmData[sigEntity].LS_Flag) && (!cemData[sigEntity].CF_Join))
		tval = TL_Min;

	else if (!pcmData[sigEntity].LS_Flag)
		tval = T_Out;

	else
		tval = 0;

	SetCSPTimer (tval, PCM_SIGNALS, sigEntity, &pcmData[sigEntity].TPC);

	return;
}


/*********************************************************************
	PCM State Machine Actions
*********************************************************************/

static void
Off_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform action for a transition to OFF.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:		
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Off_Action.", sigEntity);
	/* disable PMD */
	TransmitPMD (sigEntity, DISABLE);

	/* transmit quiet */
	SendLineState (sigEntity, QLS);
	
	/* clear CF_Loop */
	SMTSendSignal (SIG_CF_Loop, sigEntity, (uInt32) CLEAR);

	/* clear CF_Join */
	SMTSendSignal (SIG_CF_Join, sigEntity, (uInt32) CLEAR);
	
	/*
	*	910605-004	LJP
	*	Clear PC_Neighbor and report change to MSP.
	*/
	phyData[sigEntity].PC_Neighbor = PC_Type_None;
	SendCSPEvent (fddiPORTPC_Neighbor, sigEntity);

	/* clear BS_Flag */
	pcmData[sigEntity].BS_Flag = CLEAR;
	SendCSPEvent (fddiPORTBS_Flag, sigEntity);
	
	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_OFF;
	SendCSPEvent (fddiPORTPCMState, sigEntity);

	/* turn off LEM */
	SMTSendSignal (SIG_LEM_Stop, sigEntity, (uInt32) 0);

	/* turn off scrubbing */
	SetCSPTimer ((uTime) 0, PCM_SIGNALS, sigEntity,
		&pcmData[sigEntity].TScrub);

	/*
	*	Connection process has stopped.
	*	Disable detection of SILS.
	*/
	SetSILSDetect (sigEntity, DISABLE);        /* jlin */

	/* enter OFF state */
	Off_Entry (sigEntity);

	return;
}

static void
Break_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to BREAK.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Break_Action.", sigEntity);
	/* disable PMD */
	TransmitPMD (sigEntity, DISABLE);

	/* transmit quiet */
	SendLineState (sigEntity, QLS);
	
	/* clear CF_Loop */
	SMTSendSignal (SIG_CF_Loop, sigEntity, (uInt32) CLEAR);

	/* clear CF_Join */
	SMTSendSignal (SIG_CF_Join, sigEntity, (uInt32) CLEAR);
	
	/* clear BS_Flag */
	pcmData[sigEntity].BS_Flag = CLEAR;
	SendCSPEvent (fddiPORTBS_Flag, sigEntity);
	
	/*
	*	910605-004	LJP
	*	Clear PC_Neighbor and report change to MSP.
	*/
	phyData[sigEntity].PC_Neighbor = PC_Type_None;
	SendCSPEvent (fddiPORTPC_Neighbor, sigEntity);

	/* clear LS_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;

	/* set PC_Mode */
	phyData[sigEntity].PC_Mode = PC_Mode_None;

	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_BREAK;
	SendCSPEvent (fddiPORTPCMState, sigEntity);

	/* turn off LEM */
	SMTSendSignal (SIG_LEM_Stop, sigEntity, (uInt32) 0);

	/* turn off scrubbing */
	SetCSPTimer ((uTime) 0, PCM_SIGNALS, sigEntity,
		&pcmData[sigEntity].TScrub);

	/*
	*	Connection process has stopped.
	*	Disable detection of SILS.
	*/
	SetSILSDetect (sigEntity, DISABLE);

	/* enter BREAK state */
	Break_Entry (sigEntity);

	return;
}

static void 
Hold_Break_Required (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Test recover enabled flag on hold policy to see if
		break should be performed. This test only applies to the
		attachment ports.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Hold_Break_Required.", sigEntity);
	/*
	*	Check RE_Flag.
	*/
	if (ecmData.RE_Flag)
		/* recovery allowed */
		Break_Actions (sigEntity);
	else
		/* hold until allowed */
		pcmData[sigEntity].RE_Needed = SET;

	return;
}

static void
Trace_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to TRACE.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Trace_Action.", sigEntity);
	/* clear LS_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;

	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_TRACE;
	SendCSPEvent (fddiPORTPCMState, sigEntity);

	/* turn off LEM */
	SMTSendSignal (SIG_LEM_Stop, sigEntity, (uInt32) 0);

	/* turn off scrubbing */
	SetCSPTimer ((uTime) 0, PCM_SIGNALS, sigEntity,
		&pcmData[sigEntity].TScrub);

	/* read current line state */
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);

	/*
	*	Since the line state flag has not been set,
	*	test for line states that may already exist.
	*/
	if ((pcmData[sigEntity].PC_LS & (QLS | HLS))
		&& (ecmData.Path_Test == PT_Passed))
	{
		/*
		*	PC(22a)
		*/
		SMTSendSignal (SIG_Path_Test, 0, (uInt32) PT_Pending);
	}

	else if ((pcmData[sigEntity].PC_LS & MLS)
		&& !pcmData[sigEntity].TR_Flag)
	{
		/*
		*	PC(22b)
		*/
		pcmData[sigEntity].LS_Flag = SET;
		SMTSendSignal (SIG_Trace_Prop, sigEntity,
			(uInt32) PHY_TYPE);
	}

	/* transmit Master */
	SendLineState (sigEntity, MLS);

	/* enter TRACE state */
	Trace_Entry (sigEntity);

	return;
}

static void
Connect_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to CONNECT.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:		
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Connect_Action.", sigEntity);
	/* clear LS_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;
	
	/* clear BS_Flag */
	pcmData[sigEntity].BS_Flag = CLEAR;
	SendCSPEvent (fddiPORTBS_Flag, sigEntity);

	/*
	*	Connection is starting. Enable detection of SILS
	*	during the connection process.
	*/
	SetSILSDetect (sigEntity, ENABLE);

	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_CONNECT;
	SendCSPEvent (fddiPORTPCMState, sigEntity);

	/*
	*	Check for a other line state transitions.
	*/
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);

	/*
	*	If HALT is seen, then continue with connection.
	*/
	if (pcmData[sigEntity].PC_LS & HLS)
		/*
		*	PC(33)
		*/
		pcmData[sigEntity].LS_Flag = SET;

	/*
	*	Else if IDLE is seen before HALT, then restart connection.
	*/
	/*	910920-001	LJP */
	else if (pcmData[sigEntity].PC_LS & SILS)
	{
		/*
		*	PC(31)
		*/
		Break_Actions (sigEntity);
		return;
	}

	/* enable PMD transmit */
	TransmitPMD (sigEntity, ENABLE);

	/* transmit Halt */
	SendLineState (sigEntity, HLS);
	
	/* enter CONNECT state */
	Connect_Entry (sigEntity);
	
	return;
}

static void
Next_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to NEXT.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:		
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Next_Action.", sigEntity);
	/* clear LS_FLag, RC_Flag, TC_Flag, TD_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;
	pcmData[sigEntity].RC_Flag = CLEAR;
	pcmData[sigEntity].TC_Flag = CLEAR;
	pcmData[sigEntity].TD_Flag = CLEAR;

	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_NEXT;
	SendCSPEvent (fddiPORTPCMState, sigEntity);

	/*
	*	Check for other line state transitions.
	*/
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);

	/*
	*	PC(44a)
	*/
	/*	910920-001	LJP */
	if (pcmData[sigEntity].PC_LS & SILS)
	{
		/*
		*	910122-001	LJP
		*	Turn off SILS detect for n=6 or n=8.
		*	911023-001	LJP
		*	Turn off every time.
		*/
		SetSILSDetect (sigEntity, DISABLE);
		pcmData[sigEntity].LS_Flag = SET;
	}

	/* transmit Idle */
	SendLineState (sigEntity, ILS);

	/* enter NEXT state */
	Next_Entry (sigEntity);

	/*
	*	Recheck for QLS. QLS may occur in CONNECT just prior
	*	to the PC(34) transition. If this happens, the CONNECT
	*	state fields the line state change interrupt which
	*	results in no action. When the timer expires for PC(34)
	*	the state machine gets here expecting the line state
	*	to have remained stable and does not check for QLS.
	*
	*	This test for QLS must be done after transmitting ILS
	*	which would force the neighbor from CONNECT to BREAK.
	*	Also, since ReadLineState() has been called above,
	*	the code is assured that the QLS occurred while
	*	in NEXT. This means that if QLS appears in PC_LS
	*	the transition to BREAK can be taken.
	*/
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);
	if (pcmData[sigEntity].PC_LS & QLS)
		Break_Actions (sigEntity);

	return;
}

static void
Signal_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to SIGNAL.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:		
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Signal_Action.", sigEntity);
	/* clear LS_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;
	
	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_SIGNAL;
	SendCSPEvent (fddiPORTPCMState, sigEntity);
	
	/*
	*	Check for other line state transitions.
	*/
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);

	/*
	*	PC(55a)
	*/
	if (pcmData[sigEntity].PC_LS & HLS)
	{
		pcmData[sigEntity].LS_Flag = SET;
		MSetBit (pcmData[sigEntity].R_Val,
			pcmData[sigEntity].n, SET);
	}

	/*
	*	PC(55b)
	*/
	else if (pcmData[sigEntity].PC_LS & MLS)
	{
		pcmData[sigEntity].LS_Flag = SET;
		MSetBit (pcmData[sigEntity].R_Val,
			pcmData[sigEntity].n, CLEAR);
	}

	/* transmit HLS for 1 or MLS for 0 */
	if (MReadBit (pcmData[sigEntity].T_Val, pcmData[sigEntity].n))
		SendLineState (sigEntity, HLS);
	else
		SendLineState (sigEntity, MLS);

	/* enter SIGNAL */
	Signal_Entry (sigEntity);

	/*
	*	911023-001	LJP
	*	Turn on SILS detection after transmitting line state.
	*       Read line state to clear history register in
	*       case ILS was received before enabling the intr.
	*       This will set the history register to just SILS
	*       (instead of MLS-SILS or HLS-SILS) for Next_Actions().
	*       This is for systems that do not generate the
	*       interrupt when SILS is being received and the SILS
	*       interrupt is enabled.
	*/
	SetSILSDetect (sigEntity, ENABLE);
	ReadLineState (sigEntity);

	return;
}

static void
Join_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to JOIN.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Join_Action.", sigEntity);
	/* clear LS_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;
	
	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_JOIN;
	SendCSPEvent (fddiPORTPCMState, sigEntity);
	
	/*
	*	Check for other line state transitions.
	*/
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);

	/*
	*	PC(66)
	*/
	if (pcmData[sigEntity].PC_LS & HLS)
		pcmData[sigEntity].LS_Flag = SET;

	/*
	*	911023-001	LJP
	*	Turn on SILS detection.
	*/
	SetSILSDetect (sigEntity, ENABLE);

	/* transmit Halt */
	SendLineState (sigEntity, HLS);

	/* enter JOIN */
	Join_Entry (sigEntity);

	return;
}

static void
Verify_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to VERIFY.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Verify_Action.", sigEntity);
	/* clear LS_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;

	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_VERIFY;
	SendCSPEvent (fddiPORTPCMState, sigEntity);

	/*
	*	Check for other line state transitions.
	*/
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);

	/*
	*	PC(77)
	*/
	if (pcmData[sigEntity].PC_LS & MLS)
		pcmData[sigEntity].LS_Flag = SET;

        /*
	*	910920-001	LJP
        *       Else if IDLE is seen before MLS, then restart connection.
        */
        else if (pcmData[sigEntity].PC_LS & SILS)
        {
                /*
                *       PC(71)
                */
                Break_Actions (sigEntity);
                return;
        }

	/* transmit Master */
	SendLineState (sigEntity, MLS);
	
	/* enter VERIFY */
	Verify_Entry (sigEntity);
	
	return;
}

static void
Active_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to ACTIVE.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:		
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Active_Action.", sigEntity);
	/* clear LS_Flag, TR_Flag */
	pcmData[sigEntity].LS_Flag = CLEAR;
	pcmData[sigEntity].TR_Flag = CLEAR;

	/* clear RE_Needed flag */
	pcmData[sigEntity].RE_Needed = CLEAR;

	/* clear scrubbing timer to indicate no scrubbing in progress */
	pcmData[sigEntity].TScrub = 0;

	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_ACTIVE;
	SendCSPEvent (fddiPORTPCMState, sigEntity);
	
	/*
	*	Check for other line state transitions.
	*/
	pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);

	/*
	*	PC(88a)
	*/
	/*	910920-001	LJP */
	if (pcmData[sigEntity].PC_LS & SILS)
	{
		pcmData[sigEntity].LS_Flag = SET;

		/*
		*	Connection process has completed.
		*	Disable SILS detection.
		*/
		SetSILSDetect (sigEntity, DISABLE);
	}

	/* transmit Idle */
	SendLineState (sigEntity, ILS);

	/* enter ACTIVE */
	Active_Entry (sigEntity);

	return;
}

static void
Maint_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perfrom actions for a transition to MAINT.
Parameters:	sigEntity	= entity (PHY) to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:
Modification History:
*********************************************************************/
{
        PCMDPT1("Port%d:Maint_Action.", sigEntity);
	/* set current PCM state */
	pcmData[sigEntity].pcState = PC_MAINT;
	SendCSPEvent (fddiPORTPCMState, sigEntity);

	/* set PMD */
	if (pcmData[sigEntity].Maint_LS == QLS)
		TransmitPMD (sigEntity, DISABLE);
	else
		TransmitPMD (sigEntity, ENABLE);

	/* turn off scrubbing */
	SetCSPTimer ((uTime) 0, PCM_SIGNALS, sigEntity,
		&pcmData[sigEntity].TScrub);

	/* transmit requested line state */
	SendLineState (sigEntity, pcmData[sigEntity].Maint_LS);
	
	return;
}

/*********************************************************************
	PCM State Machine
*********************************************************************/

void
SignalPCM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process a PCM signal.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Notes:		
Modification History:
*********************************************************************/
{
	/*
	*	If Port is not operational, then ignore the signal.
	*/
	if (!phyData[sigEntity].operational)
		return;

	/*
	*	For more efficient processing, some signals are preprocessed
	*	here to avoid duplicate code throughout the state machine.
	*/
	switch (sigType)
	{
	/*
	*	Set line state changes.
	*/
	case SIG_Line_State:
		pcmData[sigEntity].PC_LS = (uInt16) sigData;
		break;
	
	/*
	*	If timer has expired, then verify this timer is the current
	*	event being timed.
	*/
	case SIG_PC_Timer:
		if (pcmData[sigEntity].TPC != sigData)
		{
			/* check noise timer */
			if (pcmData[sigEntity].TNE == sigData)
			{
				sigType = SIG_Noise_Timer;
				pcmData[sigEntity].TNE = 0;
			}

			/* check scrub timer */
			else if (pcmData[sigEntity].TScrub == sigData)
			{
				/*
				*	Convert signal type so that other
				*	scrubbing mechanisms can also
				*	signal scrubbing completion allowing
				*	PCM to change the PHY to transmit
				*	PDRs.
				*/
				sigType = SIG_PC_Scrub_Complete;
				pcmData[sigEntity].TScrub = 0;
			}

			/* not current timer */
			else
				return;
		}

		else			/* 910503-001	LJP */
			/* mark timer as expired */
			pcmData[sigEntity].TPC = 0;
		        PCMDPT1("Port%d:TPC Timeout.", sigEntity);
		break;

	/*
	*	Handle general transitions. PC_Start and PC_Stop signals
	*	are processed from all states except PC9:MAINT.
	*/
	/* PC(01) - PC(81) */
	case SIG_PC_Start:
		if (pcmData[sigEntity].pcState != PC_MAINT)
			/* go to BREAK state */
			Break_Actions (sigEntity);

		return;

	/* PC(00) - PC(80) */
	case SIG_PC_Stop:
		if (pcmData[sigEntity].pcState != PC_MAINT)
			/* go to OFF state */
			Off_Actions (sigEntity);

		return;

	/*
	*	Handle general transitions. PC_Disable signal
	*	processed from all states.
	*/
	/* PC(09) - PC(99) */
	case SIG_PC_Disable:
		Off_Actions (sigEntity);
		/* set current PCM state */
		pcmData[sigEntity].pcState = PC_MAINT;
		SendCSPEvent (fddiPORTPCMState, sigEntity);
		return;
		break;
	}
	
	/*
	*	Select PCM state for processing.
	*/
	switch (pcmData[sigEntity].pcState)
	{
	/*
	*	PC0:OFF
	*/
	case PC_OFF:
	        PCMDPT1("\nPort%d:PC0_OFF: ", sigEntity);
		switch (sigType)
		{
		/* PC(09) */
		case SIG_PC_Maint:
			/* set maintenance line state */
			pcmData[sigEntity].Maint_LS = (LineState) sigData;
			/* go to MAINT state */
			Maint_Actions (sigEntity);
			break;
			
		case SIG_Line_State:
			PCMDPT1("Port%d:Line_State_Changed.", sigEntity);
			/*
			*	If line states change while in this state, 
			*	ECM may be checking for a stuck bypass. 
			*	Since line state changes have no effect when 
			*	PCM is in this state, simply notify ECM to 
			*	recheck the line states. This signal is 
			*	issued only when ALL attach PHYs are OFF.
			*/
			{
			uInt16		i, test = TRUE;

				for (i = 0; i < stationData.attachCount; i++)
					test &= (pcmData[i].pcState == PC_OFF);
				if (test)
					SMTSendSignal (SIG_EC_Test_BP, 0,
						(uInt32) 0);
			}
			break;
		}
		break;

	/*
	*	PC1:BREAK
	*/
	case PC_BREAK:
	        PCMDPT1("\nPort%d:PC1_Break: ", sigEntity);
		/*
		*	Note: In this state, the LS_Flag is used to determine if
		*	TB_Min time has expired.
		*/ 
		switch (sigType)
		{
		case SIG_PC_Timer:
			/* if TB_Max has expired */
			if (pcmData[sigEntity].LS_Flag)
			{
			        PCMDPT1("Port%d:PC11:Set_BS_Flag.", sigEntity);
				/* PC(11) Set BS_Flag */
				pcmData[sigEntity].BS_Flag = SET;
				SendCSPEvent (fddiPORTBS_Flag, sigEntity);
			}
			else
			{
				/* TB_Min has expired */
				/* set LS_Flag */
				pcmData[sigEntity].LS_Flag = SET;

				/* if current line state is QLS or HLS */
				pcmData[sigEntity].PC_LS =
					ReadLineState (sigEntity);
				if (pcmData[sigEntity].PC_LS & (QLS | HLS))
					/* PC(13) */
					Connect_Actions (sigEntity);
				else
					/* re-enter BREAK */
				        Break_Entry (sigEntity);
			}
			break;
			
		case SIG_Line_State:
			/* if enough time has passed */
			if (pcmData[sigEntity].LS_Flag)
			{
				/* if current line state is QLS or HLS */
				pcmData[sigEntity].PC_LS = 
					ReadLineState (sigEntity);
				if (pcmData[sigEntity].PC_LS & (QLS | HLS))
					/* PC(13) */
					Connect_Actions (sigEntity);
			}
			break;
		}
		break;

	/*
	*	PC2:TRACE
	*/
	case PC_TRACE:
	        PCMDPT1("\nPort%d:PC2_TRACE: ", sigEntity);
		switch (sigType)
		{
		case SIG_Line_State:
			/* PC(22a) */
			if ((pcmData[sigEntity].PC_LS & (QLS | HLS))
					&& (!pcmData[sigEntity].LS_Flag)
					&& (ecmData.Path_Test == PT_Passed))
				SMTSendSignal (SIG_Path_Test, 0, 
					(uInt32) PT_Pending);
			
			/* PC(22b) */
			else if ((pcmData[sigEntity].PC_LS & (MLS))
					&& (!pcmData[sigEntity].LS_Flag)
					&& (!pcmData[sigEntity].TR_Flag))
			{
				pcmData[sigEntity].LS_Flag = SET;
				SMTSendSignal (SIG_Trace_Prop, sigEntity, 
					(uInt32) PHY_TYPE);
			}
			break;
		}
		break;

	/*
	*	PC3:CONNECT
	*/
	case PC_CONNECT:
	        PCMDPT1("\nPort%d:PC3_CONNECT: ", sigEntity);
		switch (sigType)
		{
		case SIG_Line_State:
			/* check line states only if LS_Flag is not set */
			if (!pcmData[sigEntity].LS_Flag)
			{
				/*
				*	PC(33)
				*/
				if (pcmData[sigEntity].PC_LS & HLS)
				{
					pcmData[sigEntity].LS_Flag = SET;
					Connect_Entry (sigEntity);
				}

				/*
				*	PC(31)
				*	If IDLE seen before HALT, 
				*	restart connection.
				*/
				/* 910920-001 LJP */
				else if (pcmData[sigEntity].PC_LS & SILS)
					Break_Actions (sigEntity);
			}
			break;
			
		case SIG_PC_Timer:
			/*
			*	PC(34)
			*/
			if (pcmData[sigEntity].LS_Flag)
			{
				pcmData[sigEntity].n = 0;
				Next_Actions (sigEntity);
			}
			break;
		}
		break;

	/*
	*	PC4:NEXT
	*/
	case PC_NEXT:
	        PCMDPT1("\nPort%d:PC4_NEXT: ", sigEntity);
		switch (sigType)
		{
		case SIG_Line_State:
			/*
			*	PC(41)
			*/
			if (pcmData[sigEntity].PC_LS & QLS)
			{                  /* Advisory 6.2.3-018 */
			   if ((pcmData[sigEntity].n == 7)
			       && pcmData[sigEntity].TD_Flag
			       && !pcmData[sigEntity].TC_Flag)
			   {
			      StopLCT(sigEntity);
			   }
			   Break_Actions (sigEntity);
			}
				
			/*
			*	PC(44a)
			*/
			else if ((pcmData[sigEntity].PC_LS & SILS)
					&& (!pcmData[sigEntity].LS_Flag))
			{
				/*
				*	910122-001	LJP
				*	Turn off SILS detect for n=6 or n=8.
				*	911023-001	LJP
				*	Turn off every time.
				*/
				SetSILSDetect (sigEntity, DISABLE);
				pcmData[sigEntity].LS_Flag = SET;
				Next_Entry (sigEntity);
			}
			
			/*
			*	PC(44d)
			*/
			else if ((pcmData[sigEntity].PC_LS & (HLS | MLS))
					&& pcmData[sigEntity].TD_Flag
					&& !pcmData[sigEntity].TC_Flag)
			{
				pcmData[sigEntity].TC_Flag = SET;
				PC_TCode_Actions (sigEntity);
				Next_Entry (sigEntity);
			}
			break;
			
		case SIG_PC_Timer:
			/* 
			*	PC(41)
			*	Check for break condition first (TPC > T_Out).
			*	Note that no other flags can get set until
			*	LS_Flag is set.
			*/
			if (!pcmData[sigEntity].LS_Flag)
			{
				/*
				* Break only if n != 0
				*/
				if (pcmData[sigEntity].n != 0)
				{
				   Break_Actions (sigEntity);
				}
			}

			/*
			*	PC(44b)
			*	Otherwise next timer set is TPC > TL_Min.
			*/
			else if (!pcmData[sigEntity].RC_Flag)
			{
				pcmData[sigEntity].RC_Flag = SET;
				PC_RCode_Actions (sigEntity);
				Next_Entry (sigEntity);
			}

			/*
			*	PC(44d)
			*	Otherwise check for transmit delay 
			*	TPC > T_Next(n).
			*/
			else if (pcmData[sigEntity].TD_Flag
					&& !pcmData[sigEntity].TC_Flag)
			{
				pcmData[sigEntity].TC_Flag = SET;
				PC_TCode_Actions (sigEntity);
				Next_Entry (sigEntity);
			}
			break;
			
		/*
		*	PC(44c)
		*/
		case SIG_PC_PDR:
			SendLineState (sigEntity, PDRLS);
			Next_Entry (sigEntity);
			break;
			
		/*
		*	PC(45)
		*/
		case SIG_PC_Signal:
			Signal_Actions (sigEntity);
			break;
			
		/*
		*	PC(46)
		*/
		case SIG_PC_Join:
			Join_Actions (sigEntity);
			break;
		}
		break;

	/*
	*	PC5:SIGNAL
	*/
	case PC_SIGNAL:
	        PCMDPT1("\nPort%d:PC5_SIGNAL: ", sigEntity);
		switch (sigType)
		{
		case SIG_Line_State:
			/*
			*	PC(51)
			*/
			if (pcmData[sigEntity].PC_LS & QLS)
				Break_Actions (sigEntity);
			
			/*
			*	Only process changes if LS_Flag is not set.
			*/
			else if (!pcmData[sigEntity].LS_Flag)
			{
				/*
				*	PC(55a)
				*/
				if (pcmData[sigEntity].PC_LS & HLS)
				{
					pcmData[sigEntity].LS_Flag = SET;
					MSetBit (pcmData[sigEntity].R_Val,
						pcmData[sigEntity].n,
						SET);
					Signal_Entry (sigEntity);
				}
				
				/*
				*	PC(55b)
				*/
				if (pcmData[sigEntity].PC_LS & MLS)
				{
					pcmData[sigEntity].LS_Flag = SET;
					MSetBit (pcmData[sigEntity].R_Val,
						pcmData[sigEntity].n,
						CLEAR);
					Signal_Entry (sigEntity);
				}
			}
			break;
			
		case SIG_PC_Timer:
			/*
			*	PC(51)
			*/
			if (!pcmData[sigEntity].LS_Flag)
				Break_Actions (sigEntity);
				
			/*
			*	PC(54)
			*/
			else
			{
				++pcmData[sigEntity].n;
				Next_Actions (sigEntity);
			}
			break;
		}
		break;

	/*
	*	PC6:JOIN
	*/
	case PC_JOIN:
	        PCMDPT1("\nPort%d:PC6_JOIN: ", sigEntity);
		switch (sigType)
		{
		case SIG_Line_State:
			/*
			*	PC(61)
			*/
			if (pcmData[sigEntity].PC_LS & QLS)
			{
				Break_Actions (sigEntity);
			}

			/*
			*	PC(66)
			*/
			else if (!pcmData[sigEntity].LS_Flag
					&& (pcmData[sigEntity].PC_LS & HLS))
			{
				pcmData[sigEntity].LS_Flag = SET;
				Join_Entry (sigEntity);
			}

			/*
			*	Second part to PC(61)
			*	Note: PC_LS must be set to the current line 
			*	state to check ILS because it could show 
			*	both ILS and HLS from a legitimate PC(46) 
			*	transition.
			*/
			/*	910920-001	LJP */
			else 
			{
				pcmData[sigEntity].PC_LS = 
					ReadLineState (sigEntity);
				if (pcmData[sigEntity].LS_Flag
					&& (pcmData[sigEntity].PC_LS & SILS))
					Break_Actions (sigEntity);
			}
				
			break;
			
		case SIG_PC_Timer:
			/*
			*	PC(61)
			*/
			if (!pcmData[sigEntity].LS_Flag)
				Break_Actions (sigEntity);
				
			/*
			*	PC(67)
			*/
			else
				Verify_Actions (sigEntity);
				
			break;
		}
		break;

	/*
	*	PC7:VERIFY
	*/
	case PC_VERIFY:
	        PCMDPT1("\nPort%d:PC7_VERIFY: ", sigEntity);
		switch (sigType)
		{
		case SIG_Line_State:
			/*
			*	PC(71)
			*/
			/*
			*	910920-001	LJP
			*	Code used to check for QLS or LS_Flag and
			*	ILS. Since ACTIVE sends ILS, this code
			*	must check for QLS, then MLS, then ILS
			*	to insure sequencing proceeds correctly.
			*/
			if (pcmData[sigEntity].PC_LS & QLS)
				Break_Actions (sigEntity);

			else if (!pcmData[sigEntity].LS_Flag)
			{
				if (pcmData[sigEntity].PC_LS & MLS)
				{
					/*
					*	PC(77)
					*/
					pcmData[sigEntity].LS_Flag = SET;
					Verify_Entry (sigEntity);
				}

				else if (pcmData[sigEntity].PC_LS & SILS)
					/*
					*	PC(71)
					*/
					Break_Actions (sigEntity);
			}
			break;
			
		case SIG_PC_Timer:
			/*
			*	PC(71)
			*/
			if (!pcmData[sigEntity].LS_Flag)
				Break_Actions (sigEntity);
				
			/*
			*	PC(78)
			*/
			else
				Active_Actions (sigEntity);
				
			break;
		}
		break;

	/*
	*	PC8:ACTIVE
	*/
	case PC_ACTIVE:
	        PCMDPT1("\nPort%d:PC8_ACTIVE: ", sigEntity);
		switch (sigType)
		{
		case SIG_PC_Timer:
			/*
			*	PC(81)
			*/
			if (!pcmData[sigEntity].LS_Flag
					&& !pcmData[sigEntity].TR_Flag)
			{
				Hold_Break_Required (sigEntity);
			}

			/*
			*	TL_Min expired.
			*/
			else if (pcmData[sigEntity].LS_Flag
					&& !cemData[sigEntity].CF_Join)
			{
				/*
				*	910627-001	LJP
				*	PC(88c)
				*	Check for MLS received. All line
				*	states read here have occurred since
				*	entering ACTIVE due to read on
				*	entry.
				*/
				pcmData[sigEntity].PC_LS =
					ReadLineState (sigEntity);
				if ((pcmData[sigEntity].PC_LS & MLS)
					&& !pcmData[sigEntity].TR_Flag)
				{
					pcmData[sigEntity].TR_Flag = SET;
					SMTSendSignal (SIG_Trace_Prop, sigEntity, 
						(uInt32) PHY_TYPE);
				}

				/*
				*	PC(88b1)
				*	Refer to SIG_PC_PDR in PC(88b2)
				*	for explanation about dividing
				*	this transition into two parts.
				*/
				else
				{
					SMTSendSignal (SIG_CF_Join, sigEntity,
						(uInt32) SET);
					SMTSendSignal (SIG_PC_PDR,
						sigEntity, (uInt32) 0);
				}

				Active_Entry (sigEntity);
			}
			break;

		case SIG_Line_State:
			/*
			*	PC(81)
			*	Only transition to BREAK if TR_Flag not set.
			*	Take transition on HLS,
			*	otherwise check hold policy on QLS.
			*/
			if (!pcmData[sigEntity].TR_Flag
				&& (pcmData[sigEntity].PC_LS & HLS))
			{
				Break_Actions (sigEntity);
			}

			else if (!pcmData[sigEntity].TR_Flag
				&& (pcmData[sigEntity].PC_LS & QLS))
			{
				Hold_Break_Required (sigEntity);
			}

			/*
			*	PC(88c)
			*	Use CF_Join flag instead of LS_Flag
			*	because the line state change interrupt
			*	may be due to the PC(78) transition.
			*/
			else if (cemData[sigEntity].CF_Join
					&& !pcmData[sigEntity].TR_Flag
					&& (pcmData[sigEntity].PC_LS & MLS))
			{
				pcmData[sigEntity].TR_Flag = SET;
				SMTSendSignal (SIG_Trace_Prop, sigEntity, 
					(uInt32) PHY_TYPE);
				Active_Entry (sigEntity);
			}

			/*
			*	PC(88na)
			*	This is an additional state to allow PCM to 
			*	time the noise condition on a link. When 
			*	noise is detected by the PHY, PCM may begin 
			*	the noise timer.
			*/
			else if ((pcmData[sigEntity].PC_LS & NLS)
					&& (pcmData[sigEntity].TNE == 0))
			{
				SetCSPTimer (NS_Max, PCM_SIGNALS, sigEntity, 
					&pcmData[sigEntity].TNE);
			}

			else if (pcmData[sigEntity].PC_LS & SILS)
			{
				/*
				*	If the noise timer is currently 
				*	running, it can be stopped as soon
				*	as Idle is detected on the link.
				*/
				if (pcmData[sigEntity].TNE > 0)
					SetCSPTimer ((uTime) 0, 0, 0,
						&pcmData[sigEntity].TNE);

				/*
				*	PC(88a)
				*/
				if (!pcmData[sigEntity].LS_Flag)
				{
					pcmData[sigEntity].LS_Flag = SET;

					/*
					*	Connection process is complete.
					*	Disable SILS detection.
					*/
					SetSILSDetect (sigEntity, DISABLE);

					Active_Entry (sigEntity);
				}
			}
			
			break;
			
		case SIG_PC_Trace:
			/*
			*	PC(82)
			*/
			Trace_Actions (sigEntity);
			break;
			
		case SIG_Noise_Event:
			/*
			*	PC(81)
			*	This signal replaces the TNE > NS_Max term 
			*	on PC(81) to allow noise detection either 
			*	by hardware or PCM. This signal is used when 
			*	a link is to be disconnected due to excessive 
			*	noise.
			*/
			if (!pcmData[sigEntity].TR_Flag)
				Hold_Break_Required (sigEntity);
			break;
			
		case SIG_Noise_Timer:
			/*
			*	PC(88nb)
			*	When the timer event is associated with 
			*	the expiration of the noise timer, the 
			*	event type is changed to SIG_Noise_Timer. 
			*	This indicates that PCM has timed the 
			*	noise condition on the link. If no Idle 
			*	symbols have been seen since the timer 
			*	began, then the link must be disconnected.
			*/
			/*	910920-001	LJP */
			pcmData[sigEntity].PC_LS = ReadLineState (sigEntity);
			if (!(pcmData[sigEntity].PC_LS & (ILS | SILS)))
				SMTSendSignal (SIG_Noise_Event, sigEntity,
					(uInt32) 0);
			break;
			
		case SIG_PC_PDR:
			/*
			*	PC(88b2)
			*	This is the second half of PC(88b). Since
			*	a software implementation of PCM cannot
			*	guarantee how long it will take for CFM
			*	to respond to the CF_Join flag, the
			*	SIG_PC_PDR signal is used. By placing the
			*	two events sequentially in the signal list,
			*	PCM is guaranteed that the station
			*	configuration has been completed before
			*	allowing the PHY to transmit PDUs.
			*
			*	Also, for Idle-based scrubbing, this
			*	transition must be held off when scrubbing
			*	is in progress (TScrub != 0). Having
			*	SIG_PC_PDR as a separate transition makes
			*	this operation easier to implement.
			*/
			if (pcmData[sigEntity].TScrub == 0)
			{
				SendLineState (sigEntity, PDRLS);
				SMTSendSignal (SIG_LEM_Start, sigEntity,
					(uInt32) 0);
				Active_Entry (sigEntity);
			}
			break;

		case SIG_PC_Scrub_Start:
			/*
			*	PC(88scruba)
			*	Additional state change to handle scrubbing
			*	using Idle symbols. This state change is
			*	used to start Idle-based scrubbing. If
			*	another scrubbing method is used, then
			*	this signal should not be used.
			*/
			SendLineState (sigEntity, ILS);
			SetCSPTimer (T_Scrub, PCM_SIGNALS, sigEntity,
				&pcmData[sigEntity].TScrub);
			Active_Entry (sigEntity);
			break;

		case SIG_PC_Scrub_Complete:
			/*
			*	PC(88scrubb)
			*	Additional state change to handle scrubbing
			*	using Idle symbols. When Idle-based
			*	scrubbing completes, the timer expiration
			*	signal is converted into this signal to
			*	restart sending PDRs through the PHY.
			*	Other scrubbing mechanisms may also send
			*	SIG_PC_Scrub_Complete to change the PHY
			*	transmission back to sending PDRs.
			*/
			SMTSendSignal (SIG_PC_PDR, sigEntity, (uInt32) 0);
			Active_Entry (sigEntity);
			break;

		case SIG_PC_LEM_Fail:
			/*
			*	PC(81)
			*/
			if (!pcmData[sigEntity].TR_Flag)
				Hold_Break_Required (sigEntity);
			break;

		case SIG_PC_RE_Flag:
			/*
			*	PC(81)
			*	If the RE_Flag changes, recheck if
			*	a break condition exists.
			*/
			if (pcmData[sigEntity].RE_Needed)
				Break_Actions (sigEntity);
			break;
		}
		break;

	/*
	*	PC9:MAINT
	*/
	case PC_MAINT:
	        PCMDPT1("\nPort%d:PC9_MAINT: ", sigEntity);
		/*
		*	PC(99)
		*	This is the only event recognized for this state at this
		*	point in the state machine.
		*/
		if (sigType == SIG_PC_Maint)
		{
			pcmData[sigEntity].Maint_LS = (LineState) sigData;
			Maint_Actions (sigEntity);
		}

		/*
		*	PC(90)
		*/
		else if (sigType == SIG_PC_Enable)
			Off_Actions (sigEntity);

		break;
	}
	
	return;
}

