#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) csprmt.c V6.2.3:cs.910829:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module
	
	RMT State Machine
	
	File:		csprmt.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.910829	09/04/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module implements the RMT state machine listed in the
	ANSI X3T9.5 standard.
		
	This file consists of two parts. The first part consists of
	a set of functions that perform the various actions associated
	with a transition into a state. The second part implements the
	RMT state machine.

	Modification History:

	*** Updated to SMT 6.2 ***

	910129-001	LJP
		Made forward declarations of Ring_OP_Actions()
		match the function definition.
	910419-002	LJP
		Test to see if RM_Join or RM_Loop is zero before
		decrementing on CLEAR signal.
	910426-001	LJP
		The code to handle multiple SETs and CLEARS for RM_Join
		did not account for all possibilities. The RM_Join
		signal to RMT now tells RMT to count the number of
		RM_Joins SET in cemData[] and is sent whenever an
		RM_Join is changed by CFM.
	910429-001	LJP
		Use new beacon type constant definitions in SetBeacon()
		calls.
	910626-001	LJP
		Removed extraneous sending of Trace_Prop signal.
	910823-001	LJP
		Only set TRM for T_Stuck if BN_Flag is set.
	910823-002	LJP
		On transition RM(33a), code was setting timer for
		handling the RM(33c) transition. The assumption was
		made that upon receiving a Beacon, the MAC goes to
		Claim. This is only true when receiving My_Beacon.
		On receiving Other_Beacon, MAC yields. In this case,
		the timer should not be set.
*********************************************************************/
#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"cspmacro.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include	"mibdefs.h"
#include        "cmtflag.h"


/*********************************************************************
	RMT Support Functions
*********************************************************************/

extern	void	SetCSPTimer ();
extern	void	SendCSPEvent ();
extern	void	MACResetRequest ();
extern	void	SetBeacon ();
extern	void	BeaconRequest ();
extern	void	SMTSendSignal ();
extern	void	MACSetInterrupts ();
extern	void	SetMACInput ();
extern	void	ChangeMACAddress ();
extern	void	ChangeMACTBid ();
extern	void	ConfigureMAC ();
extern	void	MACClaimRequest ();
extern	Flag	ValidClaimReceived ();
extern	Flag	ReadMACR_Flag ();

/* 910129-001 LJP */
static void	Ring_OP_Actions ();		/* forward declaration */


/*********************************************************************
	State Entry Functions
*********************************************************************/

static void
Non_OP_Entry (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM1:NON_OP state.
Parameters:	macID	= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* Reset TRM */
	SetCSPTimer (T_Non_OP, RMT_SIGNALS, macID, &rmtData[macID].TRM);
		
	/* set current RMT state */
	rmtData[macID].rmState = RM_NON_OP;
	SendCSPEvent (fddiMACRMTState, macID);
	
	/* if optional hold policy is active, */
	if (ecmData.Hold)
		/* enable MAC interrupts for hold policy */
		MACSetInterrupts (macID, ENABLE);
	else
		/* disable MAC interrupts */
		MACSetInterrupts (macID, DISABLE);

	/* if ring is operational, then go to RING_OP state */
	if (rmtData[macID].ringOP)
		Ring_OP_Actions (macID);

	return;
}

static void
Detect_Entry (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM3:DETECT state.
Parameters:	macID	= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
910823-008	LJP
*********************************************************************/
{
	/*
	*	910823-001	LJP
	*	TRM only needs to be set while Beaconing.
	#	Else clear timer.
	*/
	if (rmtData[macID].BN_Flag)
		/* Reset TRM */
		SetCSPTimer (T_Stuck, RMT_SIGNALS, macID, &rmtData[macID].TRM);
	else
		/* Clear TRM */
		SetCSPTimer ((uInt32) 0, RMT_SIGNALS, macID,
			&rmtData[macID].TRM);
	
	/* set current RMT state */
	rmtData[macID].rmState = RM_DETECT;
	SendCSPEvent (fddiMACRMTState, macID);

	/* (re-)enable MAC interrupts */
	MACSetInterrupts (macID, ENABLE);

	/* if ring is operational, then go to RING_OP state */
	if (rmtData[macID].ringOP)
		Ring_OP_Actions (macID);

	return;
}

static void
Non_OP_Dup_Entry (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM4:NON_OP_DUP state.
Parameters:	macID	= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* Reset TRM */
	if (rmtData[macID].BN_Flag)
		SetCSPTimer (T_Stuck, RMT_SIGNALS, macID, 
			&rmtData[macID].TRM);
	else
		SetCSPTimer (T_Announce, RMT_SIGNALS, macID, 
			&rmtData[macID].TRM);

	/* set current RMT state */
	rmtData[macID].rmState = RM_NON_OP_DUP;
	SendCSPEvent (fddiMACRMTState, macID);
	
	/* (re-)enable MAC interrupts */
	MACSetInterrupts (macID, ENABLE);

	/* if ring is operational, then go to RING_OP state */
	if (rmtData[macID].ringOP)
		Ring_OP_Actions (macID);

	return;
}

static void
Ring_OP_Dup_Entry (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM5:RING_OP_DUP state.
Parameters:	macID	= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* Reset TRM */
	SetCSPTimer ((uTime) 0, 0, 0, &rmtData[macID].TRM);
	
	/* set current RMT state */
	rmtData[macID].rmState = RM_RING_OP_DUP;
	SendCSPEvent (fddiMACRMTState, macID);
	
	/* disable MAC interrupts */
	MACSetInterrupts (macID, DISABLE);

	return;
}

/*********************************************************************
	State Transition Actions
*********************************************************************/

static void
Jam_A_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process Jam_A actions.
Parameters:	macID	= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* Disconnect MAC PH_DATA.indication */
	SetMACInput (macID, FALSE);

	/* Prepare beacon frame */
	/*
	*	910429-001	LJP
	*	Use new beacon type definitions.
	*/
	SetBeacon (macID, (uChar) BCN_CLAIM_FAILED, macData[macID].SMTAddress, 
		0, (uChar *) NULL);

	/* Issue MAC Reset */
	MACResetRequest (macID);
	
	/* BeaconRequest (macID, DA = MLA, INFO = Jam_Info) */
	BeaconRequest (macID);
	
	/* Wait for T_Jam in special substate */
	SetCSPTimer (T_Jam, RMT_SIGNALS, macID, &rmtData[macID].TRM);
	rmtData[macID].rmState = RM_JAM_A_WAIT_TJAM;

	return;
}

static void
Jam_B_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process Jam_B actions.
Parameters:	macID	= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* MAC Reset */
	MACResetRequest (macID);
	
	/* BeaconRequest (macID, DA = MLA, INFO = Jam_Info) */
	/*
	*	910429-001	LJP
	*	Use new beacon type definitions.
	*/
	SetBeacon (macID, (uChar) BCN_CLAIM_FAILED, macData[macID].SMTAddress,
		0, (uChar *) NULL);
	BeaconRequest (macID);
	
	/* Wait for T_DBJ in special substate */
	SetCSPTimer (T_DBJ, RMT_SIGNALS, macID, &rmtData[macID].TRM);
	rmtData[macID].rmState = RM_JAM_B_WAIT_TDBJ;
	
	return;
}

static void
Jam_Init_Actions (macID, jamPolicy)
	uInt16	macID;
	uChar	jamPolicy;
/*********************************************************************
Function:	Process actions for entering the RM0:ISOLATED state.
Parameters:	macID		= MAC to process.
		jamPolicy	= policy flag for Jam A or Jam B actions.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/*
	*	Configure MAC to lose the Claim process to other MACs
	*	having T_Bid <= T_Req_Max and nonzero MAC addresses.
	*/
	ChangeMACTBid (macID, Jam_T_Req_Value);
	SendCSPEvent (fddiMACT_Req, macID);

	/* Set JM_Flag */
	rmtData[macID].JM_Flag = SET;

	/* Perform Jam_A_Actions or Jam_B_Actions */
	if (jamPolicy == RMT_DUP_JAM_A)
		Jam_A_Actions (macID);
	else
		Jam_B_Actions (macID);

	return;
}

static void
Leave_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process leave actions.
Parameters:	macID		= MAC to process.
Input:		None.
Output:		None.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* disconnect from ring */
        CSPDPT1("$MAC%d: Leave_Action$", macID);
	SMTSendSignal (SIG_Disconnect, 0, (uInt32) 0);
       
	return;
}

static void
Change_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process change actions.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
Int16	compare;

	/*
	*	IF Using Local MAC Address
	*	Determine by comparing data address
	*	and SMT address. If data address is not the same
	*	as the SMT address, it will be a short address.
	*/
	MCompareAddress (macData[macID].dataAddress,
		macData[macID].SMTAddress, compare);
	if (compare != 0)
	{
		/* scrub PDUs sourced using current address */
		
		/* change MAC address to universal address */
		MCopyAddress (macData[macID].dataAddress,
			macData[macID].SMTAddress);
		ChangeMACAddress (macID, macData[macID].dataAddress);
		SendCSPEvent (xdiMACDataAddress, macID);

		/* clear DA_Flag */
		rmtData[macID].DA_Flag = CLEAR;
		SendCSPEvent (fddiMACDa_Flag, macID);
	}
	
	/* else */
	else
	{
		/* perform Jam_Init_Actions (macID) or Leave_Actions (macID) */
		if (macData[macID].dupPolicy2 == RMT_DUP_LEAVE)
			Leave_Actions (macID);
		else
			Jam_Init_Actions (macID, macData[macID].dupPolicy2);
	}

	return;
}

static void
Isolated_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM0:ISOLATED state.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* configure MAC to be disabled */
        CSPDPT1("$MAC%d: Isolated_Action$", macID);
	ConfigureMAC (macID, DISABLE, macData[macID].pathsRequested);
	SendCSPEvent (fddiMACCurrentPath, macID);

	rmtData[macID].DA_Flag = CLEAR;
	SendCSPEvent (fddiMACDa_Flag, macID);

	macData[macID].MAC_Avail = CLEAR;
	macData[macID].Loop_Avail = CLEAR;
	rmtData[macID].NO_Flag = SET;
	SMTSendSignal (SIG_EC_NO_Flag, 0, (uInt32) 0);

	/* set current RMT state */
	rmtData[macID].rmState = RM_ISOLATED;
	SendCSPEvent (fddiMACRMTState, macID);

	/* disable MAC interrupts */
	MACSetInterrupts (macID, DISABLE);

	return;
}

static void
Insert_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process insert actions.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        CSPDPT1("$MAC%d: Insert_Action$", macID);
	/* Configure MAC for operation */
	ConfigureMAC (macID, ENABLE, macData[macID].pathsRequested);
	SendCSPEvent (fddiMACCurrentPath, macID);

	/* MAC Reset */
	MACResetRequest (macID);
	
	/* start beacon or claim process */
	MACClaimRequest (macID);
		
	return;
}

static void
Ring_OP_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM2:RING_OP state.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        CSPDPT1("$MAC%d: Ring_OP_Action$", macID);
	rmtData[macID].NO_Flag = CLEAR;
	SMTSendSignal (SIG_EC_NO_Flag, 0, (uInt32) 0);

	if (rmtData[macID].RM_Loop)
	{
		macData[macID].Loop_Avail = SET;
	}

	if (rmtData[macID].RM_Join)
	{
		macData[macID].MAC_Avail = SET;
	}

	/* set current RMT state */
	rmtData[macID].rmState = RM_RING_OP;
	SendCSPEvent (fddiMACRMTState, macID);
	
	/* disable MAC interrupts */
	MACSetInterrupts (macID, DISABLE);

	/* turn off timer */
	SetCSPTimer ((uTime) 0, 0, 0, &rmtData[macID].TRM);

	return;
}

static void
New_Dup_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process new duplicate actions.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	rmtData[macID].DA_Flag = SET;
	SendCSPEvent (fddiMACDa_Flag, macID);
	rmtData[macID].BN_Flag = CLEAR;
	rmtData[macID].JM_Flag = CLEAR;

	/* Perform Change_Actions (macID), Jam_Init_Actions (macID),
		or Leave_Actions (macID) */
	if (macData[macID].dupPolicy1 == RMT_DUP_CHANGE)
		Change_Actions (macID);
	else if (macData[macID].dupPolicy1 == RMT_DUP_LEAVE)
		Leave_Actions (macID);
	else
		Jam_Init_Actions (macID, macData[macID].dupPolicy1);
		
	return;
}

static void
Dup_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process duplicate actions.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	if (rmtData[macID].JM_Flag)
	{
		/* Perform Jam_A_Actions (macID) or Jam_B_Actions (macID) */
		if (macData[macID].dupPolicy1 == RMT_DUP_JAM_A)
			Jam_A_Actions (macID);
		else
			Jam_B_Actions (macID);
	}
	
	else
	{
		/* set NO_Flag */
		rmtData[macID].NO_Flag  = SET;
		SMTSendSignal (SIG_EC_NO_Flag, 0, (uInt32) 0);

		/* Perform Change_Actions (macID), Jam_Init_Actions (macID),
			or Leave_Actions (macID) */
		if (macData[macID].dupPolicy1 == RMT_DUP_CHANGE)
			Change_Actions (macID);
		else if (macData[macID].dupPolicy1 == RMT_DUP_LEAVE)
			Leave_Actions (macID);
		else
			Jam_Init_Actions (macID, macData[macID].dupPolicy1);
	}
	
	return;
}

static void
Directed_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM6:DIRECTED state.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* Reset TRM */
	SetCSPTimer (T_Direct, RMT_SIGNALS, macID, &rmtData[macID].TRM);
	
	/* MAC Reset */
	MACResetRequest (macID);
	
	/* BeaconRequest (macID, DA = multicast, INFO = Directed_Info) */
	/*
	*	910429-001	LJP
	*	Use new beacon type definitions.
	*/
	SetBeacon (macID, (uChar) BCN_DIRECTED, (MACAddr48 *) (DIR_BCN_ADDRESS),
		(uInt16) 6, (uChar *) macData[macID].upstreamNbr);
	BeaconRequest (macID);

	/* set new RMT state */
	rmtData[macID].rmState = RM_DIRECTED;
	SendCSPEvent (fddiMACRMTState, macID);
	
	/* (re-)enable MAC interrupts */
	MACSetInterrupts (macID, ENABLE);

	return;
}

static void
Trace_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering the RM7:TRACE state.
Parameters:	macID		= MAC to process.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* Reset TRM */
	SetCSPTimer ((uTime) 0, RMT_SIGNALS, macID, &rmtData[macID].TRM);

	/* set current RMT state */
	rmtData[macID].rmState = RM_TRACE;
	SendCSPEvent (fddiMACRMTState, macID);

	/* disable MAC interrupts */
	MACSetInterrupts (macID, DISABLE);

	/* signal Trace_Started */
	rmtData[macID].traceStatus = Trace_Initiated;
	SendCSPEvent (fddiPATHClassPATHTraceStatus, macID);

	/* set Trace_Prop */
	/* 910626-001 LJP Cast MAC_TYPE to uInt32 */
	SMTSendSignal (SIG_Trace_Prop, macID, (uInt32) MAC_TYPE);

	return;
}

static void
Rmode_Actions (macID)
	uInt16	macID;
/*********************************************************************
Function:	Process actions for entering restricted transmissions.
Parameters:	macID		= MAC to process.
Input:		None.
Output:		None.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/* SM_MA_CONTROL.request(Reset) */
	MACResetRequest (macID);

	/* SM_MA_CONTROL.request(Claim) */
	MACClaimRequest (macID);

	return;
}


/*********************************************************************
	RMT State Machine
*********************************************************************/

void
SignalRMT (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process an RMT signal.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (MAC) to process.
		sigData		= any data associated with the signal.
Input:		Uses rmtData.
Output:		Changes rmtData.
Return:		No value returned.
Modification History:
910823-002	LJP
*********************************************************************/
{
uInt16	i;			/* 910426-001 LJP */

	/*
	*	If MAC is not operational, then ignore the signal.
	*	This handles configuration signals for secondary
	*	or local MACs that may not exist.
	*/
#ifdef  __CMT_ONLY
        return;
#endif        
#ifdef __FEBRIDGE
	if (sigEntity)		/* gjs: only MAC 0 is OK */
		return;
#endif
	if (!macData[sigEntity].operational)
		return;

	/*
	*	First process any changes to RM_Join or RM_Loop.
	*
	*	RM_Loop and RM_Join is the logical OR of
	*	all loop and join signals to this MAC.
	*
	*	910426-001	LJP
	*	The RM_Join & RM_Loop signals indicate a
	*	change in the flag's status. This tells
	*	RMT to recount the number of PHYs
	*	requesting a MAC.
	*/
       CSPDPT3("RMT: Signal Type: %x.  Entity: %x.  Data: %x\n",
       sigType, sigEntity, sigData);
	if ((sigType == SIG_RM_Loop) || (sigType == SIG_RM_Join))
	{
		if (sigType == SIG_RM_Loop)
		{
			/* if any ports want this MAC, then it is set */
			rmtData[sigEntity].RM_Loop = CLEAR;
			for (i = 0; (i < MAX_PORT_COUNT) 
					&& !rmtData[sigEntity].RM_Loop; i++)
				rmtData[sigEntity].RM_Loop |=
					cemData[i].RM_Loop[sigEntity];
		}

		else if (sigType == SIG_RM_Join)
		{
			/* if any ports want this MAC, then it is set */
			rmtData[sigEntity].RM_Join = CLEAR;
			for (i = 0; (i < MAX_PORT_COUNT) 
					&& !rmtData[sigEntity].RM_Join; i++)
				rmtData[sigEntity].RM_Join |=
					cemData[i].RM_Join[sigEntity];
		}

		/*
		*	General transitions.
		*/
		if (!rmtData[sigEntity].RM_Loop && !rmtData[sigEntity].RM_Join)
		{
			Isolated_Actions (sigEntity);
			return;
		}
	}

	/*
	*	If ring operational changes, set RMT flag.
	*/
	else if (sigType == SIG_Ring_OP)
	{
		rmtData[sigEntity].ringOP = sigData;
		CSPDPT2("\nMAC%d's Ring_OP:%d.\n", sigEntity, sigData);
	}
	
	/*
	*	If timer has expired, then verify this timer is the current
	*	event being timed.
	*/
	else if (sigType == SIG_RM_Timer)
	{
		/*
		*	Determine which timer expired.
		*/
		if (rmtData[sigEntity].rmState == RM_DETECT)
		{
			if (rmtData[sigEntity].TRC == sigData)
			{
				/*
				* If the state machine shows the MAC in
				* claim, then this timer has measured
				* RM34cTime amount of time and the state
				* machine must check for receiving claim
				* frame with My_Address and T_Bid != T_Req.
				* The routine ValidClaimReceived() checks if
				* the condition for transition RM(34c) is
				* met (this is very hardware dependent).
				* If the condition is met, then the
				* transition is taken. Otherwise, the timer
				* is reset to check again later.
				*/
				if (rmtData[sigEntity].inClaim)
				{
					if (ValidClaimReceived (sigEntity))
					{
						/*
						*	RM(34c)
						*/
						New_Dup_Actions (sigEntity);
						Non_OP_Dup_Entry (sigEntity);
						return;
					}

					/*
					*	Restart timer.
					*/
					else
					{
						SetCSPTimer (
						  macData[sigEntity].RM34cTime,
						  RMT_SIGNALS,
						  sigEntity,
						  &rmtData[sigEntity].TRC);
						return;
					}
				}

				/*
				*	Else, this timer has measured 2*D_Max
				*	since claiming.
				*/
				else
				{
					rmtData[sigEntity].claimTimed = SET;
					rmtData[sigEntity].TRC = 0;
					Detect_Entry (sigEntity);
				}

				return;
			}

			else if ((rmtData[sigEntity].TRB == sigData)
				&& (!rmtData[sigEntity].BN_Flag))
			{
				rmtData[sigEntity].beaconTimed = SET;
				rmtData[sigEntity].TRB = 0;
				Detect_Entry (sigEntity);
				return;
			}
		}

		if (rmtData[sigEntity].TRM != sigData)
			/* not current timer */
			return;
		else
			/* mark timer as expired */
			rmtData[sigEntity].TRM = 0;
	}

	/*
	*	Select RMT state for processing.
	*/
	switch (rmtData[sigEntity].rmState)
	{
	/*
	*	RM0:ISOLATED
	*/
	case RM_ISOLATED:
	  CSPDPT1("RM0:%d:ISOLATED\n", sigEntity);
		switch (sigType)
		{
		case SIG_RM_Loop:
		case SIG_RM_Join:
			/*
			*	RM(01)
			*/
			if (rmtData[sigEntity].RM_Loop
				|| rmtData[sigEntity].RM_Join)
			{
				Insert_Actions (sigEntity);
				Non_OP_Entry (sigEntity);
			}
			break;
		}
		break;
		
	/*
	*	RM1:NON_OP
	*/
	case RM_NON_OP:
	  CSPDPT1("RM1:%d:NON_OP\n", sigEntity);
		switch (sigType)
		{
		case SIG_Ring_OP:
			/*
			*	RM(12)
			*	sigData contains Ring_OP state.
			*/
			if (sigData)
				Ring_OP_Actions (sigEntity);
			break;
			
		case SIG_RM_Timer:
			/*
			*	RM(13)
			*/
			rmtData[sigEntity].BN_Flag = CLEAR;
			rmtData[sigEntity].inClaim = CLEAR;
			rmtData[sigEntity].claimTimed = CLEAR;
			rmtData[sigEntity].beaconTimed = CLEAR;
			SetCSPTimer ((uTime) 0, 0, 0, &rmtData[sigEntity].TRC);
			SetCSPTimer ((uTime) 0, 0, 0, &rmtData[sigEntity].TRB);
			Detect_Entry (sigEntity);
			break;

		case SIG_MAC_Interrupt:
			/*
			*	RM(11)
			*/
			if (!rmtData[sigEntity].NO_Flag
				&& (sigData &
					(MAC_TRT_In_T4T5
						| MAC_My_Beacon
						| MAC_Other_Beacon)))
			{
				/* set NO_Flag */
				rmtData[sigEntity].NO_Flag = SET;
				SMTSendSignal (SIG_EC_NO_Flag, 0, (uInt32) 0);

				/* disable MAC interrupts for now */
				MACSetInterrupts (sigEntity, DISABLE);
			}
			break;
		}
		break;
		
	/*
	*	RM2:RING_OP
	*/
	case RM_RING_OP:
	  CSPDPT1("RM2:%d:RING_OP\n", sigEntity);
		switch (sigType)
		{
		case SIG_Ring_OP:
			/*
			*	RM(21)
			*/
			if (!sigData)
			{
				macData[sigEntity].MAC_Avail = CLEAR;
				macData[sigEntity].Loop_Avail = CLEAR;
				Non_OP_Entry (sigEntity);
			}
			break;

		case SIG_Dup_Addr:
			/*
			*	RM(25)
			*/
			if (!sigData)
			{
				macData[sigEntity].MAC_Avail = CLEAR;
				macData[sigEntity].Loop_Avail = CLEAR;
				rmtData[sigEntity].DA_Flag = SET;
				SendCSPEvent (fddiMACDa_Flag, sigEntity);
				Ring_OP_Dup_Entry (sigEntity);
			}
			break;

		case SIG_MAC_Interrupt:
			/*
			*	RM(22)
			*/
#if 0
			printf("RM%d: Rmode in RM_RING_OP.\n", sigEntity);
			break;
#endif
			if (sigData & MAC_Rmode)
			{
				/* set TRM */
				SetCSPTimer ((uTime) stationData.T_Rmode,
					RMT_SIGNALS, sigEntity,
					&rmtData[sigEntity].TRM);

			}
			break;

		case SIG_RM_Timer:
			/*
			*	RM(21b)
			*/
#if 0
			printf("RM%d: RM_Timer timeout in RM_RING_OP\n",
			       sigEntity);
			break;
#endif
			if (ReadMACR_Flag (sigEntity))
			{
				/* Clear MAC_Avail and Loop_Avail are bug
					fixes from interim SMT meetings. */
				macData[sigEntity].MAC_Avail = CLEAR;
				macData[sigEntity].Loop_Avail = CLEAR;
				Rmode_Actions (sigEntity);
				Non_OP_Entry (sigEntity);
			}
			break;

		}
		break;
		
	/*
	*	RM3:DETECT
	*/
	case RM_DETECT:
	  CSPDPT1("RM3:%d:DETECT\n", sigEntity);
		switch (sigType)
		{
		/*
		*	Transition signals.
		*/
		case SIG_Ring_OP:
			/*
			*	RM(32)
			*/
			if (sigData)
				Ring_OP_Actions (sigEntity);
			break;

		case SIG_MAC_Interrupt:
			/*
			*	RM(33a)
			*/
			if ((sigData & (MAC_My_Beacon | MAC_Other_Beacon))
				&& (rmtData[sigEntity].BN_Flag))
			{
				/*
				* Leave Beacon state and set TRB.
				*/
				rmtData[sigEntity].BN_Flag = CLEAR;
				SetCSPTimer (Dup_Max, RMT_SIGNALS, sigEntity,
					&rmtData[sigEntity].TRB);

				/*
				*	910823-002	LJP
				*	Only set timer if beacon received
				*	was My_Beacon.
				*/
				if (sigData & MAC_My_Beacon)
				{
					/*
					* At this point, the state machine
					* assumes that the MAC is entering
					* the Claim state. This starts the
					* timer to detect the condition for
					* RM(34c).
					*/
					SetCSPTimer
						(macData[sigEntity].RM34cTime,
						RMT_SIGNALS, sigEntity,
						&rmtData[sigEntity].TRC);
					rmtData[sigEntity].claimTimed = CLEAR;
					rmtData[sigEntity].inClaim = SET;
				}

				Detect_Entry (sigEntity);
			}

			/*
			*	RM(33b)
			*/
			else if ((sigData & MAC_TRT_In_T4T5)
				&& (!rmtData[sigEntity].BN_Flag))
			{
				/*
				* If MAC was in Claim, then set timer to
				* time since claiming.
				*/
				rmtData[sigEntity].inClaim = CLEAR;
				SetCSPTimer (Dup_Max, RMT_SIGNALS, sigEntity,
					&rmtData[sigEntity].TRC);

				/*
				*	Enter the Beacon state.
				*/
				rmtData[sigEntity].BN_Flag = SET;
				rmtData[sigEntity].beaconTimed = CLEAR;
				Detect_Entry (sigEntity);
			}

			/*
			*	RM(34a)
			*/
			else if ((sigData & MAC_My_Claim)
				&& rmtData[sigEntity].claimTimed)
			{
				New_Dup_Actions (sigEntity);
				Non_OP_Dup_Entry (sigEntity);
			}

			/*
			*	RM(34b)
			*/
			else if ((sigData & MAC_My_Beacon)
				&& rmtData[sigEntity].beaconTimed)
			{
				New_Dup_Actions (sigEntity);
				Non_OP_Dup_Entry (sigEntity);
			}
			break;

		case SIG_RM_Timer:
			/*
			*	RM(36)
			*/
			if (rmtData[sigEntity].RM_Join
					&& rmtData[sigEntity].BN_Flag)
			{
				Directed_Actions (sigEntity);
			}
			/*
			*	NOTE: Special transitions for timing
			*	since entering Claim or 2D_Max
			*	since leaving Claim or Beacon are
			*	handled at the start of this routine.
			*/
			break;

		}
		break;

	/*
	*	RM4:NON_OP_DUP
	*/
	case RM_NON_OP_DUP:
	  CSPDPT1("RM4:%d:NON_OP_DUP\n", sigEntity);
		switch (sigType)
		{
		case SIG_RM_DA_Flag:
			/*
			*	RM(41)
			*/
			if (!(rmtData[sigEntity].DA_Flag = sigData))
				Non_OP_Entry (sigEntity);
			break;

		case SIG_MAC_Interrupt:
			/*
			*	RM(44a)
			*/
			if ((sigData & (MAC_My_Beacon | MAC_Other_Beacon))
				&& rmtData[sigEntity].BN_Flag)
			{
				rmtData[sigEntity].BN_Flag = CLEAR;
				Non_OP_Dup_Entry (sigEntity);
			}
			
			/*
			*	RM(44b)
			*/
			else if (!rmtData[sigEntity].BN_Flag
				&& (sigData & MAC_TRT_In_T4T5))
			{
				rmtData[sigEntity].BN_Flag = SET;
				Non_OP_Dup_Entry (sigEntity);
			}
			break;
			
		case SIG_Ring_OP:
			/*
			*	RM(45)
			*/
			if (sigData)
			{
				/* Clear NO_Flag */
				rmtData[sigEntity].NO_Flag = CLEAR;
				SMTSendSignal (SIG_EC_NO_Flag, 0, (uInt32) 0);
				Ring_OP_Dup_Entry (sigEntity);
			}
			break;

		case SIG_RM_Timer:
			/*
			*	RM(44c)
			*/
			if (!rmtData[sigEntity].BN_Flag)
			{
				Dup_Actions (sigEntity);
				Non_OP_Dup_Entry (sigEntity);
			}
			
			/*
			*	RM(46)
			*/
			else
			{
				if (rmtData[sigEntity].RM_Join)
					Directed_Actions (sigEntity);
			}
			break;
		}
		break;
		
	/*
	*	RM5:RING_OP_DUP
	*/
	case RM_RING_OP_DUP:
	  CSPDPT1("RM5:%d:RING_OP_DUP\n", sigEntity);
		switch (sigType)
		{
		case SIG_Dup_Addr:
			/*
			*	RM(52)
			*/
			if (sigData)
			{
				rmtData[sigEntity].DA_Flag = CLEAR;
				SendCSPEvent (fddiMACDa_Flag, sigEntity);
				Ring_OP_Actions (sigEntity);
			}
			break;
			
		case SIG_Ring_OP:
			/*
			*	RM(54)
			*/
			if (!sigData)
			{
				rmtData[sigEntity].JM_Flag = CLEAR;
				rmtData[sigEntity].BN_Flag = CLEAR;
				Non_OP_Dup_Entry (sigEntity);
			}
			break;

		case SIG_MAC_Interrupt:
			/*
			*	RM(55)
			*/
			if (sigData & MAC_Rmode)
			{
				/* set TRM */
				SetCSPTimer ((uTime) stationData.T_Rmode,
					RMT_SIGNALS, sigEntity, 
					&rmtData[sigEntity].TRM);

			}
			break;

		case SIG_RM_Timer:
			/*
			*	RM(54b)
			*/
			if (ReadMACR_Flag (sigEntity))
			{
				/* Clear MAC_Avail and Loop_Avail are bug
					fixes from interim SMT meetings. */
				macData[sigEntity].MAC_Avail = CLEAR;
				macData[sigEntity].Loop_Avail = CLEAR;
				Rmode_Actions (sigEntity);
				Non_OP_Dup_Entry (sigEntity);
			}
			break;
		}

		break;
		
	/*
	*	RM6:DIRECTED
	*/
	case RM_DIRECTED:
	  CSPDPT1("RM6:%d:DIRECTED\n", sigEntity);
		switch (sigType)
		{
		case SIG_MAC_Interrupt:
			if (sigData & (MAC_My_Beacon | MAC_Other_Beacon))
			{
				/*
				*	RM(63)
				*/
				if (!rmtData[sigEntity].DA_Flag)
				{
					/* Restore normal beacon */
					/*
					*	910429-001	LJP
					*	Use new beacon type definitions.
					*/
					SetBeacon (sigEntity,
						(uChar) BCN_CLAIM_FAILED, 
						(MACAddr48 *) NULL_ADDRESS, 
						0, (uChar *) NULL);
					rmtData[sigEntity].BN_Flag = CLEAR;
					Detect_Entry (sigEntity);
				}

				/*
				*	RM(64)
				*/
				else
				{
					/* Restore normal beacon */
					/*
					*	910429-001	LJP
					*	Use new beacon type definitions.
					*/
					SetBeacon (sigEntity,
						(uChar) BCN_CLAIM_FAILED, 
						(MACAddr48 *) NULL_ADDRESS, 
						0, (uChar *) NULL);
					rmtData[sigEntity].BN_Flag = CLEAR;
					Non_OP_Dup_Entry (sigEntity);
				}
			}
			break;

		case SIG_RM_Timer:
		case SIG_RM_RE_Flag:
			/*
			*	RM(67)
			*/
			if (ecmData.RE_Flag && (rmtData[sigEntity].TRM == 0))
			{
				/* Restore normal beacon */
				/*
				*	910429-001	LJP
				*	Use new beacon type definitions.
				*/
				SetBeacon (sigEntity, (uChar) BCN_CLAIM_FAILED,
					(MACAddr48 *) NULL_ADDRESS,
					0, (uChar *) NULL);

				/*
				*	910626-001	LJP
				*	Removed SMTSendSignal() for
				*	SIG_Trace_Prop since this is
				*	done in Trace_Actions().
				*/

				Trace_Actions (sigEntity);
			}
			break;
		}
		break;

	/*
	*	RM7:TRACE
	*/
	case RM_TRACE:
	  CSPDPT1("RM7:%d:TRACE\n", sigEntity);
		/*
		*	No specific transitions here. Only the general 
		*	transistions handled above.
		*/
		break;

	/*
	*	The following states are special substates to handle
	*	the interim wait periods during beacon jamming.
	*/

	/*
	*	Wait for T_Jam in Jam_A_Actions.
	*/
	case RM_JAM_A_WAIT_TJAM:
		if (sigType == SIG_RM_Timer)
		{
			/*
			*	Jam_A_Actions complete.
			*/

			/* Restore MAC input */
			SetMACInput (sigEntity, TRUE);

			/* Restore normal beacon */
			/*
			*	910429-001	LJP
			*	Use new beacon type definitions.
			*/
			SetBeacon (sigEntity, (uChar) BCN_CLAIM_FAILED, 
				(MACAddr48 *) NULL_ADDRESS, 0, (uChar *) NULL);

			/* Request MAC reset */
			MACResetRequest (sigEntity);

			/* Return to NON_OP_DUP state */
			Non_OP_Dup_Entry (sigEntity);
		}
		break;

	/*
	*	Wait for T_DBJ in Jam_B_Actions.
	*/
	case RM_JAM_B_WAIT_TDBJ:
		if (sigType == SIG_RM_Timer)
		{
			/* MAC Reset */
			MACResetRequest (sigEntity);
			
			/* BeaconRequest (macID, DA = MLA, INFO = Jam_Info) */
			BeaconRequest (sigEntity);
			
			/* Wait for T_Jam in second substate */
			SetCSPTimer (T_Jam, RMT_SIGNALS, sigEntity, 
				&rmtData[sigEntity].TRM);
			rmtData[sigEntity].rmState = RM_JAM_B_WAIT_TJAM;
		}
		break;

	/*
	*	Wait for T_Jam in Jam_B_Actions.
	*/
	case RM_JAM_B_WAIT_TJAM:
		if (sigType == SIG_RM_Timer)
		{
			/* MAC Reset */
			MACResetRequest (sigEntity);
			
			/* Restore normal beacon */
			/*
			*	910429-001	LJP
			*	Use new beacon type definitions.
			*/
			SetBeacon (sigEntity, (uChar) BCN_CLAIM_FAILED, 
				(MACAddr48 *) NULL_ADDRESS, 0, (uChar *) NULL);

			/* Return to NON_OP_DUP state */
			Non_OP_Dup_Entry (sigEntity);
		}
		break;

	}
	
	return;
}
