#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) cspsig.c V6.2.3:cs.622c:5:5 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module
	
	Signal Queue Module
	
	File:		cspsig.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.622c	08/02/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module implements the signal queue used by the CSP
	system. There are four functions associated with the signal
	queue: SMTSendSignal (add a signal to the queue), SMTReadSignal 
	(read a signal from the queue), InitSignalQueue (initializes 
	queue), and EmptySignalQueue (checks if signal queue is empty).
		
	Throughout the CSP, various events occur asynchronously. In order
	to process these events in the order in which they occur, they
	are placed in a FIFO queue. Each entry in the queue contains
	the event type (i.e., the signal), the entity to which this
	signal is being sent (the PHY or MAC index if applicable), and
	a 32-bit value containing data related to the event.
		
	The FIFO is implemented here as a circular queue in an array.
	The head is the index to the next entry to be added and the 
	tail is the next entry to be read. The empty condition
	is denoted by the head and tail being equal.

	Modification History:

	*** Updated to SMT 6.2 ***

*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmsg.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"cspmacro.h"
#include	"csphdr.h"

/*********************************************************************
	External Functions
*********************************************************************/

extern	void	SendCSPMessage ();

/*********************************************************************
	Signal Queue Data Structures
*********************************************************************/
extern int fault_cnt;

struct	SigQueueEntry {
	uInt16	type;
	uInt16	entity;
	uInt32	data;
};


/*********************************************************************
	Signal Queue Macros
*********************************************************************/

/*
*	The NextQueueEntry(n) macro returns the next index with a
*	wrap to 0 if the array size will be exceeded.
*/
#define	NextQueueEntry(n)	((n == MAX_SIG_QUEUE_SIZE - 1) ? 0 : n + 1)

/*********************************************************************
	Signal Queue Global Data
*********************************************************************/

static	sqHead,					/* head index value */
	sqTail;					/* tail index value */
		
static	struct SigQueueEntry
		sigQueue[MAX_SIG_QUEUE_SIZE];	/* FIFO array */


/*********************************************************************
	Signal Queue Functions
*********************************************************************/

uInt32
InitSignalQueue ()
/*********************************************************************
Function:	Initialize the signal queue structure.
Parameters:	None.
Input:		None.
Output:		Set sqHead and sqTail to 0.
Return:		0 if successful. Otherwise an error code is returned.
Modification History:
*********************************************************************/
{
ProcState	pState;
Int16		i;

	/*
	*	Disable interrupts while looking at the pointers.
	*/
	MDisableCSPInterrupts (&pState);
	

	/*
	*	Clear pointers.
	*/
	sqHead = sqTail = 0;
	
	/*
	*	Clear queue.
	*/
	for (i = 0; i < MAX_SIG_QUEUE_SIZE; i++)
	{
		sigQueue[i].type = 0;
		sigQueue[i].entity = 0;
		sigQueue[i].data = 0;
	}

	/*
	*	Enable interrupts.
	*/
	MRestoreCSPInterrupts (&pState);
	
	return (0);
}

uInt32
EmptySignalQueue ()
/*********************************************************************
Function:	Checks if signal queue is empty.
Parameters:	None.
Input:		sqHead and sqTail.
Output:		None.
Return:		FALSE if queue has items in it. TRUE if the queue is empty.
*********************************************************************/
{
uInt32		result;
ProcState	pState;

	/*
	*	Disable interrupts while looking at the pointers.
	*/
	MDisableCSPInterrupts (&pState);

	/*
	*	Get queue empty status.
	*/
	result = (sqHead == sqTail);

	/*
	*	Enable interrupts.
	*/
	MRestoreCSPInterrupts (&pState);

	return (result);
}

uInt32 Send_SignalTypeEntity;
uInt32 Send_SignalData;
uInt32 Read_SignalTypeEntity;
uInt32 Read_SignalData;

uInt32
SMTReadSignal (sigType, sigEntity, sigData)
	uInt16 *sigType;
	uInt16 *sigEntity;
	uInt32 *sigData;
/*********************************************************************
Function:	Reads next signal from the queue.
Parameters:	sigType		= pointer to buffer for signal type.
		sigEntity	= pointer to buffer for signal entity index.
		sigData		= pointer to buffer for signal data.
Input:		sqHead, sqTail, and sigQueue.
Output:		sigType, sigEntity, and sigData set. sigHead incremented.
Return:		0		if signal is retured successfully.
		ECSP_SIG_Q_MT	if queue is empty.
*********************************************************************/
{
ProcState	pState;

	/*
	*	Check for something to read.
	*/
	if (EmptySignalQueue ())
		return (ECSP_SIG_Q_MT);

	/*
	*	Disable interrupts while reading queue.
	*/
	MDisableCSPInterrupts (&pState);
	
	/*
	*	Set return values.
	*/
	*sigType = sigQueue[sqHead].type;
	*sigEntity = sigQueue[sqHead].entity;
	*sigData = sigQueue[sqHead].data;

   Read_SignalTypeEntity = ((uInt32) *sigType) << 16 | (uInt32) *sigEntity;
   Read_SignalData = *sigData;
	
	/*
	*	Increment head pointer.
	*/
	sqHead = NextQueueEntry (sqHead);
	
	/*
	*	Enable interrupts.
	*/
	MRestoreCSPInterrupts (&pState);
	
	return (0);
}

void
SMTSendSignal (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Reads next signal from the queue.
Parameters:	sigType		= signal type to send.
		sigEntity	= signal entity index.
		sigData		= signal data.
Input:		sqHead, sqTail, and sigQueue.
Output:		siqQueue entry added. sigTail incremented.
Return:		No value returned.
Note:		In the event that the signal queue becomes full, the signal
		will be lost and the connection process should be restarted.
*********************************************************************/
{
ProcState	pState;
extern int fault_cnt;
extern inline strobe_wdt();
	/*
	*	Disable interrupts while updating queue.
	*/
	MDisableCSPInterrupts (&pState);

	/*
	*	Check for queue full condition.
	*/
	if (NextQueueEntry (sqTail) == sqHead)
	{
	SMTMessage smtmsg;

		sqTail = sqHead = 0;

		/* report error condition */
		smtmsg.destination = MAP_MSG_ID;
		smtmsg.source = CSP_MSG_ID;
		smtmsg.type = CSP_ERROR_MSG;
		smtmsg.typeInfo = ECSP_SIG_Q_FULL;
		smtmsg.localID = 0;
		smtmsg.len1 = 0;
		smtmsg.len2 = 0;
#if 1
	        fault_cnt++;
#   if 1
	        strobe_wdt();
		SendCSPMessage (&smtmsg);
	        printf("Signal full: Signal Type: %x.  Entity: %x.  Data: %x\n",
		       sigType, sigEntity, sigData);    /* jlin */
#   else
	        {
		   int i, type, entity, data, count;

	        printf("Signal full: Signal Type: %x.  Entity: %x.  Data: %x\n",
		       sigType, sigEntity, sigData);    /* jlin */

		   type = sigQueue[i].type;
		   entity = sigQueue[i].entity;
		   data = sigQueue[i].data;
		   count = 1;
		   for (i=NextQueueEntry(sqHead);
			i != sqTail; i = NextQueueEntry(i), count++)
		   {
		      strobe_wdt();

		      if ((type != sigQueue[i].type) ||
			  (entity != sigQueue[i].entity) ||
			  (data != sigQueue[i].data))
		      {
			 printf ("Signal- type %x - entity %x - data %x "
				 "- count %x \n", type, entity, data, count);
			 type = sigQueue[i].type;
			 entity = sigQueue[i].entity;
			 data = sigQueue[i].data;
			 count = 0;
		      }
		   }
		}
#   endif
	        fault_cnt--;
#else
	        asm("fmark");
#endif

		MRestoreCSPInterrupts (&pState);
		return;
	}

	/*
	*	Add event.
	*/
	sigQueue[sqTail].type = sigType;
	sigQueue[sqTail].entity = sigEntity;
	sigQueue[sqTail].data = sigData;
	sqTail = NextQueueEntry (sqTail);

   Send_SignalTypeEntity = ((uInt32) sigType) << 16 | (uInt32) sigEntity;
   Send_SignalData = sigData;
	/*
	*	Enable interrupts.
	*/
	MRestoreCSPInterrupts (&pState);

	return;
}
