/*********************************************************************
	Frame Based Management Module
	
	Message Interface Routines

	THIS FILE IS TO BE PORTED BY THE IMPLEMENTOR.

	File:		fbmmsgif.c
	Created:	12/01/89

	SID:		1.4
	Last Modified:	1/28/91
	
	Copyright 1990,1991 XLNT Designs Inc.

	This module provides the interface routines to handle message
	passing between FBM and the other processes in the SMT system.

	Modification History:

	*** Updated to SMT 6.2 ***

*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smtmacro.h"
#include	"smterror.h"
#include	"fddihdr.h"
#include	"fbmmacro.h"
#include	"smtmsg.h"
#include	"fbmhdr.h"


/*********************************************************************
	Message Interface External Functions
*********************************************************************/


/*********************************************************************
	Message Interface Defined Values
*********************************************************************/


/*********************************************************************
	Message Interface Global Variables
*********************************************************************/


/*********************************************************************
	MIB Interface Global Variables
*********************************************************************/


/*********************************************************************
	Message Interface Routines
*********************************************************************/

void
SendFBMMessage (msg)
	SMTMessage	*msg;
/*********************************************************************
Function:	Send a message from FBM to another process.
Parameters:	msg	= address of buffer containing message.
Input:		msg	= contains SMT message structure that defines the
				message to send.
Output:		None.
Return:		None.
*********************************************************************/
{
   switch (msg->destination) {
   case CSP_MSG_ID:
      ReceiveCSPMessage (msg);         /* Call directly without allocate
					  extra memory and copy the data 
				        */
      break;

   case MAP_MSG_ID:
      ProcessMAPMessage (msg);         /* Call directly without allocate
					  extra memory and copy the data 
				        */
      break;

   case MIB_MSG_ID:
      ProcessMIBMessage (msg);         /* Call directly without allocate
					  extra memory and copy the data 
				        */
      break;
   }
   
   return;
}

void
FreeMessageBuffer (buf)
	SMTMessage	*buf;
/*********************************************************************
Function:	Release the memory used by a message received by FSP.
Parameters:	buf	= pointer to message buffer.
Input:		None.
Output:		Implementor defined.
Return:		None.
*********************************************************************/
{
   FREEMEMORY((uChar*) buf);
   return;
}

uInt32
InitFBMMessage ()
/*********************************************************************
Function:	Initialize system for sending and receiving SMT messages.
Parameters:	None.
Input:		None.
Output:		Set system message processing.
Return:		0 if successful, error number if failed.
*********************************************************************/
{
	return (0);
}

/*********************************************************************
        MIB Interface Routines

        The MIB interface routines are placed with the message
        interface routines because the MIB interface may require
        strong support from the message passing facilities provided
        by the system. In cases where this process is separate and
        unique from the MIB process, the message passing facilities
        will implement some form of interprocess communication.
                The strong support will be required when this process
        is waiting for a response from the MIB. In this case, any other
        messages will need to be held off or enqueued for storage
        until the MIB response is received.
                FBM does not set MIB attributes through the standard
        change interface. Therefore, only the get interface needs to
        be implemented.
*********************************************************************/

uInt32
FBMGetMIBAttr (bufSize, bufData, setCount)
        uInt16          bufSize;
        uChar           *bufData;
        SetCountType    *setCount;
/*********************************************************************
Function:       Get information from the MIB. This function provides
                FBM with access to the MIB. In cases where the MIB and
                FBM reside in the same executable image, this function
                can simply call GetMIBAttr(). If FBM and MIB are separated,
                this function will perform the necessary communication
                functions to access the MIB and wait for a reply.
Parameter:      bufSize         = maximum length of buffer bufData
                                        measured in chars.
                bufData         = pointer to buffer containing the
                                        requested attribute information
                                        and used to hold attribute value
                                        returned.
                setCount        = pointer to buffer to hold current value
                                        of the fddiSMTSetCountAttribute
                                        (may be NULL if current SetCount
                                        is not required).
Input:          bufData         = contains the requested attribute ID and
                                        object index (if required) using
                                        the MIB TLV encoding for a Get
                                        Request (see SMT standard).
Output:         bufData         = holds the attribute value requested
                                        using the MIB TLV encoding specified
                                        in the SMT standard.
                setCount        = if not NULL, this buffer gets set to the
                                        current value of fddiSMTSetCount,
                                        also using the MIB TLV encoding.
Return:         None.
Note:           The operation of this function must cause FBM to block
                (or effectively stop processing) until the response
                is received from the MIB. This routine is called
                throughout the protocol processing of FBM.
*********************************************************************/
{
extern  uInt32  GetMIBAttr ();

	return (GetMIBAttr (bufSize, bufData, setCount));
}


uInt32
FBMChangeMIBAttr (bufSize, bufData, setCount, stationID)
	uInt16			bufSize;
	uChar			*bufData;
	TLVSetCountType		*setCount;
	SMTStationIdType	*stationID;
/*********************************************************************
Function:	MIB access routine to change a MIB attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					new attribute information
					and used to hold attribute value
					returned.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= pointer to buffer holding ID of station
					requesting the change.
Input:		bufData		= contains the new attribute value using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
		setCount	= contains the verification value for
					fddiSMTSetCount.
		stationID	= contains station ID of the station
					requesting the attribute change.
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
		setCount	= set to the value of fddiSMTSetCount after
					attempting the attribute change,
					also using the MIB TLV encoding.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_BAD_SET_COUNT
			the value of setCount does not match the current
			MIB value
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_OUT_OF_RANGE
			value given is out of bounds
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
*********************************************************************/
{
extern	ChangeMIBAttr ();

	return (ChangeMIBAttr (bufSize, bufData, setCount, stationID));
}

uInt32
FBMAddMIBAttr (bufSize, bufData, setCount, stationID)
	uInt16			bufSize;
	uChar			*bufData;
	TLVSetCountType		*setCount;
	SMTStationIdType	*stationID;
/*********************************************************************
Function:	MIB access routine to add items to the MIB attributes
		fddiMACLongAliases, fddiMACShortAliases, fddMACLongGrpAddrs,
		or fddiMACShortGrpAddrs.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing a single
					item to add to the attribute
					information and used to hold
					attribute value returned.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= pointer to buffer holding the ID
					of the station requesting the add.
Input:		bufData		= contains one and only one item to be
					added to the attribute value using
					the MIB TLV encoding.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= contains the ID of the station requesting
					the add operation.
Output:		bufData		= holds the complete attribute value
					after the addition has been attempted
					using the MIB TLV encoding specified
					in the SMT standard.
		setCount	= set to the value of fddiSMTSetCount after
					attempting the attribute change,
					also using the MIB TLV encoding.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_BAD_SET_COUNT
			the value of setCount does not match the current
			MIB value
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_NO_MORE
			this attribute cannot hold any more values
		RC_OUT_OF_RANGE
			value given is out of bounds
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
		RC_LENGTH_ERROR
			length of attribute does not correspond to
			a single item value
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
Notes:		AddMIBAttr() may only be used for the attributes
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		and fddiMACShortGrpAddrs.
*********************************************************************/
{
extern	AddMIBAttr ();

	return (AddMIBAttr (bufSize, bufData, setCount, stationID));
}

uInt32
FBMRemoveMIBAttr (bufSize, bufData, setCount, stationID)
	uInt16			bufSize;
	uChar			*bufData;
	TLVSetCountType		*setCount;
	SMTStationIdType	*stationID;
/*********************************************************************
Function:	MIB access routine to remove an itme from attribute
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		or fddiMACShortGrpAddrs.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					attribute information to be removed
					and used to hold attribute value
					returned.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= pointer to buffer holding the ID
					of the station requesting the remove.
Input:		bufData		= contains one and only one value to be
					removed represented in MIB TLV
					encoding.
		setCount	= contains the verification value for
					fddiSMTSetCount.
		stationID	= contains the ID of the station requesting
					the remove operation.
Output:		bufData		= holds the attribute value after attempting
					the deletion in the MIB TLV encoding
					specified in the SMT standard.
		setCount	= set to the value of fddiSMTSetCount after
					attempting the attribute delete,
					also using the MIB TLV encoding.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_BAD_SET_COUNT
			the value of setCount does not match the current
			MIB value
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_NO_MORE
			no more items to be removed
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
		RC_LENGTH_ERROR
			length of attribute does not correspond to
			a single item value
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
Notes:		RemoveMIBAttr() may only be used with attributes
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		and fddiMACShortGrpAddrs. Specifying the removal of zero
		members results in removing all members of the attribute.
*********************************************************************/
{
extern	RemoveMIBAttr ();

	return (RemoveMIBAttr (bufSize, bufData, setCount, stationID));
}
