/**********************************************************************
	Management Information Base Module

	System Interface Module

	THIS FILE IS TO BE PORTED BY THE IMPLEMENTOR.

	File:		mibsysif.c
	Created:	12/01/89

	SID:		1.3
	Last Modified:	1/28/91
	
	Copyright 1990,1991 XLNT Designs Inc.

	This module is responsible for interfacing to the system
	environment for providing information used and supplied by
	the Management Information Base (MIB). This module provides
	routines to access hardware- and system-level data.

	Modification History:

	*** Updated to SMT 6.2 ***

*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"fddihdr.h"
#include	"smtmacro.h"
#include	"mibdefs.h"
#include	"mibtypes.h"
#include	"mibglbl.h"
#include        "mibconf.h"

#include        <krnl.h>
#include        "eeprecs.h"
#include        "edfddinvr.h"
#include        "drv.h"
#include        "msgutil.h"


/*********************************************************************
	Defined Values
*********************************************************************/
#define TIMER_TIMESTAMP_TICK             1      /* 10 ms tick */
#define NULLMACADDRS                     "\0\0\0\0\0\0"

#define MACADDR48_NONULL(addr) (*((uInt32 *) addr) || *((uInt16 *) addr[4]))

#define MACADDR16_NONULL(addr) (*((uInt16 *) addr))

#define COPYMACADDR48(dest,src) *((uInt32 *) dest) = *(uInt32 *) src; \
                                *((uInt16 *) &dest[4]) = *(uInt16*) src;

#define COPYMACADDR16(dest,src) *((uInt16 *) dest) = *((uInt16 *) src);

#define MACADDR48_MATCH(a1, a2) ((*((uInt32 *) a1) == *((uInt32 *) a2)) \
			  && (*((uInt16 *) a1[4]) == *((uInt16 *) a2[4])))

#define MACADDR16_MATCH(a1, a2) (*((uInt16 *) a1) == *((uInt16 *) a2))

/*********************************************************************
	Defined Structures
*********************************************************************/
typedef void (*GetMACAddrFuncVector()) (uChar**, uChar**, uInt16, uInt16 *);
typedef void (*NullCompareFuncVector()) (uChar**);

typedef void (*CopyFuncVector()) (uChar**, uChar**);
typedef void (*CompareFuncVector()) (uChar**, uChar**);
    
/*********************************************************************
	External Variables
*********************************************************************/

/*********************************************************************
          Variables
 *********************************************************************/

uInt32 TimeStamp_10msTicks;
uInt16 LongAliasesIndex, ShortAliasesIndex,
       LongGrpAddrsIndex, ShortGrpAddrsIndex;
uChar  TimeStampEnable=FALSE;

MACAddr48 LongAliases  [MAX_MAC_COUNT][MAX_ALIASES_COUNT];
MACAddr16 ShortAliases [MAX_MAC_COUNT][MAX_ALIASES_COUNT];

MACAddr48 LongGrpAddrs  [MAX_MAC_COUNT][MAX_ALIASES_COUNT];
MACAddr16 ShortGrpAddrs [MAX_MAC_COUNT][MAX_ALIASES_COUNT];


/*********************************************************************
	System Interface Support Functions.
*********************************************************************/
/* 
 *   Called by TimerTicks in krnl.c
 */
void FDDITimerTicksInt ( void )
{
   GenericMsgType *msg;

   TimeStamp_10msTicks++;
   if (TimeStampEnable && FBMTimerInterrupt())
   {
      msg = (GenericMsgType *) AcptFDDIMsgHdr();
      if (msg)
      {
	 msg->msg_type = MSG_FBM_TIMER_TIMEOUT;
	 SendMessage((MSGHDR *) msg, &RxSMTFrameMBox);
      }
   }
}

/*
 *
 */


void GetMACAddr48 ( buffer, addrTable, index, length )
uChar **buffer;
MACAddr48 *addrTable;
uInt16 index, *length;
{


   if (MACADDR48_NONULL(addrTable[index]))
   {
      COPYMACADDR48(*buffer, addrTable[index]);
      *length += 6;
      *buffer += 6;
   }
}

/*
 *
 */
void GetMACAddr16 ( buffer, addrTable, index, length )
uChar **buffer;
MACAddr16 *addrTable;
uInt16 index, *length;
{

   if (MACADDR16_NONULL(addrTable[index]))
   {
      COPYMACADDR48(*buffer, addrTable[index]);
      *length += 2;
      *buffer += 2;
   }
}

/*
 *
 */
uInt32 GetMACAddressGroup ( buffer, blength, index, addrTable, 
			    CopyMACAddrFunc)
uChar *buffer;
uInt16 *blength, index;
uChar  **addrTable;
GetMACAddrFuncVector CopyMACAddrFunc;
{
   uInt16 i;


   *blength = 0;

   if (index)
   {
      for (i=0; i < index; i++)
      {
         CopyMACAddrFunc ( &buffer, addrTable, i, blength );
      }
      return RC_SUCCESS;
   }
   else
   {
      return RC_NO_MORE;
   }
}

/*
 *
 */
uChar IsMACAddr48Nonull ( addr )
MACAddr48 addr;
{
   return MACADDR48_NONULL(addr);
}

uChar IsMACAddr16Nonull ( addr )
MACAddr16 addr;
{
   return MACADDR16_NONULL(addr);
}

uChar SameMACAddr48 ( addr1, addr2 )
MACAddr48 addr1, addr2;
{
   return MACADDR48_MATCH (addr1, addr2);
}

uChar SameMACAddr16 ( addr1, addr2 )
MACAddr48 addr1, addr2;
{
   return MACADDR16_MATCH (addr1, addr2);
}

void CopyMACAddr48 ( dest, src )
MACAddr48 dest, src;
{
   COPYMACADDR48 (dest, src);
}

void CopyMACAddr16 ( dest, src )
MACAddr16 dest, src;
{
   COPYMACADDR16 (dest, src);
}

/*
 *
 */
uInt32 AddMACAddrGrp (addr, addrTable, index, NullCompareFunc, CopyFunc)
uChar *addr, **addrTable;
uInt16 *index;
NullCompareFuncVector NullCompareFunc;
CopyFuncVector CopyFunc;
{
   uInt16 i;

   for (i=0; i < *index; i++)
   {
      if (!NullCompareFunc(addr, addrTable[i]))
      {
         break;
      } 
   }

   if (i < *index)
   {
      CopyFunc(addrTable[i], addr);
      return RC_SUCCESS;
   }
   else if (i < MAX_ALIASES_COUNT)
   {
      *index = i;
      CopyFunc(addrTable[i], addr);
      return RC_SUCCESS;
   }

   return RC_NO_MORE;
}

/*
 *
 */
uInt32 RemoveMACAddrGrp (addr, addrTable, index, NullCompareFunc, 
			 CopyFunc, CompareFunc)
uChar *addr, **addrTable;
uInt16 *index;
NullCompareFuncVector NullCompareFunc;
CopyFuncVector CopyFunc;
CompareFuncVector CompareFunc;
{
   uInt16 i;

   if (!*index)
   {
      return RC_NO_MORE;           /* Table is empty */
   }

   for (i=0; i < *index; i++)
   {
      if (CompareFunc(addr, addrTable[i]))
      {
         break;
      }      
   }

   if (i == *index)
   {
      return RC_OUT_OF_RANGE;
   }
   else
   {
      CopyFunc ( addrTable[i], NULLMACADDRS );
   }

   if (++i == *index)
   {
      for (--i; i ; )
      {
         if (NullCompareFunc(addrTable[--i]))
	 {
            *index = i + 1;
	    break;                   /* if the entry is not null */
         }
      }
   }
   return RC_SUCCESS;
}

/*
 *
 */

/*********************************************************************
	System Interface Functions.
*********************************************************************/
uInt32
InitTimeStamp ()
/*********************************************************************
Function:	This function initializes the SMT time stamp function.
		The SMT time stamp is a 64-bit value measuring the
		number of 80 ns intervals since the station started.
		This function is called at station start up and should
		initialize the starting time stamp to 0.
		The accuracy of the time stamp is not critical. However,
		consecutive calls to the time stamp function must return
		unique, increasing values (until the the count wraps
		around).
Parameters:	None.
Input:		None.
Output:		None.
Return:		0 if successful, otherwise an error code is returned.
*********************************************************************/
{
   extern uInt32 currentIntState;

   currentIntState = TRUE; /* Used in the MDisableCSPInterrupts() &
			      MRestoreCSPInterrupts() */
   TimeStamp_10msTicks = 0;
   TimeStampEnable = FALSE;
   return (0);
}

void
GetTimeStamp (buf)
	SMTTimeStamp	*buf;
/*********************************************************************
Function:	This function generates the next SMT time stamp.
		The SMT time stamp is a 64-bit value measuring the
		number of 80 ns intervals since the station started.
		The accuracy of the time stamp is not critical. However,
		consecutive calls to the time stamp function must return
		unique, increasing values (until the the count wraps
		around).
Parameters:	buf	= buffer to hold time stamp value.
Input:
Output:
Return:		No value returned.
*********************************************************************/
{
   static unsigned long prev_ticks;
   static unsigned long unique = 0;

   /*
    * convert 10 millisecond ticks to 80 nanosecond ticks
    */
   buf->hiword = TimeStamp_10msTicks / 34360;
   buf->loword = TimeStamp_10msTicks * 125000;
   buf->loword += unique;

   /*
    * ensure that a unique value is always generated
    */
   if (prev_ticks == TimeStamp_10msTicks)
      unique++;
   else
      unique = 0;

   prev_ticks = TimeStamp_10msTicks;

      return;
}


/*********************************************************************
	MAC Interface Routines
*********************************************************************/

uInt32
GetMACData (attr, macID, buffer)
	uInt16	attr;
	uInt16	macID;
	uChar	*buffer;
/*********************************************************************
Function:	Get a MAC attribute.
		Many of the MAC-related attributes are contained
		within the MAC or its associated hardware and software.
		To get the most current data, GetMIBMACAttr() calls
		GetMACData() to retrieve the data from the system itself.
Parameters:	attr	= MAC attribute to retrieve.
		macID	= index of MAC to retrieve data from. This value
			ranges from 0 to one less than the number of 
			MACs in the system.
		buffer	= pointer to a buffer to hold the requested attribute.
Input:		None.
Output:		Sets buffer to contain requested attribute.
Return:         The value returned corresponds to the results codes used
                by the Parameter Manager Protocol:
                RC_SUCCESS
                        attribute retrieved successfully
                RC_NO_PARAM
                        attribute not supported
                RC_ILLEGAL_PARAMETER
                        illegal parameter requested
*********************************************************************/
{
uInt32	result = RC_SUCCESS, value;
uInt16  bufLen;

        if (macID >= mib->SMTStationConfigGrp.MAC_Ct)
	{
           return RC_OUT_OF_RANGE;
        }
        

	/*
	*	Read the appropriate MAC attribute.
	*/
	switch (attr)
	{
	/*
	*	fddiMACOperationGrp
	*/
	case fddiMACT_Neg:
	        result = DRV_StatusReq( macID, SM_MA_STATUS_T_NEG,
					&value);
	        *(uInt32*) buffer = value;
                if (result == RC_SUCCESS)
		   mib->MACOperationGrp[macID].T_Neg = (uInt32) value;
		break;

	/*
	*	fddiMACCountersGrp
	*/
        case fddiMACFrame_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_FRAME_CT, 
				       &value);
	        *(uInt32*) buffer = value;
/*
                if (result == RC_SUCCESS)
   		   *(uInt32*) buffer += mib->MACCountersGrp[macID].Frame_Ct;
*/
		break;

        case fddiMACCopied_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_COPIED_CT, 
				       &value );
	        *(uInt32*) buffer = value;
/*
                if (result == RC_SUCCESS)
		   *(uInt32*) buffer += mib->MACCountersGrp[macID].Copied_Ct;
*/
		break;

        case fddiMACTransmit_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_TX_CT,
				       &value );
	        *(uInt32*) buffer = value;
/*                if (result == RC_SUCCESS)
		   *(uInt32*) buffer += mib->MACCountersGrp[macID].Transmit_Ct;
*/
		break;

        case fddiMACToken_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_TOKEN_CT,
				       &value );	
	        *(uInt32*) buffer = value;
/*                if (result == RC_SUCCESS)
		   *(uInt32*) buffer += mib->MACCountersGrp[macID].Token_Ct;
*/
		break;

	/*
	*	fddiMACErrorCtrsGrp
	*/
        case fddiMACError_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_ERROR_CT,
				       &value );
	        *(uInt32*) buffer = value;
/*                if (result == RC_SUCCESS)
		   *(uInt32*) buffer += mib->MACErrorCtrsGrp[macID].Error_Ct;
*/
		break;

        case fddiMACLost_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_LOST_CT,
				       &value);
	        *(uInt32*) buffer = value;
/*                if (result == RC_SUCCESS)
		   *(uInt32*) buffer += mib->MACErrorCtrsGrp[macID].Lost_Ct;
*/
		break;

	case fddiMACTvxExpired_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_TVX_EXPIRED_CT,
				       &value );
	        *(uInt32*) buffer = value;
/*                if (result == RC_SUCCESS)
		   *(uInt32*) buffer += mib->MACErrorCtrsGrp[macID].TvxExpired_Ct;
*/
		break;

        case fddiMACNotCopied_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_NOT_COPIED_CT,
				       &value );
	        *(uInt32*) buffer = value;
/*                if (result == RC_SUCCESS)
		   *(uInt32*) buffer += mib->MACErrorCtrsGrp[macID].NotCopied_Ct;
*/
		break;

	case fddiMACRingOp_Ct:
		*(uInt32*) buffer = mib->MACErrorCtrsGrp[macID].RingOp_Ct;
		break;

        case fddiMACLate_Ct:
                result = DRV_StatusReq( macID, 
				        SM_MA_STATUS_LATE_CT,
				       &value );
	        *(uInt16*) buffer = value;
/*                if (result == RC_SUCCESS)
		   *(uInt16*) buffer += mib->MACErrorCtrsGrp[macID].Late_Ct;
*/
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

uInt32
SetMACData (attr, macID, buffer)
	uInt16	attr;
	uInt16	macID;
	uChar	*buffer;
/*********************************************************************
Function:	Set a MAC attribute.
		Many of the MAC-related attributes are contained
		within the MAC or its associated hardware and software.
		To set the data values, SetMIBMACAttr() calls SetMACData()
		to place the data into the appropriate location.
Parameters:	attr	= MAC attribute to set.
		macID	= index of MAC to set. This value ranges from
			0 to one less than the number of MACs in the system.
		buffer	= pointer to a buffer holding the data value.
Input:		buffer	= contents of buffer are used for the new value.
Output:		Hardware or system software is set with new value.
		buffer	= value actually set in the hardware. This applies
			primarily to values T_Req and T_Min
			where the requested value may not match the
			hardware's granularity. In these cases, the value
			used must be equal to or greater than the value
			requested.
Return:         The value returned corresponds to the results codes used
                by the Parameter Manager Protocol:
                RC_SUCCESS
                        attribute changed successfully
                RC_NO_PARAM
                        attribute not supported
                RC_OUT_OF_RANGE
                        value given is out of bounds
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
                RC_ILLEGAL_PARAMETER
                        illegal parameter requested
*********************************************************************/
{
uInt32	result = RC_SUCCESS;
uInt32  value;


        if (macID >= mib->SMTStationConfigGrp.MAC_Ct)
	{
           return RC_OUT_OF_RANGE;
        }

	switch (attr)
	{
        /*
        *       fddiMACAddressGrp
        */
        /*
        *       Normally, the MAC's address is not writeable according to
        *       the standard. For this version of the XLNT Manager, the
        *       address is writeable.
        */
        case fddiMACSMTAddress:               /* buffer: MACAddr48  */
	        result = DRV_ControlReq ( macID, 
					  SM_MA_CTRL_SET_LONG_ADDR,
					 (uInt32) buffer );
		break;

        /*
	*	fddiMACOperationGrp
	*/
        case fddiMACT_Req:                    /* buffer: uInt32 */
		/*
		*	Set the T_Req value in the specified MAC. If the
		*	value requested cannot be represented by the MAC,
		*	then use the closest time value not to exceed
		*	the requested value. Return the time value used
		*	in the parameter buffer.
		*/
	        if ((result = DRV_ControlReq ( macID, 
					  SM_MA_CTRL_SET_T_REQ,
					 (uInt32) *buffer)) == RC_SUCCESS)
		{
		   mib->MACOperationGrp[macID].T_Req = (uInt32) *buffer;
		}

		break;

	 case fddiMACT_Min:                    /* buffer: uInt32 */
		/*	Operate setting this value in the same manner as
		*	setting T_Req. In most cases, T_Min is not
		*	a chip-based value and does not need to be
		*	adjusted.
		*/
		mib->MACOperationGrp[macID].T_Min = (uInt32) *buffer;
#if 0		
	        if (result = DRV_ControlReq ( macID, 
					  SM_MA_CTRL_SET_TMIN,
					 (uInt32) *buffer);
#endif
		break;

        case fddiMACFrameStatus:
	        if ((result = DRV_ControlReq ( macID, 
					  SM_MA_CTRL_MAC_FRAME_STATUS,
					 (uInt32) *buffer)) == RC_SUCCESS)
		{
		   mib->MACOperationGrp[macID].FrameStatus = 
		      (uInt16) *buffer;
		}
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}


/*********************************************************************
	MAC Add/Remove Interface Routines
*********************************************************************/

uInt32
AddMIBMACAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to add items to the MIB attributes
		fddiMACLongAliases, fddiMACShortAliases, fddMACLongGrpAddrs,
		or fddiMACShortGrpAddrs.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing a single
					item to add to the attribute
					information and used to hold 
					attribute value returned.
Input:		bufData		= contains one and only one item to be
					added to the attribute value using
					the MIB TLV encoding.
Output:		bufData		= holds the complete attribute value
					after the addition has been attempted
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_NO_PARAM
			attribute not supported
		RC_NO_MORE
			this attribute cannot hold any more values
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
Notes:		AddMIBAttr() may only be used for the attributes
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		and fddiMACShortGrpAddrs.
*********************************************************************/
{
   uInt32	result = RC_SUCCESS, request;
   TLVParamType *p;
   uInt16 macID;

   return RC_NO_PARAM;  /* not supported by this concentrator currently */

   p = (TLVParamType *) bufData;
   macID = p->MACINDEX;

   if (macID >= mib->SMTStationConfigGrp.MAC_Ct)
   {
      return RC_OUT_OF_RANGE;
   }
        
   switch (p->paramType)
   {
      case fddiMACLongAliases:
           request = SM_MA_CTRL_ADD_LONG_ALIASES;
           result = AddMACAddrGrp (p->MACOTHER, LongAliases[macID], 
				  &LongAliasesIndex, IsMACAddr48Nonull, 
				   CopyMACAddr48);
           break;
      case fddiMACLongGrpAddrs:
           request = SM_MA_CTRL_ADD_LONG_GRPADDR;
           result = AddMACAddrGrp (p->MACOTHER, LongGrpAddrs[macID], 
				  &LongGrpAddrsIndex, IsMACAddr48Nonull, 
				   CopyMACAddr48);
           break;

      case fddiMACShortAliases:
           request = SM_MA_CTRL_ADD_LONG_ALIASES;
           result = AddMACAddrGrp (p->MACOTHER, ShortAliases[macID], 
				  &ShortAliasesIndex, IsMACAddr16Nonull, 
				   CopyMACAddr16);
           break;

      case fddiMACShortGrpAddrs:
           request = SM_MA_CTRL_ADD_SHORT_GRPADDR;
           result = AddMACAddrGrp (p->MACOTHER, ShortGrpAddrs[macID], 
				  &ShortGrpAddrsIndex, IsMACAddr16Nonull, 
				   CopyMACAddr16);
           break;

      default:
           return (RC_NO_PARAM);
   }

   if (result == RC_SUCCESS)
   {
     DRV_ControlReq (macID, request, (uInt32) p->MACOTHER);
   }

   return result;
}

uInt32
RemoveMIBMACAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to remove an itme from attribute
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		or fddiMACShortGrpAddrs.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					attribute information to be removed
					and used to hold attribute value
					returned.
Input:		bufData		= contains one and only one value to be
					removed represented in MIB TLV
					encoding.
Output:		bufData		= holds the attribute value after attempting
					the deletion in the MIB TLV encoding 
					specified in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_NO_PARAM
			attribute not supported
		RC_NO_MORE
			no more items to be removed
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
Notes:		RemoveMIBAttr() may only be used with attributes
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		and fddiMACShortGrpAddrs. Specifying the removal of zero
		members results in removing all members of the attribute.
*********************************************************************/
{
   uInt32	result, request;
   TLVParamType *p;
   uInt16 macID;

   return RC_NO_PARAM;  /* not supported by this concentrator currently */

   p = (TLVParamType *) bufData;
   macID = p->MACINDEX;

   if (macID >= mib->SMTStationConfigGrp.MAC_Ct)
   {
      return RC_OUT_OF_RANGE;
   }
        
   switch (p->paramType)
   {
      case fddiMACLongAliases:
           request = SM_MA_CTRL_REMOVE_LONG_ALIASES;
           result = RemoveMACAddrGrp (p->MACOTHER, LongAliases[macID], 
				      &LongAliasesIndex, 
				      IsMACAddr48Nonull, 
				      CopyMACAddr48, SameMACAddr48);
           break;

      case fddiMACLongGrpAddrs:
           request = SM_MA_CTRL_REMOVE_LONG_GRPADDR;
           result = RemoveMACAddrGrp (p->MACOTHER, LongGrpAddrs[macID], 
				      &LongGrpAddrsIndex, 
				      IsMACAddr48Nonull, 
				      CopyMACAddr48, SameMACAddr48);
           break;

      case fddiMACShortAliases:
           request = SM_MA_CTRL_REMOVE_SHORT_ALIASES;
           result = RemoveMACAddrGrp (p->MACOTHER, ShortAliases[macID], 
				      &ShortAliasesIndex, 
				      IsMACAddr16Nonull, 
				      CopyMACAddr16, SameMACAddr16);
           break;

      case fddiMACShortGrpAddrs:
           request = SM_MA_CTRL_REMOVE_SHORT_GRPADDR;
           result = RemoveMACAddrGrp (p->MACOTHER, ShortGrpAddrs[macID], 
				      &ShortGrpAddrsIndex, 
				      IsMACAddr16Nonull, 
				      CopyMACAddr16, SameMACAddr16);
           break;

      default:
  	   return (RC_NO_PARAM);
   }

   if (result == RC_SUCCESS)
   {
      DRV_ControlReq (macID, request, (uInt32) p->MACOTHER);
   }

   return result;
}


uInt32
GetMACAddresses ( bufSize, paramData )
uInt16 bufSize; 
TLVParamType *paramData;
{
uInt32	result = RC_SUCCESS;
uChar *buffer, macID;

   return RC_NO_PARAM;  /* not supported by this concentrator currently */

   macID  = paramData->MACINDEX;
   buffer = paramData->MACADDR;

   if (macID >= mib->SMTStationConfigGrp.MAC_Ct)
   {
      return RC_OUT_OF_RANGE;
   }

   switch (paramData->paramType)
   {
        case fddiMACLongAliases:
                result = GetMACAddressGroup ( buffer, &bufSize,
					      LongAliasesIndex,
					      LongAliases[macID], 
					      GetMACAddr48);
                break;

        case fddiMACShortAliases:
                result = GetMACAddressGroup ( buffer, &bufSize, 
					      ShortAliasesIndex,
					      ShortAliases[macID],
					      GetMACAddr16);
                break;

	case fddiMACLongGrpAddrs:
                result = GetMACAddressGroup ( buffer, &bufSize,
					      LongGrpAddrsIndex,
					      LongGrpAddrs[macID],
					      GetMACAddr48);
                break;

        case fddiMACShortGrpAddrs:
                result = GetMACAddressGroup( buffer, &bufSize,
					     ShortGrpAddrsIndex,
					     ShortGrpAddrs[macID],
					     GetMACAddr16);
                break;
	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);

}

/*********************************************************************
	PATH Interface Routines
*********************************************************************/

/*
*	NOTE: A copy of all PATH values is maintained in the MIB
*	data base. Therefore, no GetPATHData() routine is required.
*/

uInt32
SetPATHData (attr, pathID, buffer)
	uInt16	attr;
	uInt16	pathID;
	uChar	*buffer;
/*********************************************************************
Function:	Set a PATH attribute.
		Many of the PATH-related attributes are contained 
		within the PATH or its associated hardware and software.
		To set the data values, SetMIBPATHAttr() calls SetPATHData()
		to place the data into the appropriate location.
Parameters:	attr	= PATH attribute to set.
		pathID	= index of PATH to set. This value will be either
			PATH_P or PATH_S for the primary or secondary path
			and is only used for PATH Class PATH attributes.
		buffer	= pointer to a buffer holding the data value.
Input:		buffer	= contents of buffer are used for the new value.
Output:		Hardware or system software is set with new value.
		buffer	= value actually set in the hardware.
			If the hardware does not support the actual value
			requested, then the closest value supported is
			used and returned through parameter buffer.
Return:         The value returned corresponds to the results codes used
                by the Parameter Manager Protocol:
                RC_SUCCESS
                        attribute changed successfully
                RC_NO_PARAM
                        attribute not supported
                RC_OUT_OF_RANGE
                        value given is out of bounds
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
                RC_ILLEGAL_PARAMETER
                        illegal parameter requested
*********************************************************************/
{
uInt32	result = RC_SUCCESS, temp;
uInt16 i, mask;

   if (pathID >= mib->SMTStationConfigGrp.MAC_Ct)
   {
      return RC_OUT_OF_RANGE;
   }
        
	switch (attr)
	{
        /*
        *       fddiPATHConfigGrp
	*
	*	Setting the lower bound values for the PATHs uses the
	*	following protocol. The lower bound is used to counter
	*	the problems with granularity. When setting the lower
	*	bound, all MACs that have the primary or secondary
	*	PATH available (it doesn't have to be requested) will
	*	have the appropriate value set. The value used in the
	*	MAC will represent a time value less than or equal
	*	to the requested value.
	*	Remember, the timer values are 2s complements.
        */
        case fddiPATHClassTrace_MaxExpiration:
	        if (((uTime) MFddi2Usecs(-*(uInt32*) buffer)) < Min_Trace_Max)
		{
		   return RC_OUT_OF_RANGE;
		}

                break;

        case fddiPATHClassTVXLowerBound:
		/* Verify time value is in range */
		/* For each MAC that has access to P or S path. */
		/* Determine appropriate TVX */
		/* Set MAC */
		/* Store in mib->MACOperationGrp[pathID].TvxValue */	
                if (((uTime) MFddi2Usecs (-*(uInt32*) buffer)) > TVX_MAX)
		{
                   return RC_OUT_OF_RANGE;
		}

		temp = *(uInt32*) buffer;
		mib->ClassConfigGrp.TVXLowerBound = temp;
		for (i=0; i < eep_mfg_rec.eep_fddi.smtMac_Ct; i++)
		{
		   if (temp < (mib->MACCapabilitiesGrp[i].TVXGreatestLowerBound))
		   {
		      mib->MACCapabilitiesGrp[i].TVXGreatestLowerBound = temp;
		      if (temp < (mib->MACOperationGrp[i].TvxValue))
		      {	
		           mib->MACOperationGrp[i].TvxValue = temp;
                           result = DRV_ControlReq (i, SM_MA_CTRL_SET_TVX,
						    temp);
		      }
		   }
		}
		break;

        case fddiPATHClassT_MaxLowerBound:
		/* Verify time value is in range */
		/* For each MAC that has access to P or S path. */
		/* Determine appropriate T_Max */
		/* Set MAC */
		/* buffer contains FddiTimeTwosComplement format */
		/* Store in mib->MACOperationGrp[pathID].T_Max */

                if (((uTime) MFddi2Usecs (-*(uInt32*) buffer)) > T_MAX_MAX)
		{
                   return RC_OUT_OF_RANGE;
		}

		temp = *(uInt32 *) buffer;
		mib->ClassConfigGrp.T_MaxLowerBound = temp;

		for (i=0; i < eep_mfg_rec.eep_fddi.smtMac_Ct; i++)
		{
		   if (temp < (mib->MACCapabilitiesGrp[i].T_MaxGreatestLowerBound))
		   {
		      mib->MACCapabilitiesGrp[i].T_MaxGreatestLowerBound =
		        temp;
		      if (temp < (mib->MACOperationGrp[i].T_Max))
		      {	
			 mib->MACOperationGrp[i].T_Max = temp;
                         result = DRV_ControlReq (i, SM_MA_CTRL_SET_TMAX,
						    temp);

		      }
		   }
		 }

		break;


        /*
        *       fddiPATHClassPATHConfigGrp
        */
        case fddiPATHClassPATHRingLatency:
		/*
		*	The reasons for being able to set the ring
		*	latency are not clear in the standard. There
		*	is no protocol for handling this operation, so
		*	no action needs to be taken here.
		*/
		break;

        case fddiPATHClassPATHSba:
		/*
		*	Similar to ring latency.
		*/
		break;

        case fddiPATHClassPATHSbaOverhead:
		/*
		*	Similar to ring latency.
		*/
		break;
	}

	return (result);
}


/*********************************************************************
	PORT Interface Routines
*********************************************************************/

uInt32
GetPORTData (attr, portID, buffer)
	uInt16	attr;
	uInt16	portID;
	uChar	*buffer;
/*********************************************************************
Function:	Get a PORT attribute.
		Many of the PORT-related attributes are contained 
		within the PORT or its associated hardware and software.
		To get the most current data, GetMIBPORTAttr() calls
		GetPORTData() to retrieve the data from the system itself.
Parameters:	attr	= PORT attribute to retrieve.
		portID	= index of PORT to retrieve data from. This value
			ranges from 0 to one less than the number of 
			PORTs in the system.
		buffer	= pointer to a buffer to hold the requested attribute.
Input:		None.
Output:		Sets buffer to contain requested attribute.
Return:         The value returned corresponds to the results codes used
                by the Parameter Manager Protocol:
                RC_SUCCESS
                        attribute retrieved successfully
                RC_NO_PARAM
                        attribute not supported
                RC_ILLEGAL_PARAMETER
                        illegal parameter requested
*********************************************************************/
{
uInt32	result = RC_SUCCESS, value;

   if (portID >= (mib->SMTStationConfigGrp.NonMaster_Ct + 
		  mib->SMTStationConfigGrp.Master_Ct))
   {
      return RC_OUT_OF_RANGE;
   }
        
	/*
	*	Read the appropriate PORT attribute.
	*/
	switch (attr) 
	{
        /*
        *       fddiPORTErrorCtrsGrp
        */
	case fddiPORTEBError_Ct:
		result = DRV_StatusReq (portID, SM_PH_STATUS_ELBUF_ERROR, 
					&value);
		*(uInt32*) buffer = value;
/*
                if (result == RC_SUCCESS)
		   *(uInt32 *) buffer += mib->PORTErrorCtrsGrp[portID].EBErr_Ct;
  
*/
              break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

/*
*	All PORT values that can be set do not directly affect the
*	hardware and are handled through other interfaces so no
*	SetPORTData() routine is required.
*/

/*
*	ATTACHMENT values have not been finalized in the standard. The
*	current values only need to be set at initialization.
*/
/*********************************************************************
	jlin : ATTACHMENT Interface Routines
*********************************************************************/

uInt32
GetATTACHMENTData (attr, attachID, buffer)
	uInt16	attr;
	uInt16	attachID;
	uChar	*buffer;
/*********************************************************************
Function:	Get a PORT attribute.
		Many of the PORT-related attributes are contained 
		within the PORT or its associated hardware and software.
		To get the most current data, GetMIBPORTAttr() calls
		GetPORTData() to retrieve the data from the system itself.
Parameters:	attr	= PORT attribute to retrieve.
		attachID	= index of PORT to retrieve data from. This value
			ranges from 0 to one less than the number of 
			PORTs in the system.
		buffer	= pointer to a buffer to hold the requested attribute.
Input:		None.
Output:		Sets buffer to contain requested attribute.
Return:         The value returned corresponds to the results codes used
                by the Parameter Manager Protocol:
                RC_SUCCESS
                        attribute retrieved successfully
                RC_NO_PARAM
                        attribute not supported
                RC_ILLEGAL_PARAMETER
                        illegal parameter requested
*********************************************************************/
{
uInt32	result = RC_SUCCESS, value, i;
volatile uChar* iop;

   if (attachID >= (mib->SMTStationConfigGrp.NonMaster_Ct + 
		  mib->SMTStationConfigGrp.Master_Ct))
   {
      return RC_OUT_OF_RANGE;
   }
        
	/*
	*	Read the appropriate PORT attribute.
	*/
	switch (attr) 
	{
        /*
        *       fddiPORTErrorCtrsGrp
        */
	case fddiATTACHMENTInsertedStatus:
#ifdef __FEBRIDGE
	      if (attachID > PHY_A)		/* optical bypass bit in reg of PHY A */
#else
	      if (attachID > PHY_B)
#endif
	      {
		 *buffer = CLEAR;
	      }
	      else
	      {

		 iop = FDDI_BYPASS_SWITCH_PRESENT;
		 *iop &= 0x48;

		 for (i=0; i < 10; i++); /* wait loop */
#ifdef __FEBRIDGE
		 *buffer = (~(*iop)) & 0x10; /* invert and mask off all but 3rd bit */
#else
		 *buffer = *iop;
#endif
	      }
              break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}





