#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmrcv.c V6.2.3:cs.622c:5:5 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Based Management Module

	Receive SMT Frames Module

	File:		fbmrcv.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.622c	08/02/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module is responsible for receiving SMT frames and taking
	whatever action is necessary for processing each frame.  For 
	frames which require a require a response one is generated 
	immediately.  

	Modification History:

	*** Updated to SMT 6.2 ***
	910122-002	LJP
		After validating the frame contents, if the frame is not
		valid and it is a request, the code now checks if this 
		frame was going to be processed by this station before
		issuing an RDF.
	910204-004	LJP
		ValidateFrameContents() checks for plen==0. This condition
		will never exist. Instead, when looping through the parameters
		it needs to check if the parameter type is 0.
	910315-003	LJP
		When processing an invalid frame, the code checks to see
		if this frame was to be processed by this station. To do
		this, it was checking the SA for my address or broadcast.
		It now checks the DA.
	910326-003	LJP
		ValidateFrameContents() was reading the version from the
		frame using htons(). Now it uses ntohs().
	910605-002	LJP
		SMT Should not respond to frames with FC=NSA and
		DA=Broadcast.
	910605-003	LJP
		Removed changes 910122-002 and 910315-003. These were
		replaced with new code to check whether an invalid
		Request frame needs an RDF.
	910618-001	LJP
		Added code to ValidateFrameContents() to check if
		parameter lengths are even multiples of 4. This
		prevents parameters from starting on odd addresses
		which may cause data accessing problems.
*********************************************************************/
#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"fbmhdr.h"
#include	"fbmframe.h"
#include	"fbmglbl.h"


/*********************************************************************
	External functions used.
*********************************************************************/

extern	void	PassMAPFrame ();
extern	uInt32	BuildNIFResponse ();
extern	uInt32	BuildSIFConfigResponse ();
extern	uInt32	BuildSIFOpResponse ();
extern	uInt32	BuildEchoResponse ();
extern	uInt32	BuildRDFResponse ();
extern	void	ClearFrameBuffer ();
extern	void	ProcessNIFReceived ();
extern	void	ProcessNIFResponse ();
extern	void	ProcessESFAnnouncement ();
extern	void	ProcessESFRequest ();
extern	void	ProcessESFResponse ();
extern	void	ProcessPMFGet ();
extern	void	ProcessPMFChange ();
extern	void	ProcessPMFAdd ();
extern	void	ProcessPMFRemove ();
extern	void	SendSMTFrame ();
extern	uInt16	ntohs ();
extern	uInt32	ntohl ();
extern	uInt16	htons ();
extern	uInt32	htonl ();
extern	uInt32	LocalFBMFrames ();


/*********************************************************************
	FBM Global Variables
*********************************************************************/


/*********************************************************************
	FBM Receive Routines
*********************************************************************/

#if 1  /* jlin: used in fbmintif.c  06/22/93 */
uInt32
#else
static uInt32
#endif
ValidateFrameContents (frameHdr, frameLen)
	SMTFrameHdrType	*frameHdr;
	uInt16		frameLen;
/*********************************************************************
Function:	This function checks the contents of the frame to
		verify that its basic format and size are correct.
Parameters:	frameHdr	= address of frame buffer.
		frameLen	= length of frame (in bytes).
Input:		frameHdr	= contains frame to process. This must
				include the three pad bytes at the start
				of the frame.
Output:		None.
Return:		Reason code indicating result of test:
		RC_SUCCESS
			Contents of frame are OK so far.
		RC_LENGTH_ERROR
			Length of frame and info field do not match or
			frame parameter length incorrect.
Modification History:
*********************************************************************/
{
uInt16		infoLen,	/* info field length */
		plen,		/* parameter length */
		len;		/* length counter */
TLVHdrType	*param;		/* parameter ptr */

	/*
	*	Verify there is enough data to form a valid SMT header
	*	and the amount of data received is what the frame
	*	says should be there.
	*/
	infoLen = ntohs (frameHdr->smtHdr.InfoField_Length);
	if ((frameLen < SMT_FRAME_HDR_SIZE) ||
		(frameLen != (infoLen + SMT_FRAME_HDR_SIZE)))
	{
		/*
		*	Lengths do not match or frame is too small.
		*/
	        FBMDPT4("$ infoLen = %x   HDR_SIZE = %x   "
		       "frameLen = %x   (info+hdr) = %x $",
		       infoLen,  SMT_FRAME_HDR_SIZE, frameLen, 
		       infoLen+ SMT_FRAME_HDR_SIZE);
		return (RC_LENGTH_ERROR);
	}

	/*
	*	Skip through parameters and verify that the InfoField
	*	length ends where a parameter ends.
	*/
	param = (TLVHdrType *) (frameHdr + 1);
	len = 0;
	/*
	*	910618-001	LJP
	*	This code did not account for the possibility that
	*	odd parameter lengths would cause problems. However,
	*	adding an odd length to the current parameter pointer
	*	causes the next parameter to start at an odd address.
	*	This causes memory accessing problems on some
	*	processors.
	*
	*	To prevent this, all parameter lengths must be an even
	*	multiple of 4 except for Echo Data and Rejected Frame
	*	Beginning. These two exceptions may be any length as long
	*	as they are the last parameters in the frame.
	*
	*	The code below allows any parameter to be an odd length
	*	at the end of the frame. If it is not the last parameter
	*	then the frame is invalid (i.e., the next parameter will
	*	not start on a 32-bit boundary).
	*/
	plen = 0;
	while (len < infoLen)
	{
		/*
		*	910618-001	LJP
		*	This plen is for the previous parameter. Since len
		*	is not less than infoLen, it was not the last
		*	parameter. If its length is not a multiple of 4,
		*	then the frame is invalid.
		*/
		if (plen & 3)
			return (RC_LENGTH_ERROR);

		plen = (ntohs (param->paramLen) + sizeof (TLVHdrType));
		/* 910204-004 LJP */
		if (param->paramType == 0)
			return (RC_LENGTH_ERROR);
		len += plen;
		param = (TLVHdrType *) (((uChar *) param) + plen);
	}

	/*
	*	Check if sizes match.
	*/
	if (len != infoLen)
	  {
	        FBMDPT2("$len %d != info %d$", len, infoLen);
		return (RC_LENGTH_ERROR);
	  }

	/*
	*	Verify version.
	*/
	/* 910326-003	LJP */
	if ((ntohs (frameHdr->smtHdr.Version_ID)
			< fbmStationData.loVersionID)
		|| (ntohs (frameHdr->smtHdr.Version_ID)
			> fbmStationData.hiVersionID))
	{
		/* version is not in range */
	        FBMDPT1("$VersionID rxed: %d$", frameHdr->smtHdr.Version_ID);
		return (RC_FRAME_VERSION);
	}

	return (RC_SUCCESS);
}

uInt32
ProcessFBMFrame (memPtr, frameLen, MACNum, EACbits)
	uChar	*memPtr;
	uInt16	frameLen;
	uInt16	MACNum;
	uChar	EACbits;
/*********************************************************************
Function:	This function is called from the frame service
		routine when a frame has been received by the station.
		This function checks the EAC bits to see if the frame
		needs to be processed by this station and takes
		appropriate action based upon the frame type.
Parameters:	memPtr		= address of frame buffer.
		frameLen	= length of frame (in bytes).
		MACNum		= MAC index that received the frame.
		EACbits		= EAC indicators as received by the MAC.
Input:		memPtr		= contains frame to process. This must
				include the three pad bytes at the start
				of the frame.
Output:		May issue new frames, set MIB attributes, pass frame to
		MAP, or do nothing.
Return:		0.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*frameHdr;	/* header for frames to process */
uInt16		len;		/* frame length */
uInt32		reasonCode;	/* validation result */

	/*
	*	Don't process until initialized.
	*/
	if (!fbmStationData.FBMInitialized)
		return (0);

	/*
	*	Check E indicator.
	*/
	if (EACbits & E_Indicator)
		/*
		*	Ignore this frame.
		*/
		return (0);

	/*
	*	Get frame header.
	*/
	frameHdr = (SMTFrameHdrType *) memPtr;

	/*
	*	For all request frames, clear buffer for responses
	*/
	if (frameHdr->smtHdr.Frame_Type == SMTREQUEST)
		ClearFrameBuffer (fbmFrameBuffer);

	/*
	*	Validate frame contents and format.
	*/
	reasonCode = ValidateFrameContents (frameHdr, frameLen);
	if (reasonCode != RC_SUCCESS)
	{
	  FBMDPT("Invalid frame rxed. ");
		/*
		*	Frame is not valid. If this is a request,
		*	then issue a RDF.
		*/
		if (frameHdr->smtHdr.Frame_Type == SMTREQUEST)
		{
			/*
			*	910605-003	LJP
			*	Removed changes 910122-002 & 910315-003.
			*	RDFs are sent to Request frames only.
			*	Of the Request frames, this code assumes
			*	that the MAC and frame interface only
			*	pass SMT frames to be processed by
			*	this station. Of these frames, the only
			*	ones that SMT would not have responded to are
			*	secondary NSAs (FC=0x4F and Ar=S). This
			*	code now checks for Request frames that
			*	would have been processed by SMT by looking
			*	at the FC and Ar indicator.
			*/
			if ((frameHdr->macHdr.FC == SMT_INFO_FC)
				|| ((frameHdr->macHdr.FC == SMT_NSA_FC) && 
					!(EACbits & A_Indicator)))
			{
				/* build RDF frame */
				BuildRDFResponse (memPtr, fbmFrameBuffer,
					MACNum, reasonCode, frameLen);

				/* send frame */
				len = ntohs 
				    (fbmFrameHeader->smtHdr.InfoField_Length)
					+ SMT_FRAME_HDR_SIZE;
				SendSMTFrame (fbmFrameBuffer, len, MACNum);
			}
		}

		return (0);
	}

	/*
	*	Select process based on frame type.
	*/
	switch (frameHdr->smtHdr.Frame_Type)
	{
	/*
	*	Announcement frames.
	*/
	case SMTANNOUNCE:
	   FBMDPT("SMTAnnounce_");
		/*
		*	Process frame classes.
		*/
		switch (frameHdr->smtHdr.Frame_Class)
		{
		case NIF_CLASS:
		     FBMDPT("NIF.");
			/* handle through NN protocol */
			ProcessNIFReceived (memPtr, frameLen, MACNum, EACbits);
			break;

		case RAF_CLASS:
		     FBMDPT("RAF.");
			/* RAF protocol undefined */
			break;

		case ESF_CLASS:
		     FBMDPT("ESF.");
			/* do local processing */
			ProcessESFAnnouncement
				(memPtr, frameLen, MACNum, EACbits);
			break;

		case SRF_CLASS:
		     FBMDPT("SRF.");
			/* send this frame to the MAP */
			PassMAPFrame (FBM_EVENT_FRAME_RECEIVED,
				memPtr, frameLen, MACNum, EACbits);
			break;

		default:
			break;
		}
		break;

	case SMTREQUEST:
	   FBMDPT("SMTRequest_");
		/*
		*	910605-002	LJP
		*	Do not respond to secondary NSAs.
		*/
		if ((frameHdr->macHdr.FC == SMT_NSA_FC) && 
				(EACbits & A_Indicator))
#if 0
			return (0);
#else
	                break;   /* jlin: break here so that smt statistics
				    can be collected. 06/01/93 */
#endif
					
		/*
		*	Process frame classes.
		*/
		switch (frameHdr->smtHdr.Frame_Class)
		{
		case NIF_CLASS:
		     FBMDPT("NIF.");
			/* handle through NN protocol */
			ProcessNIFReceived (memPtr, frameLen, MACNum, EACbits);
			break;

		case SIFCONFIG_CLASS:
		     FBMDPT("SIFConfig.");
			/* build response frame */
			BuildSIFConfigResponse (memPtr, fbmFrameBuffer,
				MACNum);

			/* send frame */
			len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
				+ SMT_FRAME_HDR_SIZE;
			SendSMTFrame (fbmFrameBuffer, len, MACNum);

			break;

		case SIFOP_CLASS:
		     FBMDPT("SIFOp.");
			/* build response frame */
			BuildSIFOpResponse (memPtr, fbmFrameBuffer, MACNum);

			/* send frame */
			len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
				+ SMT_FRAME_HDR_SIZE;
			SendSMTFrame (fbmFrameBuffer, len, MACNum);

			break;

		case ECHO_CLASS:
		     FBMDPT("ECHO.");
			/* build response frame */
			BuildEchoResponse (memPtr, fbmFrameBuffer, MACNum);

			/* send frame */
			len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
				+ SMT_FRAME_HDR_SIZE;
			SendSMTFrame (fbmFrameBuffer, len, MACNum);

			break;

		case ESF_CLASS:
		     FBMDPT("ESF.");
			/* handle through local protocol */
			ProcessESFRequest (memPtr, frameLen, MACNum, EACbits);
			break;

		case PMFGET_CLASS:
		     FBMDPT("PMFGET.");
			/* handle through PMF protocol */
			ProcessPMFGet (memPtr, frameLen, MACNum, EACbits);
			break;

		case PMFCHANGE_CLASS:
		     FBMDPT("PMFCHANGE.");
			/* handle through PMF protocol */
			ProcessPMFChange (memPtr, frameLen, MACNum, EACbits);
			break;

		case PMFADD_CLASS:
		     FBMDPT("PMFADD.");
			/* handle through PMF protocol */
			ProcessPMFAdd (memPtr, frameLen, MACNum, EACbits);
			break;

		case PMFREMOVE_CLASS:
		     FBMDPT("PMFREMOVE.");
			/* handle through PMF protocol */
			ProcessPMFRemove (memPtr, frameLen, MACNum, EACbits);
			break;

		case RAF_CLASS:
		     FBMDPT("RAF.");
		default:
			/* build RDF frame */
			BuildRDFResponse (memPtr, fbmFrameBuffer,
				MACNum, RC_FRAME_CLASS, frameLen);

			/* send frame */
			len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
				+ SMT_FRAME_HDR_SIZE;
			SendSMTFrame (fbmFrameBuffer, len, MACNum);

			break;
		}
		break;

	case SMTRESPONSE:
	   FBMDPT("SMTResponse_");
		/*
		*	Process frame classes.
		*/
		switch (frameHdr->smtHdr.Frame_Class)
		{
		case NIF_CLASS:
		     FBMDPT("NIF.");
			/* handle through NN protocol */
			ProcessNIFResponse (memPtr, frameLen, MACNum, EACbits);
			break;

		case ESF_CLASS:
		     FBMDPT("ESF.");
			/* handle through local interface */
			ProcessESFResponse (memPtr, frameLen, MACNum, EACbits);
			break;

		case SIFCONFIG_CLASS:
		case SIFOP_CLASS:
		case ECHO_CLASS:
		case PMFGET_CLASS:
		case PMFCHANGE_CLASS:
		case PMFADD_CLASS:
		case PMFREMOVE_CLASS:
		case RDF_CLASS:
		     FBMDPT("PassedToMAP.");
			/* send response frame to MAP */
			PassMAPFrame (FBM_EVENT_FRAME_RECEIVED,
				memPtr, frameLen, MACNum, EACbits);
			break;

		case RAF_CLASS:
		default:
			break;
		}
		break;
	}

	LocalFBMFrames (memPtr, frameLen, MACNum, EACbits);

	return (0);
}

uInt32
InitFBMReceive ()
/*********************************************************************
Function:	This function initialize static global values used
		for received frame processing.
Parameters:	None.
Input:		None.
Output:		Sets static globals to initial values.
Return:		0 if successful, otherwise error code.
Modification History:
*********************************************************************/
{
	return (0);
}
