#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmnn.c V6.2.3:cs.910829:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Based Management Module

	Neighbor Notification Protocol Module

	File:		fbmnn.c
	Created:	09/05/90

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.910829	09/04/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	This module is responsible for processing the Neighbor Notification
	protocol specified by the SMT standard. Every FBM_CLOCK_PULSE
	seconds, ProcessNNTick() is executed. When the amount of time passed
	reaches T_Notify, then the protocol is executed.

	The NN protocol uses NIFs. All NIF requests and announcements 
	received by this station are processed through ProcessNIFReceived().
	The NIF responses are handled through ProcessNIFResponse().

	Modification History:
	910315-005	LJP
		SMT should respond to all primary NSA NIF Requests.
		This change adjusts to code to drop only secondary
		NSAs.
	910823-004	LJP
		Corrected logic for determining T_Notify period.
*********************************************************************/
#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"fbmhdr.h"
#include	"fbmframe.h"
#include	"fbmglbl.h"
#include	"mibdefs.h"


/*********************************************************************
	Defined Values
*********************************************************************/


/*********************************************************************
	Global Data
*********************************************************************/


/*********************************************************************
	External Functions
*********************************************************************/

extern	uInt32	BuildNIFRequest ();
extern	void	SendFBMEvent ();
extern	uChar	*FindFrameParam ();
extern	void	ClearFrameBuffer ();
extern	uInt16	ntohs ();
extern	uInt32	ntohl ();
extern	uInt16	htons ();
extern	uInt32	htonl ();
extern	uInt32	BuildRDFResponse ();
extern	uInt32	BuildNIFRequest ();
extern	uInt32	BuildNIFResponse ();
extern	uInt32	GenTransID ();
extern	void	SendSMTFrame ();
extern	void	PassMAPFrame ();
extern	uInt32	FBMGetMIBAttr ();


/*********************************************************************
	FBM NN Protocol Routines
*********************************************************************/

void
ProcessNIFReceived (memPtr, frameLen, MACNum, EACbits)
	uChar	*memPtr;
	uInt16	frameLen;
	uInt16	MACNum;
	uChar	EACbits;
/*********************************************************************
Function:	This function is called from ProcessmFBMFrame() to
		handle NIF Requests and Announcements in conjunction
		with the Neighbor Notification protocol.
Parameters:	memPtr		= address of frame buffer.
		frameLen	= length of frame (in bytes).
		MACNum		= MAC index that received the frame.
		EACbits		= EAC indicators as received by the MAC.
Input:		memPtr		= contains frame to process. This must
				include the three pad bytes at the start
				of the frame.
Output:		May issue new frames, set MIB attributes, pass frame to
		MAP, or do nothing.
Return:		None.
Modification History:
*********************************************************************/
{
SMTFrameHdrType		*frameHdr;	/* header for received frame */
Int16			compare;	/* address compare result */
StationStateParamType	*ssp;		/* pointer to station state param */
Flag			frameError = CLEAR;/* error condition found in frame */
ParamHdrType		*phdr;
uChar			*p;
uInt16			len;
Flag                    tempFlag;

	/*
	*	Get frame header.
	*/
	frameHdr = (SMTFrameHdrType *) memPtr;

	/*
	*	910315-005	LJP
	*	Any NSA with the Ar set is a secondary NSA,
	*	no work needs to be done.
	*/
	if ((frameHdr->macHdr.FC == SMT_NSA_FC) && (EACbits & A_Indicator))
		return;

	/*
	*	Check NIF_Received condition.
	*	ProcessFBMFrame() has already checked this frame for
	*	NIF Request or Announcement.
	*/
	MCompareAddress (frameHdr->macHdr.DA, BROADCAST_ADDRESS, compare);
	if ((frameHdr->macHdr.FC == SMT_NSA_FC) && (compare == 0)
		&& (!(EACbits & (E_Indicator | A_Indicator | C_Indicator))))
	{
		/*
		*	NIF_Received condition met, now do
		*	Respond_Actions.
		*/

		/* reset TVU */
		fbmMACData[MACNum].TVU = 0;

		/* check for new upstream neighbor */
		MCompareAddress (fbmMACData[MACNum].UNA,
			frameHdr->macHdr.SA, compare);
		if (compare != 0)
		{
			/* set new UNA */
			MCopyAddress (fbmMACData[MACNum].UNA,
				frameHdr->macHdr.SA);

			/* notify MIB of new UNA */
			SendFBMEvent (fddiMACUpstreamNbr, MACNum);
		}

		/* locate Station State parameter */
		ssp = (StationStateParamType *) FindFrameParam
			(STATION_STATE_PARAM_TYPE, memPtr);
		if (ssp)
		{
			/* Change upstream neighbor DA flag if needed. */
		        tempFlag = (ssp->DuplAddr & DuplAddr_My_Duplicate)
			           ? SET : CLEAR;
			
			if (fbmMACData[MACNum].myUNADuplicate != tempFlag)
			{
				/* set new condition */
				fbmMACData[MACNum].myUNADuplicate = tempFlag;
					
				SendFBMEvent (fddiMACUnaDa_Flag, MACNum);
			}
		}
		else
			frameError = SET;
	}

	/*
	*	Respond to all request frames (as opposed to 
	*	announcement frames).
	*/
	if (frameHdr->smtHdr.Frame_Type == SMTREQUEST)
	{
		/* clear buffer for the response frame */
		ClearFrameBuffer (fbmFrameBuffer);

		/* test for my address */
		MCompareAddress (frameHdr->macHdr.DA,
			fbmMACData[MACNum].SMTAddress, compare);

		/*
		*	If frame could not be processed, then
		*	send RDF.
		*/
		if (frameError)
		{
			/* build RDF frame */
			BuildRDFResponse (memPtr, fbmFrameBuffer,
				MACNum, RC_ERROR_UNKNOWN, frameLen);
			/*
			*	Send frame.
			*/
			SendSMTFrame (fbmFrameBuffer,
				ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
					+ SMT_FRAME_HDR_SIZE, MACNum);
		}

		/*
		*	Else send response frame.
		*
		*	910315-005	LJP
		*	Respond to all NIF Requests at this point.
		*/
		else
		{
			/*
			*	Build response frame.
			*/
			BuildNIFResponse (memPtr, fbmFrameBuffer, MACNum);

			if ((frameHdr->macHdr.FC == SMT_NSA_FC) 
				&& (compare == 0))
			{
				phdr = (ParamHdrType *) FindFrameParam
					(ECHO_PARAM_TYPE, memPtr);
				if (phdr)
				{
				  p = fbmFrameBuffer + (ntohs 
				    (fbmFrameHeader->smtHdr.InfoField_Length)
						+ sizeof (SMTFrameHdrType));
				  len = ntohs (phdr->length)
						+ sizeof (ParamHdrType);
				  MEMCOPY (p, phdr, len);
				  fbmFrameHeader->smtHdr.InfoField_Length =
				    htons 
				     (ntohs 
				      (fbmFrameHeader->smtHdr.InfoField_Length) 
				      + len);
				}
			}

			/*
			*	Send frame.
			*/
			SendSMTFrame (fbmFrameBuffer,
				ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
					+ SMT_FRAME_HDR_SIZE, MACNum);
		}

	}

	return;
}

void
ProcessNIFResponse (memPtr, frameLen, MACNum, EACbits)
	uChar	*memPtr;
	uInt16	frameLen;
	uInt16	MACNum;
	uChar	EACbits;
/*********************************************************************
Function:	This function processes an NIF response frame and performs
		duplicate address detection, available communication
		detection, and downstream neighbor detection.
Parameters:	memPtr		= address of buffer containing frame.
		frameLen	= total length of frame in bytes.
		MACNum		= MAC index which received the frame.
		EACbits		= EAC bits received by MAC.
Input:		memPtr		= NIF response frame received.
Output:		Performs duplicate address detection and notifies RMT if
		a duplicate address is found. Sets the MAC's downstream 
		neighbor, if available.
Return:		No value is returned.
Modification History:
*********************************************************************/
{
uInt32		transID;		/* Frame's xaction ID */
SMTFrameHdrType	*frameHdr;		/* header to frame */
Int16		compare;		/* result of address compare */

	/*
	*	Get header addresses.
	*/
	frameHdr = (SMTFrameHdrType *) memPtr;

	/*
	*	Check if this response frame can be used for NN.
	*	ProcessFBMFrame() has already check type, class,
	*	version and E indicator.
	*/
	MCompareAddress (frameHdr->macHdr.DA, fbmMACData[MACNum].SMTAddress,
		compare);
	if ((frameHdr->macHdr.FC == SMT_INFO_FC)
		&& (compare == 0))
	{
		/*
		*	Check if this is the response frame to the
		*	request sent out from this station.
		*/
		transID = ntohl (frameHdr->smtHdr.Transaction_ID);
		/*
		 *	This is the response frame.
		 *	Do NN processing.
		 */
		/* Check for duplicate address */
		 if (EACbits & A_Indicator)
		 {
		                FBMDPT1("FBMNN%d:My Duplicate found.\n",
					MACNum);
				/*
				*	Duplicate address condition.
				*	Do Fail_Actions.
				*/
				if (fbmMACData[MACNum].dupAddrTest
					!= DA_Test_Fail)
				{
					fbmMACData[MACNum].dupAddrTest
						= DA_Test_Fail;
					SendFBMEvent (fddiMACDup_Addr_Test,
						MACNum);
				}

				/* set station state */
				if (!fbmMACData[MACNum].myDuplicate)
				{
					fbmMACData[MACNum].myDuplicate = SET;
					SendFBMEvent (fddiMACDa_Flag, MACNum);
				}

				/* clear DNA */
				MCompareAddress (fbmMACData[MACNum].DNA,
					NULL_ADDRESS, compare);
				if (compare != 0)
				{
					MCopyAddress (fbmMACData[MACNum].DNA,
						NULL_ADDRESS);
					SendFBMEvent (fddiMACDownstreamNbr,
						MACNum);
				}

				if (transID != 
				      fbmMACData[MACNum].NNTransactionID)
				{
				   PassMAPFrame (FBM_EVENT_FRAME_RECEIVED,
						 memPtr, frameLen, MACNum,
						 EACbits);
				}
		}
		else
		{
			if (transID == fbmMACData[MACNum].NNTransactionID)
			{
				/*
				*	No duplicate condition.
				*	Do Pass_Actions.
				*/
				if (fbmMACData[MACNum].dupAddrTest
					!= DA_Test_Pass)
				{
					fbmMACData[MACNum].dupAddrTest
						= DA_Test_Pass;
					SendFBMEvent (fddiMACDup_Addr_Test,
						MACNum);
				}

				/* set station state */
				if (fbmMACData[MACNum].myDuplicate)
				{
				        FBMDPT1("FBMNN%d:My Duplicate disappeared.\n",
					MACNum);
					fbmMACData[MACNum].myDuplicate = CLEAR;
					SendFBMEvent (fddiMACDa_Flag, MACNum);
				}

				/* set DNA */
				MCompareAddress (fbmMACData[MACNum].DNA,
					frameHdr->macHdr.SA, compare);
				if (compare != 0)
				{
					MCopyAddress (fbmMACData[MACNum].DNA,
						frameHdr->macHdr.SA);
					SendFBMEvent (fddiMACDownstreamNbr,
						MACNum);
				}

				/* reset TVD */
				fbmMACData[MACNum].TVD = 0;

				/* set new transaction ID */
				fbmMACData[MACNum].NNTransactionID
					= GenTransID ();
			}
			else
			{
			   /*
			    *	This response was not sent for the NN
			    *	protocol. Assume it came from the MAP and
			    *	pass the frame up.
			    */
			    PassMAPFrame (FBM_EVENT_FRAME_RECEIVED,
					  memPtr, frameLen, MACNum, EACbits);
			}
	       }
	}

	return;
}

void
ProcessNNTick ()
/*********************************************************************
Function:	For every clock tick in FBM, this routine is called
		as one of the timer service routines. When T_Notify
		time has passed, the NN protocol is executed.
Parameters:	None.
Input:		timeToNotify	= amount of time left.
Output:		timeToNotify	= reset or decremented.
Return:		No value returned.
Modification History:
*********************************************************************/
{
Int16		i;			/* loop counter */
TLVParamType	param;			/* MIB parameter */
Int16		compare;		/* address comparison result */
uInt16		len;			/* length of send frame */

	/*
	*	Process all MACs.
	*/
	for (i = 0; i < fbmStationData.macCount; i++)
	{
		/*
		*	Skip invalid MACs.
		*/
		if (!fbmMACData[i].operational)
			continue;

		/*
		*	Increment timers.
		*	Only increment TNN if it has not expired yet.
		*/
		/*
		*	910823-004	LJP
		*	Increment TNN without testing for expiration.
		*/
		fbmMACData[i].TNN += FBM_CLOCK_TICK;
		fbmMACData[i].TVU += FBM_CLOCK_TICK;
		fbmMACData[i].TVD += FBM_CLOCK_TICK;

		/*
		*	If TNN has timed out,
		*	send an NIF request.
		*/
		/*
		*	910823-004	LJP
		*	Test if TNN >= T_Notify, not TNN > T_Notify.
		*/
		if (fbmMACData[i].TNN >= fbmStationData.T_Notify)
		{
			/*
			*	Get RMT state.
			*/
			param.paramType = fddiMACRMTState;
			param.paramLen = 4;
			param.MACINDEX = i + 1;
			FBMGetMIBAttr (sizeof (TLVParamType),
				(uChar *) &param,
				(SetCountType *) NULL);

			/*
			*	Ring is up if in RM2 or RM5.
			*/
			if (param.MACPARAM8 == RM_RING_OP
				|| param.MACPARAM8 == RM_RING_OP_DUP)
			{
				/*
				*	Clear frame buffer.
				*/
				ClearFrameBuffer (fbmFrameHeader);

				/*
				*	Send an NIF Request.
				*	Set initial trans ID if 0.
				*/
				fbmMACData[i].NNTransactionID =
					BuildNIFRequest (fbmFrameBuffer,
						SMT_NSA_FC,
						BROADCAST_ADDRESS, i,
						fbmMACData[i].NNTransactionID);

				/* send frame */
				len = ntohs
				    (fbmFrameHeader->smtHdr.InfoField_Length)
					    + SMT_FRAME_HDR_SIZE;
				SendSMTFrame (fbmFrameBuffer, len, i);

				/*
				*	Reset TNN.
				*/
				fbmMACData[i].TNN = 0;
			}
		}

		/*
		*	If TVU has timer out,
		*	clear current UNA, reset TVU.
		*/
		if (fbmMACData[i].TVU > T_NN_Out)
		{
			MCompareAddress (fbmMACData[i].UNA, NULL_ADDRESS,
				compare);
			if (compare != 0)
			{
				MCopyAddress (fbmMACData[i].UNA, NULL_ADDRESS);
				SendFBMEvent (fddiMACUpstreamNbr, i);
				fbmMACData[i].TVU = 0;
			}
		}

		/*
		*	If TVD has timer out,
		*	clear current DNA, reset TVD.
		*/
		if (fbmMACData[i].TVD > T_NN_Out)
		{
			MCompareAddress (fbmMACData[i].DNA, NULL_ADDRESS,
				compare);
			if (compare != 0)
			{
				MCopyAddress (fbmMACData[i].DNA, NULL_ADDRESS);
				SendFBMEvent (fddiMACDownstreamNbr, i);
				fbmMACData[i].TVD = 0;
			}
		}
	}

	return;
}

uInt32
InitFBMNN ()
/*********************************************************************
Function:	This function initializes the Frame Based Management
		Neighbor Notification protocol.
Parameters:	None.
Input:		None.
Output:		None.
Return:		0 if successful, otherwise an error code is returned.
Modification History:
*********************************************************************/
{
Int16	i;

	for (i = 0; i < MAX_MAC_COUNT; i++)
	{
		/*
		*	Clear transaction ID.
		*/
		fbmMACData[i].NNTransactionID = (uInt32) 0;

		/*
		*	Init timers.
		*/
		fbmMACData[i].TNN = (uInt32) T_Notify_Max;
		fbmMACData[i].TVU = (uInt32) 0;
		fbmMACData[i].TVD = (uInt32) 0;
	}

	return (0);
}

