#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) mibchng.c V6.2.3:cs.910829:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/**********************************************************************
	Management Information Base Module

	MIB maintenance module.

	File:		mibchng.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.910829	09/04/91

	Copyright 1990,1991 XLNT Designs, Inc.

	This module supplies the change operation for each MIB object
	group: SMT, MAC, PATH, PORT, and ATTACHMENT. For any writable
	attribute, the new value must take effect immediately. This
	requires a relatively close coupling between the change routines
	provided here and underlying interface routines to perform the
	change in the station. The following algorithm is used to change
	MIB attributes:

	For attributes related to SMT operation (e.g. policies, etc.),
	the XLNT Manager handles the change appropriately. For attributes
	affecting station operation (like, T_Req), the change request
	is passed to the interface routine. If the interface routine
	is successful at implementing the change, then the operation
	routine changes the value in the MIB.
	
	Modification History:

	*** Updated to SMT 6.2 ***
	910122-003	LJP
		In fddiPORTActions, need to set typeInfo of notification
		message to p->PORTPARAM16 (action ID).
	910122-004	LJP
		In fddiMACSMTAddress, use p->MACADDR as the source instead
		of p->MACPARAM8.
	910122-005	LJP
		Remove unused XDI parameters: xdiSMTStationType,
		xdiSMTPathLatency*, xdiSMTOutIndex*
	910128-001	LJP
		Corrected logical parentheses placement.
	910204-002	LJP
		fddiMACPathsAvailable & fddiMACPathsRequested were being
		compared and assigned with 16-bit values. These have been
		changed to use 8-bit values as stated in the standard.
	910204-003	LJP
		General corrections to writeable attributes:
		fddiSMTConfigPolicy: correct verification logic
		fddiMACPathsRequested: correct bounds check
		fddiMACT_Min: add bounds check
		fddiMACFrameStatus: add validation; use 16-bits not 32-bits
		fddiPORTConnectionPolicies: add bounds check
		fddiPORTPathsRequested: add validation; return error code
		fddiPORTMACLoop_Time: add bounds check
		fddiPORTTB_Max: add bounds check
		fddiPORTLer_Cutoff: return error code
		fddiPORTLer_Alarm: return error code
		fddiATTACHMENTInsertPolicy: add bounds check
	910301-001	LJP
		Put in range checking in ChangeMIBPORTAttr() and
		ChangeMIBATTACHMENTAttr().
	910315-001	LJP
		Removed "&" from in front of variables that are actually
		arrays. Some compilers complain about this.
	910429-002	LJP
		In the MAC, PORT, PATH, and ATTACHMENT change routines,
		the notification message to other Manager tasks did not
		have its entity member set. This is now done at the start
		of each routine when the notification message is
		initialized.
		Also removed extra assignment that reset the parameter
		index. This was a hold-over from unused code structure.
	910529-002	LJP
		In ChangeMIBMACAttr(), the variable temp was declared
		as a uChar. It is now a uInt16 to match the size of
		fddiMACFrameStatus.
	910823-005	LJP
		Added default case to SMT, MAC, and PORT actions for
		values out of range.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"fddihdr.h"
#include	"smtmacro.h"
#include	"smtmsg.h"
#include	"mibdefs.h"
#include	"mibtypes.h"
#include	"mibglbl.h"


/*********************************************************************
	Global Data
*********************************************************************/


/*********************************************************************
	External function definitions.
*********************************************************************/

extern	void	SendMIBNotification();
extern	void	SendMIBMessage ();
extern	uInt32	SetMACData ();
extern	uInt32	SetPATHData ();


/*********************************************************************
	MIB Functions.
*********************************************************************/

uInt32
ChangeMIBSMTAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to change an fddiSMT attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					new attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the new attribute value using 
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_OUT_OF_RANGE
			value given is out of bounds
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
*********************************************************************/
{
TLVParamType    *p;			/* attribute pointer */
SMTMessage	notifyMsg;		/* message to notify other tasks */
uInt32		result = RC_SUCCESS;	/* return code */

        /*
        *       Get format for buffer.
        */
        p = (TLVParamType *) bufData;

	/*
	*	Set notification message in case other tasks need
	*	to be notified of the change.
	*/
	notifyMsg.source = MIB_MSG_ID;
	notifyMsg.type = MIB_EVENT_NOTIFY_CHANGE;
	notifyMsg.typeInfo = p->paramType;
	notifyMsg.entity = 0;		/* 910429-001 LJP */
	notifyMsg.localID = 0;
	notifyMsg.len1 = 0;
	notifyMsg.len2 = 0;

        /*
        *       Change value in MIB.
        */
        switch (p->paramType)
        {
	/*****
	*	Filter READ-ONLY attributes
	*****/
	case fddiSMTStationIdGrp:
	case fddiSMTStationId:
	case fddiSMTHiVersionId:
	case fddiSMTLoVersionId:
	case fddiSMTManufacturerData:

	case fddiSMTStationConfigGrp:
	case fddiSMTMAC_Ct:
	case fddiSMTNonMaster_Ct:
	case fddiSMTMaster_Ct:
	case fddiSMTPathsAvailable:
	case fddiSMTConfigCapabilities:
	case fddiSMTStatusReporting:

	case fddiSMTStatusGrp:
	case fddiSMTECMState:
	case fddiSMTCF_State:
	case fddiSMTHoldState:
	case fddiSMTRemoteDisconnectFlag:

	case fddiSMTMIBOperationGrp:
	case fddiSMTMsgTimeStamp:
	case fddiSMTTransitionTimeStamp:
	case fddiSMTSetCount:
	case fddiSMTLastSetStationId:

	case xdiSMTBothWrapCapability:
	case xdiSMTTopology:
	case hlsSMTRemoteDisconnectStationID:
	/* 910122-005 LJP Removed unsed XDI attributes */

		result = RC_READ_ONLY;
		break;

	/*****
	*	READ-WRITE attributes
	*****/
	/* 
	*	fddiSMTStationIdGrp 
	*/
	case fddiSMTOpVersionId:
		if ((p->SMTPARAM16 < mib->SMTStationIdGrp.LoVersionId)
		  || (p->SMTPARAM16 > mib->SMTStationIdGrp.HiVersionId))
			result = RC_OUT_OF_RANGE;
		else
		{
			/* set attribute */
			mib->SMTStationIdGrp.OpVersionId = p->SMTPARAM16;

			/* send notification message */
			notifyMsg.destination = FBM_MSG_ID;
			notifyMsg.data.b16 =
				mib->SMTStationIdGrp.OpVersionId;
			SendMIBMessage (&notifyMsg);
		}
		break;

	case fddiSMTUserData:
		if (p->paramLen != sizeof (SMTUserDataType))
			result = RC_LENGTH_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (mib->SMTStationIdGrp.UserData, 
				p->SMTOTHER, p->paramLen);
		break;

	/* 
	*	fddiSMTStationConfigGrp
	*/
	case fddiSMTConfigPolicy:
		/* 910204-003 LJP */
		if ((mib->SMTStationConfigGrp.ConfigCapabilities
				& p->SMTPARAM16) != p->SMTPARAM16)
			result = RC_OUT_OF_RANGE;
		else
		{
			/* set attribute */
			mib->SMTStationConfigGrp.ConfigPolicy
				= p->SMTPARAM16;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b16 =
				mib->SMTStationConfigGrp.ConfigPolicy;
			SendMIBMessage (&notifyMsg);
		}
		break;

	case fddiSMTConnectionPolicy:
		/*
		*	Connection policy must always reject M-M.
		*/
		if (p->SMTPARAM16 & Policy_rejectM_M)
		{
			/* set attribute */
			mib->SMTStationConfigGrp.ConnectionPolicy
				= p->SMTPARAM16;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b16 =
				mib->SMTStationConfigGrp.ConnectionPolicy;
			SendMIBMessage (&notifyMsg);
		}
		else
			result = RC_OUT_OF_RANGE;
		break;

	case fddiSMTReportLimit:
		if (p->SMTPARAM32 == 0)
			result = RC_OUT_OF_RANGE;
		else
		{
			/* set attribute */
			mib->SMTStationConfigGrp.ReportLimit
				= p->SMTPARAM32;

			/* send notification message */
			notifyMsg.destination = FBM_MSG_ID;
			notifyMsg.data.b16 =
				mib->SMTStationConfigGrp.ReportLimit;
			SendMIBMessage (&notifyMsg);
		}
		break;

	case fddiSMTT_Notify:
		if ((p->SMTPARAM16 < T_Notify_Min)
				|| (p->SMTPARAM16 > T_Notify_Max))
			result = RC_OUT_OF_RANGE;
		else
		{
			/* set attribute */
			mib->SMTStationConfigGrp.T_Notify
				= p->SMTPARAM16;

			/* send notification message */
			notifyMsg.destination = FBM_MSG_ID;
			notifyMsg.data.b16 =
				mib->SMTStationConfigGrp.T_Notify;
			SendMIBMessage (&notifyMsg);
		}
		break;

	/*
	*	XDI added attributes
	*/
	case xdiSMTBothWrapPolicy:
		if ((mib->XDISMTGrp.BothWrapCapability & p->SMTPARAM8)
				!= p->SMTPARAM8)
			result = RC_OUT_OF_RANGE;
		else
		{
			/* set attribute */
			mib->XDISMTGrp.BothWrapPolicy= p->SMTPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b8 =
				mib->XDISMTGrp.BothWrapPolicy;
			SendMIBMessage (&notifyMsg);
		}
		break;

	/*
	*	fddiSMTStationAction
	*/
	case fddiSMTStationAction:
		switch (p->SMTPARAM16)
		{
		case StationAction_Connect:
		case StationAction_Disconnect:
		case StationAction_Path_Test:
		case StationAction_Self_Test:
			/* Pass message to CSP */
		        MIBDPT("fddiSMTStationAction.\n");
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.source = MIB_MSG_ID;
			notifyMsg.type = CSP_ACTION_FDDI_SMT;
			notifyMsg.typeInfo = p->SMTPARAM16;
			notifyMsg.localID = 0;
			notifyMsg.len1 = 0;
			notifyMsg.len2 = 0;
			SendMIBMessage (&notifyMsg);
			break;
		
		/*
		*	910823-005	LJP
		*	Add default for out of range.
		*/
		default:
			result = RC_OUT_OF_RANGE;
			break;
		}
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

uInt32
ChangeMIBMACAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to change an fddiMAC attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					new attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the new attribute value using 
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_OUT_OF_RANGE
			value given is out of bounds
		RC_NOT_AUTHORIZED
			station does not permit changing this attribute
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
*********************************************************************/
{
TLVParamType    *p;
SMTMessage	notifyMsg;
uInt16		entity;				/* MAC index */
uInt32		result = RC_SUCCESS;
uInt16		temp;				/* 910204-003 for FrameStatus */
						/* 910529-002 LJP */

        /*
        *       Get format for buffer.
        */
        p = (TLVParamType *) bufData;

        /*
        *       Get index
        */
        entity = p->MACINDEX;

        /*
        *       Adjust for MIB array element.
        */
        if (entity < 1 || entity > mib->SMTStationConfigGrp.MAC_Ct)
                return (RC_ILLEGAL_PARAMETER);
        entity--;

	/*
	*	Set up notification message.
	*/
	notifyMsg.source = MIB_MSG_ID;
	notifyMsg.type = MIB_EVENT_NOTIFY_CHANGE;
	notifyMsg.typeInfo = p->paramType;
	notifyMsg.entity = entity;		/* 910429-001	LJP */
	notifyMsg.localID = 0;
	notifyMsg.len1 = 0;
	notifyMsg.len2 = 0;

        /*
        *       Change value in MIB.
        */
        switch (p->paramType)
        {
	/*****
	*	Filter READ-ONLY attributes
	*****/
	case fddiMACCapabilitiesGrp:
	case fddiMACT_MaxGreatestLowerBound:
		/*
		*	The r/w status of this attribute is under question
		*	in the SMT committee.
		*/
	case fddiMACTVXGreatestLowerBound:
	case fddiMACFrameStatusCapabilities:
	case fddiMACBridgeFunction:

	case fddiMACConfigGrp:
	case fddiMACPathsAvailable:
	case fddiMACCurrentPath:
	case fddiMACUpstreamNbr:
	case fddiMACDownstreamNbr:
	case fddiMACOldUpstreamNbr:
	case fddiMACOldDownstreamNbr:
	case fddiMACRootConcentratorMAC:
	case fddiMACDup_Addr_Test:
	case fddiMACDownstreamPORTType:

	case fddiMACAddressGrp:

	case fddiMACOperationGrp:
	case fddiMACT_Max:
	case fddiMACTvxValue:
	case fddiMACT_Neg:
	case fddiMACT_Pri0:
	case fddiMACT_Pri1:
	case fddiMACT_Pri2:
	case fddiMACT_Pri3:
	case fddiMACT_Pri4:
	case fddiMACT_Pri5:
	case fddiMACT_Pri6:

	case fddiMACCountersGrp:
	case fddiMACFrame_Ct:
	case fddiMACCopied_Ct:
	case fddiMACTransmit_Ct:
	case fddiMACToken_Ct:

	case fddiMACErrorCtrsGrp:
	case fddiMACError_Ct:
	case fddiMACLost_Ct:
	case fddiMACTvxExpired_Ct:
	case fddiMACNotCopied_Ct:
	case fddiMACLate_Ct:
	case fddiMACRingOp_Ct:

	case fddiMACFrameErrorConditionGrp:
	case fddiMACBaseFrame_Ct:
	case fddiMACBaseError_Ct:
	case fddiMACBaseLost_Ct:
	case fddiMACBaseTimeFrameError:
	case fddiMACFrameErrorRatio:

	case fddiMACNotCopiedConditionGrp:
	case fddiMACBaseNotCopied_Ct:
	case fddiMACBaseTimeNotCopied:
	case fddiMACBaseCopied_Ct:
	case fddiMACNotCopiedRatio:

	case fddiMACStatusGrp:
	case fddiMACRMTState:
	case fddiMACDa_Flag:
	case fddiMACUnaDa_Flag:
	case fddiMACFrameErrorCondition:
	case fddiMACNotCopiedCondition:
	case fddiMACLLCServiceAvailable:

	case fddiMACRootMACStatusGrp:
	case fddiMACMasterSlaveLoopStatus:
	case fddiMACRootMACDownstreamPORTType:
	case fddiMACRootMACCurrentPath:

	/* XDI added attributes */
	case xdiMACResourceIndex:
	case xdiMACConnectedIndex:
		result = RC_READ_ONLY;
		break;

	/*****
	*	READ-WRITE attributes
	*****/
	/*
	*	fddiMACConfigGrp
	*/
	case fddiMACPathsRequested:
		/* 910204-002 LJP */
		/* 910204-003 LJP */
		if (p->MACPARAM8 && ((mib->MACConfigGrp[entity].PathsAvailable 
			& p->MACPARAM8) == p->MACPARAM8))
		{
			/* set attribute */
			mib->MACConfigGrp[entity].PathsRequested
				= p->MACPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b8 =
				mib->MACConfigGrp[entity].PathsRequested;
			SendMIBMessage (&notifyMsg);
		}
		else
			result = RC_OUT_OF_RANGE;
		break;

	/* 
	*	fddiMACAddressGrp
	*/
	/*
	*	Normally, the MAC's address is not writeable according to
	*	the standard. For this version of the XLNT Manager, the
	*	address is writeable.
	*/
	case fddiMACSMTAddress:
		/*
		*	910122-004	LJP
		*	Use p->MACADDR as the source instead of
		*	p->MACPARAM8.
		*/
		/* 910315-001 LJP */
		if ((result = SetMACData (p->paramType, entity,
				p->MACADDR)) == RC_SUCCESS)
		{
			/* set MIB value if OK */
			/* 910315-001 LJP */
			MCopyAddress
			    (mib->MACAddressGrp[entity].SMTAddress,
				p->MACADDR);

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			MCopyAddress (notifyMsg.data.addr,
			    mib->MACAddressGrp[entity].SMTAddress);
			SendMIBMessage (&notifyMsg);
		}
		break;

	/*
	*	fddiMACOperationGrp
	*/
	case fddiMACT_Req:
		/*
		*	NOTE: The T_Req value should be controlled in
		*	a manner similar to T_Max and TVX where there
		*	is a greatest lower bound value. Since there
		*	is no T_Req greatest lower bound attribute,
		*	setting T_Req works like setting the greatest
		*	lower bound so that the SetMACData() routine
		*	changes the T_Req to an appropriate value for
		*	the MAC and the GetMACData() routine returns
		*	the current setting of T_Req. This requires
		*	the SetMACData() routine to return the value
		*	used to set T_Req.
		*
		*	T_Req must be between T_Max and T_Min as an
		*	initial check. Note: these are 2s complement
		*	values.
		*/
		if ((mib->MACOperationGrp[entity].T_Max <= p->MACPARAM32) &&
			(mib->MACOperationGrp[entity].T_Min >= p->MACPARAM32))
		{
			/* 910128-001 LJP */
			if ((result = SetMACData (p->paramType, entity,
					&p->MACPARAM32)) == RC_SUCCESS)
			{
				/* set T_Req */
				mib->MACOperationGrp[entity].T_Req
					= p->MACPARAM32;
			}
		}
		else
			result = RC_OUT_OF_RANGE;
		break;

	case fddiMACT_Min:
		/* 
		*	910204-003 LJP 
		*	Test if T_Min < T_Max. Since this is an
		*	FddiTimerTwosComplement, the test is reversed.
		*/
		if (mib->MACOperationGrp[entity].T_Max < p->MACPARAM32)
		{
			/* 910128-001 LJP */
			if ((result = SetMACData (p->paramType, entity,
					&p->MACPARAM32)) == RC_SUCCESS)
				mib->MACOperationGrp[entity].T_Min = 
					p->MACPARAM32;
		}
		else
			result = RC_OUT_OF_RANGE;
		break;

	case fddiMACFrameStatus:
		/*
		*	910204-003	LJP
		*	Add validation against capabilities.
		*/
		/* get programmable capabilitiles */

#ifdef REM
		printf("In smt change mib\n");
#endif
		temp = mib->MACCapabilitiesGrp[entity].FrameStatusCapabilities;
		temp >>= 8;		/* shift to low byte */
		temp &= 0x07;		/* clear unused bits */
		/* test if change is valid and programmable */
		if ((temp 
		  & mib->MACCapabilitiesGrp[entity].FrameStatusCapabilities) 
			& (temp & p->MACPARAM16))
		{
#ifdef REM
			printf("Inside if test statment with temp changed\n");
#endif
			/* 910128-001 LJP */
			if ((result = SetMACData (p->paramType, entity,
					&p->MACPARAM16)) == RC_SUCCESS)
				mib->MACOperationGrp[entity].FrameStatus
					/* 910204-003 LJP */
					= p->MACPARAM16;
#ifdef REM
			printf("result in SMT = %d \n", result);
#endif
		}
		else
			result = RC_OUT_OF_RANGE;
		break;


	/*
	*	fddiMACFrameErrorConditionGrp
	*/
	case fddiMACFrameErrorThreshold:
		/*
		*	This value is read only if initialized to 0.
		*	Otherwise the lower bound is 1.
		*/
		if (mib->MACFrameErrorConditionGrp[entity].FrameErrorThreshold)
		{
			if (p->MACPARAM16 > 0)
				mib->MACFrameErrorConditionGrp[entity].FrameErrorThreshold
					= p->MACPARAM16;
			else
				result = RC_OUT_OF_RANGE;
		}
		else
			result = RC_READ_ONLY;
		break;


	/*
	*	fddiMACNotCopiedConditionGrp
	*/
	case fddiMACNotCopiedThreshold:
		/*
		*	This value is read only if initialized to 0.
		*	Otherwise the lower bound is 1.
		*/
		if (mib->MACNotCopiedConditionGrp[entity].NotCopiedThreshold)
		{
			if (p->MACPARAM16 > 0)
				mib->MACNotCopiedConditionGrp[entity].NotCopiedThreshold
					= p->MACPARAM16;
			else
				result = RC_OUT_OF_RANGE;
		}
		else
			result = RC_READ_ONLY;
		break;

	/*
	*	XDI added attributes
	*/
	case xdiMACRMTDupPolicy1:
		/*
		*	Check for valid policy.
		*/
		if ((p->MACPARAM8 < RMT_DUP_CHANGE)
				|| (p->MACPARAM8 > RMT_DUP_LEAVE))
			result = RC_OUT_OF_RANGE;
		else
		{
			/* set attribute */
			mib->XDIMACGrp[entity].RMTDupPolicy1 = p->MACPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b8
				= mib->XDIMACGrp[entity].RMTDupPolicy1;
			SendMIBMessage (&notifyMsg);
		}
		break;

	case xdiMACRMTDupPolicy2:
		/*
		*	Check for valid policy.
		*/
		if ((p->MACPARAM8 < RMT_DUP_CHANGE)
				|| (p->MACPARAM8 > RMT_DUP_LEAVE))
			result = RC_OUT_OF_RANGE;
		else
		{
			/* set attribute */
			mib->XDIMACGrp[entity].RMTDupPolicy2 = p->MACPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b8
				= mib->XDIMACGrp[entity].RMTDupPolicy2;
			SendMIBMessage (&notifyMsg);
		}
		break;

	case xdiMACDataAddress:
		/* set attribute */
		MCopyAddress (mib->XDIMACGrp[entity].DataAddress,
			&p->MACPARAM8);

		/* send notification message */
		notifyMsg.destination = CSP_MSG_ID;
		MCopyAddress (notifyMsg.data.addr,
			mib->XDIMACGrp[entity].DataAddress);
		SendMIBMessage (&notifyMsg);
		break;

	case xdiMACOperational:
		mib->XDIMACGrp[entity].Operational = p->MACPARAM8;

		/* send notification message */
		notifyMsg.destination = CSP_MSG_ID;
		notifyMsg.data.b8
			= mib->XDIMACGrp[entity].Operational;
		SendMIBMessage (&notifyMsg);
		break;

	/*
	*	fddiMACAction
	*/
	case fddiMACAction:     /* jlin : a group added by hls fddi_con */
		switch (p->MACPARAM16)
		{
		case MACAction_EnableLLCService:
		     ENABLE_LLC_SERVICE(entity);
		     break;
		case MACAction_DisableLLCService:
		     DISABLE_LLC_SERVICE(entity);
		     break;
		case MACAction_ConnectMAC:
		     ENABLE_MAC_SERVICE(entity);
		     break;
		case MACAction_DisconnectMAC:
		     DISABLE_MAC_SERVICE(entity);
		     break;

		/*
		*	910823-005	LJP
		*	Add default for out of range.
		*/
		default:
			result = RC_OUT_OF_RANGE;
			break;
		}
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

uInt32
ChangeMIBPATHAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to change an fddiPATH attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					new attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the new attribute value using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_OUT_OF_RANGE
			value given is out of bounds
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
*********************************************************************/
{
TLVParamType    *p;				/* attribute pointer */
SMTMessage	notifyMsg;			/* notification message */
uInt16		entity;                         /* PATH index */
uInt32		result = RC_SUCCESS;		/* return code */

        /*
        *       Get format for buffer.
        */
        p = (TLVParamType *) bufData;

        /*
        *       Get index
        */
        entity = p->PATHINDEX;

        /*
        *       Adjust for MIB array element.
        */
        if (entity < 1 || entity > MAX_PATH_COUNT)
                return (RC_ILLEGAL_PARAMETER);
        entity--;

	/*
	*	Set up notification message.
	*/
	notifyMsg.source = MIB_MSG_ID;
	notifyMsg.type = MIB_EVENT_NOTIFY_CHANGE;
	notifyMsg.typeInfo = p->paramType;
	notifyMsg.entity = entity;		/* 910429-001	LJP */
	notifyMsg.localID = 0;
	notifyMsg.len1 = 0;
	notifyMsg.len2 = 0;

	/*
	*	Change value in MIB.
	*/
	switch (p->paramType)
	{
	/*****
	*	Filter READ-ONLY attributes
	*****/
	case fddiPATHConfigGrp:

	case fddiPATHClassPATHConfigGrp:
	case fddiPATHClassPATHClassType:
	case fddiPATHClassPATHPORTOrder:
	case fddiPATHClassPATHTraceStatus:
	case fddiPATHClassPATHStatus:
	case fddiPATHClassPATHConfiguration:

		result = RC_READ_ONLY;
		break;

	/*****
	*	READ-WRITE attributes
	******/
	/*
	*	fddiPATHConfigGrp
	*/
	case fddiPATHClassTrace_MaxExpiration:
		/* convert from FddiTimerTwosComplement */
		if (MFddi2Usecs (-p->PATHPARAM32) < Min_Trace_Max)
			result = RC_OUT_OF_RANGE;
		else
		{
			if ((result = SetPATHData (p->paramType, entity,
					&p->PATHPARAM32)) == RC_SUCCESS)
			{
				/* set attribute */
				mib->ClassConfigGrp.Trace_MaxExpiration
					= p->PATHPARAM32;

				/* send notification message */
				notifyMsg.destination = CSP_MSG_ID;
				notifyMsg.data.b32 =
					mib->ClassConfigGrp.Trace_MaxExpiration;
				SendMIBMessage (&notifyMsg);
			}
		}
		break;

	case fddiPATHClassTVXLowerBound:
		if ((result = SetPATHData (p->paramType, entity,
				&p->PATHPARAM32)) == RC_SUCCESS)
			mib->ClassConfigGrp.TVXLowerBound = p->PATHPARAM32;
		break;

	case fddiPATHClassT_MaxLowerBound:
		if ((result = SetPATHData (p->paramType, entity,
				&p->PATHPARAM32)) == RC_SUCCESS)
			mib->ClassConfigGrp.T_MaxLowerBound = p->PATHPARAM32;

		/* send notification message */
		notifyMsg.destination = CSP_MSG_ID;
		notifyMsg.data.b32 = mib->ClassConfigGrp.T_MaxLowerBound;
		
		SendMIBMessage (&notifyMsg);
		
		break;


	/*
	*	fddiPATHClassPATHConfigGrp
	*/
	case fddiPATHClassPATHRingLatency:
		if ((result = SetPATHData (p->paramType, entity,
				&p->PATHPARAM32)) == RC_SUCCESS)
			mib->ClassPATHConfigGrp[entity].RingLatency 
				= p->PATHPARAM32;
		break;

	case fddiPATHClassPATHSba:
		if ((result = SetPATHData (p->paramType, entity,
				&p->PATHPARAM32)) == RC_SUCCESS)
			mib->ClassPATHConfigGrp[entity].Sba = p->PATHPARAM32;
		break;

	case fddiPATHClassPATHSbaOverhead:
		if ((result = SetPATHData (p->paramType, entity,
				&p->PATHPARAM32)) == RC_SUCCESS)
			mib->ClassPATHConfigGrp[entity].SbaOverhead 
				= p->PATHPARAM32;
		break;

	case fddiPATHClassPATHT_Rmode:
		mib->ClassPATHConfigGrp[entity].T_Rmode 
			= p->PATHPARAM32;

		/* send notification message */
		notifyMsg.destination = CSP_MSG_ID;
		notifyMsg.data.b32 =
			mib->ClassPATHConfigGrp[entity].T_Rmode;
		SendMIBMessage (&notifyMsg);

		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

uInt32
ChangeMIBPORTAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to change an fddiPORT attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					new attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the new attribute value using 
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_OUT_OF_RANGE
			value given is out of bounds
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
*********************************************************************/
{
TLVParamType    *p;				/* attribute pointer */
SMTMessage	notifyMsg;			/* notification message */
uInt16		entity;                         /* PORT index */
uInt32		result = RC_SUCCESS;		/* return code */

        /*
        *       Get format for buffer.
        */
        p = (TLVParamType *) bufData;

        /*
        *       Get index
        */
        entity = p->PORTINDEX;

	/*
	*	910301-001	LJP
	*	Put in range checking for entity value.
	*/
        if (entity < 1 || entity > (mib->SMTStationConfigGrp.NonMaster_Ct 
			+ mib->SMTStationConfigGrp.Master_Ct))
                return (RC_ILLEGAL_PARAMETER);
        entity--;

	/*
	*	Set up notification message.
	*/
	notifyMsg.source = MIB_MSG_ID;
	notifyMsg.type = MIB_EVENT_NOTIFY_CHANGE;
	notifyMsg.typeInfo = p->paramType;
	notifyMsg.entity = entity;		/* 910429-001	LJP */
	notifyMsg.localID = 0;
	notifyMsg.len1 = 0;
	notifyMsg.len2 = 0;

        /*
        *       Change value in MIB.
        */
        switch (p->paramType)
        {
	/*****
	*	Filter READ-ONLY attributes
	*****/
	case fddiPORTConfigGrp:
	case fddiPORTPC_Type:
	case fddiPORTPC_Neighbor:
	case fddiPORTRemoteMACIndicated:
	case fddiPORTCE_State:
	case fddiPORTMACPlacement:
	case fddiPORTAvailablePaths:
	case fddiPORTFotxClass:

	case fddiPORTOperationGrp:
	case fddiPORTBS_Flag:

	case fddiPORTErrorCtrsGrp:
	case fddiPORTEBError_Ct:
	case fddiPORTLCTFail_Ct:

	case fddiPORTLerGrp:
	case fddiPORTLer_Estimate:
	case fddiPORTLem_Reject_Ct:
	case fddiPORTLem_Ct:
	case fddiPORTBaseLer_Estimate:
	case fddiPORTBaseLem_Reject_Ct:
	case fddiPORTBaseLem_Ct:
	case fddiPORTBaseLer_TimeStamp:

	case fddiPORTStatusGrp:
	case fddiPORTConnectState:
	case fddiPORTPCMState:
	case fddiPORTPC_Withhold:
	case fddiPORTLerCondition:

	/* XDI added attributes */
	case xdiPORTResourceIndex:
	case xdiPORTConnectedIndex:

		result = RC_READ_ONLY;
		break;

	/*****
	*	READ-WRITE attributes
	*****/
	/* fddiPORTConfigGrp */
	case fddiPORTConnectionPolicies:
		if (p->PORTPARAM8 < 8)
		{
		   mib->PORTConfigGrp[entity].ConnectionPolicies
		        = p->PORTPARAM8;
		   /* send notification message */
		   notifyMsg.destination = CSP_MSG_ID;
		   notifyMsg.data.b8 =
		        mib->PORTConfigGrp[entity].ConnectionPolicies;
		   SendMIBMessage (&notifyMsg);
		}
		else
		   result = RC_OUT_OF_RANGE;
		break;

	case fddiPORTPathsRequested:
		/* 910204-003 LJP */
		if (p->PORTPARAM8 &&
			((mib->PORTConfigGrp[entity].AvailablePaths 
			& p->PORTPARAM8) == p->PORTPARAM8))
		{
			/* set attribute */
			mib->PORTConfigGrp[entity].PathsRequested
				= p->PORTPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b8 =
				mib->PORTConfigGrp[entity].PathsRequested;
			SendMIBMessage (&notifyMsg);
		}
		/* 910204-003 LJP */
		else
			result = RC_OUT_OF_RANGE;
		break;

	case fddiPORTMACLoop_Time:
		/* 
		*	910204-003 LJP
		*	Test against min time in standard.
		*/
		if (p->PORTPARAM32 >= 0x002625A0)
		{
			mib->PORTConfigGrp[entity].MACLoop_Time
				= p->PORTPARAM32;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b32 =
				mib->PORTConfigGrp[entity].MACLoop_Time;
			SendMIBMessage (&notifyMsg);
		}
		else
			result = RC_OUT_OF_RANGE;
		break;

	/* fddiPORTOperationGrp */
	case fddiPORTMaintLineState:
		if (p->PORTPARAM8 >= Maint_QLS && p->PORTPARAM8 <= Maint_PDR)
		{
			mib->PORTOperationGrp[entity].MaintLineState
				= p->PORTPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b8 =
				mib->PORTOperationGrp[entity].MaintLineState;;
			SendMIBMessage (&notifyMsg);
		}
		break;

	case fddiPORTTB_Max:
		/*
		*	910204-003	LJP
		*	MIB does not specify a minimum value but CMT does.
		*	Use the default min value here.
		*	NOTE: TB_Max is in FddiTimerTwosComplement.
		*/
		if ((MFddi2Usecs(-p->PORTPARAM32)) >= 30000)
		{
			mib->PORTOperationGrp[entity].TB_Max
				= p->PORTPARAM32;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b32 =
				mib->PORTOperationGrp[entity].TB_Max;
			SendMIBMessage (&notifyMsg);
		}
		else
			result = RC_OUT_OF_RANGE;
		break;

	/* fddiPORTLerGrp */
	case fddiPORTLer_Cutoff:
		if (p->PORTPARAM8 >= LER_MIN && p->PORTPARAM8 <= LER_MAX)
		{
			mib->PORTLerGrp[entity].Ler_Cutoff
				= p->PORTPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b8 =
				mib->PORTLerGrp[entity].Ler_Cutoff;
			SendMIBMessage (&notifyMsg);
		}
		/* 910204-003 LJP */
		else
			result = RC_OUT_OF_RANGE;
		break;

	case fddiPORTLer_Alarm:
		if (p->PORTPARAM8 >= LER_MIN && p->PORTPARAM8 <= LER_MAX)
		{
			mib->PORTLerGrp[entity].Ler_Alarm
				= p->PORTPARAM8;
		}
		/* 910204-003 LJP */
		else
			result = RC_OUT_OF_RANGE;
		break;

	/* XDI added attributes */
	case xdiPORTOperational:
		mib->XDIPORTGrp[entity].Operational
			= p->PORTPARAM8;

		/* send notification message */
		notifyMsg.destination = CSP_MSG_ID;
		notifyMsg.data.b8
			= mib->XDIPORTGrp[entity].Operational;
		SendMIBMessage (&notifyMsg);
		break;

	/*
	*	fddiPORTAction
	*/
	case fddiPORTAction:
		switch (p->PORTPARAM16)
		{
		case PORTAction_Maint:
			/* Put maintenance line state in data */
			notifyMsg.data.b8
			  = mib->PORTOperationGrp[entity].MaintLineState;
			/* rest of message same as other actions */
#if 0
		case PORTAction_Enable:
		case PORTAction_Disable:
#else           /* 
		   06/07/93
		   jlin: Send SNMP trap whenever the port is disabled 
		*/
		case PORTAction_Disable:
			if (p->PORTPARAM16 == PORTAction_Disable)
			{
			   send_portdisable_trap(entity+1, 2);
			}
		case PORTAction_Enable:
#endif
		case PORTAction_Start:
		case PORTAction_Stop:
			/* Pass message to CSP */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.source = MIB_MSG_ID;
			notifyMsg.type = CSP_ACTION_FDDI_PORT;
			/*
			*	910122-003	LJP
			*	Include action identifier in message.
			*/
			notifyMsg.typeInfo = p->PORTPARAM16;
			/* adjust index for CSP */
			notifyMsg.entity = p->PORTINDEX - 1;
			SendMIBMessage (&notifyMsg);
			break;

		/*
		*	910823-005	LJP
		*	Add default for out of range.
		*/
		default:
			result = RC_OUT_OF_RANGE;
			break;
		}
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

uInt32
ChangeMIBATTACHMENTAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to change an fddiATTACHMENT attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					new attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the new attribute value using 
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_OUT_OF_RANGE
			value given is out of bounds
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
*********************************************************************/
{
TLVParamType    *p;				/* attribute pointer */
SMTMessage	notifyMsg;			/* notification message */
uInt16		entity;                         /* ATTACHMENT index */
uInt32		result = RC_SUCCESS;		/* return code */

        /*
        *       Get format for buffer.
        */
        p = (TLVParamType *) bufData;

        /*
        *       Get index
        */
        entity = p->ATTACHINDEX;

	/*
	*	910301-001	LJP
	*	Put in range checking for entity value.
	*/
        if (entity < 1 || entity > (mib->SMTStationConfigGrp.NonMaster_Ct 
			+ mib->SMTStationConfigGrp.Master_Ct))
                return (RC_ILLEGAL_PARAMETER);
        entity--;

	/*
	*	Set up notification message.
	*/
	notifyMsg.source = MIB_MSG_ID;
	notifyMsg.type = MIB_EVENT_NOTIFY_CHANGE;
	notifyMsg.typeInfo = p->paramType;
	notifyMsg.entity = entity;		/* 910429-001	LJP */
	notifyMsg.localID = 0;
	notifyMsg.len1 = 0;
	notifyMsg.len2 = 0;

        /*
        *       Get value from MIB.
        */
        switch (p->paramType)
        {
	/*****
	*	Filter READ-ONLY attributes.
	*****/
	case fddiATTACHMENTConfigGrp:
	case fddiATTACHMENTClass:
	case fddiATTACHMENTOpticalBypassPresent:
	case fddiATTACHMENTI_MaxExpiration:
	case fddiATTACHMENTInsertedStatus:
		result = RC_READ_ONLY;
		break;

	/*****
	*	READ-WRITE attributes.
	*****/
	case fddiATTACHMENTInsertPolicy:
		/* 910204-003 LJP */
		if (p->ATTACHPARAM8 == SET || p->ATTACHPARAM8 == CLEAR)
		{
			mib->ATTACHMENTConfigGrp[entity].InsertPolicy
				= p->ATTACHPARAM8;

			/* send notification message */
			notifyMsg.destination = CSP_MSG_ID;
			notifyMsg.data.b16 =
				mib->PORTLerGrp[entity].Ler_Alarm;
			SendMIBMessage (&notifyMsg);
		}
		else
			result = RC_OUT_OF_RANGE;
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

