#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) mibmsg.c V6.2.3:cs.911022:7:7 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/**********************************************************************
	Management Information Base Module

	Message processing module.

	File:		mibmsg.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911022	10/28/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	This module provides for message services for the Management 
	Information Base (MIB).

	Modification History:

	*** Updated to SMT 6.2 ***

	910326-001	LJP
		Pass changes to Dup_Addr_Test from FSP to CSP. Note that
		this notification is used instead of MACDa_Flag.
	910412-001	LJP
		Corrected detection and reporting of 
		fddiPORTEBErrorCondition and fddiPORTLerErrorCondition.
	910605-004	LJP
		Clear RemoteMACIndicated when PCM enters OFF or BREAK.
	910613-001	LJP
		Changed constant 0xFFFF to 0x0FFFFL for portability.
	910807-002	LJP
		Notify CSP of change to UNA.
	910807-004	LJP
		Change definition of temp variable in ProcessMIBEvent()
		for portability.
	911022-002	LJP
		Changed element [0][3] in undesirableTable from 1 to 0
		since A-M connections are controlled through MIB
		attribute, not general SMT selection.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"mibdefs.h"
#include	"mibtypes.h"
#include	"mibglbl.h"
#include	"smtmsg.h"
#include        "msgutil.h"


/*********************************************************************
	External function definitions.
*********************************************************************/

extern	void	SendMIBMessage ();
extern	uInt32	GetMIBAttr ();
extern	uInt32	ChangeMIBAttr ();
extern	uInt32	AddMIBAttr ();
extern	uInt32	RemoveMIBAttr ();
extern	uInt32	GetMIBMACAttr ();
extern	uInt32	GetMIBPORTAttr ();
extern	uInt32	GetMACData ();
extern	uInt32	GetPORTData ();
extern	void	LocalMIBMessages ();
extern	Flag	UndesirableConnection ();
extern	void	GetTimeStamp ();

/*********************************************************************
	Global Variables
*********************************************************************/

/*
*	These event values are here to avoid a large stack
*	in SendSMTEvent().
*/

static EvtCfgChgType	evtCfgChg;	/* configuration change structure */
static CondDAType	condDA;		/* duplicate address structure */
static CondFrErrType	condFrErr;	/* frame error condition struct */
static CondNotCopiedType condNotCopied;	/* not copied condition struct */
static EvtNbrChgType	evtNbrChg;	/* MAC neighbor change structure */
static EvtTrStatType	evtTrStat;	/* trace status structure */
static CondLerType	condLer;	/* LER condition structure */
static EvtConnectType	evtConnect;	/* undesirable connection attempt */
static CondEBErrType	condEBError;	/* EB Error condition struct */

/*
*	Undesirable connection table. Any position set to 1 indicates
*	an undesirable connection as defined by the FDDI Connection Rules.
*/
/*	911022-002	LJP	Allow A-M here. */
static uChar	undesirableTable[4][4] = {
	/*	A	B	S	M */
/* A */	{	1,	0,	1,	0	},
/* B */	{	0,	1,	1,	0	},
/* S */	{	1,	1,	0,	0	},
/* M */	{	0,	0,	0,	1	}
};


/*********************************************************************
	Process MIB Messages Support Routines
*********************************************************************/

void
SetConnectState (portID)
	uInt16	portID;
/*********************************************************************
Function:	When the PCM state or PC_Withhold for a PORT changes,
		a new connect state must be determined. This routine
		selects the appropriate connect state.
Parameters:	portID	= MIB index of port to change.
Input:		mib	= uses current PCM and PC_Withhold for the PORT.
Output:		mib	= sets connect state for PORT.
Return:		No value returned.
*********************************************************************/
{
#if 0
MIBMsgType	*mibInfo;
#endif
uInt16		newConnect;

	/* determine connect state */
	switch (mib->PORTStatusGrp[portID].PCMState)
	{
	case PC_OFF:
	case PC_MAINT:
		newConnect = Connect_Disabled;
		break;

	case PC_TRACE:
	case PC_ACTIVE:
		newConnect = Connect_Active;
		break;

	default:
		newConnect =
			(mib->PORTStatusGrp[portID].PC_Withhold == PC_WH_None)
			? Connect_Connecting : Connect_Standby;
		break;
	}

	/* set connect state */
	mib->PORTStatusGrp[portID].ConnectState = newConnect;

	return;
}

/*********************************************************************
	MIB Event Message Generation
*********************************************************************/

void
SendSMTEvent (eventType, entity)
	uInt16	eventType;
	uInt16	entity;
/*********************************************************************
Function:	Send a condition or event notification to the appropriate
		Manager tasks and to the MAP.
Parameters:	eventType	= event identifier as defined in the
				SMT MIB.
		entity		= Manager index of entity reporting the
				event (0 if not needed).
Input:		mib		= uses current information in MIB to
				report the event or condition.
Output:		mib		= may update the MIB for condition status
				and time stamping of events.
Return:		No value returned.
*********************************************************************/
{
SMTMessage	msg;			/* message to send */
#if 0
uInt16		eventLen,		/* length of event attribute */
#else
uInt16
#endif
		pathLen;		/* length of path list */


	/*
	*	Enter default message data.
	*/
	msg.source = MIB_MSG_ID;
	msg.type = MIB_EVENT_NOTIFY_SMT;
	msg.typeInfo = eventType;
	msg.entity = entity;
	msg.localID = 0;
	msg.len1 = 0;
	msg.len2 = 0;

	/*
	*	Select event processing.
	*/
	switch (eventType)
	{
	case fddiSMTConfigurationChgEvent:
		/* determine size of path list */
#ifdef OPTIONAL_PARAMETER
		pathLen = sizeof (TLVHdrType) + sizeof (TLV16BitType)
			+ (sizeof (PathConfigType)
				* (mib->XDISMTGrp.Port_Ct
					+ mib->SMTStationConfigGrp.MAC_Ct));
#else
		pathLen = 0;
#endif

		/* determine total attribute length */
		msg.len1 = sizeof (TLVHdrType)
			+ sizeof (TLVHdrType) + sizeof (TLV8BitType)
			+ pathLen;


		/* set message pointer */
		msg.p1.ptr = (uChar *) &evtCfgChg;

		/* fill event header data */
		evtCfgChg.eventHdr.paramType = eventType;
		evtCfgChg.eventHdr.paramLen = 0;

		/* fill CF_State data */
		evtCfgChg.CF_StateHdr.paramType = fddiSMTCF_State;
		evtCfgChg.CF_StateHdr.paramLen = 4;
		evtCfgChg.CF_State.data = mib->SMTStatusGrp.CF_State;

#ifdef OPTIONAL_PARAMETER
		/* fill in path list data */
		evtCfgChg.PathListHdr.paramType
			= fddiPATHClassPATHConfiguration
		evtCfgChg.PathListHdr.paramLen = pathLen;
			/*****
				To be completed.
			*****/
#endif

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);

		break;

	case fddiMACDuplicateAddressCondition:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &condDA;

		/* fill event header data */
		condDA.condHdr.paramType = eventType;
		condDA.condHdr.paramLen = 0x0010;
		msg.len1 = condDA.condHdr.paramLen + sizeof (TLVHdrType);

		/* fill condition data */
		condDA.Condition = 0;
		if (mib->MACStatusGrp[entity].Da_Flag)
		{
			condDA.Condition |= DuplAddr_My_Duplicate;
			MCopyAddress (condDA.DuplicateAddr,
				mib->MACAddressGrp[entity].SMTAddress);
		}
		else
			MCopyAddress (condDA.DuplicateAddr, UNKNOWN_ADDRESS);

		if (mib->MACStatusGrp[entity].UnaDa_Flag)
		{
			condDA.Condition |= DuplAddr_My_UNA_Duplicate;
			MCopyAddress (condDA.UNADuplicateAddr,
				mib->MACConfigGrp[entity].UpstreamNbr);
		}
		else
			MCopyAddress (condDA.UNADuplicateAddr,
				UNKNOWN_ADDRESS);

		/* set MIB based index */
		condDA.MACIndex = entity + 1;

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);

		break;

	case fddiMACFrameErrorConditionEvent:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &condFrErr;

		/* fill event header data */
		condFrErr.condHdr.paramType = eventType;
		condFrErr.condHdr.paramLen = 0x0028;
		msg.len1 = condFrErr.condHdr.paramLen + sizeof (TLVHdrType);

		/* fill condition data */
		condFrErr.Condition_State
			= mib->MACStatusGrp[entity].FrameErrorCondition;
		condFrErr.MACIndex = entity + 1;
		GetMACData (fddiMACFrame_Ct, entity,
			&condFrErr.Frame_Ct);
		GetMACData (fddiMACError_Ct, entity,
			&condFrErr.Error_Ct);
		GetMACData (fddiMACLost_Ct, entity,
			&condFrErr.Lost_Ct);
		condFrErr.BaseFrame_Ct
			= mib->MACFrameErrorConditionGrp[entity].BaseFrame_Ct;
		condFrErr.BaseError_Ct
			= mib->MACFrameErrorConditionGrp[entity].BaseError_Ct;
		condFrErr.BaseLost_Ct
			= mib->MACFrameErrorConditionGrp[entity].BaseLost_Ct;
		condFrErr.BaseTimeStamp.hiword
			= mib->MACFrameErrorConditionGrp[entity].BaseTimeFrameError.hiword;
		condFrErr.BaseTimeStamp.loword
			= mib->MACFrameErrorConditionGrp[entity].BaseTimeFrameError.loword;
		condFrErr.FrameErrorRatio
			= mib->MACFrameErrorConditionGrp[entity].FrameErrorRatio;

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);
		break;

	case fddiMACNotCopiedConditionEvent:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &condNotCopied;

		/* fill event header data */
		condNotCopied.condHdr.paramType = eventType;
		condNotCopied.condHdr.paramLen = 0x0020;
		msg.len1 = condNotCopied.condHdr.paramLen + sizeof (TLVHdrType);

		/* fill condition data */
		condNotCopied.Condition_State =
			mib->MACStatusGrp[entity].NotCopiedCondition;
		condNotCopied.MACIndex = entity + 1;
		GetMACData (fddiMACNotCopied_Ct, entity,
			&condNotCopied.NotCopied_Ct);
		GetMACData (fddiMACCopied_Ct, entity,
			&condNotCopied.Copied_Ct);
		condNotCopied.BaseCopied_Ct =
			mib->MACNotCopiedConditionGrp[entity].BaseCopied_Ct;
		condNotCopied.BaseTimeStamp.hiword =
			mib->MACNotCopiedConditionGrp[entity].BaseTimeNotCopied.hiword;
		condNotCopied.BaseTimeStamp.loword =
			mib->MACNotCopiedConditionGrp[entity].BaseTimeNotCopied.loword;
		condNotCopied.BaseCopied_Ct =
			mib->MACNotCopiedConditionGrp[entity].BaseCopied_Ct;
		condNotCopied.NotCopiedRatio =
			mib->MACNotCopiedConditionGrp[entity].NotCopiedRatio;

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);
		break;

	case fddiMACNeighborChangeEvent:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &evtNbrChg;

		/* fill event header data */
		evtNbrChg.eventHdr.paramType = eventType;
		evtNbrChg.eventHdr.paramLen = 0x001C;
		msg.len1 = evtNbrChg.eventHdr.paramLen + sizeof (TLVHdrType);

		/* fill condition data */
		evtNbrChg.Condition
			= mib->XDIMACGrp[entity].NeighborChange;
		evtNbrChg.MACIndex = entity + 1;
		MCopyAddress (evtNbrChg.Old_UNA,
			mib->MACConfigGrp[entity].OldUpstreamNbr);
		MCopyAddress (evtNbrChg.New_UNA,
			mib->MACConfigGrp[entity].UpstreamNbr);
		MCopyAddress (evtNbrChg.Old_DNA,
			mib->MACConfigGrp[entity].OldDownstreamNbr);
		MCopyAddress (evtNbrChg.New_DNA,
			mib->MACConfigGrp[entity].DownstreamNbr);

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);

		break;

	case fddiPATHTraceStatusEvent:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &evtTrStat;

		/* fill event header data */
		evtTrStat.eventHdr.paramType = eventType;
		evtTrStat.eventHdr.paramLen = 0x0004;
		msg.len1 = evtTrStat.eventHdr.paramLen + sizeof (TLVHdrType);

		/* fill condition data */
		evtTrStat.TraceStarted = 0;
		evtTrStat.TraceTerminated = 0;
		evtTrStat.TracePropagated = 0;
		switch (mib->ClassPATHConfigGrp[entity].TraceStatus)
		{
		case Trace_Initiated:
			evtTrStat.TraceStarted = 1;
			break;

		case Trace_Propagated:
			evtTrStat.TracePropagated = 1;
			break;

		case Trace_Terminated:
			evtTrStat.TraceTerminated = 1;
			break;
		}

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);
		break;

	case fddiPORTLerConditionEvent:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &condLer;

		/* fill event header data */
		condLer.condHdr.paramType = eventType;
		condLer.condHdr.paramLen = 0x0024;
		msg.len1 = condLer.condHdr.paramLen + sizeof (TLVHdrType);

		/* fill condition data */
		condLer.PORTIndex = entity + 1;
		condLer.ConditionState =
			mib->PORTStatusGrp[entity].LerCondition;
		condLer.Ler_Cutoff =
			mib->PORTLerGrp[entity].Ler_Cutoff;
		condLer.Ler_Alarm =
			mib->PORTLerGrp[entity].Ler_Alarm;
		condLer.Ler_Estimate =
			mib->PORTLerGrp[entity].Ler_Estimate;
		condLer.Lem_Reject_Ct =
			mib->PORTLerGrp[entity].Lem_Reject_Ct;
		condLer.Lem_Ct =
			mib->PORTLerGrp[entity].Lem_Ct;
		condLer.BaseLer_Estimate =
			mib->PORTLerGrp[entity].BaseLer_Estimate;
		condLer.BaseLem_Reject_Ct =
			mib->PORTLerGrp[entity].BaseLem_Reject_Ct;
		condLer.BaseLem_Ct =
			mib->PORTLerGrp[entity].BaseLem_Ct;
		condLer.BaseLer_TimeStamp.hiword =
			mib->PORTLerGrp[entity].BaseLer_TimeStamp.hiword;
		condLer.BaseLer_TimeStamp.loword =
			mib->PORTLerGrp[entity].BaseLer_TimeStamp.loword;

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);

		break;

	case fddiPORTUndesiredConnectionAttempt:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &evtConnect;

		/* fill event header data */
		evtConnect.eventHdr.paramType = eventType;
		evtConnect.eventHdr.paramLen = 0x0008;
		msg.len1 = evtConnect.eventHdr.paramLen + sizeof (TLVHdrType);

		/* fill condition data */
		evtConnect.PORTIndex = entity + 1;
		evtConnect.PC_Type =
			mib->PORTConfigGrp[entity].PC_Type;
		evtConnect.connectState =
			mib->PORTStatusGrp[entity].ConnectState;
		evtConnect.PC_Neighbor =
			mib->PORTConfigGrp[entity].PC_Neighbor;
		evtConnect.connectionAccepted =
			(mib->PORTStatusGrp[entity].PC_Withhold != PC_WH_None)
			? 0 : 1;

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);

		break;

	case fddiPORTEBErrorConditionEvent:
		/* set message pointer */
		msg.p1.ptr = (uChar *) &condEBError;

		/* fill event header data */
		condEBError.condHdr.paramType = eventType;
		condEBError.condHdr.paramLen = 0x0008;

		/* fill condition data */
		condEBError.PORTIndex = entity + 1;
		/*
		*	910412-001	LJP
		*	Use new EB condition member in MIB.
		*/
		condEBError.ConditionState =
			mib->XDIPORTGrp[entity].EBErrorCondition;
		GetPORTData (fddiPORTEBError_Ct, entity,
			&condEBError.EbError_Ct);

		/* send message to FBM */
		msg.destination = FBM_MSG_ID;
		SendMIBMessage (&msg);
		break;
	}

	/*
	*	Send a copy to the MAP.
	*/
	msg.destination = MAP_MSG_ID;
	SendMIBMessage (&msg);

	return;
}

void
DetectMIBConditions ()
/*********************************************************************
Function:	When requested, either based on a timer or request from
		another process, MSP checks the MIB values for the
		sample intervale conditions to be reported by SRFs.
Parameters:	None.
Input:		Uses the MIB.
Output:		Updates MIB condition values.
Return:		No value returned.
*********************************************************************/
{
uInt32		lost,
		error,
		frame,
		notCopied,
		copied,
		deltaTop,
		deltaBottom,
		reason;
uChar		state;
uInt16		i;
TLVParamType	param;

	/*
	*	MAC Frame Error Condition
	*/
	for (i = 0; i < mib->SMTStationConfigGrp.MAC_Ct; i++)
	{
		/* get lost count values */
		param.paramType = fddiMACLost_Ct;
		param.paramLen = 4;
		param.MACINDEX = i + 1;
		GetMIBMACAttr (sizeof (param), &param);
		lost = param.MACPARAM32;

		/* get error count values */
		param.paramType = fddiMACError_Ct;
		param.paramLen = 4;
		param.MACINDEX = i + 1;
		GetMIBMACAttr (sizeof (param), &param);
		error = param.MACPARAM32;

		/* get frame count values */
		param.paramType = fddiMACFrame_Ct;
		param.paramLen = 4;
		param.MACINDEX = i + 1;
		GetMIBMACAttr (sizeof (param), &param);
		frame = param.MACPARAM32;

		/* get deltas */
		deltaTop =
		  (lost - mib->MACFrameErrorConditionGrp[i].BaseLost_Ct) +
		  (error - mib->MACFrameErrorConditionGrp[i].BaseError_Ct);
		deltaBottom =
		  (frame - mib->MACFrameErrorConditionGrp[i].BaseFrame_Ct) +
		  (error - mib->MACFrameErrorConditionGrp[i].BaseError_Ct);

		/* get condition state */
		state = CLEAR;
		/* 910613-001 LJP */
		if (deltaTop > 0x0FFFFL)
			/* condition exists when numerator >= 2^16 */
			state = SET;

		/* calculate only when values are present */
		else if (deltaBottom)
		{
			/* shift to multiply by 2^16 */
			deltaTop <<= 16;
			deltaTop /= deltaBottom;
			state = deltaTop >
				mib->MACFrameErrorConditionGrp[i].FrameErrorThreshold;
		}

		/* check for change of state */
		if (state != mib->MACStatusGrp[i].FrameErrorCondition)
		{
			/* update state values */
			mib->MACStatusGrp[i].FrameErrorCondition = state;
			mib->MACFrameErrorConditionGrp[i].FrameErrorRatio =
				deltaTop;
			GetTimeStamp (&mib->SMTMIBOperationGrp.TransitionTimeStamp);
			SendSMTEvent (fddiMACFrameErrorConditionEvent, i);

			if (state)
			{
				/* update base values */
				mib->MACFrameErrorConditionGrp[i].BaseLost_Ct = lost;
				mib->MACFrameErrorConditionGrp[i].BaseError_Ct = error;
				mib->MACFrameErrorConditionGrp[i].BaseFrame_Ct = frame;
			}
		}
	}

	/*
	*	MAC Not Copied Condition
	*/
	for (i = 0; i < mib->SMTStationConfigGrp.MAC_Ct; i++)
	{
		/* get not copied count values */
		param.paramType = fddiMACNotCopied_Ct;
		param.paramLen = 4;
		param.MACINDEX = i + 1;
		reason = GetMIBMACAttr (sizeof (param), &param);
		if (reason != RC_SUCCESS)
			/* optional parameter */
			continue;
		notCopied = param.MACPARAM32;

		/* get copied count values */
		param.paramType = fddiMACCopied_Ct;
		param.paramLen = 4;
		param.MACINDEX = i + 1;
		reason = GetMIBMACAttr (sizeof (param), &param);
		if (reason != RC_SUCCESS)
			/* optional parameter */
			continue;
		copied = param.MACPARAM32;

		/* get deltas */
		deltaTop = notCopied -
			mib->MACNotCopiedConditionGrp[i].BaseNotCopied_Ct;
		deltaBottom = deltaTop +
		  (copied - mib->MACNotCopiedConditionGrp[i].BaseCopied_Ct);

		/* get condition state */
		state = CLEAR;
		/* 910613-001 LJP */
		if (deltaTop > 0x0FFFFL)
			/* condition exists when numerator >= 2^16 */
			state = SET;

		/* calculate only when values are present */
		else if (deltaBottom)
		{
			/* shift to multiply by 2^16 */
			deltaTop <<= 16;
			deltaTop /= deltaBottom;
			state = deltaTop >
				mib->MACNotCopiedConditionGrp[i].NotCopiedThreshold;
		}

		/* check for change of state */
		if (state != mib->MACStatusGrp[i].NotCopiedCondition)
		{
			/* update state values */
			mib->MACStatusGrp[i].NotCopiedCondition = state;
			mib->MACNotCopiedConditionGrp[i].NotCopiedRatio =
				deltaTop;
			GetTimeStamp (&mib->SMTMIBOperationGrp.TransitionTimeStamp);
			SendSMTEvent (fddiMACNotCopiedConditionEvent, i);

			if (state)
			{
				/* update base values */
				mib->MACNotCopiedConditionGrp[i].BaseNotCopied_Ct = notCopied;
				mib->MACNotCopiedConditionGrp[i].BaseCopied_Ct = copied;
			}
		}
	}

	/*
	*	PHY EB Error Condition
	*/
	for (i = 0; i < mib->XDISMTGrp.Port_Ct; i++)
	{
		/* get copied count values */
		param.paramType = fddiPORTEBError_Ct;
		param.paramLen = 4;
		param.PORTINDEX = i + 1;
		reason = GetMIBPORTAttr (sizeof (param), &param);
		if (reason != RC_SUCCESS)
			/* optional parameter */
			continue;

		state =
		  (param.PORTPARAM32 > mib->XDIPORTGrp[i].BaseEBError_Ct)
		  ? SET : CLEAR;
		/*
		*	910412-001	LJP
		*	The following code used test and report the Ler
		*	condition. Now it correctly reports the EB condition.
		*/
		if (state != mib->XDIPORTGrp[i].EBErrorCondition)
		{
			/* update state values */
			mib->XDIPORTGrp[i].EBErrorCondition = state;
			GetTimeStamp 
				(&mib->SMTMIBOperationGrp.TransitionTimeStamp);
			SendSMTEvent (fddiPORTEBErrorConditionEvent, i);
		}
		mib->XDIPORTGrp[i].BaseEBError_Ct = param.PORTPARAM32;
	}

	return;
}


/*********************************************************************
	Process MIB Messages Routines
*********************************************************************/

void
ProcessMIBAction (smtmsg)
	SMTMessage	*smtmsg;
/*********************************************************************
Function:	Process an action message sent to the MIB. The only
		actions handled by the MIB are actions to set attributes
		in the MIB.
Parameters:	smtmsg	= buffer for message received by MIB for processing.
Input:		smtmsg	= contains action message to process.
Output:		Depends upon action.
Return:		No value returned.
*********************************************************************/
{
MIBMsgType	*mibInfo;
#if 0
TLVParamType	*mibAttr;
SMTMessage	newMsg;			/* output message */
#endif

	/* Perform requested action. */
	switch (smtmsg->type)
	{
	/*
	*	MIB Operations
	*/
	case MIB_ACTION_GET_ATTR:
		/* Fill response message. */
		smtmsg->destination = smtmsg->source;
		smtmsg->source = MIB_MSG_ID;
		smtmsg->type = MIB_RESPONSE_GET_ATTR;

		/* Get MIB information. */
		mibInfo = smtmsg->p1.mibMsgBuf;
		mibInfo->result = GetMIBAttr (smtmsg->len2,
			smtmsg->p2, &mibInfo->setCount);

		/* Return response. */
		SendMIBMessage (smtmsg);
		break;

	case MIB_ACTION_CHANGE_ATTR:
		/* Fill response message. */
		smtmsg->destination = smtmsg->source;
		smtmsg->source = MIB_MSG_ID;
		smtmsg->type = MIB_RESPONSE_CHANGE_ATTR;

		/* Change MIB information. */
		mibInfo = smtmsg->p1.mibMsgBuf;
		mibInfo->result = ChangeMIBAttr (smtmsg->len2,
			smtmsg->p2, &mibInfo->setCount,
			&mibInfo->setStationID);

		/* Return response. */
		SendMIBMessage (smtmsg);
		break;

	case MIB_ACTION_ADD_ATTR:
		/* Fill response message. */
		smtmsg->destination = smtmsg->source;
		smtmsg->source = MIB_MSG_ID;
		smtmsg->type = MIB_RESPONSE_ADD_ATTR;

		/* Add MIB information. */
		mibInfo = smtmsg->p1.mibMsgBuf;
		mibInfo->result = AddMIBAttr (smtmsg->len2,
			smtmsg->p2, &mibInfo->setCount,
			&mibInfo->setStationID);

		/* Return response. */
		SendMIBMessage (smtmsg);
		break;

	case MIB_ACTION_REMOVE_ATTR:
		/* Fill response message. */
		smtmsg->destination = smtmsg->source;
		smtmsg->source = MIB_MSG_ID;
		smtmsg->type = MIB_RESPONSE_REMOVE_ATTR;

		/* Remove MIB information. */
		mibInfo = smtmsg->p1.mibMsgBuf;
		mibInfo->result = RemoveMIBAttr (smtmsg->len2,
			smtmsg->p2, &mibInfo->setCount,
			&mibInfo->setStationID);

		/* Return response. */
		SendMIBMessage (smtmsg);
		break;

	case MIB_ACTION_DETECT_CONDITIONS:
		/* Look for conditions */
		DetectMIBConditions ();
		break;

	default:
		LocalMIBMessages (smtmsg);
		break;
	}

	return;
}

void
ProcessMIBEvent (smtmsg)
	SMTMessage	*smtmsg;
/*********************************************************************
Function:	Process an event message sent to the MIB.
Parameters:	smtmsg	= buffer for message received by MIB for processing.
Input:		smtmsg	= contains request message to process.
Output:		Issues indication message.
Return:		No value returned.
*********************************************************************/
{
/* 910807-004 LJP Use 32-bit value for portability. */
Int32	temp;					/* comparison result */

	switch (smtmsg->type)
	{
	case CSP_EVENT_NOTIFY_MIB:
	case FBM_EVENT_NOTIFY_MIB:
		switch (smtmsg->typeInfo)
		{
		/*
		*	Process MIB information changes.
		*/
		case fddiSMTECMState:
			mib->SMTStatusGrp.ECMState = smtmsg->data.b8;
			break;

		case fddiSMTCF_State:
			/* 910807-004 LJP Removed extraneous statement */
			/* set new state */
			mib->SMTStatusGrp.CF_State = smtmsg->data.b8;
			break;

		case fddiSMTHoldState:
			mib->SMTStatusGrp.HoldState = smtmsg->data.b16;
			break;

		case fddiSMTConfigurationChgEvent:
			/* report event to SMT */
			SendSMTEvent (fddiSMTConfigurationChgEvent,
				smtmsg->entity);
			break;

		case fddiMACCurrentPath:
			mib->MACConfigGrp[smtmsg->entity].CurrentPath
				= smtmsg->data.b16;
			break;

		case fddiMACUpstreamNbr:
			/*
			*	Check for new upstream neighbor.
			*/
			MCompareAddress
				(mib->MACConfigGrp[smtmsg->entity].UpstreamNbr,
				smtmsg->data.addr, temp);
			if (temp != 0)
		{
			/* Save old address */
			MCopyAddress
			(mib->MACConfigGrp[smtmsg->entity].OldUpstreamNbr,
				mib->MACConfigGrp[smtmsg->entity].UpstreamNbr);

			/* set new address */
			MCopyAddress
				(mib->MACConfigGrp[smtmsg->entity].UpstreamNbr,
				smtmsg->data.addr);

			/* set event type */
			mib->XDIMACGrp[smtmsg->entity].NeighborChange =
				EVENT_UNA_CHANGED;

			/*
			*	910807-002	LJP
			*	Notify CSP of new UNA.
			*/
			smtmsg->destination = CSP_MSG_ID;
			smtmsg->source = MIB_MSG_ID;
			smtmsg->type = MIB_EVENT_NOTIFY_CHANGE;
			SendMIBMessage (smtmsg);

			/* notify SMT */
			SendSMTEvent (fddiMACNeighborChangeEvent, smtmsg->entity);
		}
			break;

		case fddiMACDownstreamNbr:
			/*
			*	Check for new downstream neighbor.
			*/
			MCompareAddress
			(mib->MACConfigGrp[smtmsg->entity].DownstreamNbr,
				smtmsg->data.addr, temp);
			if (temp != 0)
		{
			/* Save old address */
			MCopyAddress
			(mib->MACConfigGrp[smtmsg->entity].OldDownstreamNbr,
			mib->MACConfigGrp[smtmsg->entity].DownstreamNbr);

			/* set new address */
			MCopyAddress
			(mib->MACConfigGrp[smtmsg->entity].DownstreamNbr,
			smtmsg->data.addr);

			/* set event type */
			mib->XDIMACGrp[smtmsg->entity].NeighborChange =
				EVENT_DNA_CHANGED;

			/* notify SMT */
			SendSMTEvent (fddiMACNeighborChangeEvent,
				smtmsg->entity);
		}
			break;

		case fddiMACDup_Addr_Test:
			mib->MACConfigGrp[smtmsg->entity].Dup_Addr_Test
				= smtmsg->data.b16;

			/*
			*	910326-001	LJP
			*	Pass the message on to CSP for RMT
			*	processing.
			*/
			smtmsg->destination = CSP_MSG_ID;
			smtmsg->source = MIB_MSG_ID;
			smtmsg->type = MIB_EVENT_NOTIFY_CHANGE;
			SendMIBMessage (smtmsg);
			break;

		case fddiMACRMTState:
			if (smtmsg->data.b8 == RM_RING_OP)
			{
			   SendMsgToRiMapMBox((uInt16) EV_OTHER, 
					      (uInt16) smtmsg->entity);
			}
			else if ((mib->MACStatusGrp[smtmsg->entity].RMTState
				   == RM_ISOLATED) 
				 && (smtmsg->data.b8 != RM_ISOLATED))
			{
			   SendMsgToRiMapMBox( (uInt16) EV_MAC_ENABLED, (uInt16) smtmsg->entity);
			}
			else if ((smtmsg->data.b8 == RM_ISOLATED) 
				 && (mib->MACStatusGrp[smtmsg->entity].RMTState
				     != RM_ISOLATED))
			{
			   SendMsgToRiMapMBox( (uInt16) EV_MAC_DISABLED, (uInt16) smtmsg->entity);
			}
			mib->MACStatusGrp[smtmsg->entity].RMTState
				= smtmsg->data.b8;
			mib->MACStatusGrp[smtmsg->entity].LLCServiceAvailable
				= (smtmsg->data.b8 == RM_RING_OP) ? 1 : 0;
			break;

		case fddiMACDa_Flag:
			if (mib->MACStatusGrp[smtmsg->entity].Da_Flag
				!= smtmsg->data.b8)
			{
				mib->MACStatusGrp[smtmsg->entity].Da_Flag
					= smtmsg->data.b8;

				/* notify SMT */
				SendSMTEvent (fddiMACDuplicateAddressCondition,
					smtmsg->entity);
			}

			break;

		case fddiMACUnaDa_Flag:
			if (mib->MACStatusGrp[smtmsg->entity].UnaDa_Flag
				!= smtmsg->data.b8)
			{
				mib->MACStatusGrp[smtmsg->entity].UnaDa_Flag
					= smtmsg->data.b8;

				/* notify SMT */
				SendSMTEvent (fddiMACDuplicateAddressCondition,
					smtmsg->entity);
			}

			break;

		case fddiPATHClassPATHTraceStatus:
			mib->ClassPATHConfigGrp[smtmsg->entity].TraceStatus
				= smtmsg->data.b16;

			/* notify SMT */
			SendSMTEvent (fddiPATHTraceStatusEvent,
				smtmsg->entity);

			break;

		case fddiPATHClassPATHStatus:
			mib->ClassPATHConfigGrp[smtmsg->entity].Status
				= smtmsg->data.b16;
			break;

		case fddiPORTPC_Neighbor:
			mib->PORTConfigGrp[smtmsg->entity].PC_Neighbor
				= smtmsg->data.b8;

			/* Note: undesirable connection checked when
				PC_Withhold is reported */
			break;

		case fddiPORTRemoteMACIndicated:
			mib->PORTConfigGrp[smtmsg->entity].RemoteMACIndicated
				= smtmsg->data.b8;
			break;

		case fddiPORTCE_State:
			mib->PORTConfigGrp[smtmsg->entity].CE_State
				= smtmsg->data.b8;
			break;

		case fddiPORTMACPlacement:
			mib->PORTConfigGrp[smtmsg->entity].MACPlacement
				= smtmsg->data.b16;
			break;

		case fddiPORTLCTFail_Ct:
			mib->PORTErrorCtrsGrp[smtmsg->entity].LCTFail_Ct
				= smtmsg->data.b32;
			break;

		case fddiPORTLer_Estimate:
			mib->PORTLerGrp[smtmsg->entity].Ler_Estimate
				= smtmsg->data.b8;

			/*
			*	910412-001	LJP
			*	Correct and report proper condition state.
			*/
			/* check for alarm condition
				(comparing abs value of exponents) */
			temp = (smtmsg->data.b8 <=
				mib->PORTLerGrp[smtmsg->entity].Ler_Alarm)
				? SET : CLEAR;
			if (temp != mib->PORTStatusGrp[smtmsg->entity].LerCondition)
			{
				/* update condition */
				mib->PORTStatusGrp[smtmsg->entity].LerCondition
					= temp;

				/* notify SMT */
				SendSMTEvent (fddiPORTLerConditionEvent,
					smtmsg->entity);
			}
			break;

		case fddiPORTLem_Reject_Ct:
			mib->PORTLerGrp[smtmsg->entity].Lem_Reject_Ct
				= smtmsg->data.b32;
			break;

		case fddiPORTLem_Ct:
			mib->PORTLerGrp[smtmsg->entity].Lem_Ct
				= smtmsg->data.b32;
			break;

		case fddiPORTPCMState:
			mib->PORTStatusGrp[smtmsg->entity].PCMState
				= smtmsg->data.b8;
			SetConnectState (smtmsg->entity);
			/*
			*	910605-004	LJP
			*	Clear remote MAC indicated when PCM goes
			*	to OFF or BREAK.
			*/
			if ((smtmsg->data.b8 == PC_OFF)
					|| (smtmsg->data.b8 == PC_BREAK))
 				mib->PORTConfigGrp[smtmsg->entity].RemoteMACIndicated
					= CLEAR;
			break;

		case fddiPORTPC_Withhold:
			mib->PORTStatusGrp[smtmsg->entity].PC_Withhold
				= smtmsg->data.b8;
			SetConnectState (smtmsg->entity);

			/* check for undesirable connections */
			if (undesirableTable
				[mib->PORTConfigGrp[smtmsg->entity].PC_Type]
				[mib->PORTConfigGrp[smtmsg->entity].PC_Neighbor])
			{
				/* notify SMT */
				SendSMTEvent (fddiPORTUndesiredConnectionAttempt,
					smtmsg->entity);
			}

			break;

		case fddiPORTBS_Flag:
			mib->PORTOperationGrp[smtmsg->entity].BS_Flag
				= smtmsg->data.b8;
			break;

		case fddiATTACHMENTInsertedStatus:
			mib->ATTACHMENTConfigGrp[smtmsg->entity].InsertedStatus
				= smtmsg->data.b8;
			break;

		case xdiSMTSB_Flag:
			mib->XDISMTGrp.SB_Flag
				= smtmsg->data.b8;
			break;

		case xdiMACDataAddress:
			MCopyAddress (
				mib->XDIMACGrp[smtmsg->entity].DataAddress,
				smtmsg->data.addr);
			break;

		default:
			break;
		}
		break;

	default:
		LocalMIBMessages (smtmsg);
		break;
	}

	return;
}

void
ProcessMIBMessage (smtmsg)
	SMTMessage	*smtmsg;
/*********************************************************************
Function:	Process a message received by MIB.
Parameters:	smtmsg	= address of buffer containing message.
Input:		smtmsg	= complete contents of message.
Output:		Performs actions appropriate to message type and info.
Return:		None.
Modification History:
*********************************************************************/
{

	/*
	*	Select message processing.
	*/
	switch (MESSAGE_TYPE (smtmsg->type))
	{
	case EVENT_MSG:
		ProcessMIBEvent (smtmsg);
		break;

	case ACTION_MSG:
		ProcessMIBAction (smtmsg);
		break;
	}

	return;
}

