#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) mibproc.c V6.2.3:cs.622c:5:5 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/**********************************************************************
	Management Information Base Module

	MIB maintenance module.

	File:		mibproc.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.622c	08/02/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	This module provides the basic interface to the MIB maintenance
	routines. The maintenance routines come in four varieties: get,
	change, add, and remove. The get interface performs a get action
	on a MIB attribute or attribute group. A get operation should 
	work for all supported parameters. The change interface sets
	a MIB attribute's value. Only read-write attributes may be changed.
	Group or alias addresses may only be changed through the add or
	remove operations.

	A get operation generally reads the current value from the MIB
	data base and returns it to the calling procedure. Externally
	located values (i.e., counters), are retrieved through an 
	interface routine. The associated MIB attributes in the data base
	are not changed. This allows the implementor to use the MIB
	locations for interim storage.

	A change operation requires that the new value become active
	within the station's operational parameters. The change operation
	routine calls a change interface routine to implement the new
	attribute value (e.g., T_Req or Ler_Cutoff). The interface routine
	implements the new value and operation routine places the new value
	in the MIB. Thus all current operational values are contained
	in the MIB data base. The change operation routine filters out
	read-only attributes and group attributes since these values
	may not be changed.

	The add and remove operations change the contents of the group
	and alias addresses attributes. Since the capacity, storage, and
	format for handling these attributes may vary among implementations,
	the MIB operations call interface routines to perform the actual
	operation requested.

	Modification History:

	*** Updated to SMT 6.2 ***

*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"fddihdr.h"
#include	"smtmacro.h"
#include	"smtmsg.h"
#include	"mibdefs.h"
#include	"mibtypes.h"
#include	"mibglbl.h"


/*********************************************************************
	Global Data 
*********************************************************************/


/*********************************************************************
	External function definitions.
*********************************************************************/

extern	uInt32	GetMIBSMTAttr ();
extern	uInt32	GetMIBMACAttr ();
extern	uInt32	GetMIBPATHAttr ();
extern	uInt32	GetMIBPORTAttr ();
extern	uInt32	GetMIBATTACHMENTAttr ();

extern	uInt32	ChangeMIBSMTAttr ();
extern	uInt32	ChangeMIBMACAttr ();
extern	uInt32	ChangeMIBPATHAttr ();
extern	uInt32	ChangeMIBPORTAttr ();
extern	uInt32	ChangeMIBATTACHMENTAttr ();

extern	uInt32	AddMIBMACAttr ();
extern	uInt32	RemoveMIBMACAttr ();

extern	void	GetTimeStamp ();
extern  void    UpdateFddiMibInNvr();


/*********************************************************************
	Local function definitions.
*********************************************************************/

static Flag
ValidSetCount (setCount)
	TLVSetCountType	*setCount;
/*********************************************************************
Function:	MIB access routine to verify correct fddiSMTSetCount
		value is given when changing the value of an attribute.
Parameter:	setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
Input:		setCount	= contains the verification value for
					fddiSMTSetCount.
Output:		None.
Return:		TRUE if the set count value matches the current value
		in the MIB. Otherwise, FALSE is returned.
*********************************************************************/
{
TLVSetCountType	currentSetCount;	/* current MIB value */

	/*
	*	Get current fddiSMTSetCount.
	*/
	if (setCount == NULL)
	{
	   return TRUE; /* jlin: NULL for internal use */
	}

	currentSetCount.paramType = fddiSMTSetCount;
	currentSetCount.paramLen = 0;
	GetMIBSMTAttr (sizeof (TLVSetCountType), &currentSetCount);

	return (currentSetCount.paramValue.count == setCount->paramValue.count);
}

/*********************************************************************
	MIB Access Routines
*********************************************************************/

uInt32
GetMIBAttr (bufSize, bufData, setCount)
	uInt16		bufSize;
	uChar		*bufData;
	TLVSetCountType	*setCount;
/*********************************************************************
Function:	MIB access routine to get MIB attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute value
					returned.
		setCount	= pointer to buffer to hold current value
					of the fddiSMTSetCountAttribute
					(may be NULL if current SetCount
					is not required).
Input:		bufData		= contains the requested attribute ID and
					object index (if required) using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
		setCount	= if not NULL, this buffer gets set to the
					current value of fddiSMTSetCount,
					also using the MIB TLV encoding.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute retrieved successfully
		RC_NO_PARAMETER
			attribute not supported
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
		RC_BUFFER_ERROR
			if the buffer is not large enough to hold the
			requested attribute or attribute group.
Notes:	For the attributes fddiMACLongAliases, fddiMACShortAliases,
	fddiMACShortGrpAddrs, and fddiMACLongGrpAddrs, this routine
	returns as many itmes in the group as will fit in the buffer.
*********************************************************************/
{
TLVHdrType	*phdr;			/* Parameter header */
uInt32		result;			/* function return value */

	/*
	*	Get header info.
	*/
	phdr = (TLVHdrType *) bufData;

	/*
	*	Select proper processing routine.
	*/
	switch (phdr->paramType & MIB_OBJECT_MASK)
	{
	case fddiSMT:
		result = GetMIBSMTAttr (bufSize, bufData);
		break;

	case fddiMAC:
		result = GetMIBMACAttr (bufSize, bufData);
		break;

	case fddiPATHClass:
		result = GetMIBPATHAttr (bufSize, bufData);
		break;

	case fddiPORT:
		result = GetMIBPORTAttr (bufSize, bufData);
		break;

	case fddiATTACHMENT:
		result = GetMIBATTACHMENTAttr (bufSize, bufData);
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	/*
	*	Get new fddiSMTSetCount.
	*/
	if (result == RC_SUCCESS)
	{
		if (setCount != (TLVSetCountType *) NULL)
		{
			setCount->paramType = fddiSMTSetCount;
			setCount->paramLen = 0;
			GetMIBSMTAttr (sizeof (TLVSetCountType), setCount);
		}
	}
        else if (result == RC_ILLEGAL_PARAMETER)
	{
	        result = GetHLSMIBSMTAttr (bufSize, bufData);
	}

	return (result);
}

uInt32
ChangeMIBAttr (bufSize, bufData, setCount, stationID)
	uInt16			bufSize;
	uChar			*bufData;
	TLVSetCountType		*setCount;
	SMTStationIdType	*stationID;
/*********************************************************************
Function:	MIB access routine to change a MIB attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					new attribute information
					and used to hold attribute value
					returned.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= pointer to buffer holding ID of station
					requesting the change.
Input:		bufData		= contains the new attribute value using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
		setCount	= contains the verification value for
					fddiSMTSetCount.
		stationID	= contains station ID of the station
					requesting the attribute change.
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
		setCount	= set to the value of fddiSMTSetCount after
					attempting the attribute change,
					also using the MIB TLV encoding.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_BAD_SET_COUNT
			the value of setCount does not match the current
			MIB value
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_OUT_OF_RANGE
			value given is out of bounds
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
*********************************************************************/
{
TLVHdrType	*phdr;			/* Parameter header */
uInt32		result;			/* function return value */
#if 0
uInt16		paramType;
#endif

	if (ValidSetCount (setCount))
	{
	        MIBDPT("ChangeMIBAttr\n");
		/*
		*	Get header info.
		*/
		phdr = (TLVHdrType *) bufData;

		/*
		*	Select proper processing routine.
		*/
		switch (phdr->paramType & MIB_OBJECT_MASK)
		{
		case fddiSMT:
			result = ChangeMIBSMTAttr (bufSize, bufData);
			break;

		case fddiMAC:
			result = ChangeMIBMACAttr (bufSize, bufData);
			break;

		case fddiPATHClass:
			result = ChangeMIBPATHAttr (bufSize, bufData);
			break;

		case fddiPORT:
			result = ChangeMIBPORTAttr (bufSize, bufData);
			break;

		case fddiATTACHMENT:
			result = ChangeMIBATTACHMENTAttr (bufSize, bufData);
			break;

		default:
			result = RC_ILLEGAL_PARAMETER;
			break;
		}

		/*
		*	If change successful, update set count and set
		*	last set station ID.
		*/
		if (result == RC_SUCCESS)
		{
		        /* to update FDDI NVR record */
		        UpdateFddiMibInNvr((TLVParamType*) bufData, 
					   stationID);

			mib->SMTMIBOperationGrp.SetCount.count++;
			GetTimeStamp
			  (&mib->SMTMIBOperationGrp.SetCount.setTimeStamp);

			MEMCOPY (&mib->SMTMIBOperationGrp.LastSetStationId,
				stationID, sizeof (SMTStationIdType));
		}
	}

	else
	{
		/*
		*	Invalid set count, set it to current value.
		*/
	        MIBDPT("\nBad Set Count.\n");
		result = RC_BAD_SET_COUNT;
	}

	/*
	*	Return current set count.
	*/
	if (setCount != (TLVSetCountType *) NULL) /* jlin: NULL for
	internal use */
	{
		setCount->paramType = fddiSMTSetCount;
		setCount->paramLen = 0;
		GetMIBSMTAttr (sizeof (TLVSetCountType), setCount);
	}	

	return (result);
}

uInt32
AddMIBAttr (bufSize, bufData, setCount, stationID)
	uInt16			bufSize;
	uChar			*bufData;
	TLVSetCountType		*setCount;
	SMTStationIdType	*stationID;
/*********************************************************************
Function:	MIB access routine to add items to the MIB attributes
		fddiMACLongAliases, fddiMACShortAliases, fddMACLongGrpAddrs,
		or fddiMACShortGrpAddrs.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing a single
					item to add to the attribute
					information and used to hold
					attribute value returned.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= pointer to buffer holding the ID
					of the station requesting the add.
Input:		bufData		= contains one and only one item to be
					added to the attribute value using
					the MIB TLV encoding.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= contains the ID of the station requesting
					the add operation.
Output:		bufData		= holds the complete attribute value
					after the addition has been attempted
					using the MIB TLV encoding specified
					in the SMT standard.
		setCount	= set to the value of fddiSMTSetCount after
					attempting the attribute change,
					also using the MIB TLV encoding.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_BAD_SET_COUNT
			the value of setCount does not match the current
			MIB value
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_NO_MORE
			this attribute cannot hold any more values
		RC_OUT_OF_RANGE
			value given is out of bounds
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
		RC_LENGTH_ERROR
			length of attribute does not correspond to 
			a single item value
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
Notes:		AddMIBAttr() may only be used for the attributes
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		and fddiMACShortGrpAddrs.
*********************************************************************/
{
TLVHdrType	*phdr;			/* Parameter header */
uInt		result;			/* function return value */

	/*
	*	Check if specified SetCount is equivalent to
	*	current fddiSMTSetCount.
	*/
	if (ValidSetCount (setCount))
	{
		/*
		*	Get header info.
		*/
		phdr = (TLVHdrType *) bufData;

		/*
		*	Verify proper attributes.
		*/
		switch (phdr->paramType)
		{
		case fddiMACLongAliases:
		case fddiMACLongGrpAddrs:
			/*
			*	Check length, only item allowed.
			*/
			if (phdr->paramLen == 0x0C)
				result = AddMIBMACAttr (bufSize, bufData);
			else
				result = RC_LENGTH_ERROR;
			break;

		case fddiMACShortAliases:
		case fddiMACShortGrpAddrs:
			/*
			*	Check length, only item allowed.
			*/
			if (phdr->paramLen == 0x08)
				result = AddMIBMACAttr (bufSize, bufData);
			else
				result = RC_LENGTH_ERROR;
			break;

		default:
			result = RC_ILLEGAL_PARAMETER;
			break;
		}

		/*
		*	If add was successful, update set count and set
		*	last set station ID.
		*/
		if (result == RC_SUCCESS)
		{
			mib->SMTMIBOperationGrp.SetCount.count++;
			GetTimeStamp
			  (&mib->SMTMIBOperationGrp.SetCount.setTimeStamp);

			MEMCOPY (&mib->SMTMIBOperationGrp.LastSetStationId,
				stationID, sizeof (SMTStationIdType));
		}
	}

	else
	{
		/*
		*	Invalid set count, set it to current value.
		*/
		result = RC_BAD_SET_COUNT;
	}

	/*
	*	Return current set count.
	*/
	setCount->paramType = fddiSMTSetCount;
	setCount->paramLen = 0;
	GetMIBSMTAttr (sizeof (TLVSetCountType), setCount);

	return (result);
}

uInt32
RemoveMIBAttr (bufSize, bufData, setCount, stationID)
	uInt16			bufSize;
	uChar			*bufData;
	TLVSetCountType		*setCount;
	SMTStationIdType	*stationID;
/*********************************************************************
Function:	MIB access routine to remove an itme from attribute
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		or fddiMACShortGrpAddrs.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					attribute information to be removed
					and used to hold attribute value
					returned.
		setCount	= pointer to buffer holding the current value
					of the fddiSMTSetCount attribute
		stationID	= pointer to buffer holding the ID
					of the station requesting the remove.
Input:		bufData		= contains one and only one value to be
					removed represented in MIB TLV
					encoding.
		setCount	= contains the verification value for
					fddiSMTSetCount.
		stationID	= contains the ID of the station requesting
					the remove operation.
Output:		bufData		= holds the attribute value after attempting
					the deletion in the MIB TLV encoding
					specified in the SMT standard.
		setCount	= set to the value of fddiSMTSetCount after
					attempting the attribute delete,
					also using the MIB TLV encoding.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute changed successfully
		RC_BAD_SET_COUNT
			the value of setCount does not match the current
			MIB value
		RC_READ_ONLY
			attempted to change a read-only parameter
		RC_NO_PARAMETER
			attribute not supported
		RC_NO_MORE
			no more items to be removed
                RC_NOT_AUTHORIZED
                        station does not permit changing this attribute
		RC_LENGTH_ERROR
			length of attribute does not correspond to
			a single item value
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
Notes:		RemoveMIBAttr() may only be used with attributes
		fddiMACLongAliases, fddiMACShortAliases, fddiMACLongGrpAddrs,
		and fddiMACShortGrpAddrs. Specifying the removal of zero
		members results in removing all members of the attribute.
*********************************************************************/
{
TLVHdrType	*phdr;			/* Parameter header */
uInt32		result;			/* function return value */

	/*
	*	Check if specified SetCount is equivalent to
	*	current fddiSMTSetCount.
	*/
	if (ValidSetCount (setCount))
	{
		/*
		*	Get header info.
		*/
		phdr = (TLVHdrType *) bufData;

		/*
		*	Verify proper attributes.
		*/
		switch (phdr->paramType)
		{
		case fddiMACLongAliases:
		case fddiMACLongGrpAddrs:
			/*
			*	Check length, only item allowed.
			*/
			if (phdr->paramLen == 0x0C)
				result = RemoveMIBMACAttr (bufSize, bufData);
			else
				result = RC_LENGTH_ERROR;
			break;

		case fddiMACShortAliases:
		case fddiMACShortGrpAddrs:
			/*
			*	Check length, only item allowed.
			*/
			if (phdr->paramLen == 0x08)
				result = RemoveMIBMACAttr (bufSize, bufData);
			else
				result = RC_LENGTH_ERROR;
			break;

		default:
			result = RC_ILLEGAL_PARAMETER;
			break;
		}

		/*
		*	If remove was successful, update set count and set
		*	last set station ID.
		*/
		if (result == RC_SUCCESS)
		{
			mib->SMTMIBOperationGrp.SetCount.count++;
			GetTimeStamp
			  (&mib->SMTMIBOperationGrp.SetCount.setTimeStamp);

			MEMCOPY (&mib->SMTMIBOperationGrp.LastSetStationId,
				stationID, sizeof (SMTStationIdType));
		}
	}

	else
	{
		/*
		*	Invalid set count, set it to current value.
		*/
		result = RC_BAD_SET_COUNT;
	}

	/*
	*	Return current set count.
	*/
	setCount->paramType = fddiSMTSetCount;
	setCount->paramLen = 0;
	GetMIBSMTAttr (sizeof (TLVSetCountType), setCount);

	return (result);
}
