#include <types.h>
#include <eeprom.h>
#include <eeprecs.h>
#include <nvram.h>
#include <nvrecs.h>
#include <time.h>
#include <nim960h.h>
#include <flash.h>
#include <sys.h>
#include <uart.h>
#include <tcpip.h>
#include <bootp.h>
#include <product.h>
#include <udp.h>
#include <tftpboot.h>

extern int program_id;
extern int timer_1_ticks_per_second;
extern NVR_PORT nvr_port;

static EEP_MFG ManuRec;
static NVR_BOOT BooterRec;

int media = 0;

initialize_configuration()
{
	int error_code;
	EEP_MFG *MRec = &eep_mfg_rec;
	EEP_MFG *MfgRec = &ManuRec;
	NVR_BOOT *BootRec = &BooterRec;
	char buffer[16];

#ifdef notdef		/* GJS already read in */
	if((error_code = read_eeprom((u_short *)MRec)) != EEP_OK)
	{
		printf("\nCannot read the CONFIGURATION record\n");
		print_eeprom_error(error_code);
		printf("\nDefault CONFIGURATION record being used\n");

		MfgErrSetDefault();
		PutMfgRec(MRec);
	}

	/*
	* initializing the nvram here wont hurt anything
	* if nvram has already been initialized this will check the header
	* and reload the directory information
	* if the nvram is bad a new nvram header will be generated
	*/
	error_code = InitNvram(NVRAM, eep_mfg_rec.eep_nvram_size);
	if (error_code)
		{
		printf("\nCannot initialize NVRAM\n");
		print_nvram_error(error_code);
		}

	/*
	* get the boot record from the nvram
	* if an error of any kind happens write a new record
	*/
	if ((error_code = GetBootRec(BRec)) != NVRAM_AOK)
		{
		printf("\nCannot read the BOOT record\n");
		print_nvram_error(error_code);
		printf("\nDefault BOOT record being used\n");

		PutBootRec(BRec);
		}
#endif

	memcpy(MfgRec, MRec, sizeof(EEP_MFG));

#ifdef notdef
	/*
	* write zeros in all unused locations
	*/
	memset(&MfgRec->eep_spare_0, 0x0, sizeof(MfgRec->eep_spare_0));
	memset(&MfgRec->eep_spare_1, 0x0, sizeof(MfgRec->eep_spare_1));
	memset(MfgRec->eep_spare_3, 0x0, sizeof(MfgRec->eep_spare_3));
#endif


	printf("\r\n");
	PrintManuRecord((EEP_MFG *)MfgRec);
	printf("\r\n");

	if (AskYesNo("Do you want to modify the records ([y]/n) ? "))
	{
		MfgRec->eep_srlno = AskUnsigned("Enter S/N",
					MfgRec->eep_srlno);
		MfgRec->eep_level = AskLevel("Enter Rev Level",
					MfgRec->eep_level);
		MfgRec->eep_mfgdt = AskDate("Enter Mfg date",
					MfgRec->eep_mfgdt);
		MfgRec->eep_altdt = AskDate("Enter FCO date",
					MfgRec->eep_altdt);

		printf("\r\n");
		if (!(AskYesNo("Do you want to keep default memory sizes ([y]/n) ? ")))
			ChangeMemorySize(MfgRec);

		printf("\r\n");
		if (!(AskYesNo("Do you want to keep default FDDI configuration ([y]/n) ? ")))
			ChangeFDDIRecord(MfgRec);
		printf("\r\n");
		PrintManuRecord((EEP_MFG *)MfgRec);
		printf("\r\n");

		if (AskYesNo("Do you want to update the records ([y]/n) ? "))
		{
			memcpy(MRec, MfgRec, sizeof(EEP_MFG));
			PutMfgRec(MfgRec);
		}
	}

}


#ifdef notdef
ChangeMemorySize(MfgRec)
EEP_MFG *MfgRec;
{
	printf("\r\n");

	MfgRec->eep_dram_size = AskUnsigned("Size of DRAM in Kbytes",
					MfgRec->eep_dram_size / 1024) * 1024;

	MfgRec->eep_nvram_size = AskUnsigned("Size of NVRAM in Kbytes",
					MfgRec->eep_nvram_size / 1024) * 1024;
	MfgRec->eep_eeprom_size = AskUnsigned("Size of EEPROM in bytes",
					MfgRec->eep_eeprom_size );
	MfgRec->eep_fprom_size = AskUnsigned("Size of FEPROM in Kbytes",
						MfgRec->eep_fprom_size / 1024) * 1024;
	MfgRec->eep_sram_size = AskUnsigned("Size of SRAM in Kbytes",
						MfgRec->eep_sram_size / 1024) * 1024;
}

Load_Configuration(int *eeprom_error, int *nvram_error, int *boot_error)
{

	*eeprom_error = read_eeprom((u_short *)&eep_mfg_rec);
	if (*eeprom_error != EEP_OK)
		MfgErrSetDefault();

	/*
	* JIM JIM
	* remove the following if in version 103
	*/
	if (eep_mfg_rec.eep_ports== 0)
		eep_mfg_rec.eep_ports = DEFAULT_PORTS;

	*nvram_error = InitNvram(NVRAM, eep_mfg_rec.eep_nvram_size);

	*boot_error = GetBootRec(&nvr_boot_rec);

	return;
}

print_configuration_errors(int eeprom_error, int nvram_error, int boot_error)
{
	if (eeprom_error)
		{
		/*
		* print error message
		*/
		printf("\nCannot read the CONFIGURATION record - EEPROM error\n");
		print_eeprom_error(eeprom_error);
		printf("\nDefault CONFIGURATION record being used\n");
		}

	if (nvram_error || boot_error)
		{
		/*
		* print error message
		*/
		printf("\nCannot read the BOOT record - NVRAM error\n");
		print_nvram_error(boot_error);
		printf("\nDefault BOOT record being used\n");
		}

	return;
}

PutMfgRec(MfgRec)
EEP_MFG *MfgRec;
{
	int error_code;

	MfgRec->eep_marker = EEP_MARKER;

	if ((error_code = write_eeprom((u_short *)MfgRec)) != EEP_OK)
		{
		printf("EEPROM Error: Cannot update the Configuration record\n", error_code);
		print_eeprom_error(error_code);
		}

	return (error_code);
}

PutBootRec(BootRec)
NVR_BOOT *BootRec;
{
	int error_code;

	BootRec->nvr_boot_marker = BOOT_MARKER;
	if ((error_code = Nvram_Updt(NVR_BOOT_ADDR, BootRec, NVR_BOOT_SIZE)) != NVRAM_AOK)
		{
		printf("ERROR: Cannot update the BOOT record\n");
		print_nvram_error(error_code);
		}

	return (error_code);
}

GetBootRec(BootRec)
NVR_BOOT *BootRec;
{
	int error_code;

	error_code = Nvram_Load(NVR_BOOT_ADDR, BootRec, NVR_BOOT_SIZE);
	if ((error_code != NVRAM_AOK) || (BootRec->nvr_boot_marker != BOOT_MARKER))
		{
		/*
		* initialize the boot record to the default values
		* we cannot run without good values in the boot record
		*/
		initialize_boot_record(BootRec);

		/*
		* if the boot record does not exist write one to NVRAM
		*/
		if (error_code == NVRAM_NOSUCH)
			PutBootRec(BootRec);
		}
	else if (BootRec->nvr_boot_marker != BOOT_MARKER)
		error_code = NVRAM_NO_MARKER;
	
	media = BootRec->nvr_bootp_media;

	/*
	* JIM JIM
	* remove the following if in version 103
	*/
	if (BootRec->nvr_request_file[0] == 0)
		{
#ifdef __TRC
		strcpy(BootRec->nvr_request_file, "h516");
#endif
#ifdef __FDDI_CON
		strcpy(BootRec->nvr_request_file, "fddi_con");
#endif
		}


	return (error_code);
}

initialize_boot_record(BootRec)
NVR_BOOT *BootRec;
{
	/*
	* initialize a default boot rercord
	*/
	memset(BootRec, 0, sizeof(NVR_BOOT));
	BootRec->nvr_boot_marker   = BOOT_MARKER;
	BootRec->nvr_tty_baud      = -1;	/* null entry (9600)*/
	BootRec->nvr_tty_parity    =  1;        /* default to none */
	BootRec->nvr_tty_stop_bits =  1;        /* default to 1    */
	BootRec->nvr_tty_data_length = 8;       /* default to 8    */
	strcpy(BootRec->nvr_tty_prompt, "Console>");
	BootRec->nvr_max_reset_level =  3;	/* reset level 3 */
	BootRec->nvr_save_mode_type = -2;	/* null entry */
	BootRec->nvr_mode_type     = -1;	/* null entry */
	BootRec->nvr_ring_number   = -1;	/* null entry */
	BootRec->nvr_ring_rate     = -1;	/* null entry */
	BootRec->nvr_bootp_request = -1;	/* null entry */
	BootRec->nvr_bootp_error   = -1;	/* null entry */
	BootRec->nvr_debug         = -1;	/* null entry */
	BootRec->SNMPAuthentication = 1;	/* default none(1) */
	BootRec->sysAdmAccess	   =  1;	/* default none(1) */
#ifdef __TRC
	strcpy(BootRec->nvr_request_file, "h516");
#endif
#ifdef __FDDI_CON
	strcpy(BootRec->nvr_request_file, "fddi_con");
#endif

	strncpy(BootRec->nvr_BurnDate, "00/00/00", sizeof(BootRec->nvr_BurnDate));
	strncpy(BootRec->nvr_BurnTime, "00:00:00", sizeof(BootRec->nvr_BurnTime));
	copy_flash_info(BootRec, FEPROM);

	return;
}

build_flash_record(struct flash_info *new_flash_record, struct flash_info *old_flash_record)
{
	struct tm *tm;
	time_t time;
	char burn_date[12];
	char burn_time[12];
	u_long seventy = 0x83aa7e80;
	u_short csum;

	/*
	* get the time from the server
	*/
	time = GetTimeFromServer(nvr_boot_rec.nvr_TimeServer);

	/*
	* convert the time to an ascii format
	* if no time returned from server - use zero as default
	*/
	if (time)
		{
		time += nvr_boot_rec.nvr_OffsetGMT - seventy;
		tm = gmtime(&time);
		sprintf(burn_date, "%02d/%02d/%02d", tm->tm_mon + 1, tm->tm_mday, tm->tm_year);
		sprintf(burn_time, "%02d:%02d:%02d", tm->tm_hour, tm->tm_min, tm->tm_sec);
		}
	else
		{
		strncpy(burn_date, "00/00/00", sizeof(burn_date));
		strncpy(burn_time, "00:00:00", sizeof(burn_time));
		}

	/*
	* clear entire new flash record
	*/
	memset(new_flash_record, 0, sizeof(struct flash_info));

	/*
	* set marker
	*/
	strncpy(new_flash_record->marker, "FLASH   ", sizeof(new_flash_record->marker));
	/*
	* set name of file written to the flash
	*/
	strncpy(new_flash_record->file_name, nvr_boot_rec.nvr_image_file, sizeof(new_flash_record->file_name));

	/*
	* set time information
	*/
	memcpy(new_flash_record->burn_date, burn_date, 8);
	memcpy(new_flash_record->burn_time, burn_time, 8);
	new_flash_record->seconds = time;

	/*
	* get the number of flash burns from the old record if it is good
	* otherwise use the NVRAM burn count
	*/
	if (check_flash(old_flash_record, 0))
		/*
		* flash record is good
		*/
		new_flash_record->burn_cycles = old_flash_record->burn_cycles;
	else
		/*
		* flash record is bad - use info from nvram boot record
		*/
		new_flash_record->burn_cycles = nvr_boot_rec.nvr_BurnCycles;

	/*
	* increment the burn count
	*/
	new_flash_record->burn_cycles++;

	/*
	* the different size fields, the board name, copy right notice
	* and checksum must be filled in by the calling program
	*/

	return;
}

int
check_flash(struct flash_info *flash_info, int print_error)
{
	unsigned short	csum;
	unsigned short	csumx;


	if (strncmp(flash_info->marker, "FLASH   ", sizeof(flash_info->marker)) != 0)
		{
		/*
		* flash record is bad
		*/
		if (print_error)
			printf("Flash marker is bad\n");
		return(0);
		}

	/*
	* checksum entire flash data
	*/
	csum = cksum(flash_info, ((flash_info->info_size + flash_info->comp_size) >> 1));
	if ((csum != 0) && (csum != 0xffff))
		{
		/*
		* flash record is bad
		*/
		if (print_error)
			printf("Flash checksum is bad - %X\n", csum);
		return(0);
		}

	/*
	* flash record is good
	*/
	return(1);
}

copy_flash_info(NVR_BOOT *boot_record, struct flash_info *flash_info)
{
	/*
	* copy file name - clearing entire name first
	*/
	memset(boot_record->nvr_flash_file, 0, sizeof(boot_record->nvr_flash_file));
	strncpy(boot_record->nvr_flash_file, flash_info->file_name, sizeof(boot_record->nvr_flash_file));

	/*
	* copy burn date and time info
	*/
	memcpy(boot_record->nvr_BurnDate, flash_info->burn_date, sizeof(nvr_boot_rec.nvr_BurnDate));
	memcpy(boot_record->nvr_BurnTime, flash_info->burn_time, sizeof(nvr_boot_rec.nvr_BurnTime));

	/*
	* copy burn cycle information
	*/
	boot_record->nvr_BurnCycles = flash_info->burn_cycles;

	return;
}

MfgErrSetDefault()
{
	long *lp;

	/*
	* initialize the manufacturing record to default values
	* only the memory sizes need to be non-zero
	*/
	memset(&eep_mfg_rec, 0, sizeof(EEP_MFG));

	eep_mfg_rec.eep_marker      = EEP_MARKER;
	eep_mfg_rec.eep_dram_size   = DEFAULT_DRAM_SIZE;
	eep_mfg_rec.eep_nvram_size  = DEFAULT_NVRAM_SIZE;
	eep_mfg_rec.eep_eeprom_size = DEFAULT_EEPROM_SIZE;
	eep_mfg_rec.eep_fprom_size  = DEFAULT_FPROM_SIZE;
	eep_mfg_rec.eep_ports       = DEFAULT_PORTS;
	lp = (long *)&eep_mfg_rec.eep_model[0];
	*lp                         = DEFAULT_MODEL;

#ifdef __FDDI_CON
	eep_mfg_rec.eep_sram_size  = DEFAULT_SRAM_SIZE;
        SetEEP_FDDIDefaults();
#endif
}

check_header(NIM960_HDR *header, int print_errors)
{
	int ret;

	ret = 1;
	if (header->signature != NIM960_MAGIC)
		{
		if (print_errors)
			{
			printf("Error: Not a 960 image format\n");
			printf("signature is bad - signature is 0x%x it should be 0x%x\n",
				header->signature, NIM960_MAGIC);
			}
		ret = 0;
		}

	if (header->Entry == NULL)
		{
		if (print_errors)
			printf("Error: No program entry point\n");
		ret = 0;
		}

	if (header->ProgramId != program_id)
		{
		if (print_errors)
			{
			printf("Error: Program not for this board\n");
			printf("program id is bad - program id is 0x%X it should be 0x%X\n",
				header->ProgramId, program_id);
			}
		ret = 0;
		}

	return(ret);
}

check_code(NIM960_HDR *header, int print_errors)
{
	u_short csum;
	int ret;

	ret = check_header(header, print_errors);

	if (ret)
		{
		csum = cksum(header, header->FileLength >> 1);
		if ((csum != 0xFFFF) && (csum != 0))
			{
			if (print_errors)
				printf("Error: Image did not checksum - checksum = 0x%x\n", csum);
			ret = 0;
			}
		}

	return(ret);
}

check_reset(int called_from)
{
	int reset_level;
	int reset_cause;
	int error_code;

	reset_cause = nvr_boot_rec.nvr_reset_cause;
	reset_level = nvr_boot_rec.nvr_reset_level;

	if ((called_from == 0) || (nvr_boot_rec.nvr_reset_processed == 0))
		{
		nvr_boot_rec.nvr_last_reset_cause = reset_cause;
		nvr_boot_rec.nvr_last_reset_level = reset_level;

		nvr_boot_rec.nvr_reset_cause = RSTMSG0;
		nvr_boot_rec.nvr_reset_level = 0;

		if (called_from == 0)
			nvr_boot_rec.nvr_reset_processed = 1;
		}

	else
		{
		reset_cause = nvr_boot_rec.nvr_last_reset_cause;
		reset_level = nvr_boot_rec.nvr_last_reset_level;
		}

	if ((reset_cause == RSTMSG5) ||
	    ((reset_cause == RSTMSG1) && (reset_level <= nvr_boot_rec.nvr_max_reset_level)))
		{
		if ((reset_level == 4) && ((called_from == 0) || (nvr_boot_rec.nvr_reset_processed == 0)))
			{
			printf("Reset Level 4\n");
			reset_level_4(called_from, reset_cause);
			}

		else if ((reset_level == 3) && (called_from == 1))
			{
			printf("Reset Level 3\n");
			reset_level_2();
			}

		else if ((reset_level == 2) && (called_from == 1))
			{
			printf("Reset Level 2\n");
			reset_level_2();
			}
		}

	if ((called_from == 1) && (nvr_boot_rec.nvr_reset_processed == 1))
		nvr_boot_rec.nvr_reset_processed = 0;

	PutBootRec(&nvr_boot_rec);

	return;
}

/*
* copy info returned from bootp to my data structure and the nvram.
*/
void
CopyBootpInfo()
{
	int	i;

	_initp->in_me    = bootpinfo.yiaddr;
	_initp->net_mask = bootpinfo.NetMask;

	nvr_boot_rec.nvr_IP          = bootpinfo.yiaddr;
	nvr_boot_rec.nvr_ServerIP    = bootpinfo.siaddr;
	nvr_boot_rec.nvr_BootpServer = bootpinfo.siaddr;
	nvr_boot_rec.nvr_NetMask     = bootpinfo.NetMask;
	nvr_boot_rec.nvr_OffsetGMT   = bootpinfo.TimeOffset;
	nvr_boot_rec.nvr_TimeServer  = bootpinfo.TimeServer[0];

	memset(nvr_boot_rec.nvr_config_file, 0, sizeof(nvr_boot_rec.nvr_config_file));
	strncpy(nvr_boot_rec.nvr_config_file, bootpinfo.file, (sizeof(nvr_boot_rec.nvr_config_file) - 1));

	for (i = 0;  i < 2; i++)
		nvr_boot_rec.nvr_RouterIP[i] = bootpinfo.Router[i];

	_initp->net_gway  = bootpinfo.Router[0];
	_initp->net_gway2 = bootpinfo.Router[1];

	/*
	* update the hub and slot numbers in case the board was moved
	*/
	nvr_boot_rec.nvr_hubid  = GetHubID();
	nvr_boot_rec.nvr_slotid = GetSlotID();

	/*
	* write boot record to NVRAM
	*/
	PutBootRec(&nvr_boot_rec);

	return;
}

/*
* test an IP address to see if it is valid
*	ip must be non-zero
*	ip must not be broadcast
*	the non-network portion of the ip must be non-zero
* returns:
*	1 = valid
*	0 = not valid
*/

int
valid_ip(in_name ip, in_name netmask)
{
	return(!((ip == 0) || (ip == -1) || ((ip & ~netmask) == ~netmask)));
}

int
BootpReqSend(char *filename)
{
	in_name	myip;
	int	swap;
	char	*request_file;
	int	i;
	int	k;
	int	media_type[2];

	myip = _initp->in_me;
	_initp->in_me = 0;

	media = nvr_boot_rec.nvr_bootp_media;
	media_type[0] = media;
	if (media_type[0] == 0)
		media_type[1] = 1;
	else
		media_type[1] = 0;

	/*
	* first try the media of choice the try the adminbus
	*/
	for (k = 0; k < 2; k++)
		{
		/*
		* first send the request file name for the boot information
		* in NVRAM, if no answer is received send a null request
		* file name
		*/
		for (i = 0; i < 2; i++)
			{
			/*
			* first try requesting the request file, if no server
			* answers try requesting a null request file
			*/
			if (i == 0)
				{
				if ((filename == 0) || (filename[0] == 0))
					request_file = nvr_boot_rec.nvr_request_file;
				else
					request_file = filename;
				if (request_file[0] == 0)
					i++;
				}
			else
				request_file = "";


			printf("Sending BOOTP request for file %s",
				request_file[0] == NULL ? "(NULL)" : request_file);
			if (media_type[k] == 0)
				printf(" over network\n");
			else
				printf(" over admin bus\n");

			/*
			* send the bootp request two different ways
			* first send it the normal way
			* second send it with the hardware address bits in the
			* reverse order within the byte - this will get us
			* answer if we go through a bridge to the ethernet
			* where the bits of the address are reversed this is
			* kind of kludgy but should work the user must make
			* the entry in the bootptab with the hardware address
			* in the ethernet mode
			*/
			for (swap = 0; swap < 2; swap++)
				{
				if (udpboot(0, 0, request_file, swap, media_type[k]) != FALSE)
					{
					/*
					 *	We have a reply.
					 */
					media = media_type[k];
					nvr_boot_rec.nvr_bootp_media = media;
					printf("BOOTP reply received\n");
					CopyBootpInfo();
					UpdateAdmBusConfig();
					return(0);
					}
				}
			}
		}

	_initp->in_me = myip;
	printf("Warning: No response from BOOTP server\n");

	return(1);
}


char *inet_ntoa(in_name);
char *NidToString(byte *);
/*
 *	PrintManuRecord	- print out the manufacturer record.
 */
PrintManuRecord(EEP_MFG *MfgRec)
{

	printf("Manufacturer Data:\n");
	printf("\tS/N: %05d", MfgRec->eep_srlno);
	p_date("\tMfg date:", MfgRec->eep_mfgdt);
	p_date("\tFCO date:", MfgRec->eep_altdt);
	printf("\tRev Level: %d.%d\n\n", 
		MfgRec->eep_level >> 8, 
		MfgRec->eep_level & 0xff);
	printf("Configured Memory Sizes:\n");
	printf("\tDRAM: %d KB\tNVRAM: %d KB\tEEPROM: %d B\tFEPROM: %d KB\n",
		MfgRec->eep_dram_size / 1024,
		MfgRec->eep_nvram_size / 1024,
		MfgRec->eep_eeprom_size,
		MfgRec->eep_fprom_size / 1024);
#ifdef __FDDI_CON
	printf("\tSRAM:  %6d KB\n", MfgRec->eep_sram_size / 1024);
#endif
}

PrintMacAddress()
{
#ifdef __FDDI_CON
       PrintFddiMacAddress();
#else
	  
	  int speed;

	speed = get_token_ring_speed();

	printf("\nToken Ring Address:\t%s\tToken Ring Speed: %d megabit\n",
		NidToString((byte *)&sys.sys_nid[0]), speed ? 16 : 4);
#endif
}

p_date(str, date)
char		*str; 	/* tag */
unsigned	 date;	/* date */
{
	char *m;

	switch(date & 0x0f)
		{
	case 0: m = "Jan"; break;
	case 1: m = "Feb"; break;
	case 2: m = "Mar"; break;
	case 3: m = "Apr"; break;
	case 4: m = "May"; break;
	case 5: m = "Jun"; break;
	case 6: m = "Jul"; break;
	case 7: m = "Aug"; break;
	case 8: m = "Sep"; break;
	case 9: m = "Oct"; break;
	case 10:m = "Nov"; break;
	case 11:m = "Dec"; break;
	default:m = "###";
		}
	printf("%s %s %d", str, m, 1990 + (date >> 4));
}

get_model_number(char *number, int length)
{

	/*
	* zero fill model number string
	*/
	memset(number, 0, length);

	/*
	* move model number to the string
	*/
	if (eep_mfg_rec.eep_model[0] == 0)
		strncpy(number, "NULL", length);
	else
		{
		int len;

		/*
		* compute how much to copy
		*/
		len = sizeof(eep_mfg_rec.eep_model);
		if (len > length)
			len = length;

		memcpy(number, eep_mfg_rec.eep_model, len);
		}

	return;
}

static NVR_BOOT my_boot;

int
print_config(int error)
{
	printf("The following configuration information has been found:\n\n");
	printf("\tThis station's IP address: %s\n", inet_ntoa(nvr_boot_rec.nvr_IP));
	printf("\tThis sub-net's IP net-mask: %s\n", inet_ntoa(nvr_boot_rec.nvr_NetMask));
	printf("\tServer station's IP address: %s\n", inet_ntoa(nvr_boot_rec.nvr_ServerIP));
	if ((nvr_boot_rec.nvr_IP & nvr_boot_rec.nvr_NetMask) != 
		(nvr_boot_rec.nvr_ServerIP & nvr_boot_rec.nvr_NetMask))
		{
		/*
		* server is not on the same network 
		*/
		printf("\tIP router address is %s\n", 
			inet_ntoa(nvr_boot_rec.nvr_RouterIP[0]));
		}
	printf("\tConfiguration file to be loaded: %s\n", nvr_boot_rec.nvr_config_file);

	printf("\nDo you want to use this configuration information? ([y]/n) ");

	if (wait_for_reply(error) == 0)
		{
		/*
		* No, the user does not want to use the boot record 
		* stored in the nvram
		* We have to ask the user to enter boot record.
		*/
		memcpy(&my_boot, &nvr_boot_rec, sizeof(NVR_BOOT));
		get_config_info(&my_boot);
		nvr_boot_rec.nvr_IP       = my_boot.nvr_IP;
		nvr_boot_rec.nvr_ServerIP = my_boot.nvr_ServerIP;
		nvr_boot_rec.nvr_NetMask  = my_boot.nvr_NetMask;
		if ((my_boot.nvr_IP & my_boot.nvr_NetMask) != 
		    (my_boot.nvr_ServerIP & my_boot.nvr_NetMask))
			nvr_boot_rec.nvr_RouterIP[0] = my_boot.nvr_RouterIP[0];
		memset(nvr_boot_rec.nvr_config_file, 0, sizeof(nvr_boot_rec.nvr_config_file));
		strncpy(nvr_boot_rec.nvr_config_file, my_boot.nvr_config_file, (sizeof(nvr_boot_rec.nvr_config_file) - 1));
		/*
		 *	Save new records to nvram
		 */
		printf("Do you want to save this record in NVRAM ([y]/n) ");
		if (wait_for_reply(0) == 1)
			PutBootRec(&nvr_boot_rec);

		}

	set_seven_segment_display(get_ring_number());

	return;
}
/*
* wait for reply from console
* If there is no response after timeout ticks return
* returns:
*	0 = no
*	1 = yes
*	2 = timeout
*/
static char reply_buffer[16];
static int ireply;

int
wait_for_reply(int error)
{
	int	ticks;
	int	ch;
	int	i;
	int tens;
	int ones;
	int display_ticks;
	int display_time;
	int pause_time;
	int ret;

	display_time = timer_1_ticks_per_second;	/* one second */
	pause_time = display_time / 4;

	tens = error / 10;
	ones = error - 10 * tens;

	if (tens > 9)
		tens = 0xa;		/* indicate error */

	ticks = RealTimeTicks();
	ireply = 0;
	reply_buffer[i] = 0;
	for (;;)
		{
		if ((RealTimeTicks() - ticks) > 60 * timer_1_ticks_per_second)
			{
			putchar('\n');
			return(2);
			}

		/*
		* display 'E' for error
		*/
		ret = wait_for_input(0xe, display_time);
		if (ret >= 0)
			return(ret);

		ret = wait_for_input(0x12, pause_time);
		if (ret >= 0)
			return(ret);

		/*
		* display tens digit
		*/
		if (tens)
			{
			ret = wait_for_input(tens, display_time);
			if (ret >= 0)
				return(ret);

			ret = wait_for_input(0x12, pause_time);
			if (ret >= 0)
				return(ret);
			}

		/*
		* always display ones digit
		*/
		ret = wait_for_input(ones, display_time);
		if (ret >= 0)
			return(ret);

		ret = wait_for_input(0x12, pause_time);
		if (ret >= 0)
			return(ret);
		}
}

int
wait_for_input(int display, int display_time)
{
	int	ticks;
	int	ch;

	set_seven_segment_display(display);

	ticks = RealTimeTicks();
	while (RealTimeTicks() - ticks < display_time);
		{
		ch = getch();
		if (ch != -1)
			{
			putchar(ch);
			ch &= 0x7f;

			/*
			* if carriage return output a linefeed
			*/
			if (ch == '\r')
				putchar('\n');

			/*
			* if the character is a backspace 
			* delete last character
			*/
			if (ch == '\b')
				{
				ireply--;
				if (ireply < 0)
					ireply = 0;
				}

			/*
			* else put character in buffer
			*/
			else
				{
				if (ireply < sizeof(reply_buffer) - 1)
					reply_buffer[ireply++] = ch;
				else
					ireply = sizeof(reply_buffer) - 1;
				}

			/*
			* terminate string
			*/
			reply_buffer[ireply] = 0;

			/*
			* if character is a linefeed or a carriage return
			* process line
			*/
			if (ch == '\n' || ch == '\r')
			{
				/*
				* skip over blanks
				*/
				ireply = 0;
				while (reply_buffer[ireply] == ' ')
					ireply++;

				/*
				* look at first character only
				* answer must be specific
				*/
				ch = reply_buffer[ireply];
				if (ch == 'y' || ch == 'Y' || ch == '\n' || ch == '\r')
					return(1);
				if (ch == 'n' || ch == 'N')
					return(0);
				/*
				* start over
				*/
				printf("Answer not understood.  Please enter again. ([y]/n) ");
				ireply = 0;
				}
			}

		ReSchedule();
		}

	return(-1);
}

get_config_info(NVR_BOOT *boot)
{
	int	confirmed = 0;
	char	filename[TFTP_FILE_LENGTH];

	while (confirmed == 0)
		{
		/*
		 *	ask user to enter information
		 */
		AskIP("Enter this stations's IP address", 
			&boot->nvr_IP);
		AskIP("Enter this sub-net's IP net-mask", 
			&boot->nvr_NetMask);
		AskIP("Enter server station's IP address", 
			&boot->nvr_ServerIP);
		if ((boot->nvr_IP & boot->nvr_NetMask) != 
			(boot->nvr_ServerIP & boot->nvr_NetMask))
			{
			/*
			 *	server is not on the same network 
			 *	need to ask for the gateway's ip address
			 */
			AskIP("Enter this sub-net's IP router address",
				boot->nvr_RouterIP);
			}
		for (;;)
			{
			printf("Enter the configuration file to be loaded (%s) ==>", 
				boot->nvr_config_file[0] != '\0' ? boot->nvr_config_file : 
				"NULL");
			gets(filename);
			if (filename[0] != '\0')
				{
				memset(boot->nvr_config_file, 0, sizeof(boot->nvr_config_file));
				strncpy(boot->nvr_config_file, filename, (sizeof(boot->nvr_config_file) - 1));
				break;
				}
			else if (boot->nvr_config_file[0] != '\0')
				break;
			}
		confirmed = AskYesNo("Confirm ");
		}
}

/*----------------------------------------------------------------------------
 * UpdateLocationChange()
 * 	Save old hub and slot id in memory for SNMP svc_oldSlotId() and 
 * 	svc_oldHubId() SVCs.
 *	Replace old IDs by current IDs in NV RAM so that next location change
 *	can be detected after power up.
 *	
 *----------------------------------------------------------------------------
 */

int	LocationChangeFlag;
ushort	oldSlotID, oldHubID;
UpdateLocationChange()
{

        /*--------------------------------------------
         * Initialize old hub and slot id if not so.
         * NOTE: These two values are NOT updated in
         *       Boot code.
         *--------------------------------------------
         */
        if ( nvr_boot_rec.nvr_oldslotid == 0 ||
             nvr_boot_rec.nvr_oldhubid  == 0)
        {
                nvr_boot_rec.nvr_oldslotid = GetSlotID();
                nvr_boot_rec.nvr_oldhubid  = GetHubID();
        }

        /*------------------------------------
         * Save old hub and slot id in memory for
         * SNMP svc_oldSlotId() and svc_oldHubId()
         * SVCs.
         *------------------------------------
         */
        oldSlotID = nvr_boot_rec.nvr_oldslotid;
        oldHubID  = nvr_boot_rec.nvr_oldhubid;

        /*------------------------------------
         * Replace old IDs by current IDs in NV
         * RAM so that next location change can
         * be detected after power up.
         *------------------------------------
         */
        nvr_boot_rec.nvr_oldslotid = GetSlotID();
        nvr_boot_rec.nvr_oldhubid  = GetHubID();
        PutBootRec( &nvr_boot_rec );

        /*--------------------------------------------
         * Check if old hub and slot IDs differ from
         * current IDs and inform SNMP to send 
	 * changeLocation trap.
         *--------------------------------------------
         */
        if ( nvr_boot_rec.nvr_oldslotid != oldSlotID ||
             nvr_boot_rec.nvr_oldhubid  != oldHubID )
                LocationChangeFlag = 1;
	else
                LocationChangeFlag = 0;

}

/*----------------------------------------------------------------------------
 * UpdateAdmBusConfig()
 *	Update module info via admin bus to system card. Module info may have
 *	been changed after receiving a bootp reply.
 *	
 *----------------------------------------------------------------------------
 */

UpdateAdmBusConfig()
{

	AdmSetIP( nvr_boot_rec.nvr_IP );
	AdmSetNetMask( nvr_boot_rec.nvr_NetMask );
	AdmSetBootServer( nvr_boot_rec.nvr_BootpServer );
	AdmSetRouter( nvr_boot_rec.nvr_RouterIP[0] );
	AdmSetBootFile( nvr_boot_rec.nvr_config_file );

}
#endif
