/*--------------------------------------------------------------------------
 * File name: 	member.c
 *
 * Description: Contains routines for manipulating the BRIP member table.
 * 
 * Author: 	Shaun Hui
 * 
 * Module History:
 * 
 * who	when		what
 * -------------------------------------------------------------------------
 * lsh	Jun-29-92	Module creation.
 * lsh  Mar-10-93	Added code for FDDI CON support.
 *--------------------------------------------------------------------------
 */
#include <types.h>
#include <krnl.h>


#include <brip.h>
#include "member.h"
#ifdef __FDDI_CON
#include "nvrecs.h"
#endif
struct Ring_Member_Table RingMemberTbl[MAX_MAC_CHANNEL];

extern BRIP_INFO BripInfo[];

/*--------------------------------------------------------------------------
 * init_member_table(mac_chnl) 
 *
 * Initialize the member table specified by the mac channel and initialize
 * the entry corresponding to our transmit line.
 *--------------------------------------------------------------------------
 */
void init_ring_member_table( short mac_chnl)
{
byte	line;
byte	my_xmt_line;

	my_xmt_line = BripInfo[mac_chnl].xmt_line_nbr;

	for (line=0; line <= MAX_XMT_LINE; line++)
	{
		RingMemberTbl[mac_chnl].slot_nbr[line] = 0;
		RingMemberTbl[mac_chnl].ka_count[line] = 0;
		RingMemberTbl[mac_chnl].state[line] = 0;
		RingMemberTbl[mac_chnl].speed[line] = 0;
	}

	RingMemberTbl[mac_chnl].slot_nbr[my_xmt_line] = get_xmt_line();
	RingMemberTbl[mac_chnl].ka_count[my_xmt_line] = 0;
	RingMemberTbl[mac_chnl].ka_expire[my_xmt_line] = 0;
	RingMemberTbl[mac_chnl].state[my_xmt_line] = BRIP_INIT_STATE;
	RingMemberTbl[mac_chnl].speed[my_xmt_line] = BripInfo[mac_chnl].media_speed;


}

#ifdef SCC_DEBUG
void print_member_entry(short mac_chnl, byte line)
{
	printf( "%d  slot: %d ", line,RingMemberTbl[mac_chnl].slot_nbr[line]);
	printf( "ka_count: %d ",RingMemberTbl[mac_chnl].ka_count[line]);
	printf( "stat: %d ",RingMemberTbl[mac_chnl].state[line]);
	printf( "speed: %d \n",RingMemberTbl[mac_chnl].speed[line]);
}

void print_member_table(short mac_chnl)
{
byte line;
	for (line=0; line <= MAX_XMT_LINE; line++)
	{
		printf( "%d  slot: %d ", line,RingMemberTbl[mac_chnl].slot_nbr[line]);
		printf( "ka_count: %d ",RingMemberTbl[mac_chnl].ka_count[line]);
		printf( "stat: %d ",RingMemberTbl[mac_chnl].state[line]);
		printf( "speed: %d \n",RingMemberTbl[mac_chnl].speed[line]);
	}
}
#endif

/*--------------------------------------------------------------------------
 * add_ring_member(mac_chnl, frame_ptr ) 
 *
 * Add an entry in the member table, specified by the mac channel and
 * source line, to the INIT state.
 *--------------------------------------------------------------------------
 */
void add_ring_member(short mac_chnl, BRIP_FRAME *frame_ptr ) 
{
	RingMemberTbl[mac_chnl].slot_nbr[frame_ptr->src_line] = frame_ptr->src_slot;
	RingMemberTbl[mac_chnl].ka_count[frame_ptr->src_line] = 0;
	RingMemberTbl[mac_chnl].ka_expire[frame_ptr->src_line] = 0;
	RingMemberTbl[mac_chnl].state[frame_ptr->src_line] = BRIP_INIT_STATE;
	RingMemberTbl[mac_chnl].speed[frame_ptr->src_line] = frame_ptr->media_speed;
#ifdef SCC_DEBUG
	printf("\tAdd %d\t",frame_ptr->src_line);
#endif
}



/*--------------------------------------------------------------------------
 * update_ring_member(mac_chnl, line_nbr ) 
 *
 * Update an existing entry in the member table specified by the mac channel 
 * and source line to the INSERTED state.
 *--------------------------------------------------------------------------
 */
void update_ring_member( short mac_chnl, byte line_nbr ) 
{
	RingMemberTbl[mac_chnl].state[line_nbr] = BRIP_INSERTED_STATE;
	reset_ka_counter(mac_chnl, line_nbr);   /* jlin: 06/25/93 */
#ifdef SCC_DEBUG
	printf("\tUpd %d\t",line_nbr);
#endif
}


/*--------------------------------------------------------------------------
 * delete_ring_member(mac_chnl, line_nbr) 
 *
 * Delete an entry from the member table, specified by the mac channel and
 * source line, by zeroing out all the fields.
 *--------------------------------------------------------------------------
 */
void delete_ring_member( short mac_chnl, byte line_nbr) 
{
	RingMemberTbl[mac_chnl].slot_nbr[line_nbr] = 0;
	RingMemberTbl[mac_chnl].ka_count[line_nbr] = 0;
	RingMemberTbl[mac_chnl].ka_expire[line_nbr] = 0;
	RingMemberTbl[mac_chnl].state[line_nbr] = 0;
	RingMemberTbl[mac_chnl].speed[line_nbr] = 0;
#ifdef SCC_DEBUG
	printf("\nDel %d\t",line_nbr);
#endif
}


/*--------------------------------------------------------------------------
 * IsRingMember(mac_chnl, line_nbr) 
 *
 * Verify if an entry in the member table, specified by the mac channel and
 * source line, already exists.
 *--------------------------------------------------------------------------
 */
int IsRingMember( short mac_chnl, byte line_nbr) 
{
	if ((RingMemberTbl[mac_chnl].slot_nbr[line_nbr] != 0) &&
		(RingMemberTbl[mac_chnl].state[line_nbr] != 0))
		return TRUE;
	else
		return FALSE;
}


/*--------------------------------------------------------------------------
 * reset_ka_counter(mac_chnl, line_nbr) 
 *
 * Reset the keep alive counter to MAX_KA_COUNT for an entry in the member 
 * table, specified by the mac channel and source line. This counter is reset
 * to MAX_KA_COUNT whenever a keep alive frame is received.
 * The ka_counter is decremented each time when the keep alive timer expires.
 * When ka_counter reaches 0 the entry will be deleted from the member tbl.
 *--------------------------------------------------------------------------
 */
void reset_ka_counter(short mac_chnl, byte line_nbr) 
{
	RingMemberTbl[mac_chnl].ka_count[line_nbr] = MAX_KA_COUNT;
}


/*---------------------------------------------------------------------------
 * get_usn(mac_chnl) 
 *
 * Returns the up stream neighbour of our transmit line of the specified
 * mac channel.
 *---------------------------------------------------------------------------
 */
byte get_usn( short mac_chnl )
{
	register short i;

	/*----------------------------------------------
	 * Look for the first inserted line immediately 
	 * before us.
	 *----------------------------------------------
	 */
	for ( i=BripInfo[mac_chnl].xmt_line_nbr-1; i > 0;  i-- )
	{
		if ((RingMemberTbl[mac_chnl].slot_nbr[i] != 0) &&
			 (RingMemberTbl[mac_chnl].state[i] == BRIP_INSERTED_STATE))
			return (i);
	}

#ifdef __FDDI_CON
	/*----------------------------------------------
	 * For FDDI concentrator which currently supports
	 * PEER-TO-PEER configuration only, if no up 
	 * stream card, connect receive line to  the
	 * Quiet Xmt line (line 24). 
	 *----------------------------------------------
	 */

	if (nvr_fddi_rec.UI_Defaults.Port1_PC_Type != PC_Type_M)
	{
	   return ( XMT_QUIET_LINE );
	}
#endif

	/*----------------------------------------------
	 * If no lines above us, we must be first. Look 
	 * for the biggest inserted line.
	 *----------------------------------------------
	 */
	for ( i=MAX_XMT_LINE; i > BripInfo[mac_chnl].xmt_line_nbr; i--)
	{
		if ((RingMemberTbl[mac_chnl].slot_nbr[i] != 0) &&
			 (RingMemberTbl[mac_chnl].state[i] == BRIP_INSERTED_STATE))
			return (i);
	}

	/*----------------------------------------------
	 * No lines are found. We must be the only line.
	 *----------------------------------------------
	 */
	return (BripInfo[mac_chnl].xmt_line_nbr);
}


/*---------------------------------------------------------------------------
 * get_dsn(mac_chnl) 
 *
 * Returns the down stream neighbour of our transmit line of the specified
 * mac channel.
 *---------------------------------------------------------------------------
 */
byte get_dsn( short mac_chnl )
{
	register short i;

	/*----------------------------------------------
	 * Look for the first inserted line immediately 
	 * follow us.
	 *----------------------------------------------
	 */
	for ( i=BripInfo[mac_chnl].xmt_line_nbr+1; i <= MAX_XMT_LINE;  i++ )
	{
		if ((RingMemberTbl[mac_chnl].slot_nbr[i] != 0) &&
			 (RingMemberTbl[mac_chnl].state[i] == BRIP_INSERTED_STATE))
			return (i);
	}

#ifdef __FDDI_CON
	/*----------------------------------------------
	 * For FDDI concentrator which currently supports
	 * PEER-TO-PEER configuration only, if no down
	 * stream card, connect receive line to  the
	 * Quiet Xmt line (line 24). 
	 *----------------------------------------------
	 */

	if (nvr_fddi_rec.UI_Defaults.Port6_PC_Type != PC_Type_M)
	{
	   return ( XMT_QUIET_LINE );
	}
#endif	

	/*----------------------------------------------
	 * If no lines follow us, we must be last. Look 
	 * for the smallest inserted line.
	 *----------------------------------------------
	 */
	for ( i=1; i < BripInfo[mac_chnl].xmt_line_nbr; i++ )
	{
		if ((RingMemberTbl[mac_chnl].slot_nbr[i] != 0) &&
			 (RingMemberTbl[mac_chnl].state[i] == BRIP_INSERTED_STATE))
			return (i);
	}

	/*----------------------------------------------
	 * No lines are found. We must be the only line.
	 *----------------------------------------------
	 */
	return (BripInfo[mac_chnl].xmt_line_nbr);
}

/*--------------------------------------------------------------------------
 * IsStateChanged(mac_chnl, line_nbr, new_state ) 
 *
 * Verify if an entry in the member table, specified by the mac channel and
 * source line, has changed state.
 *--------------------------------------------------------------------------
 */
int IsStateChanged( short mac_chnl, byte line_nbr, byte new_state ) 
{
		if (RingMemberTbl[mac_chnl].state[line_nbr] != new_state)
			return TRUE;
		else
			return FALSE;
}

/*--------------------------------------------------------------------------
 * IsSameSpeed(mac_chnl)
 *
 * Verify if all entries in the member table, specified by the mac channel 
 * have the same speed.
 *
 * The following algorithum is used:-
 * 	If an entry contains an inserted card of the same speed, we are ok.
 *		If all entries are initializing, 
 *			check no. of entries that have the same speed and different speed.
 *			If there are more entries of the same speed than different speed, 
 *			we are ok.
 *			If the no. of entries of the same speed is the same or less then
 *			that of the different speed, not ok.
 *--------------------------------------------------------------------------
 */
int IsSameSpeed(short mac_chnl)
{


byte	line;

	/*------------------------------------
	 * If an inserted entry has the same 
	 * speed, we are ok, else we have speed
	 * conflict.
	 *------------------------------------
	 */
	for (line=1; line <= MAX_XMT_LINE; line++)
	{
		if (RingMemberTbl[mac_chnl].state[line]==BRIP_INSERTED_STATE)
		{
			if (RingMemberTbl[mac_chnl].speed[line]==
				BripInfo[mac_chnl].media_speed)
				return TRUE;
			else
				return FALSE;
		}
	}

	/*------------------------------------
	 * There is no inserted entry. Probably
	 * every card is initializing. 
	 * Then 16 Mbps TR speed has higher 
	 * priority over 4 Mbps TR speed.
	 * For FDDI devices there is no speed
	 * difference.
	 *------------------------------------
	 */

	for (line=1; line <= MAX_XMT_LINE; line++)
	{
		if (RingMemberTbl[mac_chnl].state[line] == BRIP_INIT_STATE )
		{
			if ((RingMemberTbl[mac_chnl].speed[line] == TR_16M ) && 
			    (BripInfo[mac_chnl].media_speed != TR_16M ))
				return FALSE;
		}
	}

	return TRUE;

}

word get_rsp_bmap( short mac_chnl )
{

byte	line;
word	bitmap;

	for( bitmap=0, line=1; line <= MAX_XMT_LINE; line++)
	{
		if ( RingMemberTbl[mac_chnl].slot_nbr[line] != 0 )
		{
			bitmap |= 0x01 << line;
		}
	}
	return (bitmap);
}


/*---------------------------------------------------------------------------
 * get_usn_slot(mac_chnl) 
 *
 * Returns the slot (card) number of up stream neighbour of the specified
 *  mac channel.
 *---------------------------------------------------------------------------
 */
byte get_usn_slot( short mac_chnl )
{

return (RingMemberTbl[mac_chnl].slot_nbr[ BripInfo[mac_chnl].usn_line_nbr ] );

}

/*---------------------------------------------------------------------------
 * get_dsn_slot(mac_chnl) 
 *
 * Returns the slot (card) number of down stream neighbour of the specified
 *  mac channel.
 *---------------------------------------------------------------------------
 */
byte get_dsn_slot( short mac_chnl )
{

return (RingMemberTbl[mac_chnl].slot_nbr[ BripInfo[mac_chnl].dsn_line_nbr ] );

}
