/*--------------------------------------------------------------------------
 * File name: 	scc_buff.c
 *
 * Description: Contains buffer management routines to maintain scc receive
 * 		and transmit buffer descriptor queues.
 * 
 * Author: 	Shaun Hui
 * 
 * Module History:
 * 
 * who	when		what
 * -------------------------------------------------------------------------
 * lsh	Aug-03-92	Module creation.
 *
 *--------------------------------------------------------------------------
 */
#include <types.h>
#include <defines.h>
#include <memory.h>
#include <uart.h>
#include "scc_buff.h"

byte scc_tx_buff[MAX_SCC_TX_BUFFER];
byte scc_rx_buff[MAX_SCC_RX_BUFFER];

byte *TxBuffPtr = &scc_tx_buff[0];
byte *TxDMASrcPtr = &scc_tx_buff[0];
byte *TxDMADstPtr= (byte *)(SCC_PORTA)+8;
byte *RxDMASrcPtr= (byte *)(SCC_PORTA)+8;
byte *RxDMADstPtr= &scc_rx_buff[0];
byte *rcv_dma_ptr= &scc_rx_buff[0];
byte *last_dma_ptr= &scc_rx_buff[MAX_SCC_RX_BUFFER - 1];

int TxByteCount;

SCC_BD		scc_bd[MAX_SCC_BD];

/*--------------------------------------------------------------------------
 * Scc buffer descriptor queue.
 *--------------------------------------------------------------------------
 */
SCC_BD_QUEUE	scc_bd_q;

/*--------------------------------------------------------------------------
 * set_xmt_dma()
 * set transmit dma (channel 1)
 *--------------------------------------------------------------------------
 */
void set_xmt_dma( )
{
	asm( "clrbit	1, sf2, sf2" );
	asm( "ld	_TxByteCount , g4" );
	asm( "ld	_TxDMASrcPtr, g5" );
	asm( "ld	_TxDMADstPtr, g6" );
	asm( "ldconst	0x1d0, g3");

	asm( "sdma	1, g3, g4" );
	asm( "setbit	1, sf2, sf2" );
}


/*--------------------------------------------------------------------------
 * set_rcv_dma()
 * set receive dma (channel 0)
 *--------------------------------------------------------------------------
 */
void set_rcv_dma()
{

	asm( "clrbit	0, sf2, sf2" );
	asm( "ldconst	0x1000, g4" );
	asm( "ld	_RxDMASrcPtr, g5" );
	asm( "ld	_RxDMADstPtr, g6" );
	asm( "ldconst	0x1a0, g3");

	asm( "sdma	0, g3, g4" );
	asm( "setbit	0, sf2, sf2" );
}

/*--------------------------------------------------------------------------
 * init_scc_bd() 
 * 	Initialize the scc buffer descriptors and the scc_bd_q head and tail
 *	pointers.
 *--------------------------------------------------------------------------
 */
void init_scc_bd()
{
	int	i;

	scc_bd_q.head = &scc_bd[0];
	scc_bd_q.tail = &scc_bd[MAX_SCC_BD -1];

	for (i=0; i < MAX_SCC_BD; i++)
	{
		scc_bd[i].buff_ptr	= NULL;
		scc_bd[i].length	= 0;
		scc_bd[i].bd_type	= BD_FREE;
		if ( i < MAX_SCC_BD - 1)
			scc_bd[i].next 	= &scc_bd[i+1];
		else
			scc_bd[i].next	= NULL;
			
	}
}


/*--------------------------------------------------------------------------
 * get_scc_bd( int req_type ) 
 * 	Removes a scc BD from the head of scc_bd_q indicated by the
 *	pointer scc_bd_q.head.
 *	If request is from scc_send (req_type == BD_XMT), disable interrupts.
 *	When called from scc_receive, interrupts have already been disabled.
 * 	Returns pointer to SCC_BD or NULL.
 *--------------------------------------------------------------------------
 */
SCC_BD *get_scc_bd( int req_type )
{
	SCC_BD *bd_ptr;
	int prevalue;

	if (req_type == BD_XMT)
	        prevalue = crit_on();
/*		Di(); */

	if ((bd_ptr = scc_bd_q.head) != scc_bd_q.tail )
	{
		if ( bd_ptr->next != NULL )
		{
			scc_bd_q.head	= bd_ptr->next;
			bd_ptr->next	= NULL;
		}
		else
			bd_ptr = NULL;
	}
	else
	{
		bd_ptr = NULL;
	}

	if (req_type == BD_XMT)
	        crit_off(prevalue);
/*		Ei(); */
	return (bd_ptr);
}

/*--------------------------------------------------------------------------
 * free_scc_bd() 
 * 	Returns a scc BD to the tail of scc_bd_q indicated by the
 *	pointer scc_bd_q.tail.
 *--------------------------------------------------------------------------
 */
void free_scc_bd( SCC_BD *bd_ptr )
{
        int prevalue;

	if (bd_ptr != NULL )
	{
	        prevalue = crit_on();
/*		Di(); */
		scc_bd_q.tail->next = bd_ptr;
		scc_bd_q.tail 		= bd_ptr;
		bd_ptr->next 		= NULL;
		bd_ptr->buff_ptr	= NULL;
		bd_ptr->length		= 0;
		bd_ptr->bd_type		= BD_FREE;
		crit_off(prevalue);
/*		Ei(); */
	}
}


/*--------------------------------------------------------------------------
 * get_scc_tx_buff() 
 * 	Returns a scc BD to the tail of scc_bd_q indicated by the
 *	pointer scc_bd_q.tail.
 *--------------------------------------------------------------------------
 */
byte *get_scc_tx_buff( int req_buff_size )
{
byte *tx_buff_ptr;

	if ( TxBuffPtr + req_buff_size > &scc_tx_buff[MAX_SCC_TX_BUFFER-1] )
	{
		TxBuffPtr = &scc_tx_buff[0];
		tx_buff_ptr = TxBuffPtr;
	}
	else
	{
		tx_buff_ptr = TxBuffPtr;
		TxBuffPtr += req_buff_size;
	} 
	return (tx_buff_ptr);
}
