/* admdrv.h - Administration Bus low-level device driver definitions */
/*----------------------------------------------------------------

   Hughes LAN System.
   Mountain View, CA.

   CopyRight(c) 1991.
   All Rights Reserved

   Author:  Pedro E. Rangel

   Rev    Date    Who   Comments
  -----  -------- ---   ------------------------------------------------
  00.00  08/06/91 PER   Initial software version.
  ----------------------------------------------------------------
  00.01	 10/31/91 Kwok	Adding console i/o buffer for virtual console 
			support.
			Adding Immediate-command support for the 
			FUNCARD.  The FUNCARD can issue immediate-command
			to the SYSCARD.
*/
#ifndef ADM_H
#define ADM_H
#include <types.h>

#if defined(KLHR) || defined (STTG)
#define FAR	far
#endif


#define	ADM_VERSION	1	/* this is version 1	*/
#define	ADMBUFF		16

#define	TIMEOUT		300

#define	ADM_INACTIVE	1
#define	ADM_INITREQ	2
#define	ADM_INITIALIZED	3
#define	ADM_ACTIVE	4

#define SHARED_SIZE	0x8000

#define BUFFSIZE	640
#define	ADM_CONMAP_SIZE		16
#define	ADM_BOOTFILE_SIZE	64
#define	ADM_DEVICENAME_SIZE	32
#define	ADM_DEVICEDESCR_SIZE	80
#define	ADM_COMMUNITY_SIZE	32
#define	ADM_SNMP_SECURITY_SIZE	96


#define	ADMCONB_SIZE	0x0200
#define	ADMCONB_FROM_OFFSET	0x1a00	/* from-FUNCARD offset	*/
/* to-FUNCARD offset	*/
#define	ADMCONB_TO_OFFSET	(ADMCONB_FROM_OFFSET + ADMCONB_SIZE)

#define	CMD_RSP_BUF_SIZE	0x80
#define CMDTOBUFOFF	0x1E00
#define CMDFROMBUFOFF	(CMDTOBUFOFF + CMD_RSP_BUF_SIZE)
#define	RSPFROMBUFOFF	(CMDFROMBUFOFF + CMD_RSP_BUF_SIZE)
#define	RSPTOBUFOFF	(RSPFROMBUFOFF + CMD_RSP_BUF_SIZE)
#define TXBUFFOFF	0x2000
#define RXBUFFOFF	0x4800


/*----------------------------------------------------------------
	Administration Bus Memory Layout.
  ----------------------------------------------------------------*/
typedef struct {
	short	shmInitialized;		/* non-zero if memory has been initialized */
	short	shmInitRequest;		/* non-zero to request re-initialization */
	short	shmState;		/* Card state */
#define	ADM_STATE_OPERATIONAL	1
#define	ADM_STATE_RAM_FAIL	2
#define	ADM_STATE_MVRAM_FAIL	3
#define	ADM_STATE_FLASH_FAIL	4
#define	ADM_STATE_EEPROM_FAIL	5
#define	ADM_STATE_BOOTP_FAIL	6
#define	ADM_STATE_TFTP_FAIL	7
#define	ADM_STATE_CHECKSUM_FAIL	8
	short	shmToBufFill;		/* fill state of to-FUNCARD buffers */
	short	shmFromBufAck;		/* ack state of from-FUNCARD buffers */
	short	shmToBufAck;		/* ack state of to-FUNCARD buffers */
	short	shmFromBufFill;		/* fill state of from-FUNCARD buffers */
	short	shmMaxBuffers;		/* number of buffers in each direction */
	short	shmToBuffers[ADMBUFF];	/* offsets of to-FUNCARD buffers */
	short	shmFromBuffers[ADMBUFF];/* offsets form-FUNCARD buffers */
	short	shmToImmedCmdAct;	/* non-zero if to-FUNCARD immediate-command is active */
	short	shmToImmedCmdOff;	/* offset of to-FUNCARD immediate-command area */
	short	shmToImmedRspOff;	/* offset of to-FUNCARD immediate-response area */
	short	shmFromImmedCmdAct;	/* non-zero if from-FUNCARD immediate-command is active */
	short	shmFromImmedCmdOff;	/* offset of from-FUNCARD immediate-command area */
	short	shmFromImmedRspOff;	/* offset of from-FUNCARD immediate-response area */
	short	shmImmedSize;		/* size of immediate-command area */
	short	shmConToBufOffset;	/* to-FUNCARD Console buffer offset	*/
	short	shmConFromBufOffset;	/* from FUNCARD console buffer offset	*/
	short	shmConBufSize;		/* size of the console buffer		*/
	unsigned char	shmPhysicalAddress[6];	/* Slot Physical address */
	short	shmReserved0;		/* padding		*/
	in_name shmIpAddress;		/* My Ip address	*/
	in_name shmRouterAddress;	/* My primary router address */
	in_name	shmNetMask;		/* My net maske		*/
	in_name shmBootServer;		/* My boot server ip	*/
	char	shmConMap[ADM_CONMAP_SIZE];	/* FUNCARD connectivity map */
	char	shmBootFile[ADM_BOOTFILE_SIZE];	/* name of the boot file */
	char	shmDeviceName[ADM_DEVICENAME_SIZE];	/* Card name */
	char	shmDeviceDescription[ADM_DEVICEDESCR_SIZE];/* Card Description */
	short	shmVersion;		/* this is version 1	*/
	short	shmAliveCounter;	/* increment by the function periodically */
	short	shmPower;	/* typical power consumption in Watts */
	short	shmSnmpAuth;	/* snmp auth. meth. 1 for none and 4 for community */
	char	shmSnmpCommunity[ADM_COMMUNITY_SIZE];	/* community name	*/
	char	shmSnmpSecurity[ADM_SNMP_SECURITY_SIZE]; /* for future usage */

}ADMIN;


typedef struct
	{
	unsigned long	AdmConsoleRcvdOctets;	/* hlsAdminBus 1	*/
	unsigned long	AdmConsoleSentOctets;	/* hlsAdminBus 2	*/
	unsigned long	AdmTimeouts;		/* hlsAdminBus 3	*/
	unsigned long	AdmInits;		/* hlsAdminBus 4	*/
	unsigned long	AdmRcvdICmds;		/* hlsAdminBus 5	*/
	unsigned long	AdmIssuedICmds;		/* hlsAdminBus 6	*/
	unsigned long	AdmUnknownICmds;	/* hlsAdminBus 7	*/
	unsigned long	AdmRelayerRefuseds;	/* hlsAdminBus 8	*/
	unsigned long	AdmRelayerAccepteds;	/* hlsAdminBus 9	*/
	}
	ADMIN_ENTRY;


typedef struct {
	int TimeOutCount;				/* current time out count */
#define	DEAD	0
#define	ALIVE	1					/* board is functioning */
#define	TFIRST	2					/* first keep alive failure */
#define	TSECOND	3					/* second keep alive failure */
#define	TTHIRD	4					/* card is not responding take out */
	short Counter;					/* Last Keep alive count */
	long Ticks;					/* ticks to check the keep alive counter */
#define KEEP_TICKS	5				/* number of ticks */
	unsigned long UpTime;				/* Up time-stamp */
	unsigned long DownTime;				/* Down time-stamp */
	unsigned long LastConnect;			/* Open Virtual Connection time-stamp */
	unsigned long LastDisconnect;			/* Close Virtual time-stamp */
}ADM_SLOT;

#define	BUFFIP		0x01		/* IP packet */
#define	BUFFUDP		0x02		/* an UDP packet */
#define	BUFFCONS	0x03		/* Console I/O line */

#define	MAXPROT		BUFFCONS
#define	MINPROT		BUFFIP
/*----------------------------------------------------------------
	Buffers Format
  ----------------------------------------------------------------*/
typedef struct {
	short	bufFormat;		/* format of the buffer */
	short	bufLength;		/* actual count including this struct */
	char	bufData[1];		/* data formated accordding to buffFormat */
}ADMB;
/*----------------------------------------------------------------
	Console buffer data format.
  ----------------------------------------------------------------*/
typedef struct {
	short 	length;			/* number of bytes in the line */
	short 	cReserved;		/* reserved must be 0 */
	long  	cookie;			/* the cookie value */
	char	line[1];		/* where the line starts */
}ADMCB;
/*----------------------------------------------------------------
	Immediate Command Message Format.
  ----------------------------------------------------------------*/
typedef struct _adm_icmd {
	short	iCmdType;		/* Immediate command type */
	short	iCmdLength;		/* actual count including this struct */
	union	{
		in_name iCmdIpAddr;	/* System card IP address */
		}iCmdData;		/* Immediate commands data */
}ADM_ICMD;

#define	ICMDRELAYON	0x01	/* be a relayer */
#define ICMDRELAYOFF	0x02	/* don't be a relayer anymore */
#define ICMDCHANGEIP	0x03	/* My IP address has been changed */
#define	ICMD_CONS_OPEN	0x04	/* open the console, can only be issued by
				 * the SYSTEM card */
#define ICMD_CONS_CLOSE	0x05	/* close the console, can be issued by either
			 	   the SYSTEM card or the FUNCARD	*/
#define ICMD_FUNCARD_RESET	0x06	/* This is issued from a function card
					 * to the system card. It informs the
					 * system card that I am going to 
					 * reset myself.
					 */
#define	ICMD_CHANGE_NETMASK	0x07	/* My netmask has been changed	*/
#define	ICMDMAX		ICMD_CHANGE_NETMASK	/* maximum immediate command value */
#define ICMDMIN		ICMDRELAYON

/*----------------------------------------------------------------
	Immediate Response Message Format.
  ----------------------------------------------------------------*/
typedef struct _adm_irsp {
	short	iRspType;		/* type of immediate command */
	short	iRspLength;		/* actual count including this struct */
	union {
		long	iRspReason;	/* immediate response reason */
	}iRspData;			/* immediate response data */

}ADM_IRSP;

#define IRSPACK		0x01		/* positive acknowledgement */
#define IRSPNACK	0x02		/* negative acknowledgement */

#define	IRSPNAKDOWN	0x01		/* network interface down */
#define IRSPNAKNOIP	0x02		/* doesn't yet have an IP address */
#define IRSPNOTRELAYER	0x03		/* card is not the relayer card */
#define	IRSPNAK_BUSY	0x04	/* The local console of the FUNCARD is busy.
				 * Connection refused. */
#define	IRSPNAK_NOTOPEN	0x05	/* The adm console to the FUNCARD is NOT
				 * opened. Any console operation on 
				 * this FUNCARD will be ignored.
				 */
#define IRSPNAK_NOT_SUPPORTED	0x06
/*
 *
 *	The Console input/output command area format:
 */

typedef struct
	{
	short	length;		/* size of the circular buffer	*/
	short	NextGetc;	/* 
				 * index to the circular buffer to
				 * get the next character.  The recipient
				 * updates it.
				 */
	short	NextPutc;	/* 
				 * index to the circular buffer to
				 * put the next character. The sender 
				 * updates it. 
				 */
	short	reserved;	/* for later use	*/
	char	CirBuffer[1];	/* the circular buffer, the actual
				 * size is specified in "length" 
				 */
	}
	ADMCONB;

/*
 *	Don't forget to adjust CIRBUFFER_SIZE if we add any
 *	field in the ADMCONB!!
 *	sizeof all the control data declared before the circular
 *	buffer "CirBuffer"
 *	i.e. sizeof(length) + sizeof(NextGetc) + sizeof(NextPutc)
 *		+ sizeof(reserved).
 */
#define	ADMCONB_UNPADDED_CTRL_SIZE	(4 * sizeof(short))
#define	CIRBUFFER_SIZE	256

/*
 *	offset of the NextGetc and NextPutc from the ADMCONB
 */
#define	ADM_CONB_GETC_OFF	(1 * sizeof(short)) /* length	*/
#define	ADM_CONB_PUTC_OFF	(2 * sizeof(short)) /* length + NextGetc */

/*
 *	console state
 */
#define	ADM_CON_OPENED	1
#define	ADM_CON_CLOSED	0



/*----------------------------------------------------------------
	This table is to be included in a file that may be used by
	the driver and the SNMP agent software.
  ----------------------------------------------------------------*/
typedef struct _slotconf {
	int slotIndex;			/* index to the slot config table */
	int slotDevice;			/* Adapter type */
	char slotPhysicalAddress[6];	/* Slot Physical address */
	in_name slotIpAddress;
	in_name slotRouterAddress;
	in_name	slotNetMask;
	in_name slotBootServer;
	int slotReset;			/* Enable/disable card */
	int slotState;			/* Card state */
	char slotConMap[ADM_CONMAP_SIZE];/* FUNCARD connectivity map */
	char slotBootFile[ADM_BOOTFILE_SIZE];/* name of the boot file */
	char slotDeviceName[ADM_DEVICENAME_SIZE];/* Card name */
	char slotDeviceDescription[ADM_DEVICEDESCR_SIZE];/* Card Description */
	short	slotPower;	/* typical power consumption in Watts */
	short	slotSnmpAuth;	/* snmp auth. meth. 1 for none and 4 for community */
	char	slotSnmpCommunity[ADM_COMMUNITY_SIZE];	/* community name	*/
	char	slotSnmpSecurity[ADM_SNMP_SECURITY_SIZE]; /* for future usage */
	unsigned long slotStartTime;	/* time-stamp */
	unsigned long slotDownTime;		/* time-stamp */
	unsigned long slotLastConnect;	/* time-stamp */
	unsigned long slotLastDisconnect;/* time-stamp */
}SlotConf;

#define	ADMSIZE		sizeof(ADMIN)

#ifdef SYSCARD
#define admbase		0x2B8		/* ADMIN Bus base register */
#define	ADMCMD_OFF	0x00		/* command register offset */
#define ADMDATA		0x02		/* DATA register */
#define ADMADDR_LO	0x04		/* address select register low */
#define ADMADDR_HI	0x06		/* address select register hi */

#define	CARDTYPE	0x10		/* get card type */
#define	CARDRESET	CARDTYPE	/* reset card */
#define CARDINT		CARDTYPE	/* interrupt card signal */
#define NOCARD		0xFF

#define	DISCARD		0x01		/* disable card */
#define ENACARD		0x00		/* enable card */
#define INTCARD		0x02		/* interrupt card */

#define DATAXFER	0x20		/* DATA command */
#define MAXSLOT14	14		/* 14-slot chassis */
#define MAXSLOT5	5		/* 5-slot chassis */
#define MAXSLOT		MAXSLOT14	/* always the maximun slot number */
#endif

#ifdef BASE10T
#define	BUSWIDTH	2		/* even bytes only for the 10Base-T */

#elif defined(SYSCARD)
#define	BUSWIDTH	1		/* default value for ADM bus witdh */
#define ADMADDR		(char FAR *) 0	/* address of ADM memory */
#endif
/*
 *	error code returned from functions
 */
#define	E_SUCCESS	0
#define	E_FAILURE	-1		/* any type of error */


#define	ADM_WRITE(s, b, c, val)	(b) = (val); adm_write((s), (int)((char *)&(b) - (char *)(&c)), sizeof(b), (char FAR *)&(b))
#define ADM_READ(s, b, c)	adm_read((s), (int)((char *)&(b) - (char *)(&c)), sizeof(b), (char FAR *)&((b)))

extern  int FAR adm_ip_rcv(int slot,int offset,int length);
extern  int adm_ip_send(void *p,int len,unsigned long host,void *np);
extern  int AdmConOpen(int slot);
extern  int AdmConClose(int slot);
extern  int AdmGetch(int slot);
extern  int AdmPutch(int slot,int c);
extern  int AdmPuts(int slot,char *s);
extern  int AdmGets(int slot,char *buffer);
extern  int IsAdmConOpened(int slot);
extern  int adm_send(int slot,char FAR *buff,unsigned int len);
extern  int FAR iRspCmdHandler(int slot, int Cmd, int CmdOffset, int RspOffset);
extern  int FAR DummyHandler(void );
extern  int adm_add_ImmCmdRsp(int RspCmdID,int (FAR *RspCmdHandler)());
extern  int adm_del_ImmCmdRsp(int RspCmdID);
extern  int adm_add_protocol(int ProtocolID,int (FAR *ProtocolHandler)());
extern  int adm_del_protocol(int ProtocolID);
extern  int CheckRxBufs(int slot,unsigned int FillOff,unsigned int AckOff,unsigned int BufsOff);
extern  int CheckTxCompletion(int slot,unsigned int FillOff,unsigned int AckOff);
extern  int GetFreeBuffer(unsigned int AckBuffList,unsigned int TxBuffList);
extern  int AdmImmedCommand(int slot,int CmdType,void FAR *CmdData,int DataLength);
extern  void AdmClearCounters(void );
extern  int IsValidSlot(int slot);
extern  int get_devicetype(int slot);
extern  int adm_read(int slot,unsigned int offset,unsigned int length,char FAR *BuffPtr);
extern  int adm_write(int slot,unsigned int offset,unsigned int length,char FAR *BuffPtr);
extern  int AmISysCard(void );
extern  int adm_memset(unsigned int offset,int data,unsigned int length);
extern  int FAR ImmCommandHandler(int slot,int CmdCode,
		int Cmdoffset, int RspOffset);

#ifndef SYSCARD
extern  void AdmOpenRequestInit(int slot);
extern  void AdmCloseRequestCleanUp(int slot);
extern  int adm_init(char FAR *StartAdd,int max_prot,int max_cmd);
extern  int Initialize(void );
extern  int AdmSetState(int state);
extern  int AdmSetNid(NID FAR *nid);
extern  int AdmSetIP(in_name ip);
extern  int AdmSetRouter(in_name ip);
extern  int AdmSetNetMask(in_name netmask);
extern  int AdmSetBootServer(in_name ip);
extern  int AdmSetConMap(int index,int value);
extern  int AdmSetBootFile(char FAR *filename);
extern  int AdmUpdateAliveCounter(void );
extern  int AdmSetSnmpAuth(int auth);
extern  int AdmSetCommunity(char FAR *community);
extern  int adm_poll(int slot);
extern  int adm_cmd(int slot,struct _adm_icmd *CmdPtr,int length,struct _adm_irsp *RspBuffer);
extern  int AdmFuncardReset(void );
extern  int AdmChangeIP(in_name ip);
extern  int AdmChangeNetMask(in_name NetMask);
#ifdef BASE10T
extern  void FAR _interrupt adm_irss(void );
#endif

#else
extern  void cdecl FAR _interrupt adm_isrs(void );
extern  int adm_init(int max_prot,int max_cmd);
extern  int adm_cmd(int slot,struct _adm_icmd *CmdPtr,int length,struct _adm_irsp *RspBuffer);
extern  int adm_poll(int slot);
extern  int AdmCloseRequest(int slot);
extern  int SelectRelayer(int Relayer);
extern  int BuildChassis(int *chassis);
extern  int IsSameNet(unsigned long IPaddress,unsigned long netmask,struct _slotconf FAR *Slotptr);
extern  int IsValidRelayer(int slot,unsigned long IPaddress,unsigned long netmask,struct _slotconf FAR *Slotptr);
extern  int SearchRelayer(void );
extern  int adm_FindRelayer(unsigned long IPaddress,unsigned long netmask,struct _slotconf FAR *Slotptr);
extern  int GetCard(int fromslot,int *chassis,int card);
extern  int GetOther(int fromslot,int *chassis);
extern  int BeMyRelayer(int slot);
extern  int adm_ChangeRelayer(int slot,struct _slotconf FAR *SlotPtr);
extern  int adm_SetMyIP(unsigned long ip);
extern  int adm_interrupt(int slot);
extern  int adm_reset(int slot);
extern  int adm_disable(int slot);
extern  int adm_config(int slot,struct _slotconf FAR *confptr);
extern  int GetMaxSlot(void );
#endif

extern	ADMIN_ENTRY	AdminCounters;
extern	int AdmMaxProtocols;
extern	int AdmMaxCommands;
extern	int AdmConState[];

#ifdef SYSCARD
extern ADMIN AdmPtr[MAXSLOT];
extern ADM_SLOT AdmSlotState[MAXSLOT];
extern	int (FAR *FAR *ProtocolTable)();
extern	int (FAR *FAR *ImmRspTable)();
#else
extern ADMIN AdmPtr;
extern	int (FAR **ProtocolTable)();
extern	int (FAR **ImmRspTable)();
#endif

#endif
