/******************************************************************
 * This file contains the structure definitions for
 * the National Semi SONIC ethernet controller.
 *
 * The following prefix scheme was used for naming structures
 *   snc_    is a register
 *   sncp_   is a receive pool (RRA)
 *   sncr_   is a receive descriptor (RDA)
 *   snct_   is a transmit descriptor (TDA)
 *   sncc_   are cam descriptor structures
 *   sncb_   is an actual buffer area
 *   structure fields are prefix plus root.
 *   bit fields are prefix plus root plus suffix.
 *
 * Only the least significant 16 bits are usable in all structures.
 * The most significant 16 bits are defined and ignored.
 ******************************************************************/
#ifndef SONIC
#define SONIC 0x100a	/* V1.00a  */

/*
 * This structure defines the command/status registers through
 * which the CPU and SONIC communicate.  
 */
typedef struct sonic_regs
	{
 	shrt	snc_cr; 	/* 00: command register  */
	shrt	snc_dcr;	/* 01: data configuration register  */
	shrt	snc_rcr;	/* 02: receiver control register  */
	shrt	snc_tcr;	/* 03: transmit control register  */
	shrt	snc_imr;	/* 04: interrupt mask register  */
	shrt	snc_isr;	/* 05: interrupt status register  */
	shrt	snc_utda;	/* 06: upper xmt descripter address register  */
	shrt	snc_ctda;	/* 07: current xmt descriptor addr register  */
	shrt	intrn1[5]; 	/* 08-0C internal registers  */
	shrt	snc_urda;	/* 0D: upper rcv descriptor address register  */
	shrt	snc_crda;	/* 0E: current rcv descriptor addr register  */
	shrt	intrn2[4]; 	/* 0F-12 internal registers  */
	shrt	snc_eobc;	/* 13: end of buffer word counter register  */
	shrt	snc_urra;	/* 14: upper receive resource addr register  */
	shrt	snc_rsa;	/* 15: receive resource start addr register  */
	shrt	snc_rea;	/* 16: receive resource end address register  */
	shrt	snc_rrp;	/* 17: receive resource read pntr register  */
	shrt	snc_rwp;	/* 18: receive resource write pntr register  */
	shrt	intrn3[8]; 	/* 19-20 internal registers  */
	shrt	snc_cep;	/* 21: cam entry pointer register  */
	shrt	snc_cap2;	/* 22: cam address port 2 register (rd only)  */
	shrt	snc_cap1;	/* 23: cam address port 1 register (rd only)  */
	shrt	snc_cap0;	/* 24: cam address port 0 register (rd only)  */
	shrt	snc_ce; 	/* 25: cam enable register  */
	shrt	snc_cdp;	/* 26: cam descriptor pointer register  */
	shrt	snc_cdc;	/* 27: cam descriptor count register  */
	shrt	snc_sr; 	/* 28: silicon revision register (rd only)  */
	shrt	snc_wt0;	/* 29: watch dog tmr lower 16 bits register  */
	shrt	snc_wt1;	/* 2A: watch dog tmr upper 16 bits register  */
	shrt	snc_rsc;	/* 2B: receive sequence counter register  */
	shrt	snc_crct;	/* 2C: CRC error tally register  */
	shrt	snc_faet;	/* 2D: alignment error tally register  */
	shrt	snc_mpt;	/* 2E: missed packet error tally register  */
	shrt	intrn4[16];	/* 2F-3F internal registers  */
	shrt    snc_dcr2;       /* 3F: new data configuration register */
	} SNC;

/* fields of snc_cr, the command register  */
#define SNC_CR_LCAM 	0x0200	/* load CAM command  */
#define SNC_CR_RRRA 	0x0100	/* read rra command  */
#define SNC_CR_RST  	0x0080	/* software reset command  */
#define SNC_CR_ST   	0x0020	/* start timer command  */
#define SNC_CR_STP  	0x0010	/* stop timer command  */
#define SNC_CR_RXEN 	0x0008	/* enable receiver command  */
#define SNC_CR_RXDIS	0x0004	/* disable receiver command  */
#define SNC_CR_TXP  	0x0002	/* transmit packet(s) command  */
#define SNC_CR_HTX  	0x0001	/* halt transmission command  */

/* fields of snc_dcr, the data configuration register  */
#define SNC_DCR_LBR 	0x2000	/* latched bus retry  */
#define SNC_DCR_POX 	0x1800	/* programmable outputs 1&0  */
#define SNC_DCR_STERM	0x0400	/* bus termination: 0=async, 1=sync  */
#define SNC_DCR_USRX	0x0300	/* user definable pins 1&0  */
#define SNC_DCR_WCX 	0x0060	/* number wait states  */
#define SNC_DCR_DW  	0x0020	/* data width: 0=16bit, 1=32bit  */
#define SNC_DCR_BMS 	0x0010	/* dma mode: 0=empty/fill, 1=block mode  */
#define SNC_DCR_RFTX	0x000C	/* no. longs put in rcvfifo before flush  */
#define SNC_DCR_TFTX	0x0003	/* no. longs kept in xmtfifo before fill  */

/* fields of snc_rcr (receive control register)  */
#define SNC_RCR_ERR 	0x8000	/* accept packet with errors  */
#define SNC_RCR_RNT 	0x4000	/* accept runt packets  */
#define SNC_RCR_BRD 	0x2000	/* accept broadcast packets  */
#define SNC_RCR_PRO 	0x1000	/* accept all packets (promiscuous)  */
#define SNC_RCR_AMC 	0x0800	/* accept all multicast packets  */
#define SNC_RCR_LBX 	0x0600	/* loopback mode  */
#define SNC_RCR_LB_NONE	0x0000	/*     no loopback  */
#define SNC_RCR_LB_MAC	0x0200	/*     MAC loopback  */
#define SNC_RCR_LB_ENDC	0x0400	/*     ENDEC loopback  */
#define SNC_RCR_LB_XCVR	0x0600	/*     transceiver loopback  */
#define SNC_RCR_MC  	0x0100	/* multicast packet received (rd only)  */
#define SNC_RCR_BC  	0x0080	/* broadcast packet received (rd only)  */
#define SNC_RCR_LPKT	0x0040	/* last packet in rba (rd only)  */
#define SNC_RCR_CRS 	0x0020	/* carrier sense activity (rd only)  */
#define SNC_RCR_COL 	0x0010	/* collision active (rd only)  */
#define SNC_RCR_CRCR	0x0008	/* crc error in packet (rd only)  */
#define SNC_RCR_FAER	0x0004	/* alignment error in packet (rd only)  */
#define SNC_RCR_LBK 	0x0002	/* loopback packet received (rd only)  */
#define SNC_RCR_PRX 	0x0001	/* aok packet received (rd only)  */

/* fields of snc_tcr, the transmit control register  */
#define SNC_TCR_PINT	0x8000	/* please give programmable interrupt  */
#define SNC_TCR_POWC	0x4000	/* out-of-window-clsn-timer starter:\ */
								      /* 0-frame delmtr, 1-frst preamble bit  */
#define SNC_TCR_CRCI	0x2000	/* please inhibit CRC  */
#define SNC_TCR_EXDIS	0x1000	/* please disable excessv deferal timer  */
#define SNC_TCR_EXD 	0x0400	/* excessv deferal indictn (rd only)  */
#define SNC_TCR_DEF 	0x0200	/* xmt deferal occurred (rd only)  */
#define SNC_TCR_NCRS	0x0100	/* crs never present durng xmt (rd only)  */
#define SNC_TCR_CRSL	0x0080	/* crs lost or absnt durng xmt (rd only)  */
#define SNC_TCR_EXC 	0x0040	/* excessv collisions occurred (rd only)  */
#define SNC_TCR_OWC 	0x0020	/* out-of-window collision (rd only)  */
#define SNC_TCR_PMB 	0x0008	/* xmtd pkt err: crc, bad src (rd only)  */
#define SNC_TCR_FU  	0x0004	/* fifo underrun occurred (rd only)  */
#define SNC_TCR_BMC 	0x0002	/* byte count mismatch (rd only)  */
#define SNC_TCR_PTX 	0x0001	/* packet xmtd ok (rd only)  */

/* fields of snc_imr and snc_isr, the interrupt mask and status registers  */
#define SNC_ISR_BR  	0x4000	/* bus retry interrupt  */
#define SNC_ISR_HBL 	0x2000	/* clsn dtct heartbeat lost  */
#define SNC_ISR_LCD 	0x1000	/* load cam command completed  */
#define SNC_ISR_PINT	0x0800	/* programmable interrupt occurred  */
#define SNC_ISR_PKTRX	0x0400	/* packet received  */
#define SNC_ISR_TXDN	0x0200	/* xmt done or xmt halt completed  */
#define SNC_ISR_TXER	0x0100	/* xmt error:BCM,EXC,FU,orEXD  */
#define SNC_ISR_TC  	0x0080	/* watchdog timer complete  */
#define SNC_ISR_RDE 	0x0040	/* rcv descriptors exhausted  */
#define SNC_ISR_RBE 	0x0020	/* rcv buffers exhausted  */
#define SNC_ISR_RBAE	0x0010	/* current rcv buffer area exceeded  */
#define SNC_ISR_CRC 	0x0008	/* crc error tally rollover  */
#define SNC_ISR_FAE 	0x0004	/* alignment error tally rollover  */
#define SNC_ISR_MP  	0x0002	/* missed pkt tally rollover  */
#define SNC_ISR_RFO 	0x0001	/* receive fifo overrun  */



/* 
 * receive resource structure
 *   Each RRA descriptor defines a block of contiguous memory
 *   which the sonic uses to receive one or more packets.
 *   A circular array of RRA descriptors is used to provide the
 *   sonic chip a series of receiver resources.
 */
typedef struct sonic_rra
	{
	shrt	sncp_blo;	/* lower 16 bits of rcv rsrc area address  */
	shrt	sncp0;
	shrt	sncp_bhi;	/* upper 16 bits of rcv rsrc area address  */
	shrt	sncp1;
	shrt	sncp_llo;	/* lower 16 bits of rcv rsrc area length  */
	shrt	sncp2;
	shrt	sncp_lhi;	/* upper 16 bits of rcv rsrc area length  */
	shrt	sncp3;
	} SNCP;



/* 
 * receive descriptor structure
 *   A pool of these is made available to the SONIC.  It uses one
 *   on every received packet.
*/
 
typedef struct sonic_rda
	{
	volatile shrt	sncr_sta; 	/* received packet status  */
	shrt	sncr0;
	shrt	sncr_len; 	/* received packet length  */
	shrt	sncr1;
	shrt	sncr_blo; 	/* buffer address, lo word  */
	shrt	sncr2;
	shrt	sncr_bhi; 	/* buffer address, hi word  */
	shrt	sncr3;
	shrt	sncr_seq; 	/* receive sequence number  */
	shrt	sncr4;
	shrt	sncr_nxt; 	/* next received RDA  */
	shrt	sncr5;
	volatile shrt	sncr_nus; 	/* in use flag (set by cpu, clr by sonic)  */
	shrt	sncr6;
	shrt	sncr_hom;	/* home port # */
	shrt	sncr_pli;	/* resource poll index */
	} SNCR;

/* bit fields of sncr_sta  */
#define SNCR_STA_ERR 	SNC_RCR_ERR 	
#define SNCR_STA_RNT 	SNC_RCR_RNT 	
#define SNCR_STA_BRD 	SNC_RCR_BRD 	
#define SNCR_STA_PRO 	SNC_RCR_PRO 	
#define SNCR_STA_AMC 	SNC_RCR_AMC 	
#define SNCR_STA_LBX 	SNC_RCR_LBX 	
#define SNCR_STA_MC  	SNC_RCR_MC  	
#define SNCR_STA_BC  	SNC_RCR_BC  	
#define SNCR_STA_LPKT	SNC_RCR_LPKT	
#define SNCR_STA_CRS 	SNC_RCR_CRS 	
#define SNCR_STA_COL 	SNC_RCR_COL 	
#define SNCR_STA_CRCR	SNC_RCR_CRCR	
#define SNCR_STA_FAER	SNC_RCR_FAER	
#define SNCR_STA_LBK 	SNC_RCR_LBK 	
#define SNCR_STA_PRX 	SNC_RCR_PRX 	

/* bit fields of sncr_nxt  */
#define SNCR_NXT_PTR 	0xFFFE	/* next rda pointer  */
#define SNCR_NXT_EOL 	0x0001	/* last RDA in list  */





/* 
 * transmit descriptor structure
 *   One of these is used to descrive every packet to be transmitted.
 *   The number of buffer pointers is variable, but only one is permitted
 *   here.
 *   All fields are 16 bit long, occurring every other 16 bits.
*/
 
typedef struct sonic_tda
	{
	volatile shrt	snct_sta; 	/* transmit complete status  */
	shrt	snct0;
	shrt	snct_cfg; 	/* transmit configure commands  */
	shrt	snct1;
	shrt	snct_len; 	/* transmit total length  */
	shrt	snct2;
	shrt	snct_frg; 	/* link to next tda  */
	shrt	snct3;
	shrt	snct_blo; 	/* buffer fragment address, lo word  */
	shrt	snct4;
	shrt	snct_bhi; 	/* buffer fragment address, hi word  */
	shrt	snct5;
	shrt	snct_bln; 	/* buffer fragment size  */
	shrt	snct6;
	shrt    snct_lnk[12];   /* for chain buffer purpose */
	void    *snct_pkt;      /* point to the packet structure */
	shrt	snct_nxt; 	/* link to next tda  */
	volatile shrt	snct7;
	} SNCT;

/* bit fields of snct_nxt  */
#define SNCT_NXT_PTR 	0xFFFE	/* next rda pointer  */
#define SNCT_NXT_EOL 	0x0001	/* last RDA in list  */

/* bit fields of snct_cfg  */
#define SNCT_CFG_PINT	SNC_TCR_PINT	
#define SNCT_CFG_POWC	SNC_TCR_POWC	
#define SNCT_CFG_CRCI	SNC_TCR_CRCI	
#define SNCT_CFG_EXDIS	SNC_TCR_EXDIS	

/* bit fields of snct_sta  */
#define SNCT_STA_NCX 	0xF000	/* number collisions  */
#define SNCT_STA_NCX_SHFT 12	/* number shifts to get collision cnt  */
#define SNCT_STA_EXD 	SNC_TCR_EXD 	
#define SNCT_STA_DEF 	SNC_TCR_DEF 	
#define SNCT_STA_NCRS	SNC_TCR_NCRS	
#define SNCT_STA_CRSL	SNC_TCR_CRSL	
#define SNCT_STA_EXC 	SNC_TCR_EXC 	
#define SNCT_STA_OWC 	SNC_TCR_OWC 	
#define SNCT_STA_PMB 	SNC_TCR_PMB 	
#define SNCT_STA_FU  	SNC_TCR_FU  	
#define SNCT_STA_BMC 	SNC_TCR_BMC 	
#define SNCT_STA_PTX 	SNC_TCR_PTX 	




/*
 *  CAM descriptor(s)
 */
#define SNC_MAX_CAM 16	/* maximum of 16 cam entries  */
typedef struct snc_cam_descriptor
	{
	shrt	sncc_pntr;	/* cam entry pointer  */
	shrt	sncc1;
	shrt	sncc_adr0;	/* cam address port 0  */
	shrt	sncc2;
	shrt	sncc_adr1;	/* cam address port 1  */
	shrt	sncc3;
	shrt	sncc_adr2;	/* cam address port 2  */
	shrt	sncc4;
	shrt	sncc_enabl;	/* mask for CAM enable register  */
	shrt	sncc5;
	} SNCC;


typedef struct snc_cam_descriptor_first
	{
	shrt	sncc_pntr;	/* cam entry pointer  */
	shrt	sncc1;
	shrt	sncc_adr0;	/* cam address port 0  */
	shrt	sncc2;
	shrt	sncc_adr1;	/* cam address port 1  */
	shrt	sncc3;
	shrt	sncc_adr2;	/* cam address port 2  */
	shrt	sncc4;
	} SNCC_FIRST;
#endif




