
/*
 *	Program Name:	SNMP Program
 *
 *	Filename:	decode.c
 *
 *	$Log:   /b/gregs/i960/tcpip/snmp/decode.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:43:20   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:36:38   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:16:56   gregs
 * Initial revision.
 * 
 *    Rev 1.0   16 Apr 1992 18:26:38   pvcs
 * Initial revision.
 *
 *	Comments:	Initial version for the 960 platform
 *
 *	Copyright (c) 1992 by Hughes LAN Systems
 */
/****************************************************************************
 *     Copyright (c) 1986, 1988  Epilogue Technology Corporation
 *     All rights reserved.
 *
 *     This is unpublished proprietary source code of Epilogue Technology
 *     Corporation.
 *
 *     The copyright notice above does not evidence any actual or intended
 *     publication of such source code.
 ****************************************************************************/

#include <libfuncs.h>
#include <asn1.h>
#include <localio.h>
#include <decode.h>
#include <buffer.h>
#include <snmp.h>
#include <syteksnm.h>



#define	WASTE_LIMIT	32

static void	A_DecodeOctetStringWorker(LCL_FILE *, ALENGTH_T,
ALENGTH_T,
ALENGTH_T (*)(), OCTET_P );

/****************************************************************************
NAME:  A_DecodeTypeValue

PURPOSE:  Decode the numeric part of an ASN.1 type from a stream.
	  The data stream is read using the local I/O package.
	  On exit, the stream pointer will be positioned to the byte
	  *AFTER* the type.

NOTE:	  The Class portion of the type is NOT decoded here, only the
	  value portion.
	  The user should call A_DecodeTypeClass *BEFORE* calling this
	  routine in order to get the class.

PARAMETERS:  LCL_FILE *	    A stream descriptor (already open)

RETURNS:  ATVALUE_T	    The type value

RESTRICTIONS:  It is assumed that the stream does not reach EOF before the
		end of the field.
****************************************************************************/
ATVALUE_T
A_DecodeTypeValue(lfile)
LCL_FILE *lfile;
{
	register OCTET_T oct;

	/*printf("A_DecodeTypeValue\n");*/
	oct = ((OCTET_T) Lcl_Getc(lfile)) & ~A_IDCF_MASK;

	if (oct != 0x1F)    /* Are there extension bytes? */
	{   	/* No extensions, type is in oct */
		return (ATVALUE_T) oct;
	}
	else 
	{   	/* Type is in extension octets */
		register ATVALUE_T t = 0;
		while ((oct = ((OCTET_T)Lcl_Getc(lfile))) & 0x80)
		{
			if (Lcl_Eof(lfile)) 
				break;
			t |= (ATVALUE_T)(oct & 0x7F);
			t <<= 7;
		}
		t |= (ATVALUE_T) oct;
		return t;
	}
	/* NOTREACHED */
}

/****************************************************************************
NAME:  A_DecodeLength

PURPOSE:  Decode an ASN.1 length from a stream.
	  The data stream is read using the local I/O package.
	  On exit, the stream pointer will be positioned to the byte
	  *AFTER* the length.

PARAMETERS:  LCL_FILE *	    Stream descriptor

RETURNS:  ALENGTH_T -- the length.
	  If the length is indefinite, (ALENGTH_T)-1 is returned.

RESTRICTIONS:  The stream must be open.
	       It is assumed that the stream will not reach EOF before the
	       length is decoded.
****************************************************************************/
ALENGTH_T
A_DecodeLength(lfile)
LCL_FILE *lfile;
{
	OCTET_T oct;

	/*printf("A_DecodeLength\n");*/
	oct = (OCTET_T)Lcl_Getc(lfile);

	/* Indefinite form? */
	if (oct == 0x80) 
		return (ALENGTH_T) -1;

	if (!(oct & 0x80))  /* Short or long format? */
		return (ALENGTH_T) oct;   /* Short format */
	else 
	{   /* Long format */
		register OCTET_T lsize;
		register ALENGTH_T len = 0;

		lsize = oct & 0x7F;	/* Get # of bytes comprising length field */
		while (lsize-- != 0)
		{
			len <<= 8;
			len |= (OCTET_T)Lcl_Getc(lfile);
			if (Lcl_Eof(lfile)) 
				break;
		}
		return len;
	}
	/* NOTREACHED */
}

#if defined(CSTRINGS)
/********************
A_DecodeOctetStringWorker

PURPOSE:  Pull an octet string from an ASN.1 stream.
	  The data stream is read using the local I/O package.
	  On entry stream pointer should be positioned to the first byte
	  of the data field.
	  On exit, the stream pointer will be positioned to at the start
	  of the next ASN.1 type field.

Parameters:
	LCL_FILE *	Stream descriptor
	ALENGTH_T	Length of octet string, from its ASN.1 header
	OCTET_T		FORM flags from the ASN.1 TYPE field
	ALENGTH_T (*f()) Function to be called to take generated data
	OCTET_T *	Parameter to be passed to the function.

Returns: nothing.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
	OCTET_T *	The parameter (funcparm) passed to this routine
	LCL_FILE *	The local I/O stream containing the data.
	ALENGTH_T	The number of octets in the stream.

It is very important that the function consume ALL of the indicated number
of octets from the stream.
********************/
static void
A_DecodeOctetStringWorker(stream, length, tflgs, func, funcparm)
LCL_FILE	*stream;
ALENGTH_T	length;
ALENGTH_T		tflgs;
ALENGTH_T	(*func)();
OCTET_P		funcparm;
{
	/*printf("A_DecodeOctetStringWorker\n");*/
	if (length == 0) 
		return;

	if (tflgs & A_CONSTRUCTOR)
	{
		/* Data is in constructor representation */
		int start_tell;
		ALENGTH_T inner_length;
		ATVALUE_T inner_type;
		ALENGTH_T inner_form;

		start_tell = Lcl_Tell(stream);

		for(;;)
		{
			inner_form = A_DecodeTypeClass(stream);
			inner_type = A_DecodeTypeValue(stream);
			inner_length = A_DecodeLength(stream);

			/* Is the length of this string determinate or indeterminate? */
			if (length == -1)
			{ /* Length was indeterminate, check for EOC */
				/* Does it look like an EOC? */
				if ((inner_form == 0x00) &&
					(inner_type == 0x00) &&
					(inner_length == 00))
					break;	/* EOC hit */
			}
			else 
			{ /* Length is determinate */
				/* Have we consumed enough data yet? */
				if ((Lcl_Tell(stream) - start_tell) >= length)
					break;
			}
			if (inner_length != 0)
			{
				A_DecodeOctetStringWorker(stream, inner_length,
					inner_form, func, (OCTET_P)funcparm);
			}
		}
	}
	else 
	{ /* Data is in primitive representation */
		(void) ((*func)((OCTET_P)funcparm, stream, length));
	}
}
#endif	/* CSTRINGS */

/********************
A_DecodeOctetStringData

PURPOSE:  Pull an octet string from an ASN.1 stream.
	  The data stream is read using the local I/O package.
	  On entry stream pointer should be positioned to the first byte
	  of the data field.
	  On exit, the stream pointer will be positioned to at the start
	  of the next ASN.1 type field.

Parameters:
	LCL_FILE *	Stream descriptor
	ALENGTH_T	Length of octet string, from its ASN.1 header
	OCTET_T		FORM flags from the ASN.1 TYPE field
	EBUFFER_T *	Control structure to receive the data.
	ALENGTH_T	Maximum size to receive into the buffer.

Returns: Nothing

Note:	On return, the "start_bp" component of the buffer structure
	points to a "malloc"-ed area in which the octet string is held.
	Note that the octet string is NOT null terminated, may contain
	internal nulls. A null pointer, (char *)0, is used if no area
	is malloc-ed.
********************/
void
A_DecodeOctetStringData(stream, length, tflgs, ebuffp, maxlen)
LCL_FILE	*stream;
ALENGTH_T	length;
ALENGTH_T	tflgs;
EBUFFER_P	ebuffp;
ALENGTH_T	maxlen;
{
	ALENGTH_T	buffl;
	OCTET_P		buffp;
	ALENGTH_T used;

	/*printf("A_DecodeOctetStringData\n");*/
	ebuffp->start_bp = stream->lbuf_next;   /* use the incoming data buffer */
	ebuffp->next_bp = ebuffp->start_bp;
	ebuffp->remaining = (ALENGTH_T)maxlen;
#if defined(CSTRINGS)
	A_DecodeOctetStringWorker(stream, length, tflgs,
	    A_DecodeHelper, (OCTET_P )ebuffp);
#else	/* CSTRINGS */
	A_DecodeHelper((EBUFFER_P )ebuffp, stream, length);
#endif	/* CSTRINGS */

}

/********************
A_DecodeOctetString

PURPOSE:  Pull an octet string from an ASN.1 stream.
	  The data stream is read using the local I/O package.
	  On entry stream pointer should be positioned to the first byte
	  of the octet string's type field.
	  On exit, the stream pointer will be positioned to at the start
	  of the next ASN.1 type field.

Parameters:
	LCL_FILE *	Stream descriptor
	EBUFFER_T *	Control structure to receive the data.
	ALENGTH_T	Maximum size to receive into the buffer.

Returns: Nothing

Note:	On return, the "start_bp" component of the buffer structure
	points to a "malloc"-ed area in which the octet string is held.
	Note that the octet string is NOT null terminated, may contain
	internal nulls. A null pointer, (char *)0, is used if no area
	is malloc-ed.
********************/
void
A_DecodeOctetString(stream, ebuffp, maxlen)
LCL_FILE	*stream;
EBUFFER_P	ebuffp;
ALENGTH_T	maxlen;
{
	ALENGTH_T	os_flags;
	ALENGTH_T	os_length;

	/*printf("A_DecodeOctetString\n");*/
	os_flags = A_DecodeTypeClass(stream);
	(void) A_DecodeTypeValue(stream);
	os_length = A_DecodeLength(stream);
	A_DecodeOctetStringData(stream, os_length, os_flags, (EBUFFER_P)ebuffp, 
	    maxlen);

}

/********************
A_DecodeIntegerData

PURPOSE:  Pull an integer from an ASN.1 stream.
	  The data stream is read using the local I/O package.
	  On entry stream pointer should be positioned to the first byte
	  of the data field.
	  On exit, the stream pointer will be positioned to at the start
	  of the next ASN.1 type field.

Parameters:
	LCL_FILE *	Stream descriptor
	ALENGTH_T	Length of contents field, from the ASN.1 header

Returns: INT_32_T	(See note below)

NOTE: If the received value is really unsigned, then the caller should
merely cast the value returned by this procedure to an UINT_32_T.

WARNING: If the integer occupies more than 4 octets, then high order precision
will be lost, including the sign bit.  For unsigned values in which the
basic value occupies all 4 octets, the sign octet, containing a zero sign
bit, will be lost but will not damage the returned value.
********************/
INT_32_T
A_DecodeIntegerData(stream, length)
LCL_FILE	*stream;
ALENGTH_T	length;
{
	INT_32_T ivalue = 0;
	INT_32_T cvalue = 0;
	int i;
/*printf("A_DecodeIntegerData\n");*/
	for(i=0;i<length;i++)
	{
		ivalue <<= 8;
		cvalue <<= 8;
		ivalue |= (OCTET_T) Lcl_Getc(stream);
		/*printf("ivalue is %X i is %x\n",ivalue,i);*/
		if(i == 0)
		{
			cvalue |= ivalue & 0x80;
			ivalue &= 0x7f;
		}
		if (Lcl_Eof(stream)) 
			break;
	}
	ivalue -= cvalue;

/*printf("ivalue is %X length is %X\n",ivalue,length);*/
	return ivalue;
}

/********************
A_DecodeInteger

PURPOSE:  Pull an integer from an ASN.1 stream.
	  The data stream is read using the local I/O package.
	  On entry stream pointer should be positioned to the first byte
	  of the integer's ASN.1 type field.
	  On exit, the stream pointer will be positioned to at the start
	  of the next ASN.1 type field.

Parameters:
	LCL_FILE *	Stream descriptor

Returns: INT_32_T;
********************/
INT_32_T
A_DecodeInteger(stream)
LCL_FILE	*stream;
{
/*printf("A_DecodeInteger\n");*/
	(void) A_DecodeTypeValue(stream);
	return A_DecodeIntegerData(stream, A_DecodeLength(stream));
}

/********************
A_DecodeObjectIdData

PURPOSE:  Pull an object identifier from an ASN.1 stream.
	  The data stream is read using the local I/O package.
	  On entry stream pointer should be positioned to the first byte
	  of the data field.
	  On exit, the stream pointer will be positioned to at the start
	  of the next ASN.1 type field.

Parameters:
	LCL_FILE *	Stream descriptor
	ALENGTH_T	Length of contents field, from the ASN.1 header
	OBJ_ID_T *	Object identifier structure to receive the object id.
			The "component_list" will be "malloc"ed.
			component_list == (unsigned int *)0 indicates that
			there is no component_list.

Returns: -1 if can't allocat spaces for component list 
          0 if success 
********************/
int
A_DecodeObjectIdData(stream, length, objp)
LCL_FILE	*stream;
ALENGTH_T	length;
OBJ_ID_T  *objp;
{
	int content_offset;	/* Offset in stream where the contents begins */
	int left;		/* Count of unused contents bytes */
	int subids;		/* Number of subidentifiers	*/
	int subid_num;
	unsigned int	subid_val;	/* Value of a subidentifier */
	unsigned int  *cp;
	unsigned char	c;

/*printf("A_DecodeObjectIdData\n");*/
	objp->num_components = 0;
	objp->component_list = (unsigned int  *)0;

	/* Remember where the contents begins */
	content_offset = Lcl_Tell(stream);

	/* Count the number of components */
	for(subids = 0, left = (int)length; left > 0; left--)
	{
		c = Lcl_Getc(stream);
		if (Lcl_Eof(stream)) break;

		/* Skip all bytes but ones which are the last in a subidentifier.	*/
		/* In other words skip all which have the high order bit set.	*/
		if (c & 0x80) 
			continue;
		subids++;
	}

	Lcl_Seek(stream, content_offset, 0);

	/* Null object id if no subidentifier fields */
	if (subids == 0) return 0;

	/* Try to get space for the components list */

	if ((cp = (unsigned int  *)alloc_mem(MAXOBJLEN*sizeof(int))) == (unsigned int  *)0)
		return -1;   /* no more allocated  memory is available */

	objp->num_components = subids + 1;
	if(objp->num_components > MAXOBJLEN)
		return -1;
	objp->component_list = (unsigned int  *)cp;

	/* Decode the subids and components */
	for(subid_num = 0; subid_num < subids; subid_num++)
	{
		/* Decode the subidentifier */
		for(subid_val = 0;;)
		{
			c = Lcl_Getc(stream);
			if (Lcl_Eof(stream)) break;
			subid_val <<= 7;
			subid_val |= (unsigned int)(c & 0x7F);
			if (!(c & 0x80)) break;
		}

		/* Is this the first subidentifier?			*/
		/* i.e. the one that contains TWO components?	*/
		if (subid_num == 0)
			{
				if (subid_val < 40)
					{
						*cp++ = 0;
						*cp++ = subid_val;
					}
				else {
					if (subid_val < 80)
						{
							*cp++ = 1;
							*cp++ = subid_val - 40;
						}
					else {
						*cp++ = 2;
						*cp++ = subid_val - 80;
					}
				}
			}
		else { /* subid_num != 0, i.e. this is not the first subidentifier */
			*cp++ = subid_val;
		}
	}
	return 0;
}

/********************
A_DecodeObjectId

PURPOSE:  Pull an object identifer from an ASN.1 stream.
	  The data stream is read using the local I/O package.
	  On entry stream pointer should be positioned to the first byte
	  of the object identifier's ASN.1 type field.
	  On exit, the stream pointer will be positioned to at the start
	  of the next ASN.1 type field.

Parameters:
	LCL_FILE *	Stream descriptor
	OBJ_ID_T *	Object identifier structure to receive the object id.
			The "component_list" will be "malloc"ed.
			component_list == (unsigned int *)0 indicates that
			there is no component_list.
		
Returns: if success will return The length of the object identifier 
                      (excluding tag/length fields)
		 if error return -1 
********************/
int
A_DecodeObjectId(stream, objp)
LCL_FILE	*stream;
OBJ_ID_T  *objp;
{
	ALENGTH_T leng;
/*printf("A_DecodeObjectId\n");*/
	(void) A_DecodeTypeValue(stream);
	leng = A_DecodeLength(stream);
	if( (A_DecodeObjectIdData(stream, leng,(OBJ_ID_T  *)objp)) == -1)
		return -1;   /* allocation problem */
	else
		return leng;
}

/****************************************************************************

NAME:  A_DecodeHelper

PURPOSE:  Collect string data from the ASN.1 decoding routines and
	  place it into a buffer.

PAREMETERS:
        EBUFFER_T * The "opaque" parameter (funcparm) passed to the decoding
		    routines.  This parameter is typically passed as an
		    OCTET_T *.
	LCL_FILE *  The local I/O stream containing the data.
        ALENGTH_T   The number of octets to be taken from the local stream

RETURNS:  Nothing.

RESTRICTIONS:  
	Can not handle length > 64K
****************************************************************************/
void
A_DecodeHelper(ebp, lfile, leng)
EBUFFER_P ebp;
LCL_FILE  *lfile;
ALENGTH_T leng;
{
	ALENGTH_T actual;
/*printf("A_DecodeHelper\n");*/
	actual = min(leng, ebp->remaining);
	if (actual != 0)
		{
			(void) Lcl_Read(lfile, (POINTER)ebp->next_bp, (int)actual);
			ebp->remaining -= actual;
			ebp->next_bp += (unsigned short)actual;
		}
}
