
/*
 *	Program Name:	SNMP Program
 *
 *	Filename:	mibutils.c
 *
 *	$Log:   /b/gregs/i960/tcpip/snmp/mibutils.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:43:28   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:36:46   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:17:00   gregs
 * Initial revision.
 * 
 *    Rev 1.1   07 Oct 1992 15:42:50   suresh
 * The parameter passed to memset and memcpy to copy the object id was not
 * correct. Fixed it.
 * 
 *    Rev 1.0   16 Apr 1992 18:26:42   pvcs
 * Initial revision.
 *
 *	Comments:	Initial version for the 960 platform
 *
 *	Copyright (c) 1992 by Hughes LAN Systems
 */

/****************************************************************************
 *     Copyright (c) 1988  Epilogue Technology Corporation
 *     All rights reserved.
 *
 *     This is unpublished proprietary source code of Epilogue Technology
 *     Corporation.
 *
 *     The copyright notice above does not evidence any actual or intended
 *     publication of such source code.
 ****************************************************************************/

#include <types.h>
#include <libfuncs.h>
#include <asn1.h>
#include <localio.h>
#include <buffer.h>
#include <decode.h>
#include <snmp.h>
#include <mib.h>
#include <objectid.h>
#include <syteksnm.h>

static int find_next_helper(MIBNODE_P , unsigned int, int, unsigned int  *, VB_P, int);

/****************************************************************************
NAME:  find_object_node

PURPOSE:  Locate a node in the MIB tree corresponding to a given object id.
	  The search terminates sucessfully when all the object identifer
	  components have been consumed or a leaf node is encountered.
	  (One of the status flags in the MIBLOC_T structure indicates
	  whether the search terminated at an inner node or a leaf.)

	  The search terminates unsucessfully (return code -1) if an
	  object identifier component does not match those available at
	  a given inner node.

PARAMETERS:
	OBJ_ID_T *	Object ID to be used
	MIBLOC_T *	Structure containing lots of information about
			the found object.
			(Contents will be valid only if the node is located.)

RETURNS:  int		Zero if node located
			-1 if nothing located (MIBLOC_T is not valid)
****************************************************************************/
int
find_object_node(objp, mlp)
OBJ_ID_T  *objp;
MIBLOC_T	*mlp;
{
	register MIBNODE_P	np;
	register struct MIBARC_S *ap;
	register unsigned int  *compp;	  /* Current object id component of interest */
	register		comp_num; /* Index of current object id component   */

	for(comp_num = objp->num_components,
	    compp = objp->component_list,
	    np = &mib_root;
	    (np->node_type == INNER_NODE) && (comp_num > 0);
	    comp_num--, compp++)
	{
		for(ap = np->arcs;
		    (MIBNODE_P )(ap->nodep) != (MIBNODE_P )0;
		    ap++)
		{
			if (ap->id == *compp) 
				goto found_arc;
		}
		/* If we fall through the arcs, then we have failed in the search */
		return -1;

found_arc:
		/* Save the value of the matched arc so that later, when we call	*/
		/* the procs for the various objects we can tell them the last arc	*/
		/* matched.  This will allow the routines which access tabular	*/
		/* objects to know which column (attribute) of the table is being	*/
		/* used.								*/

		mlp->ml_last_match = *compp;
		np = (MIBNODE_P )(ap->nodep);
	}

	/* Here we have either run out of object identifiers or we have hit	   */
	/* a leaf node (in which case there may be zero or more identifers left.)  */
	if (np->node_type != INNER_NODE)
	{
		mlp->ml_flags = ML_IS_LEAF;
	}
	else 
	{
		mlp->ml_flags = 0;
	}

	mlp->ml_remaining_objid.num_components = comp_num;
	mlp->ml_remaining_objid.component_list = compp;
	mlp->ml_node = np;

	return 0;
}

/****************************************************************************
NAME:  find_next_object

PURPOSE:  Given an object identifier, find the "next" SNMP object using
	  SNMP's notion of lexicographic ordering of object identifiers.
	  The "next" object may be a leaf node of the MIB tree or an object
	  within a tabular structure.  In the latter case, located object
	  id will pass through the leaf node representing the table.

	  The returned object id, unless null, will be a valid id for
	  the get/set/test routines named in the leaf node for the object.

PARAMETERS:
	OBJ_ID_T *	Object ID to be used
	OBJ_ID_T *	Object ID structure to be loaded with the result.

RETURNS:  int		 0 for success,
			>0 if nothing found,
****************************************************************************/
int
find_next_object(new_vbp,priv)
VB_P new_vbp;
int   priv;
{
	register OBJ_ID_T  *resultp;
	register unsigned int need;
	register int rtn_code;
	int tempobj[MAXOBJLEN];
	OCTET_T		type;  /* Class form & type of data */



	resultp = (OBJ_ID_T  *)&new_vbp->vb_obj_id;

	need =  resultp->num_components;
	memset((char  *)tempobj,0,MAXOBJLEN*sizeof(int));
	memcpy((char  *)tempobj,(char  *)resultp->component_list,need*sizeof(int));
	type=new_vbp->vb_data_flags_n_type;
	if ( (rtn_code = find_next_helper((MIBNODE_P)&mib_root, 0,
	    need,
	    (unsigned int  *)resultp->component_list,
	    (VB_P)new_vbp,priv)) != 0 )
	{
		resultp->num_components = need;
		memset((char  *)resultp->component_list,0,need*sizeof(int));
		memcpy((char  *)resultp->component_list,(char  *)tempobj,
		    need*2);
		new_vbp->vb_data_flags_n_type = type;

		return rtn_code;
	}
	return 0;
}


/**********************************************************************
**
**    find_next_helper routine
**
***********************************************************************/
static
int
find_next_helper(np, lastmatch, tcount, tlist, vbp, priv)
MIBNODE_P	np;
unsigned int	lastmatch;
int tcount;
unsigned int  *tlist;
VB_P  vbp;
int   priv;
{

	int rtn_code;
	if (np->node_type & INNER_NODE)
	{
		struct MIBARC_S	 *ap;

		if (tcount <= 0)
		{
			/* If we have no target object id at this level, then	*/
			/* we need to scan the available arcs, from the lowest	*/
			/* to the highest, to see if any of them can provide	*/
			/* a usable "next".					*/

			for(ap = np->arcs; (MIBNODE_P )(ap->nodep) != (MIBNODE_P )0; ap++)
			{
				if ( (rtn_code = find_next_helper((MIBNODE_P )(ap->nodep),
					ap->id,
					tcount,(unsigned int  *)(tlist+1),
					(VB_P)vbp,priv)) == 0)
				{
					*tlist = ap->id;
					vbp->vb_obj_id.num_components += 1;
					return 0;
				}
				else if ( rtn_code != NO_SUCH_NAME )
				{
					/*printf("return code %d\n",rtn_code);*/
					return rtn_code;    /* error instance, stop search */
				}
			}
		}
		else 
		{ /* tcount > 0 */
			/* If we have a target object id at this level, then	*/
			/* we need to scan the available arcs.  We ignore any	*/
			/* which come "before" the target.  If one matches the	*/
			/* target, we need to check wither it has a "next" at	*/
			/* a lower level.  For those "after" the target, we need*/
			/* to check whether any of those can provide a usable	*/
			/* "next".				*/
			for(ap = np->arcs;
				    (MIBNODE_P )(ap->nodep) != (MIBNODE_P )0;
				    ap++)
			{
				if (ap->id < *tlist) 
					continue;

				if (ap->id == *tlist)
				{
					if ( (rtn_code = find_next_helper(
						(MIBNODE_P )ap->nodep,
						ap->id,
						tcount - 1,
						(unsigned int  *)(tlist + 1),
						(VB_P)vbp,priv)) == 0)
					{
						*tlist = ap->id;
						vbp->vb_obj_id.num_components += 1;
						return 0;
					}
					else if ( rtn_code != NO_SUCH_NAME )
						return rtn_code;    /* error instance, stop search */
					continue;
				}

				if (ap->id > *tlist)
				{
					if ( (rtn_code = find_next_helper(
						(MIBNODE_P )(ap->nodep),
						ap->id,
						0,
						(unsigned int  *)(tlist+1),
						(VB_P)vbp,priv)) == 0)
					{
						*tlist = ap->id;
						vbp->vb_obj_id.num_components += 1;
						return 0;
					}
					else if ( rtn_code != NO_SUCH_NAME )
						return rtn_code;    /* error instance, stop search */
					continue;
				}
			}
		}
		/*printf("Entry not found\n");*/
		return NO_SUCH_NAME;
	}
	else 
	{ /* node_type != INNER_NODE, i.e. it must be of type LEAF_NODE */
		rtn_code= call_get_next_routine((VB_P)vbp,(MIBLEAF_P)np,tcount,
		    	(unsigned int  *)tlist,priv);
		return rtn_code;
	}
	/* NOTREACHED */
}



/****************************************************************************
NAME:  call_get_next_routine

PURPOSE:  Once a leaf node is located, this routine will invoke the get
	  next routine referenced by the node.

	  The get next routine reference by a node MUST match the data type
	  of the node as expressed by variable "expected_tag" in the node.

PARAMETERS:

RETURNS:  int	0 -  if things went OK.
			    NO_SUCH_NAME - can't get next on this obj, continue search
				else - error happened, stop search
****************************************************************************/
int
call_get_next_routine(vbp,leaf,compc,compl,priv)
VB_P		vbp;
MIBLEAF_P	leaf;
int compc;
unsigned int  *compl;
int priv;
{
	int length=0;
	register int rtn_code;

	/** AWC 2/22/91 check privilege */
	if((snmp_access() != SNMP_ACC_COMMUNITY) && 
		!(CheckPrivilege(leaf->windx,priv)))
/****
	    (boot2->cmdpriv[leaf->windx] > priv))
*****/
		return NO_SUCH_NAME;
	vbp->vb_data_flags_n_type = leaf->expected_tag;

	vbp->vb_obj_id.num_components = compc;
	switch (leaf->expected_tag)
	{
	case VT_NUMBER:
		return (*(leaf->svcproc))(
		    SNMP_GETNXT,
		    (int  *)&vbp->vb_obj_id.num_components,
		    (unsigned int  *)compl,
		    (int  *)&length,
		    (INT_32_T  *)&(vbp->value_u.v_number),
		    priv);

	case VT_COUNTER:
	case VT_GAUGE:
	case VT_TIMETICKS:
		return (*(leaf->svcproc))(
		    SNMP_GETNXT,
		    (int  *)&vbp->vb_obj_id.num_components,
		    (unsigned int  *)compl,
		    (int  *)&length,
		    (UINT_32_T  *)&vbp->value_u.v_counter,
		    priv);

	case VT_STRING:
	case VT_OPAQUE:
		{
			OCTET_P	cp;
			if ( (cp = (OCTET_P)alloc_mem(1)) == (OCTET_P)0)
				return TOO_BIG;
			if ( ( rtn_code = (*(leaf->svcproc))(
			    SNMP_GETNXT,
			    (int  *)&vbp->vb_obj_id.num_components,
			    (unsigned int  *)compl,
			    (int  *)&length,
			    (OCTET_P)cp,
			    priv)) == 0 )
			{   /* get successfully */
				alloc_mem(length);
				EBufferPreLoad((EBUFFER_P)&(vbp->value_u.v_string),
				    (OCTET_P)cp, length,length+1);

				return rtn_code;
			}
			else
			{
				return rtn_code;
			}
		}
		/**
          vbp->value_u.v_string.next_bp=(OCTET_P )0;
          vbp->value_u.v_string.start_bp=(OCTET_P )0;
          vbp->value_u.v_string.remaining=(ALENGTH_T)0;
		  **/
		break;

	case VT_OBJECT:
		{
			unsigned int  *compp;
			unsigned int i;
			if ((compp = (unsigned int  *)alloc_mem(1))
			    == (unsigned int  *)0)
				return TOO_BIG;
			else
				vbp->value_u.v_object.component_list = compp;

			if ( (rtn_code = (*(leaf->svcproc))(
			    SNMP_GETNXT,
			    (int  *)&vbp->vb_obj_id.num_components,
			    (unsigned int  *)compl,
			    (int  *)&length,
			    (unsigned int  *)vbp->value_u.v_object.component_list,
			    priv)) == 0 )
			{
				vbp->value_u.v_object.num_components =
				    length / sizeof(unsigned int);
				alloc_mem(length);
				return rtn_code;
			}
			else
			{
				return rtn_code;
			}
		}
		/****
		vbp->value_u.v_object.num_components = 0;
		vbp->value_u.v_object.component_list = (unsigned int  *)0;
		***/
		break;

	case VT_EMPTY:
		break;

	case VT_IPADDRESS:
		return (*(leaf->svcproc))(
		    SNMP_GETNXT,
		    (int  *)&vbp->vb_obj_id.num_components,
		    (unsigned int  *)compl,
		    (int  *)&length,
		    (INT_32_T  *)vbp->value_u.v_network_address,
		    priv);
		break;

	default:
		return NO_SUCH_NAME;
	}
}

