
/*
 *	Program Name:	SNMP Program
 *
 *	Filename:	snmp_e.c
 *
 *	$Log:   /b/gregs/i960/tcpip/snmp/snmp_e.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:43:38   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:36:54   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:17:06   gregs
 * Initial revision.
 * 
 *    Rev 1.0   16 Apr 1992 18:26:48   pvcs
 * Initial revision.
 *
 *	Comments:	Initial version for the 960 platform
 *
 *	Copyright (c) 1992 by Hughes LAN Systems
 */

/****************************************************************************
 *     Copyright (c) 1988  Epilogue Technology Corporation
 *     All rights reserved.
 *
 *     This is unpublished proprietary source code of Epilogue Technology
 *     Corporation.
 *
 *     The copyright notice above does not evidence any actual or intended
 *     publication of such source code.
 ****************************************************************************/

#include <libfuncs.h>
#include <asn1.h>
#include <localio.h>
#include <buffer.h>
#include <encode.h>
#include <snmp.h>
#include <syteksnm.h>


static	unsigned int	bufsize_for_normal_pdu(SNMP_PKT_P );
static	unsigned int	bufsize_for_trap_pdu(SNMP_PKT_P );
static	int		encode_snmp_normal_pdu(SNMP_PKT_P , EBUFFER_P );
static	int		encode_snmp_trap_pdu(SNMP_PKT_P , EBUFFER_P );
static	int		encode_snmp_common(unsigned int,OCTET_P, EBUFFER_P ,
ALENGTH_T, INT_32_T,
EBUFFER_P ,
ATVALUE_T, ALENGTH_T);
static	void		encode_var_bind_list(VBL_P , EBUFFER_P );
static	ALENGTH_T	set_vbl_sizes(VBL_P );

/****************************************************************************
NAME:  SNMP_Bufsize_For_Packet

PURPOSE:  Compute how much buffer space is needed to hold a packet if
	  it were encoded.

PARAMETERS:
	SNMP_PKT_T *	SNMP Packet structure

RETURNS:  unsigned int	The buffer size required.
****************************************************************************/
unsigned int
SNMP_Bufsize_For_Packet(rp)
SNMP_PKT_P	rp;
{
	if (rp->pdu_type != TRAP_PDU)
		return bufsize_for_normal_pdu((SNMP_PKT_P)rp);
	else return bufsize_for_trap_pdu((SNMP_PKT_P)rp);
}

/****************************************************************************
NAME:  SNMP_Encode_Packet

PURPOSE:  Encode an SNMP packet.

PARAMETERS:
	SNMP_PKT_T *	SNMP Packet structure
	EBUFFER_T *	A buffer structure to receive the encoded packet

RETURNS:  0		Packet processed without error
	  -1		Error encountered during packet processing

	 On a sucessful return, the ebuffer passed as a parameter will
	 contain a malloc-ed space containing the encoded packet.

****************************************************************************/
int
SNMP_Encode_Packet(rp, ebuffp)
SNMP_PKT_P	rp;
EBUFFER_P	ebuffp;
{
	register int x;
	if (rp->pdu_type != TRAP_PDU)
	{
		/** AWC 11/30/90 */
		x = encode_snmp_normal_pdu((SNMP_PKT_P)rp,(EBUFFER_P)ebuffp);
	}
	else 
	{
		x = encode_snmp_trap_pdu((SNMP_PKT_P)rp,(EBUFFER_P)ebuffp);
	}
	/* before we send out the encoded packet, we need to put all
	** the allocated object id buffer back to link list */
	return x;
}

/****************************************************************************
	ENCODE_SNMP_COMMON
****************************************************************************/
static int
encode_snmp_common(need,buffp, ebuffp, overall_length, snmp_version, community,
pdu_type, pdu_length)
unsigned int need;
OCTET_P     buffp;
EBUFFER_P	ebuffp;
ALENGTH_T	overall_length;
INT_32_T	snmp_version;
EBUFFER_P	community;
ATVALUE_T	pdu_type;
ALENGTH_T	pdu_length;
{

	EBufferSetup((EBUFFER_P)ebuffp,(OCTET_P)buffp,need);

	/* Generate the Message sequence header */
	A_EncodeType(A_SEQUENCE, A_UNIVERSAL | A_CONSTRUCTOR,
	    A_EncodeHelper, (OCTET_P)ebuffp);
	A_EncodeLength(overall_length, A_EncodeHelper, (OCTET_P)ebuffp);

	A_EncodeInt(A_INTEGER, A_UNIVERSAL | A_PRIMITIVE, snmp_version,
	    A_EncodeHelper, (OCTET_P)ebuffp);
	A_EncodeOctetString(A_OCTETSTRING, A_UNIVERSAL | A_PRIMITIVE,
	    (OCTET_P)community->start_bp, EBufferUsed((EBUFFER_P)community),
	    A_EncodeHelper, (OCTET_P)ebuffp);

	/* Generate the PDU header */
	A_EncodeType(pdu_type, A_DEFAULT_SCOPE | A_CONSTRUCTOR,
	    A_EncodeHelper, (OCTET_P)ebuffp);
	A_EncodeLength(pdu_length, A_EncodeHelper, (OCTET_P)ebuffp);

	return 0;
}

/****************************************************************************
	BUFSIZE_FOR_NORMAL_PDU
****************************************************************************/
static
unsigned int
bufsize_for_normal_pdu(rp)
SNMP_PKT_P	rp;
{
	register ALENGTH_T	alength;

	/* In the following computations, the length of various tag and length	*/
	/* fields are given as constants.  This is possible because the tag	*/
	/* values are always low enough to fit into one octet.  And for various	*/
	/* data types, in particular integers, the length field will always	*/
	/* occupy one octet.							*/

	rp->pdu_length = 2	/* Tag and length of request_id (an integer) */
	+ A_SizeOfInt(rp->pdu.std_pdu.request_id)
	    + 2	 /* Tag and length of error_status (an integer) */
	+ A_SizeOfInt(rp->pdu.std_pdu.error_status)
	    + 2	 /* Tag and length of error_index (an integer) */
	+ A_SizeOfInt(rp->pdu.std_pdu.error_index);

	alength = set_vbl_sizes((VBL_P)&(rp->pdu.std_pdu.std_vbl));

	rp->pdu_length += 1	/* Size of tag on VarBindList sequence */
	+ A_SizeOfLength(alength)
	    + alength;

	alength = A_SizeOfOctetString(EBufferUsed((EBUFFER_P)&(rp->community)));
	/* Size of tag on the PDU sequences */
	rp->overall_length = 1+A_SizeOfPduType(rp->pdu_type) /* chng for new pdu type */
	    + A_SizeOfLength(rp->pdu_length)
	    + rp->pdu_length
	    + 2    /* Tag and length of snmp_version (an integer) */
	+ A_SizeOfInt(rp->snmp_version)
	    + 1    /* Tag for the community octetstring */
	+ A_SizeOfLength(alength)
	    + alength;

	rp->buffer_needed = rp->overall_length
	    + 1 /* Size of tag for overall Message sequence */
	+ A_SizeOfLength(rp->overall_length);

	return rp->buffer_needed;
}

/****************************************************************************
	ENCODE_SNMP_NORMAL_PDU
****************************************************************************/
static int
encode_snmp_normal_pdu(rp, ebuffp)
SNMP_PKT_P	rp;
EBUFFER_P	ebuffp;
{
	register OCTET_P buffp;
	register unsigned int	need;

	need = bufsize_for_normal_pdu((SNMP_PKT_P)rp);

	/* Obtain space for the packet */
	buffp = (OCTET_P)tx_rebuffp;

	if (encode_snmp_common(need,(OCTET_P)buffp,(EBUFFER_P)ebuffp,rp->overall_length,
	    rp->snmp_version,(EBUFFER_P)&(rp->community),rp->pdu_type,
	    rp->pdu_length) == -1)
		return -1;

	/* Encode request-id */
	A_EncodeInt(A_INTEGER, A_UNIVERSAL | A_PRIMITIVE, rp->pdu.std_pdu.request_id,
	    A_EncodeHelper, (OCTET_P)ebuffp);

	/* Encode error-status */
	A_EncodeInt(A_INTEGER, A_UNIVERSAL | A_PRIMITIVE,rp->pdu.std_pdu.error_status,
	    A_EncodeHelper, (OCTET_P)ebuffp);

	/* Encode error-index */
	A_EncodeInt(A_INTEGER, A_UNIVERSAL | A_PRIMITIVE, rp->pdu.std_pdu.error_index,
	    A_EncodeHelper, (OCTET_P)ebuffp);

	encode_var_bind_list((VBL_P)&(rp->pdu.std_pdu.std_vbl),(EBUFFER_P)ebuffp);

	return 0;
}

/****************************************************************************
	BUFSIZE_FOR_TRAP_PDU
****************************************************************************/
static
unsigned int
bufsize_for_trap_pdu(rp)
SNMP_PKT_P	rp;
{
	register ALENGTH_T	alength;
	/* In the following computations, the length of various tag and length	*/
	/* fields are given as constants.  This is possible because the tag	*/
	/* values are always low enough to fit into one octet.  And for various	*/
	/* data types, in particular integers, the length field will always	*/
	/* occupy one octet.							*/

	rp->pdu_length = 2	/* Tag and length of request_id (an integer) */
	+ A_SizeOfObjectId((OBJ_ID_T  *)&(rp->pdu.trap_pdu.enterprise_objid))
	    + 2	 /* Tag and length of net_address (a string) */
	+ 4	 /* Size of IP address in SMI */
	+ 2	 /* Tag and length of generic_trap (an integer) */
	+ A_SizeOfInt(rp->pdu.trap_pdu.generic_trap)
	    + 2	 /* Tag and length of specific_trap (an integer) */
	+ A_SizeOfInt(rp->pdu.trap_pdu.specific_trap)
	    + 2	 /* Tag and length of trap_time_ticks (an integer) */
	+ A_SizeOfInt(rp->pdu.trap_pdu.trap_time_ticks);

	alength = set_vbl_sizes((VBL_P)&(rp->pdu.trap_pdu.trap_vbl));
	rp->pdu_length += 1	/* Size of tag on VarBindList sequence */
	+ A_SizeOfLength(alength)
	    + alength;

	alength = A_SizeOfOctetString(EBufferUsed((EBUFFER_P)&(rp->community)));

	/* Size of tag on the PDU sequences */
	rp->overall_length =1+A_SizeOfPduType(rp->pdu_type) /* chng for new pdu type */
	    + A_SizeOfLength(rp->pdu_length)
	    + rp->pdu_length
	    + 2    /* Tag and length of snmp_version (an integer) */
	+ A_SizeOfInt(rp->snmp_version)
	    + 1    /* Tag for the community octetstring */
	+ A_SizeOfLength(alength)
	    + alength;

	rp->buffer_needed = rp->overall_length
	    + 1 /* Size of tag for overall Message sequence */
	+ A_SizeOfLength(rp->overall_length);

	return rp->buffer_needed;
}

/****************************************************************************
	ENCODE_SNMP_TRAP_PDU
****************************************************************************/
static int
encode_snmp_trap_pdu(rp, ebuffp)
SNMP_PKT_P	rp;
EBUFFER_P	ebuffp;
{
	register OCTET_P buffp;
	register unsigned int	need;
	need = bufsize_for_trap_pdu((SNMP_PKT_P)rp);

	/* Obtain space for the packet */
	buffp = (OCTET_P)trap_rebuffp;

	if (encode_snmp_common(need,(OCTET_P)buffp,(EBUFFER_P)ebuffp,rp->overall_length,
	    rp->snmp_version,(EBUFFER_P)&(rp->community),rp->pdu_type,
	    rp->pdu_length) == -1)
		return -1;

	/* Encode enterprise */
	A_EncodeObjectId(A_OBJECTID, A_UNIVERSAL | A_PRIMITIVE,
	    (OBJ_ID_T  *)&(rp->pdu.trap_pdu.enterprise_objid),
	    A_EncodeHelper, (OCTET_P )ebuffp);

	/* Encode agent-addr */
	A_EncodeOctetString(VT_IPADDRESS & ~A_IDCF_MASK,
	    VT_IPADDRESS & A_IDCF_MASK,
	    (POINTER)rp->pdu.trap_pdu.net_address, 4,
	    A_EncodeHelper, (OCTET_P  )ebuffp);

	/* Encode generic-trap */
	A_EncodeInt(A_INTEGER, A_UNIVERSAL | A_PRIMITIVE,
	    rp->pdu.trap_pdu.generic_trap,
	    A_EncodeHelper, (OCTET_P )ebuffp);

	/* Encode specific-trap */
	A_EncodeInt(A_INTEGER, A_UNIVERSAL | A_PRIMITIVE,
	    rp->pdu.trap_pdu.specific_trap,
	    A_EncodeHelper, (OCTET_P )ebuffp);

	/* Encode time-stamp */
	A_EncodeInt(VT_TIMETICKS & ~A_IDCF_MASK,
	    VT_TIMETICKS & A_IDCF_MASK,
	    rp->pdu.trap_pdu.trap_time_ticks,
	    A_EncodeHelper, (OCTET_P )ebuffp);

	encode_var_bind_list((VBL_P)&(rp->pdu.trap_pdu.trap_vbl),(EBUFFER_P)ebuffp);

	return 0;
}

/****************************************************************************
NAME:  encode_var_bind_list

PURPOSE:  Encode a VarBindList

PARAMETERS:
	VBL_T *		The VarBindList to be encoded

RETURNS:  Nothing
****************************************************************************/
static
void
encode_var_bind_list(vblp, ebuffp)
VBL_P		vblp;
EBUFFER_P	ebuffp;
{
	register VB_P	vbp;

	/* Generate the VarBindList sequence header */
	A_EncodeType(A_SEQUENCE, A_UNIVERSAL | A_CONSTRUCTOR,
	    A_EncodeHelper, (OCTET_P )ebuffp);
	A_EncodeLength(vblp->vbl_length, A_EncodeHelper, (OCTET_P )ebuffp);

	vbp = (VB_P)vblp->vblist;
	if (vblp->vbl_count != 0)
		{
			int i;
			for (i = 0; i < vblp->vbl_count; i++)
			{
				A_EncodeType(A_SEQUENCE, A_UNIVERSAL | A_CONSTRUCTOR,
				    A_EncodeHelper, (OCTET_P )ebuffp);
				A_EncodeLength(vbp->vb_seq_size,
				    A_EncodeHelper, (OCTET_P )ebuffp);

				A_EncodeObjectId(A_OBJECTID, A_UNIVERSAL | A_PRIMITIVE,
				    (OBJ_ID_T  *)&(vbp->vb_obj_id),
				    A_EncodeHelper, (OCTET_P )ebuffp);

				switch (vbp->vb_data_flags_n_type)
				{
				case VT_NUMBER:
					A_EncodeInt(VT_NUMBER & ~A_IDCF_MASK,
					    VT_NUMBER & A_IDCF_MASK,
					    vbp->value_u.v_number,
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_COUNTER:
					A_EncodeUnsignedInt(VT_COUNTER & ~A_IDCF_MASK,
					    VT_COUNTER & A_IDCF_MASK,
					    vbp->value_u.v_counter,
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_GAUGE:
					A_EncodeUnsignedInt(VT_GAUGE & ~A_IDCF_MASK,
					    VT_GAUGE & A_IDCF_MASK,
					    vbp->value_u.v_gauge,
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_TIMETICKS:
					A_EncodeUnsignedInt(VT_TIMETICKS & ~A_IDCF_MASK,
					    VT_TIMETICKS & A_IDCF_MASK,
					    vbp->value_u.v_timeticks,
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_STRING:
					A_EncodeOctetString(VT_STRING & ~A_IDCF_MASK,
					    VT_STRING & A_IDCF_MASK,
					    vbp->value_u.v_string.start_bp,
					    EBufferUsed((EBUFFER_P)&(vbp->value_u.v_string)),
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_OPAQUE:
					A_EncodeOctetString(VT_OPAQUE & ~A_IDCF_MASK,
					    VT_OPAQUE & A_IDCF_MASK,
					    (OCTET_P)vbp->value_u.v_string.start_bp,
					    EBufferUsed((EBUFFER_P)&(vbp->value_u.v_string)),
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_OBJECT:
					A_EncodeObjectId(A_OBJECTID, A_UNIVERSAL | A_PRIMITIVE,
					    (OBJ_ID_T  *)&(vbp->value_u.v_object),
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_EMPTY:
					A_EncodeType(VT_EMPTY & ~A_IDCF_MASK,
					    VT_EMPTY & A_IDCF_MASK,
					    A_EncodeHelper, (OCTET_P )ebuffp);
					A_EncodeLength(0, A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				case VT_IPADDRESS:
					A_EncodeOctetString(VT_IPADDRESS & ~A_IDCF_MASK,
					    VT_IPADDRESS & A_IDCF_MASK,
					    (POINTER)vbp->value_u.v_network_address,
					    4,
					    A_EncodeHelper, (OCTET_P )ebuffp);
					break;
				default:
					break;
				}
				vbp=vbp->next;
			}
		}
}

/****************************************************************************
NAME:  set_vbl_sizes

PURPOSE:  Scan a VarBindList, setting the internal lengths and computing
	  the total length.

PARAMETERS:
	VBL_T *		The VarBindList structure to be scanned and set

RETURNS:  ALENGTH_T	The number of octets the VarBindList contents would
			use if ASN.1 encoded.
****************************************************************************/
static
ALENGTH_T
set_vbl_sizes(vblp)
VBL_P	vblp;
{
	register VB_P		vbp;
	register ALENGTH_T	vblist_size;

	vblist_size = 0;
	vbp = vblp->vblist;
	if(vblp->vbl_count != 0)
		{
			int i;
			for (i = 0; i < vblp->vbl_count; i++)
			{
				ALENGTH_T vb_size;	/* Accumulator of size of VarBind sequence */
				ALENGTH_T obj_size;

				obj_size = A_SizeOfObjectId((OBJ_ID_T  *)&(vbp->vb_obj_id));
				vb_size = 1 /* The object ID tag is always 1 octet long */
				+ A_SizeOfLength(obj_size)
				    + obj_size;

				switch (vbp->vb_data_flags_n_type)
				{
				case VT_NUMBER:
					vbp->vb_data_length = A_SizeOfInt(vbp->value_u.v_number);
					break;
				case VT_COUNTER:
				case VT_GAUGE:
				case VT_TIMETICKS:
					vbp->vb_data_length = A_SizeOfUnsignedInt(
					    vbp->value_u.v_number);
					break;
				case VT_STRING:
				case VT_OPAQUE:
					{
						ALENGTH_T used;
						used = EBufferUsed((EBUFFER_P)&(vbp->value_u.v_string));
						vbp->vb_data_length = A_SizeOfOctetString(used);
					}
					break;
				case VT_OBJECT:
					vbp->vb_data_length =
					    A_SizeOfObjectId((OBJ_ID_T  *)&(vbp->value_u.v_object));
					break;
				case VT_EMPTY:
					vbp->vb_data_length = 0;
					break;
				case VT_IPADDRESS:
					vbp->vb_data_length = 4;
					break;
				default:
					break;
				}
				vbp->vb_seq_size = vb_size
				    + 1 /* The data tag is always 1 octet */
				+ A_SizeOfLength(vbp->vb_data_length)
				    + vbp->vb_data_length;

				vblist_size += vbp->vb_seq_size
				    + 1 /* The sequence tag is always 1 octet */
				+ A_SizeOfLength(vbp->vb_seq_size);
				/*** AWC 1/27/91/  */
				vbp = vbp->next;
			}
		}

	vblp->vbl_length = vblist_size;

	return vblist_size;
}
